/*************************************************************************
 *
 *  $RCSfile: inetpop3.cxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: th $ $Date: 2001/05/11 12:03:55 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): Matthias Huetsch <matthias.huetsch@sun.com>
 *
 *
 ************************************************************************/

#define _INETPOP3_CXX "$Revision: 1.3 $"

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif

#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif
#ifndef _RTL_MEMORY_H_
#include <rtl/memory.h>
#endif
#ifndef _RTL_STRBUF_HXX_
#include <rtl/strbuf.hxx>
#endif
#ifndef _RTL_USTRBUF_HXX_
#include <rtl/ustrbuf.hxx>
#endif

#ifndef _VOS_REF_HXX
#include <vos/ref.hxx>
#endif

#ifndef _SOLAR_H
#include <tools/solar.h>
#endif
#ifndef _LIST_HXX
#include <tools/list.hxx>
#endif

#ifndef _INET_MACROS_HXX
#include <inet/macros.hxx>
#endif
#ifndef _INET_SOCKET_HXX
#include <inet/socket.hxx>
#endif

#ifndef _INET_CONFIG_HXX
#include <inetcfg.hxx>
#endif
#ifndef _INETDNS_HXX
#include <inetdns.hxx>
#endif
#ifndef _INETPOP3_HXX
#include <inetpop3.hxx>
#endif

#ifndef _INETCOREMSG_HXX
#include <inetmsg.hxx>
#endif
#ifndef _INETCORESTRM_HXX
#include <inetstrm.hxx>
#endif

#include <stdlib.h>

#ifdef _USE_NAMESPACE
using namespace inet;
#endif

using rtl::OString;
using rtl::OStringBuffer;

using rtl::OUString;
using rtl::OUStringBuffer;

/*=======================================================================
 *
 * INetCorePOP3Connection Implementation.
 *
 * Reference:
 *   RFC 1939 - Post Office Protocol - Version 3 (STD 53).
 *   RFC 1920 - Internet Official Protocol Standards (STD 1).
 *
 *=====================================================================*/
typedef NAMESPACE_INET(INetSocket) socket_type;

enum INetCorePOP3StreamState
{
    INETCOREPOP3_EOL_BEGIN =  0,
    INETCOREPOP3_EOL_SCR   =  1,
    INETCOREPOP3_EOL_FCR   =  2,
    INETCOREPOP3_EOL_FLF   =  3
};

#define INETCOREPOP3_SOCKET_DISPOSE(socket) \
if ((socket).isValid()) \
{ \
    (socket)->deregisterEventHandler(onSocketEvent); \
    (socket)->close(); \
    (socket).unbind(); \
}

#define INETCOREPOP3_SOCKET_WOULDBLOCK (-osl_Socket_E_WouldBlock)

//=======================================================================

inline sal_Bool ascii_isDigit( sal_Unicode ch )
{
    return ((ch >= 0x0030) && (ch <= 0x0039));
}

inline sal_Bool ascii_isWhitespace( sal_Unicode ch )
{
    return ((ch <= 0x20) && ch);
}

/*========================================================================
 *
 * OutputStreamBuffer_Impl.
 *
 *======================================================================*/
namespace inet
{
namespace pop3
{

class OutputStreamBuffer_Impl
{
    sal_uInt32  m_nSize;
    sal_uInt32  m_nGrow;
    sal_Char   *m_pBuffer;
    sal_Char   *m_pOffset;

protected:
    void ensureCapacity (sal_uInt32 nBytes);

public:
    OutputStreamBuffer_Impl (
        sal_uInt32 nSize = 0,
        sal_uInt32 nGrow = 256);
    ~OutputStreamBuffer_Impl (void);

    const sal_Char* getBuffer (void) const
    {
        return m_pBuffer;
    }
    sal_uInt32 getLength (void) const
    {
        return (m_pOffset - m_pBuffer);
    }
    void rewind (void)
    {
        m_pOffset = m_pBuffer;
    }

    void operator<< (sal_Char c);
};

inline OutputStreamBuffer_Impl::OutputStreamBuffer_Impl (
    sal_uInt32 nSize, sal_uInt32 nGrow)
    : m_nSize (nSize), m_nGrow (nGrow), m_pBuffer (0), m_pOffset (0)
{
}

inline OutputStreamBuffer_Impl::~OutputStreamBuffer_Impl (void)
{
    rtl_freeMemory (m_pBuffer);
}

inline void OutputStreamBuffer_Impl::ensureCapacity (sal_uInt32 nBytes)
{
    sal_uInt32 nOffset = m_pOffset - m_pBuffer;
    if (m_nSize < (nOffset + nBytes))
    {
        m_nSize  += ((nBytes + m_nGrow) / m_nGrow) * m_nGrow;
        m_pBuffer = (sal_Char*)rtl_reallocateMemory (m_pBuffer, m_nSize);
        m_pOffset = m_pBuffer + nOffset;
    }
}

inline void OutputStreamBuffer_Impl::operator<< (sal_Char c)
{
    ensureCapacity(1);
    *m_pOffset++ = c;
}

} // pop3
} // inet

using namespace inet::pop3;

/*=======================================================================
 *
 * INetCorePOP3ReplyStream Interface.
 *
 *=====================================================================*/
class INetCorePOP3ReplyStream : public INetCoreOStream
{
    OutputStreamBuffer_Impl m_aBuffer;
    INetCorePOP3StreamState m_eState;
    sal_Bool                m_bTransparent;

    sal_Int32               m_nReplyCode;
    OString                 m_aReplyText;

    int         ParseStatus (void *pCtx);

protected:
    virtual int PutData (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCorePOP3ReplyStream (void);
    virtual ~INetCorePOP3ReplyStream (void);

    sal_Int32       GetReplyCode (void) const;
    const sal_Char* GetReplyText (void);
};

inline sal_Int32 INetCorePOP3ReplyStream::GetReplyCode (void) const
{
    return m_nReplyCode;
}

inline const sal_Char* INetCorePOP3ReplyStream::GetReplyText (void)
{
    if (m_aReplyText.getLength())
        return m_aReplyText.getStr();
    else
        return NULL;
}

/*========================================================================
 *
 * INetCorePOP3SimpleReplyStream Interface.
 *
 *======================================================================*/
class INetCorePOP3SimpleReplyStream : public INetCorePOP3ReplyStream
{
protected:
    virtual int PutData (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCorePOP3SimpleReplyStream (void);
    virtual ~INetCorePOP3SimpleReplyStream (void);
};

/*========================================================================
 *
 * INetCorePOP3ConnectReplyStream Interface.
 *
 *======================================================================*/
class INetCorePOP3ConnectReplyStream :
    public INetCorePOP3SimpleReplyStream
{
    virtual int PutData (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCorePOP3ConnectReplyStream (void);
    virtual ~INetCorePOP3ConnectReplyStream (void);
};

/*========================================================================
 *
 * INetCorePOP3LoginReplyStream Interface.
 *
 *======================================================================*/
class INetCorePOP3LoginReplyStream :
    public INetCorePOP3SimpleReplyStream
{
    virtual int PutData (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCorePOP3LoginReplyStream (void);
    virtual ~INetCorePOP3LoginReplyStream (void);
};

/*========================================================================
 *
 * INetCorePOP3EntryReplyStream Interface.
 *
 *======================================================================*/
class INetCorePOP3EntryReplyStream :
    public INetCorePOP3SimpleReplyStream
{
    INetCorePOP3MailListEntry &m_rEntry;

    virtual int PutData (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCorePOP3EntryReplyStream (INetCorePOP3MailListEntry &rEntry);
    virtual ~INetCorePOP3EntryReplyStream (void);
};

/*========================================================================
 *
 * INetCorePOP3OutputStream Interface.
 *
 *======================================================================*/
class INetCorePOP3OutputStream : public INetCoreOStream
{
    OutputStreamBuffer_Impl m_aBuffer;
    INetCorePOP3StreamState m_eState;

    virtual int PutData (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

protected:
    virtual int PutLine (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx) = 0;

public:
    INetCorePOP3OutputStream (void);
    virtual ~INetCorePOP3OutputStream (void);
};

/*========================================================================
 *
 * INetCorePOP3ListOutputStream Interface.
 *
 *======================================================================*/
class INetCorePOP3ListOutputStream : public INetCorePOP3OutputStream
{
    List &m_rOutList;

    virtual int PutLine (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCorePOP3ListOutputStream (List& rOutList);
    virtual ~INetCorePOP3ListOutputStream (void);
};

/*========================================================================
 *
 * INetCorePOP3UIDListOutputStream Interface.
 *
 *======================================================================*/
class INetCorePOP3UIDListOutputStream : public INetCorePOP3OutputStream
{
    List &m_rOutList;

    virtual int PutLine (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCorePOP3UIDListOutputStream (List& rOutList);
    virtual ~INetCorePOP3UIDListOutputStream (void);
};

/*========================================================================
 *
 * INetCorePOP3MailOutputStream Interface.
 *
 *======================================================================*/
class INetCorePOP3MailOutputStream : public INetCorePOP3OutputStream
{
    INetCoreMessageOStream &m_rMsgStrm;

    virtual int PutLine (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCorePOP3MailOutputStream (INetCoreMessageOStream& rMsgStrm);
    virtual ~INetCorePOP3MailOutputStream (void);
};

/*========================================================================
 *
 * INetCorePOP3CmdContext Interface.
 *
 *======================================================================*/
enum INetCorePOP3CmdState
{
    INETCOREPOP3_CMD_SUCCESS = -2,
    INETCOREPOP3_CMD_ERROR   = -1,
    INETCOREPOP3_CMD_CONNECT =  0,
    INETCOREPOP3_CMD_SEND    =  1,
    INETCOREPOP3_CMD_RECV    =  2
};

#ifdef _USE_NAMESPACE
namespace inet {
#endif

struct INetCorePOP3CmdContext
{
    INetCorePOP3CmdContext (
        const OString            &rCommandLine,
        INetCorePOP3ReplyStream  *pReplyStream,
        INetCorePOP3OutputStream *pOutStream,
        INetCorePOP3Callback     *pfnCallback,
        void                     *pData);
    ~INetCorePOP3CmdContext (void);

    INetCorePOP3CmdState      m_eState;
    int                       m_nCode;

    OString                   m_aCmdLine;

    INetCorePOP3ReplyStream  *m_pReplyStream;
    INetCorePOP3OutputStream *m_pOutStream;

    INetCorePOP3Callback     *m_pfnCB;
    void                     *m_pDataCB;
};

#ifdef _USE_NAMESPACE
}
#endif

/*========================================================================
 *
 * INetCorePOP3ConnectionContext Interface.
 *
 *======================================================================*/
#ifdef _USE_NAMESPACE
namespace inet {
#endif

struct INetCorePOP3ConnectionContext
{
    INetCorePOP3ConnectionContext (sal_uInt32 nBufSiz = 4096);
    ~INetCorePOP3ConnectionContext (void);

    INetCorePOP3CmdContext  *m_pCmdCtx;

    sal_Bool                 m_bIsOpen     : 1;
    sal_Bool                 m_bIsLoggedIn : 1;
    sal_Bool                 m_bAborting   : 1;

    INetCoreDNSResolver     *m_pResolver;
    INetCoreDNSHostEntry     m_aDestAddr;
    NAMESPACE_VOS(ORef)<INetActiveTCPSocket> m_xSocket;
    sal_uInt32               m_nRetrCount;

    sal_uInt32               m_nBufSiz;
    sal_Char                *m_pBuffer;
    sal_Char                *m_pRead;
    sal_Char                *m_pWrite;

    INetCorePOP3Callback    *m_pfnRetrCB;
    void                    *m_pRetrData;
    INetCorePOP3Callback    *m_pfnTermCB;
    void                    *m_pTermData;

    /** create.
     */
    void create (INetCoreDNSHostEntry &rDstAddr);
};

#ifdef _USE_NAMESPACE
}
#endif

/*=======================================================================
 *
 * INetCorePOP3Connection Implementation.
 *
 *=====================================================================*/
/*
 * INetCorePOP3Connection.
 */
INetCorePOP3Connection::INetCorePOP3Connection (void)
    : m_pConCtx (new INetCorePOP3ConnectionContext())
{
    VOS_POSTCOND(m_pConCtx, "INetCorePOP3Connection::ctor(): no context");
}

/*
 * ~INetCorePOP3Connection.
 */
INetCorePOP3Connection::~INetCorePOP3Connection (void)
{
    VOS_PRECOND(m_pConCtx, "INetCorePOP3Connection::dtor(): no context");
    if (m_pConCtx)
    {
        INETCOREPOP3_SOCKET_DISPOSE (m_pConCtx->m_xSocket);
        delete m_pConCtx;
    }
}

/*
 * handleResolverEvent.
 */
sal_Bool INetCorePOP3Connection::handleResolverEvent (
    sal_Int32 nStatus, INetCoreDNSHostEntry *pHostEntry)
{
    // Check connection context.
    if (m_pConCtx == NULL)
        return 0;

    // Check for idle state (No command context).
    INetCorePOP3CmdContext *pCtx = m_pConCtx->m_pCmdCtx;
    if (pCtx == NULL)
        return 1;

    // Check for abort during name resolution.
    if (m_pConCtx->m_bAborting)
        nStatus = INETCOREDNS_RESOLVER_ERROR;

    switch (nStatus)
    {
        case INETCOREDNS_RESOLVER_START:
            // Notify caller.
            if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                this, INETCOREPOP3_REPLY_RESOLVER_WAIT,
                NULL, pCtx->m_pDataCB);

            // Wait for next event.
            return 1;

        case INETCOREDNS_RESOLVER_SUCCESS:
        case INETCOREDNS_RESOLVER_EXPIRED:
            // Initialize active socket to destination.
            m_pConCtx->create (*pHostEntry);
            m_pConCtx->m_xSocket->registerEventHandler (onSocketEvent, this);

            // Initiate connect.
            if (m_pConCtx->m_xSocket->connect (
                NAMESPACE_VOS(OInetSocketAddr)(
                    pHostEntry->GetDottedDecimalName(),
                    pHostEntry->GetPort())))
            {
                // Notify caller.
                if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                    this, INETCOREPOP3_REPLY_CONNECT_WAIT,
                    NULL, pCtx->m_pDataCB);

                // Wait for connect event.
                return 1;
            }
            else
            {
                // Failure. Cleanup and notify caller.
                m_pConCtx->m_xSocket.unbind();

                m_pConCtx->m_pCmdCtx = NULL;
                if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                    this, INETCOREPOP3_REPLY_CONNECT_ERROR,
                    NULL, pCtx->m_pDataCB);
                delete pCtx;
            }
            break;

        default:
            // Failure. Cleanup and notify caller.
            m_pConCtx->m_pCmdCtx = NULL;

            if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                this, INETCOREPOP3_REPLY_RESOLVER_ERROR,
                NULL, pCtx->m_pDataCB);
            delete pCtx;
            break;
    }

    // Done.
    return 0;
}

/*
 * handleSocketEvent.
 * Implements POP3 as Finite State Machine.
 */
sal_Bool INetCorePOP3Connection::handleSocketEvent (
    const NAMESPACE_VOS(ORef)<INetSocket> &rxSocket, sal_Int32 nEvent)
{
    // Check connection context.
    if (m_pConCtx == NULL)
        return 0;

    // Check command context.
    INetCorePOP3CmdContext *pCtx = m_pConCtx->m_pCmdCtx;
    if (pCtx == NULL)
    {
        // No command context (STATE_IDLE).
        if (nEvent & socket_type::EVENT_READ)
        {
            // Jump into idle handler.
            while (1)
            {
                sal_Int32 nRead = rxSocket->recv (
                    m_pConCtx->m_pBuffer, m_pConCtx->m_nBufSiz);
                if (nRead > 0)
                {
                    // Absorbing whatever comes in.
                    continue;
                }
                else if (nRead == INETCOREPOP3_SOCKET_WOULDBLOCK)
                {
                    // Wait for next event.
                    return 1;
                }
                else
                {
                    // Connection closed or Network failure.
                    rxSocket->close();
                    return 1;
                }
            }
        }

        if (nEvent & socket_type::EVENT_CLOSE)
        {
            // Connection closed.
            m_pConCtx->m_bIsOpen = sal_False;
            m_pConCtx->m_xSocket.unbind();

            // Notify caller (terminate callback).
            if (m_pConCtx->m_pfnTermCB) (m_pConCtx->m_pfnTermCB) (
                this, INETCOREPOP3_REPLY_NETWORK_ERROR, NULL,
                m_pConCtx->m_pTermData);
        }

        // Leave.
        return 1;
    }

    // Check event.
    if (nEvent & socket_type::EVENT_CLOSE)
    {
        // Premature closure.
        m_pConCtx->m_bIsOpen = sal_False;
        m_pConCtx->m_xSocket.unbind();

        // Finish with network error.
        m_pConCtx->m_pCmdCtx->m_eState = INETCOREPOP3_CMD_ERROR;
        m_pConCtx->m_pCmdCtx->m_nCode  = INETCOREPOP3_REPLY_NETWORK_ERROR;
    }

    // Jump into state machine.
    while (1)
    {
        switch (pCtx->m_eState)
        {
            case INETCOREPOP3_CMD_CONNECT:
                if (nEvent & socket_type::EVENT_CONNECT)
                {
                    if (nEvent & socket_type::EVENT_OOB)
                    {
                        // Finish with connect error.
                        pCtx->m_eState = INETCOREPOP3_CMD_ERROR;
                        pCtx->m_nCode  = INETCOREPOP3_REPLY_CONNECT_ERROR;

                        // Cleanup.
                        m_pConCtx->m_xSocket.unbind();
                    }
                    else
                    {
                        // Read greeting reply.
                        pCtx->m_eState = INETCOREPOP3_CMD_RECV;
                        nEvent = socket_type::EVENT_READ;
                    }
                }
                else
                {
                    // Ignore event.
                    return 1;
                }
                break;

            case INETCOREPOP3_CMD_SEND:
                if (nEvent & socket_type::EVENT_WRITE)
                {
                    // Write out command buffer.
                    OString aCommand (pCtx->m_aCmdLine);

                    sal_Int32 nWrite = rxSocket->send (
                        aCommand.pData->buffer, aCommand.pData->length);
                    if (nWrite > 0)
                    {
                        // Read reply.
                        pCtx->m_eState = INETCOREPOP3_CMD_RECV;
                        nEvent = socket_type::EVENT_READ;
                    }
                    else if (nWrite == INETCOREPOP3_SOCKET_WOULDBLOCK)
                    {
                        // Wait for next event.
                        return 1;
                    }
                    else
                    {
                        // Socket write error.
                        pCtx->m_eState = INETCOREPOP3_CMD_ERROR;
                        pCtx->m_nCode  = INETCOREPOP3_REPLY_NETWORK_ERROR;
                        rxSocket->close();
                    }
                }
                else
                {
                    // Ignore event.
                    return 1;
                }
                break;

            case INETCOREPOP3_CMD_RECV:
                if (nEvent & socket_type::EVENT_READ)
                {
                    sal_Int32 nRead = rxSocket->recv (
                        m_pConCtx->m_pBuffer, m_pConCtx->m_nBufSiz);
                    if (nRead > 0)
                    {
                        // Ok, analyze reply.
                        m_pConCtx->m_nRetrCount += nRead;

                        int status = pCtx->m_pReplyStream->Write (
                            m_pConCtx->m_pBuffer, nRead, m_pConCtx);
                        if (status == INETCORESTREAM_STATUS_LOADED)
                        {
                            pCtx->m_nCode =
                                pCtx->m_pReplyStream->GetReplyCode();
                            pCtx->m_eState =
                                (pCtx->m_nCode == INETCOREPOP3_REPLY_OK) ?
                                INETCOREPOP3_CMD_SUCCESS :
                                INETCOREPOP3_CMD_ERROR;
                        }
                        else
                        {
                            if (status != INETCORESTREAM_STATUS_OK)
                            {
                                pCtx->m_eState = INETCOREPOP3_CMD_ERROR;
                            }
                        }
                    }
                    else if (nRead == INETCOREPOP3_SOCKET_WOULDBLOCK)
                    {
                        if (pCtx->m_pOutStream)
                        {
                            // Notify caller.
                            if (m_pConCtx->m_pfnRetrCB)
                                (m_pConCtx->m_pfnRetrCB) (
                                    this, INETCOREPOP3_REPLY_TRANSFER_WAIT,
                                    NULL, m_pConCtx->m_pRetrData);
                        }

                        // Wait for next event.
                        return 1;
                    }
                    else
                    {
                        // Socket read error.
                        pCtx->m_eState = INETCOREPOP3_CMD_ERROR;
                        pCtx->m_nCode  = INETCOREPOP3_REPLY_NETWORK_ERROR;
                        rxSocket->close();
                    }
                }
                else
                {
                    // Ignore event.
                    return 1;
                }
                break;

            default: // (_SUCCESS || _ERROR)
                if (pCtx)
                {
                    // Restore idle state.
                    m_pConCtx->m_pCmdCtx = NULL;

                    // Notify caller.
                    if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                        this, pCtx->m_nCode,
                        pCtx->m_pReplyStream->GetReplyText(),
                        pCtx->m_pDataCB);

                    // Cleanup and leave.
                    delete pCtx;
                    return 1;
                }
                break;
        } // switch (pCtx->m_eState)
    } // while (1)
}

/*
 * StartCommand.
 */
sal_Bool INetCorePOP3Connection::StartCommand (INetCorePOP3CmdContext *pCtx)
{
    // Ensure clean destruction.
    NAMESPACE_VOS(ORef)<INetClientConnection_Impl> xThis (this);

    // Check connection context.
    if (m_pConCtx && m_pConCtx->m_bIsOpen && !m_pConCtx->m_bAborting)
    {
        // Check command context.
        if (!m_pConCtx->m_pCmdCtx && pCtx)
        {
            // Setup connection context.
            m_pConCtx->m_pCmdCtx = pCtx;
            m_pConCtx->m_nRetrCount = 0;

            // Start command.
            if (m_pConCtx->m_xSocket->postEvent (socket_type::EVENT_WRITE))
            {
                // Command started.
                return sal_True;
            }
            m_pConCtx->m_pCmdCtx = NULL;
        }
    }

    // Command failed.
    delete pCtx;
    return sal_False;
}

/*
 * Open.
 * Initiate connection to POP3D.
 */
sal_Bool INetCorePOP3Connection::Open (
    const OUString &rHost, sal_uInt16 nPort,
    INetCorePOP3Callback *pfnCallback, void *pData)
{
    // Ensure clean destruction.
    NAMESPACE_VOS(ORef)<INetClientConnection_Impl> xThis (this);

    // Check connection context.
    if (!m_pConCtx || m_pConCtx->m_bIsOpen)
        return sal_False;

    // Check arguments.
    if ((rHost.getLength() == 0) || (pfnCallback == NULL))
        return sal_False;
    if (nPort == 0)
        nPort = INETCOREPOP3_DEF_PORT;

    // Initialize command context.
    INetCorePOP3CmdContext *pCtx = new INetCorePOP3CmdContext (
        OString(),
        new INetCorePOP3ConnectReplyStream(),
        NULL, pfnCallback, pData);

    pCtx->m_eState = INETCOREPOP3_CMD_CONNECT;

    // Set connection context.
    m_pConCtx->m_pCmdCtx = pCtx;

    // Start domain name resolution.
    m_pConCtx->m_aDestAddr = INetCoreDNSHostEntry (rHost, nPort);

    if (m_pConCtx->m_pResolver->GetHostByName (
        &(m_pConCtx->m_aDestAddr), onResolverEvent, this))
    {
        // Ok. Wait for things to come.
        return sal_True;
    }
    else
    {
        // Cleanup and fail.
        m_pConCtx->m_pCmdCtx = NULL;
        delete pCtx;
        return sal_False;
    }
}

/*
 * IsOpen.
 */
sal_Bool INetCorePOP3Connection::IsOpen (void)
{
    return (m_pConCtx ? m_pConCtx->m_bIsOpen : sal_False);
}

/*
 * Close.
 * Starts QUIT command.
 */
sal_Bool INetCorePOP3Connection::Close (
    INetCorePOP3Callback *pfnCallback, void *pData)
{
    // Initialize command context.
    INetCorePOP3CmdContext *pCtx = new INetCorePOP3CmdContext (
        "QUIT\015\012", new INetCorePOP3ConnectReplyStream, NULL,
        pfnCallback, pData);

    // Start QUIT command.
    return StartCommand (pCtx);
}

/*
 * Destroy.
 */
void INetCorePOP3Connection::Destroy (void)
{
    if (m_pConCtx)
    {
        m_pConCtx->m_bIsOpen     = sal_False;
        m_pConCtx->m_bIsLoggedIn = sal_False;
        m_pConCtx->m_bAborting   = sal_True;

        if (m_pConCtx->m_pCmdCtx)
        {
            m_pConCtx->m_pCmdCtx->m_pfnCB   = NULL;
            m_pConCtx->m_pCmdCtx->m_pDataCB = NULL;
        }

        m_pConCtx->m_pfnRetrCB = NULL;
        m_pConCtx->m_pRetrData = NULL;

        m_pConCtx->m_pfnTermCB = NULL;
        m_pConCtx->m_pTermData = NULL;

        INETCOREPOP3_SOCKET_DISPOSE (m_pConCtx->m_xSocket);
    }
}

/*
 * LoginUser.
 * Starts (optional) USER command.
 */
sal_Bool INetCorePOP3Connection::LoginUser (
    const OUString &rUsername,
    INetCorePOP3Callback *pfnCallback, void *pData)
{
    // Check arguments.
    if ((rUsername.getLength() == 0) || (pfnCallback == NULL))
        return sal_False;

    // Build command line.
    OStringBuffer aBuffer ("USER ");
    aBuffer.append (OString (
        rUsername.pData->buffer,
        rUsername.pData->length,
        RTL_TEXTENCODING_ASCII_US));
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCorePOP3CmdContext *pCtx = new INetCorePOP3CmdContext (
        aBuffer.makeStringAndClear(),
        new INetCorePOP3LoginReplyStream(),
        NULL, pfnCallback, pData);

    // Start USER command.
    return StartCommand (pCtx);
}

/*
 * LoginPassword.
 * Starts (optional) PASS command.
 */
sal_Bool INetCorePOP3Connection::LoginPassword (
    const OUString &rPassword,
    INetCorePOP3Callback *pfnCallback, void *pData)
{
    // Check arguments.
    if ((rPassword.getLength() == 0) || (pfnCallback == NULL))
        return sal_False;

    // Build command line.
    OStringBuffer aBuffer ("PASS ");
    aBuffer.append (OString (
        rPassword.pData->buffer,
        rPassword.pData->length,
        RTL_TEXTENCODING_ASCII_US));
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCorePOP3CmdContext *pCtx = new INetCorePOP3CmdContext (
        aBuffer.makeStringAndClear(),
        new INetCorePOP3LoginReplyStream(),
        NULL, pfnCallback, pData);

    // Start PASS command.
    return StartCommand (pCtx);
}

/*
 * LoginSecure.
 * Starts (optional) APOP command.
 */
sal_Bool INetCorePOP3Connection::LoginSecure (
    const OUString &rUsername, const OUString &rPassword,
    INetCorePOP3Callback *pfnCallback, void *pData)
{
    // (NYI).
    return sal_False;
}

/*
 * IsLoggedIn.
 */
sal_Bool INetCorePOP3Connection::IsLoggedIn (void)
{
    return (m_pConCtx ? m_pConCtx->m_bIsLoggedIn : sal_False);
}

/*
 * GetMailListEntry.
 * Starts LIST <index> command.
 */
sal_Bool INetCorePOP3Connection::GetMailListEntry (
    INetCorePOP3MailListEntry &rMailEntry,
    INetCorePOP3Callback *pfnCallback, void *pData)
{
    // Check arguments.
    if ((rMailEntry.m_nIndex == 0) || (pfnCallback == NULL))
        return sal_False;

    // Build command line.
    OStringBuffer aBuffer ("LIST ");
    aBuffer.append (sal_Int32 (rMailEntry.m_nIndex));
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCorePOP3CmdContext *pCtx = new INetCorePOP3CmdContext (
        aBuffer.makeStringAndClear(),
        new INetCorePOP3EntryReplyStream (rMailEntry), NULL,
        pfnCallback, pData);

    // Start LIST command.
    return StartCommand (pCtx);
}

/*
 * GetMailList.
 * Starts LIST command.
 */
sal_Bool INetCorePOP3Connection::GetMailList (
    List& rMailList,
    INetCorePOP3Callback *pfnCallback, void *pData)
{
    // Check arguments.
    if (pfnCallback == NULL)
        return sal_False;

    // Initialize command context.
    INetCorePOP3CmdContext *pCtx = new INetCorePOP3CmdContext (
        "LIST\015\012",
        new INetCorePOP3ReplyStream(),
        new INetCorePOP3ListOutputStream (rMailList),
        pfnCallback, pData);

    // Start LIST command.
    return StartCommand (pCtx);
}

/*
 * GetUniqueIdList.
 * Starts (optional) UIDL command.
 */
sal_Bool INetCorePOP3Connection::GetUniqueIdList (
    List& rUniqueIdList,
    INetCorePOP3Callback *pfnCallback, void *pData)
{
    // Check arguments.
    if (pfnCallback == NULL)
        return sal_False;

    // Initialize command context.
    INetCorePOP3CmdContext *pCtx = new INetCorePOP3CmdContext (
        "UIDL\015\012",
        new INetCorePOP3ReplyStream(),
        new INetCorePOP3UIDListOutputStream (rUniqueIdList),
        pfnCallback, pData);

    // Start UIDL command.
    return StartCommand (pCtx);
}

/*
 * GetMailHeader.
 * Starts TOP command.
 */
sal_Bool INetCorePOP3Connection::GetMailHeader (
    sal_uInt32 nIndex, INetCoreMessageOStream& rMessageStream,
    INetCorePOP3Callback *pfnCallback, void *pData)
{
    // Check arguments.
    if ((nIndex == 0) || (pfnCallback == NULL))
        return sal_False;

    // Build command line (using TOP 1, as TOP 0 does not always work).
    OStringBuffer aBuffer ("TOP ");
    aBuffer.append (sal_Int32 (nIndex));
    aBuffer.append (" 1\015\012");

    // Initialize command context.
    INetCorePOP3CmdContext *pCtx = new INetCorePOP3CmdContext (
        aBuffer.makeStringAndClear(),
        new INetCorePOP3ReplyStream(),
        new INetCorePOP3MailOutputStream (rMessageStream),
        pfnCallback, pData);

    // Start TOP command.
    return StartCommand (pCtx);
}

/*
 * DeleteMail.
 * Starts DELE command.
 */
sal_Bool INetCorePOP3Connection::DeleteMail (
    sal_uInt32 nIndex,
    INetCorePOP3Callback *pfnCallback, void *pData)
{
    // Check arguments.
    if ((nIndex == 0) || (pfnCallback == NULL))
        return sal_False;

    // Build command line.
    OStringBuffer aBuffer ("DELE ");
    aBuffer.append (sal_Int32 (nIndex));
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCorePOP3CmdContext *pCtx = new INetCorePOP3CmdContext (
        aBuffer.makeStringAndClear(),
        new INetCorePOP3SimpleReplyStream(), NULL,
        pfnCallback, pData);

    // Start DELE command.
    return StartCommand (pCtx);
}

/*
 * RetrieveMail.
 * Starts RETR command.
 */
sal_Bool INetCorePOP3Connection::RetrieveMail (
    sal_uInt32 nIndex, INetCoreMessageOStream& rMessageStream,
    INetCorePOP3Callback *pfnCallback, void *pData)
{
    // Check arguments.
    if ((nIndex == 0) || (pfnCallback == NULL))
        return sal_False;

    // Build command line.
    OStringBuffer aBuffer ("RETR ");
    aBuffer.append (sal_Int32 (nIndex));
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCorePOP3CmdContext *pCtx = new INetCorePOP3CmdContext (
        aBuffer.makeStringAndClear(),
        new INetCorePOP3ReplyStream(),
        new INetCorePOP3MailOutputStream (rMessageStream),
        pfnCallback, pData);

    // Start RETR command.
    return StartCommand (pCtx);
}

/*
 * GetRetrieveCount.
 */
sal_uInt32 INetCorePOP3Connection::GetRetrieveCount (void)
{
    return (m_pConCtx ? m_pConCtx->m_nRetrCount : 0);
}

/*
 * SetRetrieveCallback.
 */
sal_Bool INetCorePOP3Connection::SetRetrieveCallback (
    INetCorePOP3Callback *pfnCallback, void *pData)
{
    if (m_pConCtx || !m_pConCtx->m_bAborting)
    {
        m_pConCtx->m_pfnRetrCB = pfnCallback;
        m_pConCtx->m_pRetrData = pData;
        return sal_True;
    }
    return sal_False;
}

/*
 * SetTerminateCallback.
 */
sal_Bool INetCorePOP3Connection::SetTerminateCallback (
    INetCorePOP3Callback *pfnCallback, void *pData)
{
    if (m_pConCtx || !m_pConCtx->m_bAborting)
    {
        m_pConCtx->m_pfnTermCB = pfnCallback;
        m_pConCtx->m_pTermData = pData;
        return sal_True;
    }
    return sal_False;
}

/*========================================================================
 *
 * INetCorePOP3ConnectionContext Implementation.
 *
 *======================================================================*/
/*
 * INetCorePOP3ConnectionContext.
 */
INetCorePOP3ConnectionContext::INetCorePOP3ConnectionContext (
    sal_uInt32 nBufSiz)
    : m_pCmdCtx     (NULL),
      m_bIsOpen     (sal_False),
      m_bIsLoggedIn (sal_False),
      m_bAborting   (sal_False),
      m_pResolver   (new INetCoreDNSResolver()),
      m_aDestAddr   (OUString(), 0),
      m_xSocket     (NULL),
      m_nRetrCount  (0),
      m_nBufSiz     (nBufSiz),
      m_pBuffer     ((sal_Char*)(rtl_allocateMemory (m_nBufSiz))),
      m_pRead       (m_pBuffer),
      m_pWrite      (m_pBuffer),
      m_pfnRetrCB   (NULL),
      m_pRetrData   (NULL),
      m_pfnTermCB   (NULL),
      m_pTermData   (NULL)
{
}

/*
 * ~INetCorePOP3ConnectionContext.
 */
INetCorePOP3ConnectionContext::~INetCorePOP3ConnectionContext (void)
{
    rtl_freeMemory (m_pBuffer);
    if (m_xSocket.isValid())
    {
        m_xSocket->close();
        m_xSocket.unbind();
    }
    delete m_pResolver;
    delete m_pCmdCtx;
}

/*
 * create.
 */
void INetCorePOP3ConnectionContext::create (INetCoreDNSHostEntry &rDstAddr)
{
    // Initialize active socket.
    m_xSocket = new INetActiveTCPSocket();

    // Check for SocksGateway.
    NAMESPACE_VOS(ORef)<INetConfig> xConfig;
    if (INetConfig::getOrCreate (xConfig))
    {
        NAMESPACE_VOS(ORef)<INetProxyPolicy> xProxyPolicy (
            xConfig->getProxyPolicy());
        if (xProxyPolicy.isValid())
        {
            OUStringBuffer aBuffer (OUString::createFromAscii ("pop3://"));
            if (rDstAddr.GetCanonicalName())
                aBuffer.append (rDstAddr.GetCanonicalName());
            else
                aBuffer.append (rDstAddr.GetDomainName());
            aBuffer.append (sal_Unicode (':'));
            aBuffer.append (rDstAddr.GetPort());
            aBuffer.append (sal_Unicode ('/'));

            OUString        aUrl (aBuffer.makeStringAndClear());
            INetProxyConfig aProxyConfig;

            if (xProxyPolicy->shouldUseProxy (aUrl, aProxyConfig))
            {
                if (aProxyConfig.hasSocksProxy())
                {
                    // Use SocksGateway.
                    m_xSocket->setSocksGateway (
                        NAMESPACE_VOS(OInetSocketAddr)(
                            aProxyConfig.getSocksProxyName(),
                            aProxyConfig.getSocksProxyPort()));
                }
            }
        }
    }
}

/*=======================================================================
 *
 * INetCorePOP3CmdContext Implementation.
 *
 *=====================================================================*/
/*
 * INetCorePOP3CmdContext.
 */
INetCorePOP3CmdContext::INetCorePOP3CmdContext (
    const OString            &rCommandLine,
    INetCorePOP3ReplyStream  *pReplyStream,
    INetCorePOP3OutputStream *pOutStream,
    INetCorePOP3Callback     *pfnCallback,
    void                     *pData)
    : m_eState (INETCOREPOP3_CMD_SEND),
      m_nCode  (0),
      m_aCmdLine (rCommandLine),
      m_pReplyStream (pReplyStream),
      m_pOutStream   (pOutStream),
      m_pfnCB        (pfnCallback),
      m_pDataCB      (pData)
{
}

/*
 * ~INetCorePOP3CmdContext.
 */
INetCorePOP3CmdContext::~INetCorePOP3CmdContext (void)
{
    delete m_pOutStream;
    delete m_pReplyStream;
}

/*=======================================================================
 *
 * INetCorePOP3ReplyStream Implementation.
 *
 *=====================================================================*/
/*
 * INetCorePOP3ReplyStream.
 */
INetCorePOP3ReplyStream::INetCorePOP3ReplyStream (void)
    : m_eState       (INETCOREPOP3_EOL_SCR),
      m_bTransparent (sal_False),
      m_nReplyCode   (0)
{
}

/*
 * ~INetCorePOP3ReplyStream.
 */
INetCorePOP3ReplyStream::~INetCorePOP3ReplyStream (void)
{
}

/*
 * PutData.
 */
int INetCorePOP3ReplyStream::PutData (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    const sal_Char *pStop = pData + nSize;
    while (!m_bTransparent && (pData < pStop))
    {
        /*
         * Non-transparent. Extract status line.
         */
        if (m_eState == INETCOREPOP3_EOL_FCR)
        {
            if (*pData == '\012')
            {
                // Found <CR><LF>.
                m_aBuffer << sal_Char('\0');
                int status = ParseStatus (pCtx);
                if (status != INETCORESTREAM_STATUS_OK) return status;

                m_aBuffer.rewind();
                m_bTransparent = sal_True;
                m_eState = INETCOREPOP3_EOL_SCR;
            }
            else
            {
                // Found <CR> only.
                m_aBuffer << sal_Char('\0');
                int status = ParseStatus (pCtx);
                if (status != INETCORESTREAM_STATUS_OK) return status;

                m_aBuffer.rewind();
                m_bTransparent = sal_True;
                m_eState = INETCOREPOP3_EOL_SCR;
            }
        }
        else if (*pData == '\015')
        {
            // Found <CR>.
            m_eState = INETCOREPOP3_EOL_FCR;
        }
        else if (*pData == '\012')
        {
            // Found <LF> only.
            m_aBuffer << sal_Char('\0');
            int status = ParseStatus (pCtx);
            if (status != INETCORESTREAM_STATUS_OK) return status;

            m_aBuffer.rewind();
            m_bTransparent = sal_True;
            m_eState = INETCOREPOP3_EOL_SCR;
        }
        else
        {
            m_aBuffer << *pData;
        }
        pData++;
    }

    nSize = (sal_uInt32)(pStop - pData);
    if (nSize > 0)
    {
        INetCorePOP3ConnectionContext *ctx =
            (INetCorePOP3ConnectionContext *)pCtx;

        // Put multi-line reply down to output stream.
        if (ctx && ctx->m_pCmdCtx && ctx->m_pCmdCtx->m_pOutStream)
            return ctx->m_pCmdCtx->m_pOutStream->Write (pData, nSize, pCtx);
        else
            return INETCORESTREAM_STATUS_WOULDBLOCK;
    }

    return INETCORESTREAM_STATUS_OK;
}

/*
 * ParseStatus.
 */
int INetCorePOP3ReplyStream::ParseStatus (void *pCtx)
{
    int status = INETCORESTREAM_STATUS_OK;

    // Check status indicator.
    const sal_Char *pBuffer = m_aBuffer.getBuffer();
    if ((*pBuffer == '+') || (*pBuffer == '-'))
    {
        // Reply code.
        if (*pBuffer == '-')
        {
            // <ERR>.
            m_nReplyCode = INETCOREPOP3_REPLY_ERROR;
            status = INETCORESTREAM_STATUS_LOADED;
        }
        else
        {
            // <OK>.
            m_nReplyCode = INETCOREPOP3_REPLY_OK;
        }

        // Reply text.
        m_aReplyText = (const sal_Char*)(pBuffer + 1);
    }
    else
    {
        // Status line w/o status indicator -> something went wrong.
        status = INETCORESTREAM_STATUS_ERROR;
    }

    // Done.
    return status;
}

/*========================================================================
 *
 * INetCorePOP3SimpleReplyStream Implementation.
 *
 *======================================================================*/
/*
 * INetCorePOP3SimpleReplyStream.
 */
INetCorePOP3SimpleReplyStream::INetCorePOP3SimpleReplyStream (void)
    : INetCorePOP3ReplyStream()
{
}

/*
 * ~INetCorePOP3SimpleReplyStream.
 */
INetCorePOP3SimpleReplyStream::~INetCorePOP3SimpleReplyStream (void)
{
}

/*
 * PutData.
 */
int INetCorePOP3SimpleReplyStream::PutData (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    int status = INetCorePOP3ReplyStream::PutData (pData, nSize, pCtx);
    if ((status == INETCORESTREAM_STATUS_OK) && GetReplyText())
    {
        // Single line reply complete.
        status = INETCORESTREAM_STATUS_LOADED;
    }
    return status;
}

/*========================================================================
 *
 * INetCorePOP3ConnectReplyStream Implementation.
 *
 *======================================================================*/
/*
 * INetCorePOP3ConnectReplyStream.
 */
INetCorePOP3ConnectReplyStream::INetCorePOP3ConnectReplyStream (void)
    : INetCorePOP3SimpleReplyStream()
{
}

/*
 * ~INetCorePOP3ConnectReplyStream.
 */
INetCorePOP3ConnectReplyStream::~INetCorePOP3ConnectReplyStream (void)
{
}

/*
 * PutData.
 */
int INetCorePOP3ConnectReplyStream::PutData (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    int status = INetCorePOP3SimpleReplyStream::PutData (pData, nSize, pCtx);
    if (pCtx && (status == INETCORESTREAM_STATUS_LOADED))
    {
        INetCorePOP3ConnectionContext *ctx =
            (INetCorePOP3ConnectionContext *)pCtx;
        ctx->m_bIsOpen = (GetReplyCode() == INETCOREPOP3_REPLY_OK);
    }
    return status;
}

/*========================================================================
 *
 * INetCorePOP3LoginReplyStream Implementation.
 *
 *======================================================================*/
/*
 * INetCorePOP3LoginReplyStream.
 */
INetCorePOP3LoginReplyStream::INetCorePOP3LoginReplyStream (void)
    : INetCorePOP3SimpleReplyStream()
{
}

/*
 * ~INetCorePOP3LoginReplyStream.
 */
INetCorePOP3LoginReplyStream::~INetCorePOP3LoginReplyStream (void)
{
}

/*
 * PutData.
 */
int INetCorePOP3LoginReplyStream::PutData (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    int status = INetCorePOP3SimpleReplyStream::PutData (pData, nSize, pCtx);
    if (pCtx && (status == INETCORESTREAM_STATUS_LOADED))
    {
        INetCorePOP3ConnectionContext *ctx =
            (INetCorePOP3ConnectionContext *)pCtx;
        ctx->m_bIsLoggedIn = (GetReplyCode() == INETCOREPOP3_REPLY_OK);
    }
    return status;
}

/*========================================================================
 *
 * INetCorePOP3EntryReplyStream Implementation.
 *
 *======================================================================*/
/*
 * INetCorePOP3EntryReplyStream.
 */
INetCorePOP3EntryReplyStream::INetCorePOP3EntryReplyStream (
    INetCorePOP3MailListEntry& rEntry)
    : INetCorePOP3SimpleReplyStream(),
      m_rEntry (rEntry)
{
}

/*
 * ~INetCorePOP3EntryReplyStream.
 */
INetCorePOP3EntryReplyStream::~INetCorePOP3EntryReplyStream (void)
{
}

/*
 * PutData.
 */
int INetCorePOP3EntryReplyStream::PutData (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    int status = INetCorePOP3SimpleReplyStream::PutData (pData, nSize, pCtx);
    if (status != INETCORESTREAM_STATUS_LOADED)
        return status;

    if (GetReplyCode() == INETCOREPOP3_REPLY_OK)
    {
        // Parse ReplyText into Index and Size.
        const sal_Char *p1 = GetReplyText();
        sal_Char       *p2 = NULL;

        while (*p1 && !ascii_isDigit (*p1)) p1++;
        p2 = (sal_Char*)p1;
        while (*p2 && ascii_isDigit (*p2)) p2++;
        *p2 = '\0';
        m_rEntry.m_nIndex = atol (p1);

        p1 = ++p2;
        while (*p1 && !ascii_isDigit (*p1)) p1++;
        p2 = (sal_Char*)p1;
        while (*p2 && ascii_isDigit (*p2)) p2++;
        *p2 = '\0';
        m_rEntry.m_nSize = atol (p1);
    }
    return status;
}

/*========================================================================
 *
 * INetCorePOP3OutputStream Implementation.
 *
 *======================================================================*/
/*
 * INetCorePOP3OutputStream.
 */
INetCorePOP3OutputStream::INetCorePOP3OutputStream (void)
    : m_eState (INETCOREPOP3_EOL_SCR)
{
}

/*
 * ~INetCorePOP3OutputStream.
 */
INetCorePOP3OutputStream::~INetCorePOP3OutputStream (void)
{
}

/*
 * PutData.
 */
int INetCorePOP3OutputStream::PutData (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    INetCorePOP3ConnectionContext *ctx =
        (INetCorePOP3ConnectionContext *)pCtx;
    if (!ctx || ctx->m_bAborting)
        return INETCORESTREAM_STATUS_ERROR;

    const sal_Char *pStop = pData + nSize;
    while (pData < pStop)
    {
        if (m_eState == INETCOREPOP3_EOL_FCR)
        {
            if (*pData != '\012') m_aBuffer << sal_Char('\012');
            m_eState = INETCOREPOP3_EOL_BEGIN;
        }
        else if (*pData == '\015')
        {
            m_eState = INETCOREPOP3_EOL_FCR;
        }
        else
        {
            if (*pData == '\012')
            {
                m_aBuffer << sal_Char('\015');
                m_eState = INETCOREPOP3_EOL_BEGIN;
            }
        }

        m_aBuffer << *pData++;

        if (m_eState == INETCOREPOP3_EOL_BEGIN)
        {
            if (*m_aBuffer.getBuffer() == '.')
            {
                if (*(m_aBuffer.getBuffer() + 1) == '\015')
                {
                    return INETCORESTREAM_STATUS_LOADED;
                }
                else
                {
                    int status = PutLine (
                        (m_aBuffer.getBuffer() + 1),
                        (m_aBuffer.getLength() - 1), pCtx);
                    if (status != INETCORESTREAM_STATUS_OK)
                        return status;
                }
            }
            else
            {
                int status = PutLine (
                    m_aBuffer.getBuffer(), m_aBuffer.getLength(), pCtx);
                if (status != INETCORESTREAM_STATUS_OK)
                    return status;
            }
            m_aBuffer.rewind();
            m_eState = INETCOREPOP3_EOL_SCR;
        }
    }
    return INETCORESTREAM_STATUS_OK;
}

/*========================================================================
 *
 * INetCorePOP3ListOutputStream Implementation.
 *
 *======================================================================*/
/*
 * INetCorePOP3ListOutputStream.
 */
INetCorePOP3ListOutputStream::INetCorePOP3ListOutputStream (List& rOutList)
    : INetCorePOP3OutputStream(),
      m_rOutList (rOutList)
{
}

/*
 * ~INetCorePOP3ListOutputStream.
 */
INetCorePOP3ListOutputStream::~INetCorePOP3ListOutputStream (void)
{
}

/*
 * PutLine.
 */
int INetCorePOP3ListOutputStream::PutLine (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    // Parse line into Index and Size.
    sal_uInt32 nMailIndex, nMailSize;
    sal_Char *p1;

    while (*pData && !ascii_isDigit (*pData)) pData++;
    p1 = (sal_Char *)pData;
    while (*p1 && ascii_isDigit (*p1)) p1++;
    *p1 = '\0';
    nMailIndex = atol (pData);

    pData = ++p1;
    while (*pData && !ascii_isDigit (*pData)) pData++;
    p1 = (sal_Char *)pData;
    while (*p1 && ascii_isDigit (*p1)) p1++;
    *p1 = '\0';
    nMailSize = atol (pData);

    // Create ListEntry.
    INetCorePOP3MailListEntry *pEntry = new INetCorePOP3MailListEntry;
    pEntry->m_nIndex = nMailIndex;
    pEntry->m_nSize  = nMailSize;

    // Append to List.
    m_rOutList.Insert (pEntry, LIST_APPEND);

    // Done.
    return INETCORESTREAM_STATUS_OK;
}

/*========================================================================
 *
 * INetCorePOP3UIDListOutputStream Implementation.
 *
 *======================================================================*/
/*
 * INetCorePOP3UIDListOutputStream.
 */
INetCorePOP3UIDListOutputStream::INetCorePOP3UIDListOutputStream (
    List& rOutList)
    : INetCorePOP3OutputStream(),
      m_rOutList (rOutList)
{
}

/*
 * ~INetCorePOP3UIDListOutputStream.
 */
INetCorePOP3UIDListOutputStream::~INetCorePOP3UIDListOutputStream (void)
{
}

/*
 * PutLine.
 */
int INetCorePOP3UIDListOutputStream::PutLine (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    // Parse into Index and UniqueId.
    sal_Char *p1, *p2;
    p1 = p2 = (sal_Char*)pData;

    while (*p2 && !ascii_isDigit (*p2)) p2++;
    p1 = p2;
    while (*p2 && ascii_isDigit (*p2)) p2++;
    *p2 = '\0';
#ifdef DBG_UTIL
    sal_uInt32 nMailIndex = atol (pData);
#endif
    *p2 = ' ';

    p1 = p2;
    while (*p2 && ascii_isWhitespace (*p2)) p2++;
    p1 = p2;
    while (*p2 && !ascii_isWhitespace (*p2)) p2++;
    *p2 = '\0';

    // Append UniqueIdListEntry to List.
    OUString *entry = new OUString (p1, (p2 - p1), RTL_TEXTENCODING_ASCII_US);
    m_rOutList.Insert (entry, LIST_APPEND);

    // Done.
    return INETCORESTREAM_STATUS_OK;
}

/*========================================================================
 *
 * INetCorePOP3MailOutputStream Implementation.
 *
 *======================================================================*/
/*
 * INetCorePOP3MailOutputStream.
 */
INetCorePOP3MailOutputStream::INetCorePOP3MailOutputStream (
    INetCoreMessageOStream& rMsgStrm)
    : INetCorePOP3OutputStream(),
      m_rMsgStrm (rMsgStrm)
{
}

/*
 * INetCorePOP3MailOutputStream.
 */
INetCorePOP3MailOutputStream::~INetCorePOP3MailOutputStream (void)
{
}

/*
 * PutLine.
 */
int INetCorePOP3MailOutputStream::PutLine (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    INetCorePOP3ConnectionContext *ctx =
        (INetCorePOP3ConnectionContext *)pCtx;
    if (!ctx || ctx->m_bAborting)
        return INETCORESTREAM_STATUS_ERROR;

    return m_rMsgStrm.Write (pData, nSize, NULL);
}

