/*************************************************************************
 *
 *  $RCSfile: imapcimp.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: mhu $ $Date: 2001/03/19 12:16:34 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef TOOLS_INETMIME_HXX
#include <tools/inetmime.hxx>
#endif

#ifndef _INETCORESTRM_HXX
#include <inetstrm.hxx>
#endif
#ifndef _INETTCP_HXX
#include <inettcp.hxx>
#endif

#ifndef INET_IMAPCIMP_HXX
#include <imapcimp.hxx>
#endif
#ifndef INET_IMAPCMDS_HXX
#include <imapcmds.hxx>
#endif

//============================================================================
//
//  class INetIMAPCommandArgument
//
//============================================================================

INetIMAPCommandArgument::~INetIMAPCommandArgument()
{
	delete m_pStream;
}

//============================================================================
//
//  class INetIMAPClient_Impl
//
//============================================================================

// static
int INetIMAPClient_Impl::connectionTerminationCallback(INetCoreTCPConnection *
													       pTheConnection,
													   int nStatus,
													   void * pData)
{
	DBG_ASSERT(pData,
			   "INetIMAPClient_Impl::connectionTerminationCallback():"
			       " Null data");
	DBG_ASSERT(nStatus == INETCORETCP_STATUS_NETWORK_ERROR,
			   "INetIMAPClient_Impl::connectionTerminationCallback():"
			       " Invalid status");

	INetIMAPClient_Impl & rThis
		= *static_cast< INetIMAPClient_Impl * >(pData);
	if (rThis.m_eState == STATE_CLOSED)
	{
		DBG_ASSERT(rThis.m_xConnection.isEmpty(),
				   "INetIMAPClient_Impl::connectionTerminationCallback():"
			           " Invalid connection");
		return 0;
	}

	DBG_ASSERT(pTheConnection == rThis.m_xConnection.getBodyPtr(),
			   "INetIMAPClient_Impl::connectionTerminationCallback():"
			       " Invalid connection");
	DBG_ASSERT(rThis.m_eState > STATE_INITIAL
			   && rThis.m_eState < STATE_CLOSED,
			   "INetIMAPClient_Impl::connectionTerminationCallback():"
			       " Invalid state");

	CallbackType eCallbackType;
	{
		vos::OGuard aGuard = rThis.m_aMutex;
		eCallbackType = rThis.m_eState == STATE_IDLE ? CALLBACK_UNILATERAL :
			                                           CALLBACK_COMMAND;
		rThis.m_xCommandStream = 0;
		rThis.m_eState = STATE_CLOSED;
		rThis.m_xConnection = 0;
	}

	INetIMAPClosedResponse aResponse;
	rThis.callBack(aResponse, eCallbackType);
	return 0;
}

//============================================================================
// static
int INetIMAPClient_Impl::connectionOpenCallback(INetCoreTCPConnection *
												    pTheConnection,
												int nStatus, void * pData)
{
	DBG_ASSERT(pData,
			   "INetIMAPClient_Impl::connectionOpenCallback(): Null data");
	INetIMAPClient_Impl & rThis
		= *static_cast< INetIMAPClient_Impl * >(pData);

	DBG_ASSERT(pTheConnection == rThis.m_xConnection.getBodyPtr(),
			   "INetIMAPClient_Impl::connectionOpenCallback():"
			       " Invalid connection");
	DBG_ASSERT(rThis.m_eState == STATE_OPENING,
			   "INetIMAPClient_Impl::connectionOpenCallback():"
			       " Invalid state");

	switch (nStatus)
	{
		case INETCORETCP_STATUS_NETWORK_ERROR:
		{
			{
				vos::OGuard aGuard = rThis.m_aMutex;
				rThis.m_xCommandStream = 0;
				rThis.m_eState = STATE_CLOSED;
				rThis.m_xConnection = 0;
			}

			INetIMAPClosedResponse aResponse;
			rThis.callBack(aResponse, CALLBACK_COMMAND);
			break;
		}

		case INETCORETCP_STATUS_NAMERES_WAIT:
		{
			INetIMAPOpeningResponse aResponse
				= INetIMAPOpeningResponse::STATUS_NAME_RESOLUTION_PENDING;
			rThis.callBack(aResponse, CALLBACK_COMMAND);
			break;
		}

		case INETCORETCP_STATUS_NAMERES_DONE:
		{
			INetIMAPOpeningResponse aResponse
				= INetIMAPOpeningResponse::STATUS_NAME_RESOLUTION_DONE;
			rThis.callBack(aResponse, CALLBACK_COMMAND);
			break;
		}

		case INETCORETCP_STATUS_NAMERES_ERROR:
		{
			{
				vos::OGuard aGuard = rThis.m_aMutex;
				rThis.m_eState = STATE_INITIAL;
			}

			INetIMAPOpeningResponse aResponse
				= INetIMAPOpeningResponse::STATUS_NAME_RESOLUTION_FAILED;
			rThis.callBack(aResponse, CALLBACK_COMMAND);
			break;
		}

		case INETCORETCP_STATUS_CONNECT_WAIT:
		{
			INetIMAPOpeningResponse aResponse
				= INetIMAPOpeningResponse::STATUS_CONNECT_PENDING;
			rThis.callBack(aResponse, CALLBACK_COMMAND);
			break;
		}

		case INETCORETCP_STATUS_CONNECT_ERROR:
		{
			{
				vos::OGuard aGuard = rThis.m_aMutex;
				rThis.m_eState = STATE_INITIAL;
			}

			INetIMAPOpeningResponse aResponse
				= INetIMAPOpeningResponse::STATUS_CONNECT_FAILED;
			rThis.callBack(aResponse, CALLBACK_COMMAND);
			break;
		}

		case INETCORETCP_STATUS_CONNECT_DONE:
		{
			INetIMAPOpeningResponse aResponse
				= INetIMAPOpeningResponse::STATUS_CONNECTED;
			if (rThis.callBack(aResponse, CALLBACK_COMMAND))
			{
				rThis.m_eScanState = SCAN_STATE_INITIAL;

				if (!rThis.m_xConnection->Recv(*rThis.m_xScanner,
											   connectionReceiveCallback,
											   &rThis))
				{
					{
						vos::OGuard aGuard = rThis.m_aMutex;
						rThis.m_eState = STATE_CLOSED;
						rThis.m_xConnection->Abort();
						rThis.m_xConnection = 0;
					}

					INetIMAPOpeningResponse aResponse
						= INetIMAPOpeningResponse::STATUS_READ_FAILED;
					rThis.callBack(aResponse, CALLBACK_COMMAND);
				}
			}
			break;
		}

		default:
			DBG_ERROR("INetIMAPClient_Impl::connectionOpenCallback():"
					      " Invalid status");
			break;
	}
	return 0;
}

//============================================================================
// static
int INetIMAPClient_Impl::connectionReceiveCallback(INetCoreTCPConnection *
												       pTheConnection,
												   int nStatus, void * pData)
{
	DBG_ASSERT(pData,
			   "INetIMAPClient_Impl::connectionReceiveCallback(): Null data");
	INetIMAPClient_Impl & rThis
		= *static_cast< INetIMAPClient_Impl * >(pData);

	DBG_ASSERT(pTheConnection == rThis.m_xConnection.getBodyPtr(),
			   "INetIMAPClient_Impl::connectionReceiveCallback():"
			       " Invalid connection");
	DBG_ASSERT(
		rThis.m_eState > STATE_INITIAL && rThis.m_eState < STATE_CLOSED,
		"INetIMAPClient_Impl::connectionReceiveCallback(): Invalid state");

	switch (nStatus)
	{
		case INETCORETCP_STATUS_NETWORK_ERROR:
		case INETCORETCP_STATUS_RECV_WAIT:
			break;

		case INETCORETCP_STATUS_RECV_ERROR:
		{
			INetIMAPErrorResponse aResponse = ERRCODE_IO_CANTREAD;
			rThis.callBack(aResponse);
			break;
		}

		default:
			DBG_ERROR("INetIMAPClient_Impl::connectionReceiveCallback():"
					      " Invalid status");
			break;
	}
	return 0;
}

//============================================================================
// static
int INetIMAPClient_Impl::connectionSendCallback(INetCoreTCPConnection *
												    pTheConnection,
												int nStatus, void * pData)
{
	DBG_ASSERT(pData,
			   "INetIMAPClient_Impl::connectionSendCallback(): Null data");
	INetIMAPClient_Impl & rThis
		= *static_cast< INetIMAPClient_Impl * >(pData);

	DBG_ASSERT(
		pTheConnection == rThis.m_xConnection.getBodyPtr(),
		"INetIMAPClient_Impl::connectionSendCallback(): Invalid connection");
	DBG_ASSERT(
		rThis.m_eState == STATE_OPENING
		|| rThis.m_eState > STATE_IDLE && rThis.m_eState < STATE_CLOSED,
		"INetIMAPClient_Impl::connectionSendCallback(): Invalid state");

	switch (nStatus)
	{
		case INETCORETCP_STATUS_NETWORK_ERROR:
		case INETCORETCP_STATUS_SEND_WAIT:
		case INETCORETCP_STATUS_SEND_DONE:
			break;

		default:
			DBG_ERROR("INetIMAPClient_Impl::connectionSendCallback():"
					      " Invalid status");
			break;
	}
	return 0;
}

//============================================================================
bool INetIMAPClient_Impl::callBack(INetIMAPResponse & rResponse,
								   CallbackType eType)
{
	if (eType == CALLBACK_STATE)
	{
		DBG_ASSERT(m_eState > STATE_INITIAL && m_eState < STATE_CLOSED,
				   "INetIMAPClient_Impl::callBack(): Invalid state");

		vos::OGuard aGuard = m_aMutex;
		eType = m_eState == STATE_IDLE ? CALLBACK_UNILATERAL :
			                             CALLBACK_COMMAND;
	}

	Link const * pCallback;
	void * pData;
	if (eType == CALLBACK_UNILATERAL)
	{
		pCallback = &m_aUnilateralCallback;
		pData = m_pUnilateralResponseData;
	}
	else
	{
		pCallback = &m_aCommandCallback;
		pData = m_pCommandResponseData;
	}

	SvRefBaseRef xRef(this);

	rResponse.setCallbackData(*this, pData);
	pCallback->Call(&rResponse);

	vos::OGuard aGuard = m_aMutex;
	return m_eState != STATE_CLOSED;
}

//============================================================================
void INetIMAPClient_Impl::freshTag()
{
	sal_uInt32 nID = 0;
	if (m_aTag.Len() != 0)
	{
		sal_Char const * p = m_aTag.GetBuffer() + 1;
		sal_Char const * pEnd = m_aTag.GetBuffer() + m_aTag.Len();
		if (INetMIME::scanUnsigned(p, pEnd, true, nID) && p == pEnd)
			++nID;
	}
	INetMIMEStringOutputSink aFresh(0,
									INetMIMEOutputSink::NO_LINE_LENGTH_LIMIT);
	aFresh << 'T';
	INetMIME::writeUnsigned(aFresh, nID);
	m_aTag = aFresh.takeBuffer();
}

//============================================================================
ErrCode INetIMAPClient_Impl::startCommand(Link const & rCallback,
										  void * pResponseData,
										  State eCommandState)
{
	DBG_ASSERT(rCallback.IsSet(),
			   "INetIMAPClient::startCommand(): Null callback");
	DBG_ASSERT(eCommandState > STATE_IDLE && eCommandState < STATE_CLOSED,
			   "INetIMAPClient_Impl::startCommand(): Invalid command state");
	{
		vos::OGuard aGuard = m_aMutex;
		if (m_eState != STATE_IDLE)
			return ERRCODE_IO_ACCESSDENIED;
		m_eState = eCommandState;
	}

	m_aCommandCallback = rCallback;
	m_pCommandResponseData = pResponseData;

	freshTag();
	ByteString aLine = m_aTag;
	static sal_Char const * const aCommandText[STATE_COMMAND_NAMESPACE
											       - STATE_COMMAND_CAPABILITY
											       + 1]
		= { " CAPABILITY", " NOOP", " LOGOUT", " LOGIN", " SELECT",
			" EXAMINE", " CREATE", " DELETE", " RENAME", " SUBSCRIBE",
			" UNSUBSCRIBE", " LIST", " LSUB", " STATUS", " APPEND", " CHECK",
			" CLOSE", " EXPUNGE", " SEARCH", " UID SEARCH", " FETCH",
			" UID FETCH", " STORE", " UID STORE", " COPY", " UID COPY",
			" NAMESPACE" };
	DBG_ASSERT(eCommandState >= STATE_COMMAND_CAPABILITY
			   && eCommandState <= STATE_COMMAND_NAMESPACE,
			   "INetIMAPClient_Impl::startCommand(): Invalid state");
	aLine.Append(aCommandText[eCommandState - STATE_COMMAND_CAPABILITY]);

	DBG_ASSERT(!m_xCommandStream,
			   "INetIMAPClient_Impl::startCommand(): Invalid stream");
	m_xCommandStream
		= new INetIMAPCommandStream(aLine,
									hasDeterminedCapabilities()
									&& getCapabilities()
									       & CAPABILITY_LITERAL_PLUS);
	return ERRCODE_NONE;
}

//============================================================================
ErrCode INetIMAPClient_Impl::sendCommand()
{
	DBG_ASSERT(m_xCommandStream.Is(),
			   "INetIMAPClient_Impl::sendCommand(): Null stream");

	if (!m_xConnection->Send(*m_xCommandStream, connectionSendCallback, this))
	{
		vos::OGuard aGuard = m_aMutex;
		m_xCommandStream = 0;
		m_eState = STATE_IDLE;
		return ERRCODE_IO_CANTWRITE;
	}

	return ERRCODE_IO_PENDING;
}

//============================================================================
// virtual
bool INetIMAPClient_Impl::hasOpenConnection() const
{
	return m_eState > STATE_OPENING && m_eState < STATE_CLOSED;
}

//============================================================================
// virtual
bool INetIMAPClient_Impl::hasClosedConnection() const
{
	return m_eState == STATE_CLOSED;
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::openConnection(UniString const & rHost,
											sal_uInt16 nPort,
											UniString const & rSocksProxyHost,
											sal_uInt16 nSocksProxyPort,
											Link const &
											    rTheUnilateralCallback,
											void * pTheUnilateralResponseData,
											Link const & rConnectCallback,
											void * pConnectResponseData)
{
	DBG_ASSERT(rTheUnilateralCallback.IsSet() && rConnectCallback.IsSet(),
			   "INetIMAPClient_Impl::openConnection(): Null callback");
	{
		vos::OGuard aGuard = m_aMutex;
		if (m_eState != STATE_INITIAL)
			return ERRCODE_IO_ACCESSDENIED;
		m_eState = STATE_OPENING;
		m_xConnection = new INetCoreTCPConnection;
	}

	m_xConnection->SetTerminateCallback(connectionTerminationCallback, this);

	m_aUnilateralCallback = rTheUnilateralCallback;
	m_pUnilateralResponseData = pTheUnilateralResponseData;

	m_aCommandCallback = rConnectCallback;
	m_pCommandResponseData = pConnectResponseData;

	if (!m_xConnection->Open(
		rtl::OUString::createFromAscii ("imap://"),
		rHost, nPort, connectionOpenCallback, this))
	{
		vos::OGuard aGuard = m_aMutex;
		m_eState = STATE_CLOSED;
		m_xConnection = 0;
		return ERRCODE_IO_ACCESSDENIED;
	}
	return ERRCODE_IO_PENDING;
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::abortConnection()
{
	vos::ORef< INetCoreTCPConnection > xTheConnection;
	{
		vos::OGuard aGuard = m_aMutex;
		switch (m_eState)
		{
			case STATE_INITIAL:
			case STATE_CLOSED:
				return ERRCODE_IO_ACCESSDENIED;
		}
		m_eState = STATE_CLOSED;
		xTheConnection = m_xConnection;
	}

	// Clearing m_xCommandStream and m_xConnection is missing from this
	// *asynchronous* operation:
	static_cast< INetIMAPScanner * >(&m_xScanner)->abort();
	if (xTheConnection.isValid())
		xTheConnection->Abort();
	return ERRCODE_NONE;
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::cancelCommand()
{
	vos::ORef< INetCoreTCPConnection > xTheConnection;
	{
		vos::OGuard aGuard = m_aMutex;
		switch (m_eState)
		{
			case STATE_INITIAL:
			case STATE_IDLE:
			case STATE_CLOSED:
				return ERRCODE_IO_ACCESSDENIED;
		}
		m_eState = STATE_CLOSED;
		xTheConnection = m_xConnection;
	}

	// Clearing m_xCommandStream and m_xConnection is missing from this
	// *asynchronous* operation:
	static_cast< INetIMAPScanner * >(&m_xScanner)->abort();
	if (xTheConnection.isValid())
		xTheConnection->Abort();
	return ERRCODE_NONE;
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandCapability(Link const & rCallback,
											   void * pResponseData)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_CAPABILITY);
	if (nError != ERRCODE_NONE)
		return nError;
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandNoOp(Link const & rCallback,
										 void * pResponseData)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_NOOP);
	if (nError != ERRCODE_NONE)
		return nError;
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandLogOut(Link const & rCallback,
										   void * pResponseData)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_LOGOUT);
	if (nError != ERRCODE_NONE)
		return nError;
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandLogIn(Link const & rCallback,
										  void * pResponseData,
										  UniString const & rUserID,
										  UniString const & rPassword)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_LOGIN);
	if (nError != ERRCODE_NONE)
		return nError;
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  ByteString(rUserID,
											 RTL_TEXTENCODING_UTF8)));
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  ByteString(rPassword,
											 RTL_TEXTENCODING_UTF8)));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandSelect(Link const & rCallback,
										   void * pResponseData,
										   ByteString const & rMailboxName)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_SELECT);
	if (nError != ERRCODE_NONE)
		return nError;
	m_aSelectedMailbox = rMailboxName;
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  rMailboxName));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandExamine(Link const & rCallback,
											void * pResponseData,
											ByteString const & rMailboxName)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_EXAMINE);
	if (nError != ERRCODE_NONE)
		return nError;
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  rMailboxName));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandCreate(Link const & rCallback,
										   void * pResponseData,
										   ByteString const & rMailboxName)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_CREATE);
	if (nError != ERRCODE_NONE)
		return nError;
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  rMailboxName));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandDelete(Link const & rCallback,
										   void * pResponseData,
										   ByteString const & rMailboxName)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_DELETE);
	if (nError != ERRCODE_NONE)
		return nError;
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  rMailboxName));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandRename(Link const & rCallback,
										   void * pResponseData,
										   ByteString const & rOldMailboxName,
										   ByteString const & rNewMailboxName)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_RENAME);
	if (nError != ERRCODE_NONE)
		return nError;
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  rOldMailboxName));
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  rNewMailboxName));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandSubscribe(Link const & rCallback,
											  void * pResponseData,
											  ByteString const & rMailboxName)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_SUBSCRIBE);
	if (nError != ERRCODE_NONE)
		return nError;
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  rMailboxName));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandUnSubscribe(Link const & rCallback,
												void * pResponseData,
												ByteString const &
												    rMailboxName)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_UNSUBSCRIBE);
	if (nError != ERRCODE_NONE)
		return nError;
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  rMailboxName));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandList(Link const & rCallback,
										 void * pResponseData,
										 ByteString const & rReference,
										 ByteString const & rPattern)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_LIST);
	if (nError != ERRCODE_NONE)
		return nError;
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  rReference));
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING_WILD,
								  rPattern));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandLSub(Link const & rCallback,
										 void * pResponseData,
										 ByteString const & rReference,
										 ByteString const & rPattern)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_LSUB);
	if (nError != ERRCODE_NONE)
		return nError;
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  rReference));
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING_WILD,
								  rPattern));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandStatus(Link const & rCallback,
										   void * pResponseData,
										   ByteString const & rMailboxName,
										   INetIMAPStatusAttributes
										       eAttributes)
{
	if (eAttributes == 0)
		return ERRCODE_IO_INVALIDPARAMETER;
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_STATUS);
	if (nError != ERRCODE_NONE)
		return nError;
	ByteString aAttributesText = '(';
	bool bSpace;
	if (eAttributes & INET_IMAP_STATUS_ATTRIBUTE_MESSAGES)
	{
		aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("MESSAGES"));
		bSpace = true;
	}
	if (eAttributes & INET_IMAP_STATUS_ATTRIBUTE_RECENT)
	{
		if (bSpace)
			aAttributesText += ' ';
		aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("RECENT"));
		bSpace = true;
	}
	if (eAttributes & INET_IMAP_STATUS_ATTRIBUTE_UIDNEXT)
	{
		if (bSpace)
			aAttributesText += ' ';
		aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("UIDNEXT"));
		bSpace = true;
	}
	if (eAttributes & INET_IMAP_STATUS_ATTRIBUTE_UIDVALIDITY)
	{
		if (bSpace)
			aAttributesText += ' ';
		aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("UIDVALIDITY"));
		bSpace = true;
	}
	if (eAttributes & INET_IMAP_STATUS_ATTRIBUTE_UNSEEN)
	{
		if (bSpace)
			aAttributesText += ' ';
		aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("UNSEEN"));
	}
	aAttributesText += ')';
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  rMailboxName));
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_TEXT,
								  aAttributesText));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandAppend(Link const & rCallback,
										   void * pResponseData,
										   ByteString const & rMailboxName,
										   ByteString const & rRFC822Header,
										   ByteString const & rMediaType,
										   SvStream * pBody)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_APPEND);
	if (nError != ERRCODE_NONE)
	{
		delete pBody;
		return nError;
	}
	// Sending any raw data with "Content-Transfer-Encoding: binary" should
	// work (and would be faster), but many IMAP servers seem to ignore this,
	// corrupting the data or rejecting the APPEND:
	ByteString aHeader = rRFC822Header;
	aHeader.Append(RTL_CONSTASCII_STRINGPARAM("MIME-Version: 1.0\x0D\x0A"
											      "Content-Type: "));
	aHeader += rMediaType;
	aHeader.Append(RTL_CONSTASCII_STRINGPARAM("\x0D\x0A"
											      "Content-Transfer-Encoding:"
											      " base64\x0D\x0A\x0D\x0A"));
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  rMailboxName));
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_STREAM_BASE64,
								  aHeader, pBody));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandAppend(Link const & rCallback,
										   void * pResponseData,
										   ByteString const & rMailboxName,
										   SvStream * pMessage)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_APPEND);
	if (nError != ERRCODE_NONE)
	{
		delete pMessage;
		return nError;
	}
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  rMailboxName));
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_STREAM,
								  ByteString(), pMessage));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandCheck(Link const & rCallback,
										  void * pResponseData)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_CHECK);
	if (nError != ERRCODE_NONE)
		return nError;
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandClose(Link const & rCallback,
										  void * pResponseData)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_CLOSE);
	if (nError != ERRCODE_NONE)
		return nError;
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandExpunge(Link const & rCallback,
											void * pResponseData)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_EXPUNGE);
	if (nError != ERRCODE_NONE)
		return nError;
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandSearch(Link const & rCallback,
										   void * pResponseData,
										   bool bUIDCommand,
										   rtl_TextEncoding eCharset,
										   INetIMAPSearchKeyList const &
										       rKeys)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  bUIDCommand ? STATE_COMMAND_UID_SEARCH :
								                STATE_COMMAND_SEARCH);
	if (nError != ERRCODE_NONE)
		return nError;
	if (eCharset != RTL_TEXTENCODING_DONTKNOW)
		appendCommandArgument(new INetIMAPCommandArgument(
			                          INetIMAPCommandArgument::TYPE_TEXT,
									  INetMIME::getCharsetName(eCharset)));
	for (sal_uInt32 i = 0; i < rKeys.getCount(); ++i)
		rKeys.get(i).appendCommandArguments(*this);
	return sendCommand();
}

//============================================================================
// virtual
ErrCode
INetIMAPClient_Impl::commandFetch(Link const & rCallback,
								  void * pResponseData, bool bUIDCommand,
								  INetIMAPMessageNumberSet const &
								      rMessageNumberSet,
								  FetchAttributes eAttributes,
								  INetIMAPArgumentBodySectionList const *
								      pBodySections,
								  INetIMAPHeaderFieldListList const *
								      pRFC822HeaderLines,
								  Link const & rTheStreamCallback)
{
	bool bExtra = pBodySections && pBodySections->getCount() != 0
	              || pRFC822HeaderLines
		             && pRFC822HeaderLines->getCount() != 0;
	if (eAttributes == 0 && !bExtra)
		return ERRCODE_IO_INVALIDPARAMETER;
	ErrCode nError = startCommand(rCallback, pResponseData,
								  bUIDCommand ? STATE_COMMAND_UID_FETCH :
								                STATE_COMMAND_FETCH);
	if (nError != ERRCODE_NONE)
		return nError;
	m_aStreamCallback = rTheStreamCallback;
	ByteString aAttributesText;
	if (!bExtra && eAttributes == FETCH_ALL)
		aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("ALL"));
	else if (!bExtra && eAttributes == FETCH_FULL)
		aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("FULL"));
	else if (!bExtra && eAttributes == FETCH_FAST)
		aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("FAST"));
	else
	{
		sal_uInt32 nCount = 0;
		if (eAttributes & FETCH_BODY)
		{
			if (nCount++ != 0)
				aAttributesText += ' ';
			aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("BODY"));
		}
		if (eAttributes & FETCH_BODYSTRUCTURE)
		{
			if (nCount++ != 0)
				aAttributesText += ' ';
			aAttributesText.
				Append(RTL_CONSTASCII_STRINGPARAM("BODYSTRUCTURE"));
		}
		if (eAttributes & FETCH_ENVELOPE)
		{
			if (nCount++ != 0)
				aAttributesText += ' ';
			aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("ENVELOPE"));
		}
		if (eAttributes & FETCH_FLAGS)
		{
			if (nCount++ != 0)
				aAttributesText += ' ';
			aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("FLAGS"));
		}
		if (eAttributes & FETCH_INTERNALDATE)
		{
			if (nCount++ != 0)
				aAttributesText += ' ';
			aAttributesText.
				Append(RTL_CONSTASCII_STRINGPARAM("INTERNALDATE"));
		}
		if (eAttributes & FETCH_RFC822)
		{
			if (nCount++ != 0)
				aAttributesText += ' ';
			aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("RFC822"));
		}
		if (eAttributes & FETCH_RFC822_HEADER)
		{
			if (nCount++ != 0)
				aAttributesText += ' ';
			aAttributesText.
				Append(RTL_CONSTASCII_STRINGPARAM("RFC822.HEADER"));
		}
		if (eAttributes & FETCH_RFC822_PEEK)
		{
			if (nCount++ != 0)
				aAttributesText += ' ';
			aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("RFC822.PEEK"));
		}
		if (eAttributes & FETCH_RFC822_SIZE)
		{
			if (nCount++ != 0)
				aAttributesText += ' ';
			aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("RFC822.SIZE"));
		}
		if (eAttributes & FETCH_RFC822_TEXT)
		{
			if (nCount++ != 0)
				aAttributesText += ' ';
			aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("RFC822.TEXT"));
		}
		if (eAttributes & FETCH_RFC822_TEXT_PEEK)
		{
			if (nCount++ != 0)
				aAttributesText += ' ';
			aAttributesText.
				Append(RTL_CONSTASCII_STRINGPARAM("RFC822.TEXT.PEEK"));
		}
		if (eAttributes & FETCH_UID)
		{
			if (nCount++ != 0)
				aAttributesText += ' ';
			aAttributesText.Append(RTL_CONSTASCII_STRINGPARAM("UID"));
		}
		if (pBodySections)
			for (sal_uInt32 i = 0; i < pBodySections->getCount(); ++i)
			{
				INetIMAPArgumentBodySection const & rSection
					= pBodySections->get(i);
				if (nCount++ != 0)
					aAttributesText += ' ';
				aAttributesText += rSection.toString();
			}
		if (pRFC822HeaderLines)
			for (sal_uInt32 i = 0; i < pRFC822HeaderLines->getCount(); ++i)
			{
				INetIMAPHeaderFieldList const & rLines
					= pRFC822HeaderLines->get(i);
				if (nCount++ != 0)
					aAttributesText += ' ';
				aAttributesText.
					Append(RTL_CONSTASCII_STRINGPARAM("HEADER.LINES"));
				aAttributesText += rLines.toString();
			}
		DBG_ASSERT(nCount != 0,
				   "INetIMAPClient_Impl::commandFetch(): No attributes");
		if (nCount > 1)
		{
			aAttributesText.Insert('(', 0);
			aAttributesText += ')';
		}
	}
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_TEXT,
								  rMessageNumberSet.toString()));
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_TEXT,
								  aAttributesText));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandStore(Link const & rCallback,
										  void * pResponseData,
										  bool bUIDCommand,
										  INetIMAPMessageNumberSet const &
										      rMessageNumberSet,
										  StoreAttribute eAttribute,
										  INetIMAPFlags eFlags,
										  INetIMAPFlagKeywordList const *
										      pFlagKeywords)
{
	if (eFlags == 0 && (!pFlagKeywords || pFlagKeywords->getCount() == 0))
		return ERRCODE_IO_INVALIDPARAMETER;
	ErrCode nError = startCommand(rCallback, pResponseData,
								  bUIDCommand ? STATE_COMMAND_UID_STORE :
								                STATE_COMMAND_STORE);
	if (nError != ERRCODE_NONE)
		return nError;
	ByteString aAttributeText;
	switch (eAttribute)
	{
		case STORE_FLAGS:
			aAttributeText.Append(RTL_CONSTASCII_STRINGPARAM("FLAGS"));
			break;

		case STORE_FLAGS_SILENT:
			aAttributeText.Append(RTL_CONSTASCII_STRINGPARAM("FLAGS.SILENT"));
			break;

		case STORE_ADD_FLAGS:
			aAttributeText.Append(RTL_CONSTASCII_STRINGPARAM("+FLAGS"));
			break;

		case STORE_ADD_FLAGS_SILENT:
			aAttributeText.
				Append(RTL_CONSTASCII_STRINGPARAM("+FLAGS.SILENT"));
			break;

		case STORE_REMOVE_FLAGS:
			aAttributeText.Append(RTL_CONSTASCII_STRINGPARAM("-FLAGS"));
			break;

		case STORE_REMOVE_FLAGS_SILENT:
			aAttributeText.
				Append(RTL_CONSTASCII_STRINGPARAM("-FLAGS.SILENT"));
			break;
	}
	if (eFlags & INET_IMAP_FLAG_ANSWERED)
		aAttributeText.Append(RTL_CONSTASCII_STRINGPARAM(" \\Answered"));
	if (eFlags & INET_IMAP_FLAG_FLAGGED)
		aAttributeText.Append(RTL_CONSTASCII_STRINGPARAM(" \\Flagged"));
	if (eFlags & INET_IMAP_FLAG_DELETED)
		aAttributeText.Append(RTL_CONSTASCII_STRINGPARAM(" \\Deleted"));
	if (eFlags & INET_IMAP_FLAG_SEEN)
		aAttributeText.Append(RTL_CONSTASCII_STRINGPARAM(" \\Seen"));
	if (eFlags & INET_IMAP_FLAG_DRAFT)
		aAttributeText.Append(RTL_CONSTASCII_STRINGPARAM(" \\Draft"));
	if (pFlagKeywords)
		for (sal_uInt32 i = 0; i < pFlagKeywords->getCount(); ++i)
		{
			aAttributeText += ' ';
			aAttributeText += pFlagKeywords->get(i);
		}
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_TEXT,
								  rMessageNumberSet.toString()));
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_TEXT,
								  aAttributeText));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandCopy(Link const & rCallback,
										 void * pResponseData,
										 bool bUIDCommand,
										 INetIMAPMessageNumberSet const &
										     rMessageNumberSet,
										 ByteString const & rMailboxName)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  bUIDCommand ? STATE_COMMAND_UID_COPY :
								                STATE_COMMAND_COPY);
	if (nError != ERRCODE_NONE)
		return nError;
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_TEXT,
								  rMessageNumberSet.toString()));
	appendCommandArgument(new INetIMAPCommandArgument(
		                          INetIMAPCommandArgument::TYPE_ASTRING,
								  rMailboxName));
	return sendCommand();
}

//============================================================================
// virtual
ErrCode INetIMAPClient_Impl::commandNamespace(Link const & rCallback,
											  void * pResponseData)
{
	ErrCode nError = startCommand(rCallback, pResponseData,
								  STATE_COMMAND_NAMESPACE);
	if (nError != ERRCODE_NONE)
		return nError;
	return sendCommand();
}

//============================================================================
// virtual
bool INetIMAPClient_Impl::isAuthenticated() const
{
	return m_bAuthenticated;
}

//============================================================================
// virtual
bool INetIMAPClient_Impl::hasDeterminedCapabilities() const
{
	return m_bCapabilitiesDetermined;
}

//============================================================================
// virtual
INetIMAPClient::Capabilities INetIMAPClient_Impl::getCapabilities() const
{
	DBG_ASSERT(m_bCapabilitiesDetermined,
			   "INetIMAPClient_Impl::getCapabilities(): None");
	return m_eCapabilities;
}

//============================================================================
// virtual
ByteString const & INetIMAPClient_Impl::getSelectedMailbox() const
{
	return m_aSelectedMailbox;
}

//============================================================================
// virtual
INetIMAPClient_Impl::~INetIMAPClient_Impl()
{
	if (m_xConnection.isValid())
		m_xConnection->Abort();
	delete m_pListResponseMailbox;
	delete m_pFetchResponseBodySection;
	if (m_pFetchResponseBodySectionParser)
	{
		if (m_pFetchResponseBodySectionParser->GetTargetMessage())
		{
			delete m_pFetchResponseBodySectionParser->
			        GetTargetMessage()->GetDocumentStream();
			delete m_pFetchResponseBodySectionParser->GetTargetMessage();
		}
		delete m_pFetchResponseBodySectionParser;
	}
	delete m_pResponse;
}

//============================================================================
INetIMAPClient_Impl::INetIMAPClient_Impl():
	m_eState(STATE_INITIAL),
	m_bAuthenticated(false),
	m_bCapabilitiesDetermined(false),
	m_pResponse(0),
	m_pListResponseMailbox(0),
	m_pFetchResponseBodySection(0),
	m_pFetchResponseBodySectionParser(0)
{
	m_xScanner = new INetIMAPScanner(scannerCallback, this,
									 INetIMAPScanner::MODE_ATOM);
}

//============================================================================
void INetIMAPClient_Impl::appendCommandArgument(INetIMAPCommandArgument *
												    pArgument)
{
	DBG_ASSERT(pArgument,
			   "INetIMAPClient_Impl::appendCommandArgument(): Null argument");
	DBG_ASSERT(m_xCommandStream.Is(),
			   "INetIMAPClient_Impl::appendCommandArgument(): Null stream");
	static_cast< INetIMAPCommandStream * >(&m_xCommandStream)->
		appendArgument(pArgument);
}

