/*************************************************************************
 *
 *  $RCSfile: XclImpStream.cxx,v $
 *
 *  $Revision: 1.10.6.1 $
 *
 *  last change: $Author: mh $ $Date: 2002/11/01 07:54:54 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifdef PCH
#include "filt_pch.hxx"
#endif

#pragma hdrstop

//___________________________________________________________________

#ifndef _SC_XCLIMPSTREAM_HXX
#include "XclImpStream.hxx"
#endif

#ifndef _EXCDEFS_HXX
#include "excdefs.hxx"
#endif

//___________________________________________________________________

XclImpStreamPos::XclImpStreamPos( ULONG _nPos, ULONG _nNextPos, ULONG _nCurrLen, ULONG _nRecLeft, UINT16 _nRecLen ) :
	nPos( _nPos ),
	nNextPos( _nNextPos ),
	nCurrLen( _nCurrLen ),
	nRecLeft( _nRecLeft ),
	nRecLen( _nRecLen )
{
}

void XclImpStreamPos::Set( ULONG _nPos, ULONG _nNextPos, ULONG _nCurrLen, ULONG _nRecLeft, UINT16 _nRecLen )
{
	nPos = _nPos;
	nNextPos = _nNextPos;
	nCurrLen = _nCurrLen;
	nRecLeft = _nRecLeft;
	nRecLen = _nRecLen;
}

//___________________________________________________________________

XclImpStreamPosStack::~XclImpStreamPosStack()
{
	Clear();
}

void XclImpStreamPosStack::Clear()
{
	for( XclImpStreamPos* pPos = Pop(); pPos; pPos = Pop() )
		delete pPos;
}

//___________________________________________________________________

XclImpStream::XclImpStream( SvStream& rInStrm, const CharSet* pCharSet, BOOL bContHandling ) :
	rStrm( rInStrm ),
    pChSet( pCharSet ),
	aFirstRec( STREAM_SEEK_TO_BEGIN, STREAM_SEEK_TO_BEGIN, 0, 0, 0 ),
	aPosStack(),
	aUserPos( STREAM_SEEK_TO_BEGIN, STREAM_SEEK_TO_BEGIN, 0, 0, 0 ),
	nUserRecNum( 0 ),
	bUserValidRec( FALSE ),
	bHasUserPos( FALSE ),
	nNextRecPos( STREAM_SEEK_TO_BEGIN ),
	nCurrRecLen( 0 ),
	nComplRecLen( 0 ),
	bHasComplRec( FALSE ),
	nRecNum( 0 ),
	nRecLen( 0 ),
	nRecLeft( 0 ),
	bCont( bContHandling ),
	bValidRec( FALSE ),
	bValid( FALSE ),
	bWarnings( TRUE )
{
    DBG_ASSERT( pChSet, "XclImpStream::XclImpStream - no CharSet" );
	rStrm.Seek( STREAM_SEEK_TO_END );
	nStreamLen = rStrm.Tell();
	rStrm.Seek( STREAM_SEEK_TO_BEGIN );
	DBG_ASSERT( nStreamLen < ~(0UL), "XclImpStream::XclImpStream - stream too long" );
}

XclImpStream::~XclImpStream()
{
}

BOOL XclImpStream::GetNextRecord( UINT16& rNewRecNum, UINT16& rNewRecLen )
{
	rStrm.Seek( nNextRecPos );
	if( nNextRecPos < nStreamLen )
	{
		rStrm >> rNewRecNum >> rNewRecLen;
		return TRUE;
	}
	rNewRecNum = rNewRecLen = 0;
	return FALSE;
}

void XclImpStream::SetupRecord()
{
	nRecLeft = nRecLen;
	nCurrRecLen = nComplRecLen = nRecLen;
	bHasComplRec = !bCont;
	aFirstRec.Set( rStrm.Tell(), nNextRecPos, nComplRecLen, nRecLeft, nRecLen );
}

BOOL XclImpStream::StartNextRecord()
{
	aPosStack.Clear();

    /* counter to ignore zero records (id==len==0) (i.e. the application
        "Crystal Report" writes zero records between other records) */
    sal_uInt32 nZeroRecCount = 5;
    sal_Bool bIsZeroRec = sal_False;

	do
	{
        bValidRec = GetNextRecord( nRecNum, nRecLen );
        bIsZeroRec = !nRecNum && !nRecLen;
        if( bIsZeroRec ) --nZeroRecCount;
		nNextRecPos = rStrm.Tell() + nRecLen;
	}
    while( bValidRec && ((bCont && (nRecNum == EXC_CONT)) || (bIsZeroRec && nZeroRecCount)) );

    if( bIsZeroRec )
        bValidRec = sal_False;
    bValid = bValidRec;
	SetupRecord();
	return bValidRec;
}

void XclImpStream::InitializeRecord( BOOL bContHandling )
{
	if( !bValidRec ) return;

	aPosStack.Clear();
	RestorePosition( aFirstRec );
	nCurrRecLen = nComplRecLen = nRecLen;
	bHasComplRec = !bContHandling;
	bCont = bContHandling;
}

BOOL XclImpStream::GetContinue()
{
    if( !bCont && (nRecNum != EXC_CONT) )
		return FALSE;

	UINT16 nNewNum;
	if( GetNextRecord( nNewNum, nRecLen ) && (nNewNum == EXC_CONT) )
	{
		nRecLeft = nRecLen;
		nNextRecPos = rStrm.Tell() + nRecLen;
		nCurrRecLen += nRecLen;
		return TRUE;
	}
	return FALSE;
}

void XclImpStream::PushPosition()
{
	if( bValid )
		aPosStack.Push( new XclImpStreamPos( rStrm.Tell(), nNextRecPos, nCurrRecLen, nRecLeft, nRecLen ) );
}

void XclImpStream::RestorePosition( const XclImpStreamPos& rPos )
{
	rStrm.Seek( rPos.nPos );
	nNextRecPos = rPos.nNextPos;
	nCurrRecLen = rPos.nCurrLen;
	nRecLeft = rPos.nRecLeft;
	nRecLen = rPos.nRecLen;
	bValid = TRUE;
}

void XclImpStream::PopPosition()
{
	XclImpStreamPos* pPos = aPosStack.Pop();
	if( pPos )
	{
		RestorePosition( *pPos );
		delete pPos;
	}
	else
		DBG_ERROR( "XclImpStream::PopPosition - stack empty" );
}

void XclImpStream::RejectPosition()
{
	XclImpStreamPos* pPos = aPosStack.Pop();
	if( pPos )
		delete pPos;
	else
		DBG_ERROR( "XclImpStream::PopPosition - stack empty" );
}

void XclImpStream::StoreUserPosition()
{
	aUserPos.Set( rStrm.Tell(), nNextRecPos, nCurrRecLen, nRecLeft, nRecLen );
	nUserRecNum = nRecNum;
	bUserValidRec = bValidRec;
	bHasUserPos = TRUE;
}

void XclImpStream::SeekUserPosition()
{
	DBG_ASSERT( bHasUserPos, "XclImpStream::SeekUserPosition - no position stored" );
	if( bHasUserPos )
	{
		RestorePosition( aUserPos );
		nRecNum = nUserRecNum;
		nComplRecLen = nCurrRecLen;
		bHasComplRec = !bCont;
		bValidRec = bValid = bUserValidRec;
	}
}

ULONG XclImpStream::GetRecLen()
{
	if( IsValid() && !bHasComplRec )
	{
		PushPosition();
		while( GetContinue() );		// GetContinue() adds up <nCurrRecLen>
		nComplRecLen = nCurrRecLen;
		bHasComplRec = TRUE;
		PopPosition();
	}
	return nComplRecLen;
}

ULONG XclImpStream::Read( void* pData, ULONG nBytes )
{
	if( !IsValid() || !pData || !nBytes )
		return 0;

	UINT8* pBuffer = (UINT8*) pData;
	ULONG nBytesLeft = nBytes;
	ULONG nRet = 0;

	while( IsValid() && nBytesLeft )
	{
		ULONG nReadLen = Min( nBytesLeft, nRecLeft );
		ULONG nReadRet = rStrm.Read( pBuffer, nReadLen );
		nRet += nReadRet;
		bValid = (nReadLen == nReadRet);
		DBG_ASSERT( !bWarnings || bValid, "XclImpStream::Read - stream read error" );
		pBuffer += nReadRet;
		nRecLeft -= nReadRet;
		nBytesLeft -= nReadRet;
		if( IsValid() && nBytesLeft )
			StartContinue();
	}
	return nRet;
}

ULONG XclImpStream::CopyToStream( SvStream& rOutStrm, ULONG nBytes )
{
	if( !IsValid() || !nBytes )
		return 0;

	const ULONG nMaxBuffer = 0x00001000;
	UINT8* pBuffer = new UINT8[ Min( nBytes, nMaxBuffer ) ];
	ULONG nBytesLeft = nBytes;
	ULONG nRet = 0;

	while( IsValid() && nBytesLeft )
	{
		ULONG nReadLen = Min( nBytesLeft, nMaxBuffer );
		nRet += Read( pBuffer, nReadLen );
		rOutStrm.Write( pBuffer, nReadLen );
		nBytesLeft -= nReadLen;
	}

	delete[] pBuffer;
	return nRet;
}

ULONG XclImpStream::CopyRecordToStream( SvStream& rOutStrm )
{
	if( !bValidRec )
		return 0;

	PushPosition();
	RestorePosition( aFirstRec );
	ULONG nRet = CopyToStream( rOutStrm, GetRecLen() );
	PopPosition();
	return nRet;
}

void XclImpStream::Seek( ULONG nPos )
{
	if( !bValidRec ) return;

    ULONG nCurrPos = GetRecPos();
    if( nCurrPos == nPos )
        return;
    else if( nCurrPos < nPos )
        Ignore( nPos - nCurrPos );
    else
    {
        RestorePosition( aFirstRec );
        Ignore( nPos );
    }
}

void XclImpStream::Ignore( ULONG nBytes )
{
	if( !IsValid() ) return;

	ULONG nBytesLeft = nBytes;
	while( IsValid() && nBytesLeft )
	{
		ULONG nReadLen = Min( nBytesLeft, nRecLeft );
		rStrm.SeekRel( (long) nReadLen );
		nRecLeft -= nReadLen;
		nBytesLeft -= nReadLen;
		if( nBytesLeft )
			StartContinue();
	}
}

void XclImpStream::StartStringContinue( BOOL& rb16Bit )
{
	DBG_ASSERT( !bWarnings || !nRecLeft, "XclImpStream::StartStringContinue - unexpected garbage" );

    if( bCont && GetRecLeft() )
		StartContinue();
	else if( nRecNum == EXC_CONT )		// start next CONTINUE for TXO import
	{
		UINT16 nNewNum, nNewLen;
		bValidRec = GetNextRecord( nNewNum, nNewLen ) && (nNewNum || nNewLen);
		bValid = bValidRec && (nNewNum == EXC_CONT);
		if( bValid )
		{
			nRecLen = nNewLen;
			nNextRecPos = rStrm.Tell() + nNewLen;
			SetupRecord();
		}
	}
	else
		bValid = FALSE;

	if( bValid )
		rb16Bit = (ReaduInt8() & EXC_STR_16BIT) != 0x00;
}

ULONG XclImpStream::ReadUniStringExtHeader( BOOL& rb16Bit, BOOL& rbRich, BOOL& rbFareast, UINT16& rCrun, UINT32& rExtInf, UINT8 nFlags )
{
	DBG_ASSERT( (nFlags & EXC_STR_UNKNOWN) == 0x00, "XclImpStream::ReadUniStringExt - unknown flags" );
	rb16Bit = (nFlags & EXC_STR_16BIT) != 0x00;
	rbRich = (nFlags & EXC_STR_RICH) != 0x00;
	rbFareast = (nFlags & EXC_STR_FAREAST) != 0x00;
	rCrun = rbRich ? ReaduInt16() : 0;
	rExtInf = rbFareast ? ReaduInt32() : 0;
	return (ULONG) rCrun * 4 + rExtInf;
}

ULONG XclImpStream::ReadUniStringExtHeader( BOOL& rb16Bit, UINT8 nFlags )
{
	BOOL bRich, bFareast;
	UINT16 nCrun;
	UINT32 nExtInf;
    return ReadUniStringExtHeader( rb16Bit, bRich, bFareast, nCrun, nExtInf, nFlags );
}

void XclImpStream::AppendRawUniString( String& rString, UINT16 nChars, BOOL b16Bit )
{
	if( !IsValid() || !nChars ) return;

	sal_Char* pBuffer8 = NULL;
	sal_Unicode* pBuffer16 = NULL;
	ULONG nCharsLeft = nChars;
	ULONG nReadLen, nReadRet;

	while( IsValid() && nCharsLeft )
	{
		if( b16Bit )
		{
			if( !pBuffer16 )
				pBuffer16 = new sal_Unicode[ nCharsLeft + 1 ];
			nReadLen = Min( nCharsLeft, nRecLeft >> 1 );
			DBG_ASSERT( !bWarnings || (nReadLen <= nCharsLeft) || !(nRecLeft & 0x00000001),
				"XclImpStream::ReadRawUniString - missing a byte" );
			sal_Unicode* pUniChar = pBuffer16;
			for( ULONG nIndex = 0; IsValid() && (nIndex < nReadLen); nIndex++, pUniChar++ )
				operator>>( *pUniChar );
			pBuffer16[ nIndex ] = 0x0000;
			if( nIndex )
				rString.Append( pBuffer16 );
		}
		else
		{
			if( !pBuffer8 )
				pBuffer8 = new sal_Char[ nCharsLeft + 1 ];
			nReadLen = Min( nCharsLeft, nRecLeft );
			nReadRet = Read( pBuffer8, nReadLen );
			pBuffer8[ nReadRet ] = 0x00;
			if( nReadRet )
                rString += String( pBuffer8, *pChSet );
		}

		nCharsLeft -= nReadLen;
		if( nCharsLeft )
			StartStringContinue( b16Bit );
	}

	if( pBuffer8 )
		delete[] pBuffer8;
	if( pBuffer16 )
		delete[] pBuffer16;
}

String XclImpStream::ReadRawUniString( UINT16 nChars, BOOL b16Bit )
{
    String aString;
    AppendRawUniString( aString, nChars, b16Bit );
    return aString;
}

String* XclImpStream::ReadNewRawUniString( UINT16 nChars, BOOL b16Bit )
{
    String* pString = new String;
    AppendRawUniString( *pString, nChars, b16Bit );
    return pString;
}

void XclImpStream::IgnoreRawUniString( UINT16 nChars, BOOL b16Bit )
{
	if( !IsValid() || !nChars ) return;

	ULONG nCharsLeft = nChars;
	ULONG nReadLen;

	while( IsValid() && nCharsLeft )
	{
		if( b16Bit )
		{
			nReadLen = Min( nCharsLeft, nRecLeft >> 1 );
			DBG_ASSERT( !bWarnings || (nReadLen <= nCharsLeft) || !(nRecLeft & 0x00000001),
				"XclImpStream::IgnoreRawUniString - missing a byte" );
			Ignore( nReadLen << 1 );
		}
		else
		{
			nReadLen = Min( nCharsLeft, nRecLeft );
			Ignore( nReadLen );
		}

		nCharsLeft -= nReadLen;
		if( nCharsLeft )
			StartStringContinue( b16Bit );
	}
}

void XclImpStream::AppendUniString( String& rString, UINT16 nChars, UINT8 nFlags )
{
	BOOL b16Bit;
    ULONG nExtLen = ReadUniStringExtHeader( b16Bit, nFlags );
    AppendRawUniString( rString, nChars, b16Bit );
    SkipUniStringExtData( nExtLen );
}

String XclImpStream::ReadUniString( UINT16 nChars, UINT8 nFlags )
{
    String aString;
    AppendUniString( aString, nChars, nFlags );
    return aString;
}

String* XclImpStream::ReadNewUniString( UINT16 nChars, UINT8 nFlags )
{
    String* pString = new String;
    AppendUniString( *pString, nChars, nFlags );
    return pString;
}

void XclImpStream::IgnoreUniString( UINT16 nChars, UINT8 nFlags )
{
	BOOL b16Bit;
    ULONG nExtLen = ReadUniStringExtHeader( b16Bit, nFlags );
	IgnoreRawUniString( nChars, b16Bit );
    SkipUniStringExtData( nExtLen );
}

String XclImpStream::ReadUniString( UINT16 nChars )
{
    String aString;
    AppendUniString( aString, nChars );
    return aString;
}

String* XclImpStream::ReadNewUniString( UINT16 nChars )
{
    String* pString = new String;
    AppendUniString( *pString, nChars );
    return pString;
}

String XclImpStream::ReadUniString()
{
    String aString;
    AppendUniString( aString );
    return aString;
}

String* XclImpStream::ReadNewUniString()
{
    String* pString = new String;
    AppendUniString( *pString );
    return pString;
}

String XclImpStream::ReadRawByteString( UINT16 nChars )
{
    String aString;
    AppendRawByteString( aString, nChars );
    return aString;
}

String XclImpStream::ReadByteString( BOOL b16BitLen )
{
    String aString;
    AppendByteString( aString, b16BitLen );
    return aString;
}

