/*************************************************************************
 *
 *  $RCSfile: dinfos2.cxx,v $
 *
 *  $Revision: 1.2.20.1 $
 *
 *  last change: $Author: hr $ $Date: 2002/06/05 15:03:56 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _DINFOS2_HXX_
#include <dinfos2.hxx>
#endif
#ifndef _BIGINT_HXX
#include <tools/bigint.hxx>
#endif

// ------------------------------------------------------------------------

struct PropEntry
{
	UINT32	mnId;
	UINT32	mnSize;
	BYTE*	mpBuf;

						PropEntry( UINT32 nId, const BYTE* pBuf, UINT32 nBufSize );
						PropEntry( const PropEntry& rProp );
						~PropEntry() { delete mpBuf; } ;

	const PropEntry&	operator=(const PropEntry& rPropEntry);
};

PropEntry::PropEntry( UINT32 nId, const BYTE* pBuf, UINT32 nBufSize ) :
	mnId	( nId ),
	mnSize	( nBufSize ),
	mpBuf	( new BYTE[ nBufSize ] )
{
	memcpy( (void*)mpBuf, (void*)pBuf, nBufSize );
};

PropEntry::PropEntry( const PropEntry& rProp ) :
	mnId	( rProp.mnId ),
	mnSize	( rProp.mnSize ),
	mpBuf	( new BYTE[ mnSize ] )
{
	memcpy( (void*)mpBuf, (void*)rProp.mpBuf, mnSize );
};

const PropEntry& PropEntry::operator=(const PropEntry& rPropEntry)
{
	if ( this != &rPropEntry )
	{
		delete[] mpBuf;
		mnId = rPropEntry.mnId;
		mnSize = rPropEntry.mnSize;
		mpBuf = new BYTE[ mnSize ];
		memcpy( (void*)mpBuf, (void*)rPropEntry.mpBuf, mnSize );
	}
	return *this;
}

//	-----------------------------------------------------------------------

void PropItem::Clear()
{
	Seek( STREAM_SEEK_TO_BEGIN );
	delete[] SwitchBuffer();
}

//	-----------------------------------------------------------------------

BOOL PropItem::Read( String& rString )
{
	BOOL bRetValue = FALSE;
	UINT32 nSize, nType, nPos;

	nPos = Tell();

	*this >> nType;

	switch( nType )
	{
		case VT_LPSTR :
		{
			*this >> nSize;
			if ( nSize )
			{
				// codepage ignorieren wir beim einlesen grosszuegigerweise : entweder UNICODE oder ANSI
				char* pString = new char[ nSize ];
				SvMemoryStream::Read( pString, nSize );
				if ( pString[ nSize - 2 ] )
				{
					rString = String( ByteString( pString, (sal_uInt16)( nSize - 1 ) ), RTL_TEXTENCODING_MS_1252 );
					bRetValue = TRUE;
				}
				else if ( nSize ^ 1 )
				{
					Seek( nPos + 8 );
					nSize >>= 1;
					sal_Unicode* pWString = (sal_Unicode*)pString;
					for ( UINT32 i = 0; i < nSize; i++ )
						*this >> pWString[ i ];
					rString = String( pWString, (sal_uInt16)( nSize - 1 ) );
					bRetValue = TRUE;
				}
				delete[] pString;
			}
		}
		break;

		case VT_LPWSTR :
		{
			*this >> nSize;
			if ( nSize > 1 )
			{
				sal_Unicode* pString = new sal_Unicode[ --nSize ];
				for ( UINT32 i = 0; i < nSize; i++ )
					*this >> pString[ i ];
				rString = String( pString, (sal_uInt16)nSize );
				delete[] pString;
				bRetValue = TRUE;
			}
		}
		break;
	}
	return bRetValue;
}

//	-----------------------------------------------------------------------

void PropItem::Write( String& rString )
{
	UINT32 nLen = rString.Len();
	const sal_Unicode* pString = rString.GetBuffer();

	*this << (UINT32)VT_LPWSTR << (UINT32)( nLen + 1 );
	for ( UINT32 i = 0; i < nLen; i++ )
		*this << pString[ i ];
	*this << (UINT16)0;
}

//	-----------------------------------------------------------------------

void PropItem::Write( DateTime& rDateTime )
{
	rDateTime.ConvertToUTC();
	BigInt a100nPerSecond( 10000000L );
	BigInt a100nPerDay = a100nPerSecond * BigInt( 60L * 60 * 24 );

	USHORT nYears = rDateTime.GetYear() - 1601;
	long nDays =
		nYears * 365 + nYears / 4 - nYears / 100 + nYears / 400 +
			rDateTime.GetDayOfYear() - 1;
	BigInt aTime=
		a100nPerDay * BigInt( nDays ) + a100nPerSecond *
			BigInt( (long)( rDateTime.GetSec() +
				   60 * rDateTime.GetMin() +
				   60L * 60 * rDateTime.GetHour() ) );

	BigInt aUlongMax( (ULONG)ULONG_MAX );
	aUlongMax += 1;

	*this	<< (UINT32)VT_FILETIME
			<< (UINT32)( aTime % aUlongMax )
			<< (UINT32)( aTime / aUlongMax );
}

//	-----------------------------------------------------------------------

PropItem& PropItem::operator=( PropItem& rPropItem )
{
	if ( this != &rPropItem )
	{
		Seek( STREAM_SEEK_TO_BEGIN );
		delete[] SwitchBuffer();

		UINT32 nPos = rPropItem.Tell();
		rPropItem.Seek( STREAM_SEEK_TO_END );
		SvMemoryStream::Write( rPropItem.GetData(), rPropItem.Tell() );
		rPropItem.Seek( nPos );
	}
	return *this;
}

//	-----------------------------------------------------------------------

struct Dict
{
	UINT32	mnId;
	String	aString;

			Dict( UINT32 nId, String rString ) { mnId = nId; aString = rString; };
};

//	-----------------------------------------------------------------------

Dictionary::~Dictionary()
{
	for ( void* pPtr = First(); pPtr; pPtr = Next() )
		delete (Dict*)pPtr;
}

//	-----------------------------------------------------------------------

void Dictionary::AddProperty( UINT32 nId, const String& rString )
{
	if ( rString.Len() )		// eindeutige namen bei properties
	{
		// pruefen, ob es die Propertybeschreibung in der Dictionary schon gibt
		for ( Dict* pDict = (Dict*)First(); pDict; pDict = (Dict*)Next() )
		{
			if ( pDict->mnId == nId )
			{
				pDict->aString = rString;
				return;
			}
		}
		Insert( new Dict( nId, rString ), LIST_APPEND );
	}
}

//	-----------------------------------------------------------------------

UINT32 Dictionary::GetProperty( const String& rString )
{
	for ( Dict* pDict = (Dict*)First(); pDict; pDict = (Dict*)Next() )
	{
		if ( pDict->aString == rString )
			return pDict->mnId;
	}
	return 0;
}

//	-----------------------------------------------------------------------

Dictionary& Dictionary::operator=( Dictionary& rDictionary )
{
	if ( this != &rDictionary )
	{
		for ( void* pPtr = First(); pPtr; pPtr = Next() )
			delete (Dict*)pPtr;

		for ( pPtr = rDictionary.First(); pPtr; pPtr = rDictionary.Next() )
			Insert( new Dict( ((Dict*)pPtr)->mnId, ((Dict*)pPtr)->aString ), LIST_APPEND );
	}
	return *this;
}

//	-----------------------------------------------------------------------

Section::Section( Section& rSection )
{
	for ( int i = 0; i < 16; i++ )
		aFMTID[ i ] = rSection.aFMTID[ i ];
	for ( PropEntry* pProp = (PropEntry*)rSection.First(); pProp; pProp = (PropEntry*)rSection.Next() )
		Insert( new PropEntry( *pProp ), LIST_APPEND );
}

//	-----------------------------------------------------------------------

Section::Section( const BYTE* pFMTID )
{
	for ( int i = 0; i < 16; i++ )
		aFMTID[ i ] = pFMTID[ i ];
}

//	-----------------------------------------------------------------------

BOOL Section::GetProperty( UINT32 nId, PropItem& rPropItem )
{
	if ( nId )
	{
		for ( PropEntry* pProp = (PropEntry*)First(); pProp; pProp = (PropEntry*)Next() )
		{
			if ( pProp->mnId == nId )
				break;
		}
		if ( pProp )
		{
			rPropItem.Clear();
			rPropItem.Write( pProp->mpBuf, pProp->mnSize );
			rPropItem.Seek( STREAM_SEEK_TO_BEGIN );
			return TRUE;
		}
	}
	return FALSE;
}

//	-----------------------------------------------------------------------

void Section::AddProperty( UINT32 nId, const BYTE* pBuf, UINT32 nBufSize )
{
	// kleiner id check

	if ( !nId )
		return;
	if ( nId == 0xffffffff )
		nId = 0;

	// keine doppelten PropId's zulassen, sortieren
	for ( UINT32 i = 0; i < Count(); i++ )
	{
		PropEntry* pPropEntry = (PropEntry*)GetObject( i );
		if ( pPropEntry->mnId == nId )
			delete (PropEntry*)Replace( new PropEntry( nId, pBuf, nBufSize ), i );
		else if ( pPropEntry->mnId > nId )
			Insert( new PropEntry( nId, pBuf, nBufSize ), i );
		else
			continue;
		return;
	}
	Insert( new PropEntry( nId, pBuf, nBufSize ), LIST_APPEND );
}

//	-----------------------------------------------------------------------

void Section::AddProperty( UINT32 nId, SvMemoryStream& rStrm )
{
	UINT32 nPos = rStrm.Tell();
	rStrm.Seek( STREAM_SEEK_TO_END );
	AddProperty( nId, (const BYTE*)rStrm.GetData(), rStrm.Tell() );
	rStrm.Seek( nPos );
}

//	-----------------------------------------------------------------------

void Section::AddDictionary( Dictionary& rDict )
{
	SvMemoryStream aStream;
	aStream << (UINT32)rDict.Count();
	for ( Dict* pPtr = (Dict*)rDict.First(); pPtr; pPtr = (Dict*)rDict.Next() )
	{
		aStream << (UINT32)pPtr->mnId
				<< (UINT32)pPtr->aString.Len() + 1;
		ByteString aOut( pPtr->aString, RTL_TEXTENCODING_MS_1252 );
		aStream.Write( aOut.GetBuffer(), aOut.Len() + 1 );
	}
	AddProperty( 0xffffffff, aStream );
}

//	-----------------------------------------------------------------------

BOOL Section::GetDictionary( Dictionary& rDict )
{
	BOOL bRetValue = FALSE;

	Dictionary aDict;

	for ( PropEntry* pProp = (PropEntry*)First(); pProp; pProp = (PropEntry*)Next() )
	{
		if ( pProp->mnId == 0 )
			break;
	}
	if ( pProp )
	{
		UINT32 nCount, nId, nSize, nPos;
		SvMemoryStream aStream( (char*)pProp->mpBuf, pProp->mnSize, STREAM_READ );
		aStream.Seek( STREAM_SEEK_TO_BEGIN );
		aStream >> nCount;
		for ( UINT32 i = 0; i < nCount; i++ )
		{
			aStream >> nId >> nSize;
			if ( nSize )
			{
				String aString;
				nPos = aStream.Tell();
				char* pString = new char[ nSize ];
				aStream.Read( pString, nSize );
				if ( pString[ nSize - 2 ] )
					aString = String( ByteString( pString, (sal_uInt16)( nSize - 1 ) ), RTL_TEXTENCODING_MS_1252 );
				else if ( nSize ^ 1 )
				{
					nSize >>= 1;
					aStream.Seek( nPos );
					sal_Unicode* pWString = (sal_Unicode*)pString;
					for ( i = 0; i < nSize; i++ )
						aStream >> (UINT16&)( pWString[ i ] );
					aString = String( pWString, (sal_uInt16)( nSize - 1 ) );
				}
				delete[] pString;
				if ( !aString.Len() )
					break;
				aDict.AddProperty( nId, aString );
			}
			bRetValue = TRUE;
		}
	}
	rDict = aDict;
	return bRetValue;
}

//	-----------------------------------------------------------------------

Section::~Section()
{
	for ( PropEntry* pProp = (PropEntry*)First(); pProp; pProp = (PropEntry*)Next() )
		delete pProp;
}

//	-----------------------------------------------------------------------

void Section::Read( SvStorageStream *pStrm )
{
	UINT32 nSecOfs, nSecSize, nPropCount, nPropId, nPropOfs, nPropType, nPropSize, nCurrent;
	nSecOfs = pStrm->Tell();
	*pStrm >> nSecSize >> nPropCount;
	while( nPropCount-- )
	{
		*pStrm >> nPropId >> nPropOfs;
		nCurrent = pStrm->Tell();
		pStrm->Seek( nPropOfs + nSecOfs );
		if ( nPropId )					// dictionary wird nicht eingelesen
		{
			*pStrm >> nPropType;
			switch( nPropType )
			{
				case VT_UI1 :
					nPropSize = 1;
				break;

				case VT_I2 :
				case VT_UI2 :
				case VT_BOOL :
					nPropSize = 2;
				break;

				case VT_I4 :
				case VT_R4 :
				case VT_UI4 :
				case VT_ERROR :
					nPropSize = 4;
				break;

				case VT_I8 :
				case VT_R8 :
				case VT_CY :
				case VT_UI8 :
				case VT_DATE :
				case VT_FILETIME :
					nPropSize = 8;
				break;

				case VT_BSTR :
					*pStrm >> nPropSize;
					nPropSize += 4;
				break;

				case VT_LPSTR :
					*pStrm >> nPropSize;
					nPropSize += 4;
				break;

				case VT_LPWSTR :
					*pStrm >> nPropSize;
					nPropSize <<= 1;
					nPropSize += 4;
				break;

				case VT_BLOB_OBJECT :
				case VT_BLOB :
				case VT_CF :
					*pStrm >> nPropSize;
					nPropSize += 4;
				break;

				case VT_CLSID :
				case VT_STREAM :
				case VT_STORAGE :
				case VT_STREAMED_OBJECT :
				case VT_STORED_OBJECT :
				case VT_VARIANT :
				case VT_VECTOR :
				default :
					nPropSize = 0;
			}
			if ( nPropSize )
			{
				nPropSize += 4;
				pStrm->Seek( nPropOfs + nSecOfs );
				BYTE* pBuf = new BYTE[ nPropSize ];
				pStrm->Read( pBuf, nPropSize );
				AddProperty( nPropId, pBuf, nPropSize );
				delete[] pBuf;
			}
		}
		else
		{
			UINT32 nDictCount, nSize;
			*pStrm >> nDictCount;
			for ( UINT32 i = 0; i < nDictCount; i++ )
			{
				*pStrm >> nSize >> nSize;
				pStrm->SeekRel( nSize );
			}
			nSize = pStrm->Tell();
			pStrm->Seek( nPropOfs + nSecOfs );
			nSize -= pStrm->Tell();
			BYTE* pBuf = new BYTE[ nSize ];
			pStrm->Read( pBuf, nSize );
			AddProperty( 0xffffffff, pBuf, nSize );
			delete[] pBuf;
		}
		pStrm->Seek( nCurrent );
	}
	pStrm->Seek( nSecOfs + nSecSize );
}

//	-----------------------------------------------------------------------

void Section::Write( SvStorageStream *pStrm )
{
	UINT32 nSecOfs = pStrm->Tell();
	UINT32 nPropCount = Count();
	*pStrm << (UINT32)0 << nPropCount;

	UINT32 nPropertyOffset = ( nPropCount << 3 ) + 8;

	for ( PropEntry* pProp = (PropEntry*)First(); pProp; pProp = (PropEntry*)Next() )
	{
		*pStrm << pProp->mnId
			   << nPropertyOffset;
		nPropertyOffset += ( pProp->mnSize + 3 ) & ~3;	// dword aligning
	}
	for ( pProp = (PropEntry*)First(); pProp; pProp = (PropEntry*)Next() )
	{
		pStrm->Write( pProp->mpBuf, pProp->mnSize );
		for ( int i = 3 - ( ( pProp->mnSize & 3 ) ^ 3 ); i; i-- )
			*pStrm << (BYTE)0;							// pad bytes
	}
	UINT32 nCurOfs = pStrm->Tell();
	pStrm->Seek( nSecOfs );
	*pStrm << (UINT32)( nCurOfs - nSecOfs );	// size of this section
	pStrm->Seek( nCurOfs );
}

//	-----------------------------------------------------------------------

Section& Section::operator=( Section& rSection )
{
	if ( this != &rSection )
	{
		memcpy( (void*)aFMTID, (void*)rSection.aFMTID, 16 );
		for ( PropEntry* pProp = (PropEntry*)First(); pProp; pProp = (PropEntry*)Next() )
			delete pProp;
		Clear();
		for ( pProp = (PropEntry*)rSection.First(); pProp; pProp = (PropEntry*)rSection.Next() )
			Insert( new PropEntry( *pProp ), LIST_APPEND );
	}
	return *this;
}

//	-----------------------------------------------------------------------

DInfo::DInfo( SvStorage& rStorage, const String& rName ) :
		mbStatus			( FALSE ),
		mnByteOrder			( 0xfffe ),
		mnFormat			( 0 ),
		mnVersionLo			( 4 ),
		mnVersionHi			( 2 )
{
	mpSvStream = rStorage.OpenStream( rName );
	if ( mpSvStream )
	{
		mpSvStream->SetNumberFormatInt( NUMBERFORMAT_INT_LITTLEENDIAN );
		memset( mApplicationCLSID, 0, 16 );
		mbStatus = TRUE;
	}
}

//	-----------------------------------------------------------------------

void DInfo::AddSection( Section& rSection )
{
	Insert( new Section( rSection ), LIST_APPEND );
}

//	-----------------------------------------------------------------------

Section* DInfo::GetSection( const BYTE* pFMTID )
{
	for ( Section* pSection = (Section*)First(); pSection; pSection = (Section*)Next() )
	{
		if ( memcmp( pSection->GetFMTID(), pFMTID, 16 ) == 0 )
			break;
	}
	return pSection;
}

//	-----------------------------------------------------------------------

DInfo::~DInfo()
{
	for ( Section* pSection = (Section*)First(); pSection; pSection = (Section*)Next() )
		delete pSection;
}

//	-----------------------------------------------------------------------

void DInfo::Read()
{
	for ( Section* pSection = (Section*)First(); pSection; pSection = (Section*)Next() )
		delete pSection;
	Clear();
	if ( mbStatus )
	{
		UINT32	nSections;
		UINT32	nSectionOfs;
		UINT32	nCurrent;
		BYTE*	pSectCLSID = new BYTE[ 16 ];
		*mpSvStream >> mnByteOrder >> mnFormat >> mnVersionLo >> mnVersionHi;
		mpSvStream->Read( mApplicationCLSID, 16 );
		*mpSvStream >> nSections;
		for ( UINT32 i = 0; i < nSections; i++ )
		{
			mpSvStream->Read( pSectCLSID, 16 );
			*mpSvStream >> nSectionOfs;
			nCurrent = mpSvStream->Tell();
			mpSvStream->Seek( nSectionOfs );
			Section aSection( pSectCLSID );
			aSection.Read( mpSvStream );
			AddSection( aSection );
			mpSvStream->Seek( nCurrent );
		}
		delete pSectCLSID;
	}
}

//	-----------------------------------------------------------------------

void DInfo::Write()
{
	if ( mbStatus )
	{
		// property set header
		*mpSvStream << mnByteOrder << mnFormat << mnVersionLo << mnVersionHi;
		mpSvStream->Write( mApplicationCLSID, 16 );
		*mpSvStream << (UINT32)( Count() );

		UINT32 nFMTIDStartOffset = mpSvStream->Tell() + 16;

		for ( Section* pSection = (Section*)First(); pSection; pSection = (Section*)Next() )
		{
			const BYTE* pPtr = pSection->GetFMTID();
			mpSvStream->Write( pPtr, 16 );
			*mpSvStream << (UINT32)0;					// offset to section
		}
		for ( pSection = (Section*)First(); pSection; pSection = (Section*)Next() )
		{
			UINT32 nCurrentPos = mpSvStream->Tell();
			mpSvStream->Seek( nFMTIDStartOffset );
			*mpSvStream << ( nCurrentPos );				// insert offset to this section
			nFMTIDStartOffset += 20;
			mpSvStream->Seek( nCurrentPos );
			pSection->Write( mpSvStream );
		}
	}
}

//	-----------------------------------------------------------------------

DInfo& DInfo::operator=( DInfo& rDInfo )
{
	if ( this != &rDInfo )
	{
		mbStatus = rDInfo.mbStatus;
		mpSvStream = rDInfo.mpSvStream;

		mnByteOrder = rDInfo.mnByteOrder;
		mnFormat = rDInfo.mnFormat;
		mnVersionLo = rDInfo.mnVersionLo;
		mnVersionHi = rDInfo.mnVersionHi;
		memcpy( mApplicationCLSID, rDInfo.mApplicationCLSID, 16 );

		for ( Section* pSection = (Section*)First(); pSection; pSection = (Section*)Next() )
			delete pSection;
		Clear();
		for ( pSection = (Section*)rDInfo.First(); pSection; pSection = (Section*)rDInfo.Next() )
			Insert( new Section( *pSection ), LIST_APPEND );
	}
	return *this;
}
