/*************************************************************************
 *
 *  $RCSfile: register.cxx,v $
 *
 *  $Revision: 1.8.2.2 $
 *
 *  last change: $Author: mh $ $Date: 2002/06/26 11:42:05 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/
 
#include <osl/diagnose.h>
#include <rtl/ustring.hxx>
#include "register.hxx"
#include "registry.hxx"

#include "preg4msdoc.hrc"

#include <string>
#include <exception>

//-----------------------------------------
//
//-----------------------------------------

using rtl::OUString;

//-----------------------------------------
//
//-----------------------------------------

#define OUSTR( x ) OUString::createFromAscii( x )

//-----------------------------------------
// constants
//-----------------------------------------

const OUString PRODUCT_NAME            = OUSTR("StarOffice");
const OUString PRODUCT_EXECUTABLE_NAME = OUSTR("soffice.exe");

const OUString REG_KEY_PLUGIN_PRIVATE      = OUSTR("soffice6.bak");
const HKEY	   REG_KEY_PLUGIN_PRIVATE_ROOT = HKEY_CLASSES_ROOT;

const OUString REG_KEY_EXTENSION_DOC = OUSTR(".doc"); 
const OUString REG_KEY_EXTENSION_DOT = OUSTR(".dot");

const OUString REG_KEY_EXTENSION_DOC_FORWARD = OUSTR("soffice6.doc");
const OUString REG_KEY_EXTENSION_DOT_FORWARD = OUSTR("soffice6.dot");


const OUString REG_KEY_EXTENSION_XLS = OUSTR(".xls");
const OUString REG_KEY_EXTENSION_XLT = OUSTR(".xlt");

const OUString REG_KEY_EXTENSION_XLS_FORWARD = OUSTR("soffice6.xls");
const OUString REG_KEY_EXTENSION_XLT_FORWARD = OUSTR("soffice6.xlt");

const OUString REG_KEY_EXTENSION_PPT = OUSTR(".ppt");
const OUString REG_KEY_EXTENSION_POT = OUSTR(".pot");

const OUString REG_KEY_EXTENSION_PPT_FORWARD = OUSTR("soffice6.ppt");
const OUString REG_KEY_EXTENSION_POT_FORWARD = OUSTR("soffice6.pot");

const OUString REG_KEY_EXTENSION_HTM = OUSTR(".htm");

const OUString REG_KEY_EXTENSION_HTM_OPENWITHLIST_SOFFICE  = OUSTR(".htm\\OpenWithList\\soffice.exe");

const OUString DEFAULT_WORD_DOC_DISPLAY_NAME           = OUSTR("Microsoft Word Document");
const OUString DEFAULT_WORD_TEMPL_DISPLAY_NAME         = OUSTR("Microsoft Word Template");
const OUString DEFAULT_EXCEL_SHEET_DISPLAY_NAME        = OUSTR("Microsoft Excel Worksheet");
const OUString DEFAULT_EXCEL_TEMPL_DISPLAY_NAME        = OUSTR("Microsoft Excel Template");
const OUString DEFAULT_POWERPOINT_PRESENT_DISPLAY_NAME = OUSTR("Microsoft PowerPoint Presentation");
const OUString DEFAULT_POWERPOINT_TEMPL_DISPLAY_NAME   = OUSTR("Microsoft PowerPoint Template");

const OUString DEFAULT_NEW_CMD_DISPLAY_NAME = OUSTR("&New");

const OUString WORD_DOC_ICON_INDEX       = OUSTR(",1");
const OUString WORD_TEMPL_ICON_INDEX     = OUSTR(",2");

const OUString EXCEL_SHEET_ICON_INDEX    = OUSTR(",3");
const OUString EXCEL_TEMPL_ICON_INDEX    = OUSTR(",4");

const OUString POWERP_PRESENT_ICON_INDEX = OUSTR(",7");
const OUString POWERP_TEMPL_ICON_INDEX   = OUSTR(",8");

// different constants

const OUString REG_KEY_FORWARD_PREFIX = OUSTR("soffice6");

const OUString REG_KEY_SHELL         = OUSTR("shell");
const OUString REG_KEY_SHELL_NEW     = OUSTR("shell\\new");
const OUString REG_KEY_SHELL_OPEN    = OUSTR("shell\\open");
const OUString REG_KEY_SHELL_PRINT   = OUSTR("shell\\print");
const OUString REG_KEY_SHELL_PRINTTO = OUSTR("shell\\printto");

const OUString REG_KEY_NEW  = OUSTR("new");
const OUString REG_KEY_OPEN = OUSTR("open");

const OUString REG_KEY_COMMAND = OUSTR("command");

const OUString NEW_CMD_PARAM  = OUString::createFromAscii(" -n \"%1\"");
const OUString OPEN_CMD_PARAM = OUString::createFromAscii(" -o \"%1\"");

const OUString PRINT_CMD_PARAM   = OUString::createFromAscii(" -p \"%1\"");
const OUString PRINTTO_CMD_PARAM = OUString::createFromAscii(" -pt \"%2\" \"%1\"");

const OUString REG_KEY_DEFAULT_ICON     = OUSTR("DefaultIcon");
const OUString VALUE_FRIENDLY_APP_NAME  = OUSTR("FriendlyAppName");
const OUString REG_KEY_SAVE_USER_PREF   = OUSTR("SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Uninstall\\");
const OUString REG_VALUE_REG4MSDOCSTATE = OUSTR("Reg4MsDocState");

const OUString BACKUP_VALUE_NAME = OUString::createFromAscii("Backup");

const OUString BACKUP_KEY_NAME			  = OUSTR("Software\\Classes\\soffice.bak");
const OUString BACKUP_KEY_DEF_HTML_EDITOR = OUSTR("Software\\Classes\\soffice.bak\\Default HTML Editor");
const OUString BACKUP_KEY_HTML_EDIT       = OUSTR("Software\\Classes\\soffice.bak\\HTML Edit");

const OUString DEFAULT_REG_KEY = OUSTR("");
const OUString EMPTY_STRING    = OUSTR("");
const OUString QUOTATION_MARK  = OUSTR("\"");

// shortcuts for some registry 

const HKEY HKCR = HKEY_CLASSES_ROOT;
const HKEY HKCU = HKEY_CURRENT_USER;
const HKEY HKLM = HKEY_LOCAL_MACHINE;

const sal_Unicode TILDE     = '~';
const sal_Unicode AMPERSAND = '&';

//#########################################

/** Makes String -> "String"
*/
OUString QuoteString(const OUString& aString)
{
	return (QUOTATION_MARK + aString + QUOTATION_MARK);
}

//#########################################

/**
	@descr	An exception that might be thrown in cases of registry
			errors.
*/

class CRegistryException : public std::exception
{
public:
	/** 	Constructor
	
		@param	ErrorCode 
				A Win32 Error code.
	*/
	
	CRegistryException(long ErrorCode);
	
	/**	Destructor
	*/
	
	~CRegistryException();
	
	/**
		@descr	Returns a string that describes the error if
				available, else NULL will be returned. The
				returned string is only temporary so the caller
				has to copy it if he needs the string further.
	*/
	virtual const char* what() const throw();
	
	/**
		@descr	Returns the error code.
	*/
	
	long getErrorCode() const throw();
	
private:
	long m_ErrorCode;
	void* m_lpErrorMsg;
};

/**
*/
CRegistryException::CRegistryException(long ErrorCode) :
	m_ErrorCode(ErrorCode),
	m_lpErrorMsg(0)
{
}

/**
*/
CRegistryException::~CRegistryException()
{
	if (m_lpErrorMsg)
		LocalFree(m_lpErrorMsg);
}

/**
*/
const char* CRegistryException::what() const throw()
{
	FormatMessage( 
		FORMAT_MESSAGE_ALLOCATE_BUFFER | 
		FORMAT_MESSAGE_FROM_SYSTEM | 
		FORMAT_MESSAGE_IGNORE_INSERTS,
		NULL,
		m_ErrorCode,
		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
		(LPTSTR) &m_lpErrorMsg,
		0,
		NULL);
		
	return reinterpret_cast<char*>(m_lpErrorMsg);
}

/**
*/
long CRegistryException::getErrorCode() const throw()
{
	return m_ErrorCode;
}

//#########################################

/** Open a registry key, create it if does not yet exist
*/
void EnsureRegKeyOpen(CRegistryKeyPtr& aRegKey, HKEY hRootKey, const OUString& aSubKeyName, REGSAM samDesired)
	throw(CRegistryException)
{    
	aRegKey.reset(CRegistryKey::Open(hRootKey, aSubKeyName, samDesired));

	if (!aRegKey->IsOpen())
	{
		long rc = aRegKey->Create();
		if (ERROR_SUCCESS != rc)
			throw CRegistryException(rc);
	}
}

/**
*/
void DeleteRegKey(HKEY RootKey, const OUString& Key) throw(CRegistryException)
{
	CRegistryKeyPtr RegKey(CRegistryKey::Open(RootKey, Key, KEY_READ | DELETE));
	long rc = RegKey->Delete();
	
	if (ERROR_SUCCESS != rc)
		throw CRegistryException(rc);
}

/**
*/
bool DeleteRegKeyNoThrow(HKEY RootKey, const OUString& Key) throw()
{
	bool bRet = true;
	
	try
	{
		DeleteRegKey(RootKey, Key);
	}
	catch(CRegistryException& ex)
	{
		bRet = false;
		OSL_ENSURE(false, ex.what());
	}
	
	return bRet;
}

/** Retrievs a value from a secified registry key
*/
void GetRegValue(HKEY RootKey, const OUString& Key, const OUString& ValueName, /* out */ OUString& Value)
	throw(CRegistryException)
{
	CRegistryKeyPtr RegKey(CRegistryKey::Open(RootKey, Key, KEY_READ));
		
	long rc = RegKey->GetValue(ValueName, Value);
	if (ERROR_SUCCESS != rc)
		throw CRegistryException(rc);
}

/** Retrievs a value from a secified registry key
*/
void GetRegValue(HKEY RootKey, const OUString& Key, const OUString& ValueName, const OUString& DefaultReturn, /* out */ OUString& Value)
	throw(CRegistryException)
{
	CRegistryKeyPtr RegKey(CRegistryKey::Open(RootKey, Key, KEY_READ));
		
	if (ERROR_SUCCESS != RegKey->GetValue(ValueName, Value))
		Value = DefaultReturn;
}

/** Retrievs a value from a secified registry key
*/
void GetRegValue(HKEY RootKey, const OUString& Key, const OUString& ValueName, /* out */ int& Value)
	throw(CRegistryException)
{
	CRegistryKeyPtr RegKey(CRegistryKey::Open(RootKey, Key, KEY_READ));
		
	long rc = RegKey->GetValue(ValueName, &Value);
	if (ERROR_SUCCESS != rc)
		throw CRegistryException(rc);
}

/** Retrievs a value from a secified registry key
*/
void GetRegValue(HKEY RootKey, const OUString& Key, const OUString& ValueName, int DefaultReturn, /* out */ int& Value)
	throw(CRegistryException)
{
	CRegistryKeyPtr RegKey(CRegistryKey::Open(RootKey, Key, KEY_READ));
		
	if (ERROR_SUCCESS != RegKey->GetValue(ValueName, &Value))
		Value = DefaultReturn;
}

/**
*/
void SetRegValue(HKEY RootKey, const OUString& Key, const OUString& ValueName, const OUString& Value, const OUString& DefaultValue = OUString())
	throw(CRegistryException)
{
	CRegistryKeyPtr RegKey(CRegistryKey::Open(RootKey, Key, KEY_WRITE));
		
	long rc = RegKey->SetValue(ValueName, Value, DefaultValue);
	if (ERROR_SUCCESS != rc)
		throw CRegistryException(rc);
}

/**
*/
void SetRegValueMandatory(HKEY RootKey, const OUString& Key, const OUString& ValueName, const OUString& Value, const OUString& DefaultValue = OUString())
	throw(CRegistryException)
{
	CRegistryKeyPtr RegKey;
	EnsureRegKeyOpen(RegKey, RootKey, Key, KEY_WRITE);
	
	long rc = RegKey->SetValue(ValueName, Value, DefaultValue);
	if (ERROR_SUCCESS != rc)
		throw CRegistryException(rc);
}

/**
*/
void SetRegValue(HKEY RootKey, const OUString& Key, const OUString& ValueName, int Value) 
	throw(CRegistryException)
{
	CRegistryKeyPtr RegKey(CRegistryKey::Open(RootKey, Key, KEY_WRITE));
		
	long rc = RegKey->SetValue(ValueName, Value);
	if (ERROR_SUCCESS != rc)
		throw CRegistryException(rc);
}

/**
*/
void SetRegValueMandatory(HKEY RootKey, const OUString& Key, const OUString& ValueName, int Value) 
	throw(CRegistryException)
{
	CRegistryKeyPtr RegKey;
	EnsureRegKeyOpen(RegKey, RootKey, Key, KEY_WRITE);
	
	long rc = RegKey->SetValue(ValueName, Value);
	if (ERROR_SUCCESS != rc)
		throw CRegistryException(rc);
}

/** Compares a registry value with a given value
	
	@param	RootKey
			The root key, e.g. HKEY_CLASSES_ROOT.
			
	@param	Key
			The key name.
			
	@param	ValueName
			The name of the registry value to compare with.
			
	@param	CompareValue
			The value to compare with the registry value.
			
	@throws	CRegistryException
			If the Key doesn't exist, the value doesn't exist or
			any other registry error occurs.
*/
bool RegValueEquals(HKEY RootKey, const OUString& Key, const OUString& ValueName, const OUString& CompareValue) 
	throw(CRegistryException)
{
	OUString RegValue;
	GetRegValue(RootKey, Key, ValueName, RegValue);
	return (0 == RegValue.compareTo(CompareValue, CompareValue.getLength()));
}

/** Checks if a registry value contains a certain sub-value
*/
bool RegValueContains(HKEY RootKey, const OUString& Key, const OUString& ValueName, const OUString& CompareValue)
	throw (CRegistryException)
{
	OUString RegValue;
	GetRegValue(RootKey, Key, ValueName, RegValue);	
	return (RegValue.indexOf(CompareValue) >= 0);
}

/**	Copies the value of registry key to another location
	
	@descr	Copies the value of a registry key to another 
			location
			
	@param	SrcRootKey
			The source root key, e.g. HKEY_CLASSES_ROOT

	@param	SrcKey
			The name of the key where to value exist

	@param	SrcValueName 
			The name of the value to copy

	@param	DestRootKey
			The destination root key, e.g. HKEY_CLASSES_ROOT

	@param	DestKey
			The name of the key where to place the value

	@param	DestValueName
			An alternatively name under which to save the 
			SrcValue

	@throw	CRegistryException
			When the source value doesn't exist
			When the destination key doesn't exist 
			When a security error occurs, like access denied etc.
			Any other registry error occurs
*/

void CopyRegValue(
	const HKEY SrcRootKey,
	const OUString& SrcKey,
	const OUString& SrcValueName,
	const HKEY DestRootKey,
	const OUString& DestKey,
	const OUString& DestValueName) throw(CRegistryException)
{
	OUString SrcValue;
	GetRegValue(SrcRootKey, SrcKey, SrcValueName, SrcValue);
	
	CRegistryKeyPtr destKey(CRegistryKey::Open(DestRootKey, DestKey, KEY_WRITE));
	
	long rc = destKey->SetValue(DestValueName, SrcValue);
	
	if (ERROR_SUCCESS != rc)
		throw CRegistryException(rc);
}

/**	Copies the value of registry key to another location
	
	@descr	Copies the value of a registry key to another 
			location
			
	@param	SrcRootKey
			The source root key, e.g. HKEY_CLASSES_ROOT

	@param	SrcKey
			The name of the key where to value exist

	@param	SrcValueName 
			The name of the value to copy

	@param	SrcDefaultValue
			A default value that will be used if the source value
			doesn't exist

	@param	DestRootKey
			The destination root key, e.g. HKEY_CLASSES_ROOT

	@param	DestKey
			The name of the key where to place the value

	@param	DestValueName
			An alternatively name under which to save the 
			SrcValue

	@throw	CRegistryException
			When the source value doesn't exist
			When the destination key doesn't exist 
			When a security error occurs, like access denied etc.
			Any other registry error occurs
*/

void CopyRegValue(
	const HKEY SrcRootKey,
	const OUString& SrcKey,
	const OUString& SrcValueName,
	const OUString& SrcDefaultValue,
	const HKEY DestRootKey,
	const OUString& DestKey,
	const OUString& DestValueName) throw(CRegistryException)
{
	OUString SrcValue;
	GetRegValue(SrcRootKey, SrcKey, SrcValueName, SrcDefaultValue, SrcValue);
	
	CRegistryKeyPtr destKey(CRegistryKey::Open(DestRootKey, DestKey, KEY_WRITE));
	
	long rc = destKey->SetValue(DestValueName, SrcValue);
	
	if (ERROR_SUCCESS != rc)
		throw CRegistryException(rc);
}

/** Copies the value of the specified registry key to another location, if this
	location doesn't exist, it will be created.
*/
void CopyRegValueMandatory(
	const HKEY SrcRootKey,
	const OUString& SrcKey,
	const OUString& SrcValueName,
	const HKEY DestRootKey,
	const OUString& DestKey,
	const OUString& DestValueName) throw(CRegistryException)
{
	OUString SrcValue;
	GetRegValue(SrcRootKey, SrcKey, SrcValueName, SrcValue);
	
	CRegistryKeyPtr RegKey;
	EnsureRegKeyOpen(RegKey, DestRootKey, DestKey, KEY_WRITE);
	
	long rc = RegKey->SetValue(DestValueName, SrcValue);
	
	if (ERROR_SUCCESS != rc)
		throw CRegistryException(rc);
}

/** Copies the value of the specified registry key to another location, if this
	location doesn't exist, it will be created.
*/
void CopyRegValueMandatory(
	const HKEY SrcRootKey,
	const OUString& SrcKey,
	const OUString& SrcValueName,
	const OUString& SrcDefaultValue,
	const HKEY DestRootKey,
	const OUString& DestKey,
	const OUString& DestValueName) throw(CRegistryException)
{
	OUString SrcValue;
	GetRegValue(SrcRootKey, SrcKey, SrcValueName, SrcDefaultValue, SrcValue);
	
	CRegistryKeyPtr RegKey;
	EnsureRegKeyOpen(RegKey, DestRootKey, DestKey, KEY_WRITE);
	
	long rc = RegKey->SetValue(DestValueName, SrcValue);
	
	if (ERROR_SUCCESS != rc)
		throw CRegistryException(rc);
}

//-----------------------------------------
// Save the content of SrcValue into
// DestKey\DestValue
// if SrcKey doesn't exist, an empty
// value will be saved 
//-----------------------------------------

void BackupForwardValue(
	const HKEY SrcRootKey, 
	const OUString& SrcKey, 
	const OUString& SrcValueName, 
	const HKEY DestRootKey,
	const OUString& DestKey,
	const OUString& DestValueName) throw()
{
	try
	{ 
		// we don't want to backup a 'soffice.?' in cases where a 6.0 was installed before and registered 
		// for the MS document types
		if (!RegValueEquals(SrcRootKey, SrcKey, DEFAULT_REG_KEY, REG_KEY_FORWARD_PREFIX)) 
			CopyRegValueMandatory(SrcRootKey, SrcKey, DEFAULT_REG_KEY, DestRootKey, DestKey,DestValueName);
	}
	catch(CRegistryException&) 
	{
	}
}

//-----------------------------------------
// Restore 
//-----------------------------------------

void RestoreForwardValue(
	const HKEY SrcRootKey, 
	const OUString& SrcKey, 
	const OUString& SrcValue, 
	const HKEY DestRootKey,
	const OUString& DestKey,
	const OUString& DestValue) throw()
{
	try
	{
		OUString DefaultSrcValue;
		if (RegValueEquals(DestRootKey, DestKey, DEFAULT_REG_KEY, REG_KEY_FORWARD_PREFIX))
			CopyRegValue(SrcRootKey, SrcKey, SrcValue, DefaultSrcValue, DestRootKey, DestKey, DEFAULT_REG_KEY);			
	}
	catch(CRegistryException&)
	{		
	}
}

/** Make the necessary registry entries to register 
	OO for Word documents
*/
void Register4MsWord(
	const OUString& OfficeExePath, 
	const OUString& DocDisplayName, 
	const OUString& TemplDisplayName, 
	const OUString& NewCmdDisplayName) throw(CRegistryException)
{
	// register for word documents
	
	OUString OfficeExePathQuoted = QuoteString(OfficeExePath);

	try
	{		
		SetRegValueMandatory(HKCR, OUSTR("soffice6.doc"), DEFAULT_REG_KEY, DocDisplayName, DEFAULT_WORD_DOC_DISPLAY_NAME);		
		SetRegValueMandatory(HKCR, OUSTR("soffice6.doc\\DefaultIcon"), DEFAULT_REG_KEY, OfficeExePath + WORD_DOC_ICON_INDEX);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.doc\\shell"), DEFAULT_REG_KEY, OUSTR("open"));
		SetRegValueMandatory(HKCR, OUSTR("soffice6.doc\\shell\\new"), DEFAULT_REG_KEY, NewCmdDisplayName, DEFAULT_NEW_CMD_DISPLAY_NAME);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.doc\\shell\\new\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + NEW_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.doc\\shell\\open\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + OPEN_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.doc\\shell\\print\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + PRINT_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.doc\\shell\\printto\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + PRINTTO_CMD_PARAM);	

		BackupForwardValue(HKCR, OUSTR(".doc"), DEFAULT_REG_KEY, HKCR, OUSTR("soffice6.doc"), OUSTR("Backup"));
		SetRegValueMandatory(HKCR, OUSTR(".doc"), DEFAULT_REG_KEY, OUSTR("soffice6.doc"));
	}
	catch(CRegistryException& ex)
	{
		DeleteRegKeyNoThrow(HKCR, OUSTR("soffice6.doc"));
		OSL_ENSURE(false, ex.what());
		throw;
	}
	
	// register for word templates 
	
	try
	{
		SetRegValueMandatory(HKCR, OUSTR("soffice6.dot"), DEFAULT_REG_KEY, TemplDisplayName, DEFAULT_WORD_TEMPL_DISPLAY_NAME);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.dot\\DefaultIcon"), DEFAULT_REG_KEY, OfficeExePath + WORD_TEMPL_ICON_INDEX);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.dot\\shell"), DEFAULT_REG_KEY, OUSTR("new"));
		SetRegValueMandatory(HKCR, OUSTR("soffice6.dot\\shell\\new"), DEFAULT_REG_KEY, NewCmdDisplayName, DEFAULT_NEW_CMD_DISPLAY_NAME);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.dot\\shell\\new\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + NEW_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.dot\\shell\\open\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + OPEN_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.dot\\shell\\print\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + PRINT_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.dot\\shell\\printto\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + PRINTTO_CMD_PARAM);	

		BackupForwardValue(HKCR, OUSTR(".dot"), DEFAULT_REG_KEY, HKCR, OUSTR("soffice6.dot"), OUSTR("Backup"));
		SetRegValueMandatory(HKCR, OUSTR(".dot"), DEFAULT_REG_KEY, OUSTR("soffice6.dot"));
	}
	catch(CRegistryException& ex)
	{
		DeleteRegKeyNoThrow(HKCR, OUSTR("soffice6.dot"));
		OSL_ENSURE(false, ex.what());
		throw;
	}
}

/** Make the necessary registry entries to register 
	OO for Word documents
*/
void Register4MsExcel(
	const OUString& OfficeExePath, 
	const OUString& DocDisplayName, 
	const OUString& TemplDisplayName, 
	const OUString& NewCmdDisplayName) throw(CRegistryException)
{
	// register for word documents
	
	OUString OfficeExePathQuoted = QuoteString(OfficeExePath);

	try
	{		
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xls"), DEFAULT_REG_KEY, DocDisplayName, DEFAULT_EXCEL_SHEET_DISPLAY_NAME);		
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xls\\DefaultIcon"), DEFAULT_REG_KEY, OfficeExePath + EXCEL_SHEET_ICON_INDEX);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xls\\shell"), DEFAULT_REG_KEY, OUSTR("open"));
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xls\\shell\\new"), DEFAULT_REG_KEY, NewCmdDisplayName, DEFAULT_NEW_CMD_DISPLAY_NAME);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xls\\shell\\new\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + NEW_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xls\\shell\\open\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + OPEN_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xls\\shell\\print\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + PRINT_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xls\\shell\\printto\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + PRINTTO_CMD_PARAM);	

		BackupForwardValue(HKCR, OUSTR(".xls"), DEFAULT_REG_KEY, HKCR, OUSTR("soffice6.xls"), OUSTR("Backup"));
		SetRegValueMandatory(HKCR, OUSTR(".xls"), DEFAULT_REG_KEY, OUSTR("soffice6.xls"));
	}
	catch(CRegistryException& ex)
	{
		DeleteRegKeyNoThrow(HKCR, OUSTR("soffice6.xls"));
		OSL_ENSURE(false, ex.what());
		throw;
	}
	
	// register for word templates 
	
	try
	{
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xlt"), DEFAULT_REG_KEY, TemplDisplayName, DEFAULT_EXCEL_TEMPL_DISPLAY_NAME);		
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xlt\\DefaultIcon"), DEFAULT_REG_KEY, OfficeExePath + EXCEL_TEMPL_ICON_INDEX);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xlt\\shell"), DEFAULT_REG_KEY, OUSTR("new"));
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xlt\\shell\\new"), DEFAULT_REG_KEY, NewCmdDisplayName, DEFAULT_NEW_CMD_DISPLAY_NAME);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xlt\\shell\\new\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + NEW_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xlt\\shell\\open\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + OPEN_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xlt\\shell\\print\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + PRINT_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.xlt\\shell\\printto\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + PRINTTO_CMD_PARAM);	

		BackupForwardValue(HKCR, OUSTR(".xlt"), DEFAULT_REG_KEY, HKCR, OUSTR("soffice6.xlt"), OUSTR("Backup"));
		SetRegValueMandatory(HKCR, OUSTR(".xlt"), DEFAULT_REG_KEY, OUSTR("soffice6.xlt"));
	}
	catch(CRegistryException& ex)
	{
		DeleteRegKeyNoThrow(HKCR, OUSTR("soffice6.xlt"));
		OSL_ENSURE(false, ex.what());
		throw;
	}
}

/** Make the necessary registry entries to register 
	OO for Word documents
*/
void Register4MsPowerPoint(
	const OUString& OfficeExePath, 
	const OUString& DocDisplayName, 
	const OUString& TemplDisplayName, 
	const OUString& NewCmdDisplayName) throw(CRegistryException)
{
	// register for word documents
	
	OUString OfficeExePathQuoted = QuoteString(OfficeExePath);

	try
	{		
		SetRegValueMandatory(HKCR, OUSTR("soffice6.ppt"), DEFAULT_REG_KEY, DocDisplayName, DEFAULT_POWERPOINT_PRESENT_DISPLAY_NAME);		
		SetRegValueMandatory(HKCR, OUSTR("soffice6.ppt\\DefaultIcon"), DEFAULT_REG_KEY, OfficeExePath + POWERP_PRESENT_ICON_INDEX);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.ppt\\shell"), DEFAULT_REG_KEY, OUSTR("open"));
		SetRegValueMandatory(HKCR, OUSTR("soffice6.ppt\\shell\\new"), DEFAULT_REG_KEY, NewCmdDisplayName, DEFAULT_NEW_CMD_DISPLAY_NAME);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.ppt\\shell\\new\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + NEW_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.ppt\\shell\\open\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + OPEN_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.ppt\\shell\\print\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + PRINT_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.ppt\\shell\\printto\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + PRINTTO_CMD_PARAM);	
		
		BackupForwardValue(HKCR, OUSTR(".ppt"), DEFAULT_REG_KEY, HKCR, OUSTR("soffice6.ppt"), OUSTR("Backup"));
		SetRegValueMandatory(HKCR, OUSTR(".ppt"), DEFAULT_REG_KEY, OUSTR("soffice6.ppt"));
	}
	catch(CRegistryException& ex)
	{
		DeleteRegKeyNoThrow(HKCR, OUSTR("soffice6.ppt"));
		OSL_ENSURE(false, ex.what());
		throw;
	}
	
	// register for word templates 
	
	try
	{
		SetRegValueMandatory(HKCR, OUSTR("soffice6.pot"), DEFAULT_REG_KEY, TemplDisplayName, DEFAULT_POWERPOINT_TEMPL_DISPLAY_NAME);		
		SetRegValueMandatory(HKCR, OUSTR("soffice6.pot\\DefaultIcon"), DEFAULT_REG_KEY, OfficeExePath + POWERP_TEMPL_ICON_INDEX);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.pot\\shell"), DEFAULT_REG_KEY, OUSTR("new"));
		SetRegValueMandatory(HKCR, OUSTR("soffice6.pot\\shell\\new"), DEFAULT_REG_KEY, NewCmdDisplayName, DEFAULT_NEW_CMD_DISPLAY_NAME);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.pot\\shell\\new\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + NEW_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.pot\\shell\\open\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + OPEN_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.pot\\shell\\print\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + PRINT_CMD_PARAM);
		SetRegValueMandatory(HKCR, OUSTR("soffice6.pot\\shell\\printto\\command"), DEFAULT_REG_KEY, OfficeExePathQuoted + PRINTTO_CMD_PARAM);	
		
		BackupForwardValue(HKCR, OUSTR(".pot"), DEFAULT_REG_KEY, HKCR, OUSTR("soffice6.pot"), OUSTR("Backup"));
		SetRegValueMandatory(HKCR, OUSTR(".pot"), DEFAULT_REG_KEY, OUSTR("soffice6.pot"));
	}
	catch(CRegistryException& ex)
	{
		DeleteRegKeyNoThrow(HKCR, OUSTR("soffice6.pot"));
		OSL_ENSURE(false, ex.what());
		throw;
	}
}

/**
	saves/retrieve the current settings in 
	the windows registry
	what we save is a number which will be 
	interpreted a bit field
	0x1 means registered for MS Word
	0x2 means registered for MS Excel
	0x4 means registered for MS PowerPoint
	0x8 means registered for Default Html Edt.
*/
void SetUserPreferences(int Prefs) throw(CRegistryException)
{
	SetRegValueMandatory(
		REG_KEY_PLUGIN_PRIVATE_ROOT, 
		REG_KEY_PLUGIN_PRIVATE, 
		REG_VALUE_REG4MSDOCSTATE, 
		Prefs);
}

/**
*/
int GetUserPreferences() throw(CRegistryException)
{
	int Prefs = 0;

	CRegistryKeyPtr RegKey(
		CRegistryKey::Open(
			REG_KEY_PLUGIN_PRIVATE_ROOT,
			REG_KEY_PLUGIN_PRIVATE, 
			KEY_READ));

	if (RegKey->IsOpen())
	{
		GetRegValue(
			REG_KEY_PLUGIN_PRIVATE_ROOT, 
			REG_KEY_PLUGIN_PRIVATE, 
			REG_VALUE_REG4MSDOCSTATE, 
			0, 
			Prefs);	
	}

	return Prefs;
}

//--------------------------------------------
// Registration as requires the following
// registry keys:
// .htm\OpenWithList\soffice.exe
// Applications\soffice.exe\shell\edit\command
//--------------------------------------------

void Register4HtmlEdt(const OUString& OfficeExePath, const OUString& aFriendlyAppName)
	throw(CRegistryException)
{   
	try
	{
		SetRegValueMandatory(HKCR, OUSTR(".htm\\OpenWithList\\soffice.exe"), DEFAULT_REG_KEY, OUString());
		SetRegValueMandatory(HKCR, OUSTR("Applications\\soffice.exe"), OUSTR("FriendlyAppName"), aFriendlyAppName);
		SetRegValueMandatory(HKCR, OUSTR("Applications\\soffice.exe\\shell\\edit\\command"), DEFAULT_REG_KEY, QuoteString(OfficeExePath) + OPEN_CMD_PARAM);
	}
	catch(CRegistryException& ex)
	{
		DeleteRegKeyNoThrow(HKCR, OUSTR(".htm\\OpenWithList\\soffice.exe"));
		DeleteRegKeyNoThrow(HKCR, OUSTR("Applications\\soffice.exe"));
		OSL_ENSURE(false, ex.what());
		throw;
	}	  
}

/** Backup the registry values for default html editor in Internet Explorer

	@descr	Only backup a value if it not yet our replaced value
*/
void BackupDefHtmlEditorRegValues()
{
	try
	{
		if (!RegValueContains(
			HKCU, 
			OUSTR("Software\\Microsoft\\Internet Explorer\\Default HTML Editor"), 
			OUSTR("Description"), 
			PRODUCT_NAME))
		{
			CopyRegValueMandatory(
				HKCU,
				OUSTR("Software\\Microsoft\\Internet Explorer\\Default HTML Editor"), 
				OUSTR("Description"),
				HKCR,
				REG_KEY_PLUGIN_PRIVATE + OUSTR("\\Default HTML Editor"),
				OUSTR("Description"));
		}
	}
	catch(CRegistryException& ex)
	{
		OSL_ENSURE(false, ex.what());
	}
	
	try
	{
		if (!RegValueContains(
			HKCU,
			OUSTR("Software\\Microsoft\\Internet Explorer\\Default HTML Editor\\shell\\edit\\command"),
			DEFAULT_REG_KEY,
			PRODUCT_EXECUTABLE_NAME))
		{
			CopyRegValueMandatory(
				HKCU, 
				OUSTR("Software\\Microsoft\\Internet Explorer\\Default HTML Editor\\shell\\edit\\command"), 
				DEFAULT_REG_KEY, 
				HKCR, 
				REG_KEY_PLUGIN_PRIVATE + OUSTR("\\Default HTML Editor"),
				DEFAULT_REG_KEY);
		}
	}
	catch(CRegistryException& ex)
	{
		OSL_ENSURE(false, ex.what());
	}
}

//-----------------------------------------------------------------------------------
//
//-----------------------------------------------------------------------------------

void RestoreDefHtmlEditorRegValues()
{
	try
	{
		if (RegValueContains(
			HKCU, 
			OUSTR("Software\\Microsoft\\Internet Explorer\\Default HTML Editor"), 
			OUSTR("Description"), 
			PRODUCT_NAME))
		{
			CopyRegValue(
				HKCR,
				REG_KEY_PLUGIN_PRIVATE + OUSTR("\\Default HTML Editor"),
				OUSTR("Description"),
				OUString(),
				HKCU,
				OUSTR("Software\\Microsoft\\Internet Explorer\\Default HTML Editor"),
				OUSTR("Description"));
		}
	}
	catch(CRegistryException& ex)
	{
		OSL_ENSURE(false, ex.what());
	}
	
	try
	{
		if (RegValueContains(
			HKCU,
			OUSTR("Software\\Microsoft\\Internet Explorer\\Default HTML Editor\\shell\\edit\\command"),
			DEFAULT_REG_KEY,
			PRODUCT_EXECUTABLE_NAME))
		{
			CopyRegValue(
				HKCR, 
				REG_KEY_PLUGIN_PRIVATE + OUSTR("\\Default HTML Editor"),
				DEFAULT_REG_KEY, 
				OUString(),
				HKCU, 
				OUSTR("Software\\Microsoft\\Internet Explorer\\Default HTML Editor\\shell\\edit\\command"),
				DEFAULT_REG_KEY);
		}
	}
	catch(CRegistryException& ex)
	{
		OSL_ENSURE(false, ex.what());
	}
}

//-----------------------------------------------------------------------------------
//
//-----------------------------------------------------------------------------------

void RestoreHtmlEditRegValues() throw()
{
	try
	{		
		OUString HtmFwdKeyName;
		GetRegValue(HKCR, OUSTR(".htm"), DEFAULT_REG_KEY, HtmFwdKeyName);

		CRegistryKeyPtr RegKeyHtmFwd(CRegistryKey::Open(HKCR, HtmFwdKeyName, KEY_READ | KEY_WRITE));

		OUString HtmKeyName;

		if (HtmFwdKeyName.getLength() && RegKeyHtmFwd->IsOpen())
			HtmKeyName = HtmFwdKeyName;
		else
			HtmKeyName = OUSTR(".htm");

		// check if we are registered yet
		if (RegValueContains(
			HKCR, 
			HtmKeyName + OUSTR("\\shell\\edit\\command"), 
			DEFAULT_REG_KEY, 
			PRODUCT_EXECUTABLE_NAME))
		{
			// restore old value			
			CopyRegValue(							
				HKCR,
				REG_KEY_PLUGIN_PRIVATE + OUSTR("\\HTML Edit"),
				OUSTR("Shell Edit Cmd"),
				OUString(),
				HKCR,
				HtmKeyName + OUSTR("\\shell\\edit\\command"), 
				DEFAULT_REG_KEY);

			// restore old value
			CopyRegValue(				
				HKCR,
				REG_KEY_PLUGIN_PRIVATE + OUSTR("\\HTML Edit"),
				OUSTR("Edit Display Name"),
				OUString(),
				HKCR,
				HtmKeyName + OUSTR("\\shell\\edit"), 
				DEFAULT_REG_KEY);
		}
	}
	catch(CRegistryException& ex)
	{
		OSL_ENSURE(false, ex.what());
	}
}

//-----------------------------------------------------------------------------------
// Registration as default html editor
// requires the following registry entries:
// Software\Microsoft\Internet Explorer\Default HTML Editor -> Value: Description
// Software\Microsoft\Internet Explorer\Default HTML Editor\shell\edit\command
// 
//-----------------------------------------------------------------------------------

void Register4DefHtmlEdt( const OUString& aShellEditCommand, const OUString& aFriendlyAppName, const OUString& aEditMenuName )
{
	// register as default html editor for Internet Explorer
    try
	{
		BackupDefHtmlEditorRegValues();
    
		SetRegValueMandatory(
			HKCU, 
			OUSTR("Software\\Microsoft\\Internet Explorer\\Default HTML Editor"), 
			OUSTR("Description"), 
			aFriendlyAppName);

		SetRegValueMandatory(
			HKCU, 
			OUSTR("Software\\Microsoft\\Internet Explorer\\Default HTML Editor\\shell\\edit\\command"), 
			DEFAULT_REG_KEY, 
			QuoteString(aShellEditCommand) + OPEN_CMD_PARAM);
	}
	catch(CRegistryException&)
	{
	}
    

    // register as default html editor for the Windows Explorer    
    
	try
	{
		OUString HtmFwdKeyName;
		GetRegValue(HKCR, OUSTR(".htm"), DEFAULT_REG_KEY, HtmFwdKeyName);

		CRegistryKeyPtr RegKeyHtmFwd(CRegistryKey::Open(HKCR, HtmFwdKeyName, KEY_READ | KEY_WRITE));

		OUString HtmKeyName;

		if (HtmFwdKeyName.getLength() && RegKeyHtmFwd->IsOpen())
			HtmKeyName = HtmFwdKeyName;
		else
			HtmKeyName = OUSTR(".htm");

		if (!RegValueContains(
				HKCR, 
				HtmKeyName + OUSTR("\\shell\\edit\\command"), 
				DEFAULT_REG_KEY, 
				PRODUCT_EXECUTABLE_NAME))
		{
			// backup old value			
			CopyRegValueMandatory(
				HKCR,
				HtmKeyName + OUSTR("\\shell\\edit\\command"), 
				DEFAULT_REG_KEY,
				OUString(),
				HKCR,
				REG_KEY_PLUGIN_PRIVATE + OUSTR("\\HTML Edit"),
				OUSTR("Shell Edit Cmd"));
			
			// backup old value
			CopyRegValueMandatory(
				HKCR,
				HtmKeyName + OUSTR("\\shell\\edit"), 
				DEFAULT_REG_KEY,
				OUString(),
				HKCR,
				REG_KEY_PLUGIN_PRIVATE + OUSTR("\\HTML Edit"),
				OUSTR("Edit Display Name"));
		}

		SetRegValueMandatory(
			HKCR, 
			HtmKeyName + OUSTR("\\shell\\edit\\command"), 
			DEFAULT_REG_KEY, 
			QuoteString(aShellEditCommand) + OPEN_CMD_PARAM);

		SetRegValueMandatory(
			HKCR,
			HtmKeyName + OUSTR("\\shell\\edit"),
			DEFAULT_REG_KEY,
			aEditMenuName);					
	}
	catch(CRegistryException&)
	{
	}
}

//-----------------------------------------
// registers StarOffice for MS document
// types and as default HTML editor if
// specified
//-----------------------------------------

void Register4MsDoc(sal_Int32 aUserPref, const OUString& aOfficeExePath, ResMgr* aResMgr)
{
    OSL_PRECOND(aOfficeExePath.getLength(), "invalid program path");
        
    OUString aProdName = String(ResId(STR_PRODUCTNAME, aResMgr));

    OUString aNewDispName = String(ResId(STR_NEW_DISPLAY_NAME, aResMgr));
    aNewDispName = aNewDispName.replace(TILDE, AMPERSAND);

	int CurrentUserPrefs = 0;

	try
	{
		CurrentUserPrefs = GetUserPreferences();
	}
	catch(CRegistryException& ex)
	{
		OSL_ENSURE(false, ex.what());
	}

    if (aUserPref & REG4MSWORD)
    {
		try
		{
			Register4MsWord(
				aOfficeExePath,
				String(ResId(STR_MS_WORD_DOCUMENT, aResMgr)),
				String(ResId(STR_MS_WORD_TEMPLATE, aResMgr)),
				aNewDispName);

			CurrentUserPrefs |= REG4MSWORD;
		}
		catch(CRegistryException& ex)
		{
			OSL_ENSURE(false, ex.what());
		}	
    }

    if (aUserPref & REG4MSEXCEL)
    {
		try
		{
			Register4MsExcel(
				aOfficeExePath,
				String(ResId(STR_MS_EXCEL_WORKSHEET, aResMgr)),
				String(ResId(STR_MS_EXCEL_TEMPLATE, aResMgr)),
				aNewDispName);

			CurrentUserPrefs |= REG4MSEXCEL;
		}
		catch(CRegistryException& ex)
		{
			OSL_ENSURE(false, ex.what());
		}		
    }

    if (aUserPref & REG4MSPOWERPOINT)
    {
		try
		{
			Register4MsPowerPoint(
				aOfficeExePath,
				String(ResId(STR_MS_POWERPOINT_PRESENTATION, aResMgr)),
				String(ResId(STR_MS_POWERPOINT_TEMPLATE, aResMgr)),
				aNewDispName);

			CurrentUserPrefs |= REG4MSPOWERPOINT;
		}
		catch(CRegistryException& ex)
		{
			OSL_ENSURE(false, ex.what());
		}		
    }
	
	if (aUserPref & REG4HTMLEDT)
	{
		try
		{
			Register4HtmlEdt(aOfficeExePath, aProdName); 

			CurrentUserPrefs |= REG4HTMLEDT;
		}
		catch(CRegistryException& ex)
		{
			OSL_ENSURE(false, ex.what());
		}
	}

    if (aUserPref & REG4DEFHTMLEDT)
    {
		try
		{
			OUString aEditMenuName = String(ResId(STR_EDIT, aResMgr));        
			aEditMenuName = aEditMenuName.replace(TILDE, AMPERSAND);
			
			Register4DefHtmlEdt(aOfficeExePath, aProdName, aEditMenuName);

			CurrentUserPrefs |= REG4DEFHTMLEDT;
		}
		catch(CRegistryException& ex)
		{
			OSL_ENSURE(false, ex.what());
		}
    }

    if (aUserPref > 0)
    {
		try
		{
			SetUserPreferences(CurrentUserPrefs);
		}
		catch(CRegistryException& ex)
		{
			OSL_ENSURE(false, ex.what());
		}
		
		SHChangeNotify(SHCNE_ASSOCCHANGED, SHCNF_IDLIST, 0, 0);    
    }
}

//-----------------------------------------
// restores the old value and deletes the
// forward key
//-----------------------------------------

void UnregisterDocType(const OUString& aDocType) throw()
{
	OUString aFwdKey(REG_KEY_FORWARD_PREFIX + aDocType);
	
	try
	{
		RestoreForwardValue(
			HKCR,
			aFwdKey,
			BACKUP_VALUE_NAME,
			HKCR,
			aDocType,
			DEFAULT_REG_KEY);

		DeleteRegKey(HKCR, aFwdKey);
	}
	catch(CRegistryException& ex)
	{
		OSL_ENSURE(false, ex.what());
	}
}

//-----------------------------------------
//
//-----------------------------------------

void Unregister4MsDoc(int UninstallModules, ResMgr* aResMgr)
{
	int UserPref = 0;

	try
	{
		UserPref = GetUserPreferences();
	}
	catch(CRegistryException& ex)
	{
		OSL_ENSURE(false, ex.what());
	}

	if (MOD_WRITER & UninstallModules)
	{
		if (UserPref & REG4MSWORD)
		{
			UnregisterDocType(REG_KEY_EXTENSION_DOC);
			UnregisterDocType(REG_KEY_EXTENSION_DOT);	
			
			UserPref &= ~REG4MSWORD;
		}
    
		if (UserPref & REG4HTMLEDT)
		{
			try
			{
				DeleteRegKey(HKCR, OUSTR(".htm\\OpenWithList\\soffice.exe"));
				DeleteRegKey(HKCR, OUSTR("Applications\\soffice.exe"));

				UserPref &= ~REG4HTMLEDT;
			}
			catch(CRegistryException& ex)
			{
				OSL_ENSURE(false, ex.what());
			}
		}

		if (UserPref & REG4DEFHTMLEDT)
		{			
			try
			{
				RestoreDefHtmlEditorRegValues();
				RestoreHtmlEditRegValues();

				UserPref &= ~REG4DEFHTMLEDT;				
			}
			catch(CRegistryException& ex)
			{
				OSL_ENSURE(false, ex.what());
			}
		}
	}

    if ((MOD_CALC & UninstallModules) && (UserPref & REG4MSEXCEL))
    {
		UnregisterDocType(REG_KEY_EXTENSION_XLS);
		UnregisterDocType(REG_KEY_EXTENSION_XLT);

		UserPref &= ~REG4MSEXCEL;
    }
    
    if ((MOD_IMPRESS & UninstallModules) && (UserPref & REG4MSPOWERPOINT))
    {
		UnregisterDocType(REG_KEY_EXTENSION_PPT);
		UnregisterDocType(REG_KEY_EXTENSION_POT);

		UserPref &= ~REG4MSPOWERPOINT;
    }	

	try
	{
		if (UninstallModules > 0)
		{
			SetUserPreferences(UserPref);
			SHChangeNotify(SHCNE_ASSOCCHANGED, SHCNF_IDLIST, 0, 0);    
		}
	}
	catch(CRegistryException& ex)
	{
		OSL_ENSURE(false, ex.what());
	}
}

//-----------------------------------------
// restores the entries for the selected
// registry entries
// Algorithm:
//
// 1. 
// Target key exist (e.g. '.doc')
// Default value == soffice.?
// Backup key != empty
// Action: Replace Default value with backup
// key
//
// 2. 
// Target key exist
// Default value == soffice.?
// Backup key == empty
// Action: nop
//
// 3.
// Target key exist
// Default value != soffice.?
// Action: nop
//
// 4.
// Target key does not exist
// Action: nop 
//-----------------------------------------

void Unregister4MsDocAll(ResMgr* aResMgr) throw()
{
    try
	{
		Unregister4MsDoc(
			MOD_WRITER | MOD_CALC | MOD_IMPRESS,
			aResMgr);
				
		DeleteRegKey(HKCR, REG_KEY_PLUGIN_PRIVATE);
	}
	catch(CRegistryException& ex)
	{
		OSL_ENSURE(false, ex.what());
	}
}

//-----------------------------------------
// restores lost settings formerly made
// with Register4MsDoc
//-----------------------------------------

void RepairRegister4MsDocSettings(const rtl::OUString& aOfficeExePath, ResMgr* aResMgr) throw()
{
	try
	{
		Register4MsDoc(GetUserPreferences(), aOfficeExePath, aResMgr);
	}
	catch(CRegistryException& ex)
	{
		OSL_ENSURE(false, ex.what());
	}
}

//-----------------------------------------
// retrievs the current registry state
//-----------------------------------------

sal_Int32 GetRegister4MsDocStatus(ResMgr* aResMgr) throw()
{
	int UserPrefs = 0;
	
	try
	{
		UserPrefs = GetUserPreferences();
	}
	catch(CRegistryException& ex)
	{
		OSL_ENSURE(false, ex.what());
	}

    return UserPrefs;
}

