/*************************************************************************
 *
 *  $RCSfile: solmath.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: er $ $Date: 2001/11/23 19:31:07 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/


#include <limits.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>

#include "solmath.hxx"


static const double nKorrVal[] = {
    0, 9e-1, 9e-2, 9e-3, 9e-4, 9e-5, 9e-6, 9e-7, 9e-8,
    9e-9, 9e-10, 9e-11, 9e-12, 9e-13, 9e-14, 9e-15
};


#ifdef WIN
#pragma optimize("",off)
#endif
#if defined(MSC) && defined(WNT)
// #56399# z.B. 1e88 => 1,00000000000001E+088
#pragma optimize("g",off)
#endif

// convert a floating point number
// s        the place to store the result
// n        the number to convert
// cType    'E', 'F', 'A' (Auto), 'G' (printf %G)
// nDec     decimal places, if cType=='A' && nDec==INT_MAX the maximum possible
// cDec     decimal point character
// bEraseTrailingDecZero    erase trailing zeros in decimal places and maybe decimal point as well
// static
void SolarMath::DoubleToString( XubString& s, double n, xub_Unicode cType, int nDec,
            xub_Unicode cDec, BOOL bEraseTrailingDecZero )
{
    static const double nRoundVal[] = {
        5.0e+0, 0.5e+0, 0.5e-1, 0.5e-2, 0.5e-3, 0.5e-4, 0.5e-5, 0.5e-6,
        0.5e-7, 0.5e-8, 0.5e-9, 0.5e-10,0.5e-11,0.5e-12,0.5e-13,0.5e-14
    };

    // sign adjustment, instead of testing for n<0.0 this will also fetch -0.0
    BOOL bSign = IsSignBitSet( n );
    if( bSign )
        n = -n;

	if ( IsNAN( n ) )
	{
        if ( bSign )
            s += '-';
        s += '1';
        s += cDec;
        s.AppendAscii( RTL_CONSTASCII_STRINGPARAM( "#NAN" ) );
        return ;
	}

    if ( n == HUGE_VAL || IsINF( n ) )
    {   // Ueberlauf-Wert
        if ( bSign )
            s += '-';
        s += '1';
        s += cDec;
        s.AppendAscii( RTL_CONSTASCII_STRINGPARAM( "#INF" ) );
        return ;
    }

    // find the exponent
    int nExp = 0;
    if ( n > 0.0 )
    {
        if ( n < 1e-8 || n > 1e8 )
        {   // die Schere, ob's schneller ist oder nicht, liegt zwischen 1e7 und 1e8
            nExp = (int) floor( log10( n ) );
            n /= pow( 10.0, (double) nExp );
        }
        else
        {   // man stelle sich sonst 1E+308 vor..
            while( n < 1.0 )
            {
                n *= 10.0;
                nExp--;
            }
            while( n >= 10.0 )
            {
                n /= 10.0;
                nExp++;
            }
        }
    }

    switch ( cType )
    {
        case 'A' :
        {   // Automatik
            int nPrec;
            if ( nExp <= -15 || nExp >= 15 )        // #58531# war <-16, >16
            {
                nPrec = 14;
                cType = 'E';
            }
            else
            {
                if ( nExp < 14 )
                {
                    nPrec = 15 - nExp - 1;
                    cType = 'F';
                }
                else
                {
                    nPrec = 15;
                    cType = 'F';
                }
            }
            if ( nDec == INT_MAX )
                nDec = nPrec;
        }
        break;
        case 'G' :
        {   // G-Point, wie sprintf %G
            if ( nDec == INT_MAX )
                nDec = 6;
            if ( nExp < -4 || nExp >= nDec )
            {
                nDec = Max( 1, nDec - 1 );
                cType = 'E';
            }
            else
            {
                nDec = Max( 0, nDec - nExp - 1 );
                cType = 'F';
            }
        }
        break;
    }

    int nDigits = nDec + 1;

    if( cType == 'F' )
        nDigits += nExp;

    // Round the number
    if( nDigits >= 0 )
    {
        if( ( n += nRoundVal[ nDigits > 15 ? 15 : nDigits ] ) >= 10 )
        {
            n = 1.0;
            nExp++;
            if( cType == 'F' )
                nDigits++;
        }
    }

    static const xub_StrLen nBufMax = 256;
    xub_Unicode aBuf[nBufMax];
    xub_Unicode* pBuf;
    xub_StrLen nBuf = (xub_StrLen)
        ( nDigits <= 0 ? Max( Abs(nDec), Abs(nExp) ) : nDigits + Abs(nDec) ) + 10;
    if ( nBuf > nBufMax )
        pBuf = new xub_Unicode[nBuf];
    else
        pBuf = aBuf;
    register xub_Unicode* p = pBuf;
    if ( bSign )
        *p++ = '-';

    BOOL bHasDec = FALSE;

    int nDecPos;
    // Check for F format and number < 1
    if( cType == 'F' )
    {
        if( nExp < 0 )
        {
            *p++ = '0';
            *p++ = cDec;
            bHasDec = TRUE;
            int i = ( nDigits <= 0 ? nDec : -nExp - 1 );
            while( i-- )
                *p++ = '0';
            nDecPos = 0;
        }
        else
            nDecPos = nExp + 1;
    }
    else
        nDecPos = 1;

    // print the number
    if( nDigits > 0 )
    {
        for ( int i = 0; ; i++ )
        {
            if( i < 15 )            // vorher 16
            {
                int nDigit;
                if (nDigits-1 == 0 && i > 0 && i < 14)
                    nDigit = (int) floor( n + nKorrVal[15-i] );
                else
                    nDigit = (int) ( n + 1E-15 );   // vorher 1E-14
                if (nDigit >= 10)
                {                                   // Nachbehandlung:
                    long sLen = (long)(p-pBuf)-1;
                    if (sLen == -1)
                    {
                        p = pBuf;
                        if ( cType == 'F' )
                        {
                            *p++ = '1';
                            *p++ = '0';
                        }
                        else
                        {
                            *p++ = '1';
                            *p++ = cDec;
                            *p++ = '0';
                            nExp++;
                            bHasDec = TRUE;
                        }
                    }
                    else
                    {
                        for (long j = sLen; j >= 0; j--)
                        {
                            xub_Unicode cS = pBuf[j];
                            if (cS != cDec)
                            {
                                if ( cS != '9')
                                {
                                    pBuf[j] = ++cS;
                                    j = -1;                 // abbruch
                                }
                                else
                                {
                                    pBuf[j] = '0';
                                    if (j == 0)
                                    {
                                        if ( cType == 'F' )
                                        {   // insert '1'
                                            xub_Unicode* px = p++;
                                            while ( pBuf < px )
                                            {
                                                *px = *(px-1);
                                                px--;
                                            }
                                            pBuf[0] = '1';
                                        }
                                        else
                                        {
                                            pBuf[j] = '1';
                                            nExp++;
                                        }
                                    }
                                }
                            }
                        }
                        *p++ = '0';
                    }
                    n = 0.0;
                }
                else
                {
                    *p++ = (xub_Unicode) ( nDigit + '0' );
                    n = ( n - nDigit ) * 10.0;
                }
            }
            else
                *p++ = '0';
            if( !--nDigits )
                break;  // for
            if( nDecPos )
            {
                if( !--nDecPos )
                {
                    *p++ = cDec;
                    bHasDec = TRUE;
                }
            }
        }
    }

    if ( !bHasDec && cType == 'F' )
    {
        while ( --nDecPos > 0 )
            *p++ = '0';     // Vorkomma Rundung auffuellen
    }

    if ( bEraseTrailingDecZero && bHasDec && p > pBuf )
    {
        while ( *(p-1) == '0' )
            p--;
        if ( *(p-1) == cDec )
            p--;
    }

    // print the exponent (dreistellig)
    if( cType == 'E' )
    {
        if ( p == pBuf )
            *p++ = '1';     // bei negativem nDec keine nDigits
        *p++ = cType;
        if( nExp < 0 )
        {
            nExp = -nExp;
            *p++ = '-';
        }
        else
            *p++ = '+';
//      if (nExp >= 100 )
        *p++ = (xub_Unicode) ( nExp / 100 + '0' ), nExp %= 100;
        *p++ = (xub_Unicode) ( nExp / 10 + '0' );
        *p++ = (xub_Unicode) ( nExp % 10 + '0' );
    }

    *p = 0;
    s.Append( pBuf, p - pBuf );
    if ( pBuf != &aBuf[0] )
        delete [] pBuf;
}

#if defined(MSC) && (defined(WNT) || defined(WIN))
#pragma optimize("",on)
#endif


#define SOMA_CUTOFF (LONG_MAX / 10)
#define SOMA_CUTLIM (LONG_MAX % 10)
// ob nExp * 10 + nAdd Overflow gibt
inline BOOL Long10OverFlow( long& nExp, int nAdd )
{
    if ( nExp > SOMA_CUTOFF || (nExp == SOMA_CUTOFF && nAdd > SOMA_CUTLIM) )
    {
        nExp = LONG_MAX;
        return TRUE;
    }
    return FALSE;
}
#undef SOMA_CUTOFF
#undef SOMA_CUTLIM

// static
double SolarMath::Pow10Exp( double fVal, int nExp )
{
    static const int n10Count = 16;
    static const double __FAR_DATA n10s[2][n10Count] = {
        { 1e1, 1e2, 1e3, 1e4, 1e5, 1e6, 1e7, 1e8,
          1e9, 1e10, 1e11, 1e12, 1e13, 1e14, 1e15, 1e16 },
        { 1e-1, 1e-2, 1e-3, 1e-4, 1e-5, 1e-6, 1e-7, 1e-8,
          1e-9, 1e-10, 1e-11, 1e-12, 1e-13, 1e-14, 1e-15, 1e-16 }
    };

    if ( nExp < 0 )
    {
        if ( -nExp <= n10Count )
            return fVal * n10s[1][-nExp-1];
        else
            return fVal * pow( 10.0, (double) nExp );
    }
    else if ( nExp > 0 )
    {
        if ( nExp <= n10Count )
            return fVal * n10s[0][nExp-1];
        else
            return fVal * pow( 10.0, (double) nExp );
    }
    else
        return fVal;
}

// We are only concerned about ASCII arabic numerical digits here
inline BOOL MyIsDigit( xub_Unicode c )
{
	return 0x30 <= c && c <= 0x39;
}


// static
double SolarMath::StringToDouble( const xub_Unicode* pStr,
            const xub_Unicode cGrpSep, const xub_Unicode cDecSep, int& nErrno,
			const xub_Unicode** ppEnd )
{
    double fVal = 0.0;
    nErrno = 0;

    const xub_Unicode* p0 = pStr;
    while ( *p0 == ' ' || *p0 == '\t' )
        ++p0;       // skip leading blanks and tabs
    BOOL bSign;
    if ( *p0 == '-' )
    {
        bSign = TRUE;
        ++p0;
    }
    else
    {
        bSign = FALSE;
        if ( *p0 == '+' )
            ++p0;
    }
    register const xub_Unicode* p = p0;
    register xub_Unicode c;

    // fuehrende Nullen und Tausenderseparatoren brauchen nicht berechnet zu werden
    while ( *p == '0' || *p == cGrpSep )
        ++p;

    long nValExp = 0;       // Exponent der Mantisse mitfuehren

    // Mantisse Integer
    while ( c = *p )
    {
        if ( MyIsDigit( c ) )
        {   // Ziffer
            fVal = fVal * 10.0 + (double) ( c - '0' );
            ++nValExp;
            ++p;
        }
        else if ( c == cDecSep )
        {   // Dezimalseparator
            ++p;
            break;  // while, Integer Ende
        }
        else if ( c == cGrpSep )
        {   // Tausenderseparator ignorieren
            ++p;
        }
        else
            break;  // while, ungueltiges Zeichen (oder 'E' bzw. 'e')
    }

    // Mantisse Fraction
    if ( c == cDecSep )
    {
        double fFrac = 0.0;
        long nFracExp = 0;
        while ( *p == '0' )
        {
            --nFracExp;
            ++p;
        }
        if ( nValExp == 0 )
            nValExp = nFracExp - 1;     // kein Integer-Teil => Fraction Exponent
        // eine Ziffer braucht ld(10) ~= 3.32 Bits
        static const int nSigs = (DBL_MANT_DIG / 3) + 1;
        int nDigs = 0;
        while ( (c = *p) && MyIsDigit( c ) )
        {   // nur Ziffern
            if ( nDigs < nSigs )
            {   // bei weiteren Ziffern geht jegliche Signifikanz verloren
                fFrac = fFrac * 10.0 + (double) ( c - '0' );
                --nFracExp;
                ++nDigs;
            }
            ++p;
        }
        if ( fFrac != 0.0 )
            fVal += Pow10Exp( fFrac, nFracExp );
        else if ( nValExp < 0 )
            nValExp = 0;        // keine Ziffer ungleich 0 hinterm Komma
    }

    if ( nValExp > 0 )
        --nValExp;      // bei der ersten Vorkomma-Ziffer zuviel

    // Exponent
    if ( p > p0 && (c == 'E' || c == 'e') )
    {
        ++p;
        BOOL bExpSign;
        if ( *p == '-' )
        {
            bExpSign = TRUE;
            ++p;
        }
        else
        {
            bExpSign = FALSE;
            if ( *p == '+' )
                ++p;
        }
        if ( fVal == 0.0 )
        {   // egal was hier kommt: es bleibt Null, aber Offset weiterfuehren
            while ( *p && MyIsDigit( *p ) )
                ++p;
        }
        else
        {
            BOOL bOverFlow = FALSE;
            long nExp = 0;
            while ( (c = *p) && MyIsDigit( c ) )
            {   // nur Ziffern
                int i = c - '0';
                if ( Long10OverFlow( nExp, i ) )
                    bOverFlow = TRUE;
                else
                    nExp = nExp * 10 + i;
                ++p;
            }
            if ( nExp )
            {
                if ( bExpSign )
                    nExp = -nExp;
                long nAllExp = ( bOverFlow ? 0 : nExp + nValExp );
                if ( nAllExp > DBL_MAX_10_EXP || (bOverFlow && !bExpSign) )
                {   // Ueberlauf
                    fVal = HUGE_VAL;
                    nErrno = ERANGE;
                }
                else if ( nAllExp < DBL_MIN_10_EXP || (bOverFlow && bExpSign) )
                {   // Unterlauf
                    fVal = 0.0;
                    nErrno = ERANGE;
                }
                else if ( nExp > DBL_MAX_10_EXP || nExp < DBL_MIN_10_EXP )
                {   // Exponenten ausgleichen
                    fVal = Pow10Exp( fVal, -nValExp );
                    fVal = Pow10Exp( fVal, nAllExp );
                }
                else
                    fVal = Pow10Exp( fVal, nExp );      // normal
            }
        }
    }
    else if ( p == p0+2 && *p == '#' && *(p-1) == cDecSep && *(p-2) == '1' )
	{
		if ( *(p+1) == 'I' && *(p+2) == 'N' && *(p+3) == 'F' )
    	{   // "1.#INF" bzw. "-1.#INF"
        	p += 4;
        	fVal = HUGE_VAL;
        	nErrno = ERANGE;
        	while ( *p && MyIsDigit( *p ) )
        	{   // nur Ziffern, 1.#INF000 u.ae.
            	++p;
        	}
    	}
		else if ( *(p+1) == 'N' && *(p+2) == 'A' && *(p+3) == 'N' )
    	{   // "1.#NAN" or "-1.#NAN"
        	p += 4;
			SolarMath::SetNAN( fVal, bSign );
			if ( bSign )
				bSign = FALSE;		// don't negate again
        	while ( *p && MyIsDigit( *p ) )
        	{   // only digits, 1.#INF000
            	++p;
        	}
    	}
	}

    // z.B. auch wenn mehr als DBL_MAX_10_EXP Ziffern ohne Dezimalseparator
    // oder 0. und mehr als DBL_MIN_10_EXP Ziffern u.ae.
    if ( fVal == HUGE_VAL )
        nErrno = ERANGE;

    if ( bSign )
        fVal = -fVal;

    if ( ppEnd )
        *ppEnd = p;

    return fVal;
}


// static
double SolarMath::Round( double fVal, short nDec, SolarMathRoundingMode eMode )
{
    if ( fVal == 0.0  )
        return fVal;

    // sign adjustment
    BOOL bSign = IsSignBitSet( fVal );
    if ( bSign )
        fVal = -fVal;

    double fFac;
    if ( nDec != 0 )
    {
        // max 20 decimals, we don't have unlimited precision
        // #38810# and no overflow on fVal*=fFac
        if ( nDec < -20 || 20 < nDec || fVal > (DBL_MAX / 1e20) )
            return bSign ? -fVal : fVal;

        fFac = pow( 10.0, nDec );
        fVal *= fFac;
    }
    //else  //! uninitialized fFac, not needed

    switch ( eMode )
    {
        case SolarMathRoundCorrected :
        {
            int nExp;       // exponent for correction
            if ( fVal > 0.0 )
                nExp = (int) floor( log10( fVal ) );
            else
                nExp = 0;
            int nIndex = 15 - nExp;
            if ( nIndex > 15 )
                nIndex = 15;
            else if ( nIndex <= 1 )
                nIndex = 0;
            fVal = floor( fVal + 0.5 + nKorrVal[nIndex] );
        }
        break;
        case SolarMathRoundDown :
            fVal = ApproxFloor( fVal );
        break;
        case SolarMathRoundUp :
            fVal = ApproxCeil( fVal );
        break;
        case SolarMathRoundFloor :
            fVal = bSign ? ApproxCeil( fVal ) : ApproxFloor( fVal );
        break;
        case SolarMathRoundCeiling :
            fVal = bSign ? ApproxFloor( fVal ) : ApproxCeil( fVal );
        break;
        case SolarMathRoundHalfDown :
        {
            double f = floor( fVal );
            fVal = ((fVal - f) <= 0.5) ? f : ceil( fVal );
        }
        break;
        case SolarMathRoundHalfUp :
        {
            double f = floor( fVal );
            fVal = ((fVal - f) < 0.5) ? f : ceil( fVal );
        }
        break;
        case SolarMathRoundHalfEven :
#ifdef FLT_ROUNDS
/*
    Use fast version. FLT_ROUNDS may be defined to a function by some compilers!

    DBL_EPSILON is the smallest fractional number which can be represented,
    its reciprocal is therefore the smallest number that cannot have a
    fractional part. Once you add this reciprocal to `x', its fractional part
    is stripped off. Simply subtracting the reciprocal back out returns `x'
    without its fractional component.
    Simple, clever, and elegant - thanks to Ross Cottrell, the original author,
    who placed it into public domain.

    volatile: prevent compiler from being too smart
*/
            if ( FLT_ROUNDS == 1 )
            {
                volatile double x = fVal + 1.0 / DBL_EPSILON;
                fVal = x - 1.0 / DBL_EPSILON;
            }
            else
#endif
            {
                double f = floor( fVal );
                if ( (fVal - f) != 0.5 )
                    fVal = floor( fVal + 0.5 );
                else
                {
                    double g = f / 2.0;
                    fVal = (g == floor( g )) ? f : (f + 1.0);
                }
            }
        break;
    }

    if ( nDec != 0 )
        fVal /= fFac;

    return bSign ? -fVal : fVal;
}


#if 0	// TEST
// May be used to test endians and bit fields.
// Copy to separate source and compile with simple command line.

#define WNT
#if 0	// Solaris-SPARC without svconf.h
#define __BIGENDIAN
#define __IEEEDOUBLE
#define __SIZEOFDOUBLE 8
#endif
#include "../../inc/solar.h"
#define _STRING_HXX
typedef UINT16 xub_Unicode;
class XubString;
#include "../../inc/solmath.hxx"
#include <stdio.h>
int main()
{
    SOMA_FPEXCEPTIONS_OFF();
	double x;
	x = HUGE_VAL;
    printf( " HUG:  %f, sign: %d, isINF: %d, isIND: %d, isNAN: %d, hi: %08x, lo: %08x\n", x, (int)SolarMath::IsSignBitSet( x ), (int)SolarMath::IsINF( x ), (int)SolarMath::IsIND( x ), (int)SolarMath::IsNAN( x ), ((SolarMathDouble*)&x)->w32_parts.msw, ((SolarMathDouble*)&x)->w32_parts.lsw );
	x = -HUGE_VAL;
    printf( "-HUG: %f, sign: %d, isINF: %d, isIND: %d, isNAN: %d, hi: %08x, lo: %08x\n", x, (int)SolarMath::IsSignBitSet( x ), (int)SolarMath::IsINF( x ), (int)SolarMath::IsIND( x ), (int)SolarMath::IsNAN( x ), ((SolarMathDouble*)&x)->w32_parts.msw, ((SolarMathDouble*)&x)->w32_parts.lsw );
	SolarMath::SetINF( x, FALSE );
    printf( " INF:  %f, sign: %d, isINF: %d, isIND: %d, isNAN: %d, hi: %08x, lo: %08x\n", x, (int)SolarMath::IsSignBitSet( x ), (int)SolarMath::IsINF( x ), (int)SolarMath::IsIND( x ), (int)SolarMath::IsNAN( x ), ((SolarMathDouble*)&x)->w32_parts.msw, ((SolarMathDouble*)&x)->w32_parts.lsw );
	SolarMath::SetINF( x, TRUE );
    printf( "-INF: %f, sign: %d, isINF: %d, isIND: %d, isNAN: %d, hi: %08x, lo: %08x\n", x, (int)SolarMath::IsSignBitSet( x ), (int)SolarMath::IsINF( x ), (int)SolarMath::IsIND( x ), (int)SolarMath::IsNAN( x ), ((SolarMathDouble*)&x)->w32_parts.msw, ((SolarMathDouble*)&x)->w32_parts.lsw );
	SolarMath::SetNAN( x, FALSE );
    printf( " NAN:  %f, sign: %d, isINF: %d, isIND: %d, isNAN: %d, hi: %08x, lo: %08x\n", x, (int)SolarMath::IsSignBitSet( x ), (int)SolarMath::IsINF( x ), (int)SolarMath::IsIND( x ), (int)SolarMath::IsNAN( x ), ((SolarMathDouble*)&x)->w32_parts.msw, ((SolarMathDouble*)&x)->w32_parts.lsw );
	SolarMath::SetNAN( x, TRUE );
    printf( "-NAN: %f, sign: %d, isINF: %d, isIND: %d, isNAN: %d, hi: %08x, lo: %08x\n", x, (int)SolarMath::IsSignBitSet( x ), (int)SolarMath::IsINF( x ), (int)SolarMath::IsIND( x ), (int)SolarMath::IsNAN( x ), ((SolarMathDouble*)&x)->w32_parts.msw, ((SolarMathDouble*)&x)->w32_parts.lsw );
    double y = 0;
    x = 1/y;
    printf( " 1/0:  %f, sign: %d, isINF: %d, isIND: %d, isNAN: %d, hi: %08x, lo: %08x\n", x, (int)SolarMath::IsSignBitSet( x ), (int)SolarMath::IsINF( x ), (int)SolarMath::IsIND( x ), (int)SolarMath::IsNAN( x ), ((SolarMathDouble*)&x)->w32_parts.msw, ((SolarMathDouble*)&x)->w32_parts.lsw );
    x = -1/y;
    printf( "-1/0: %f, sign: %d, isINF: %d, isIND: %d, isNAN: %d, hi: %08x, lo: %08x\n", x, (int)SolarMath::IsSignBitSet( x ), (int)SolarMath::IsINF( x ), (int)SolarMath::IsIND( x ), (int)SolarMath::IsNAN( x ), ((SolarMathDouble*)&x)->w32_parts.msw, ((SolarMathDouble*)&x)->w32_parts.lsw );
    x = 0/y;
    printf( " 0/0: %f, sign: %d, isINF: %d, isIND: %d, isNAN: %d, hi: %08x, lo: %08x\n", x, (int)SolarMath::IsSignBitSet( x ), (int)SolarMath::IsINF( x ), (int)SolarMath::IsIND( x ), (int)SolarMath::IsNAN( x ), ((SolarMathDouble*)&x)->w32_parts.msw, ((SolarMathDouble*)&x)->w32_parts.lsw );
    x = -0/y;
    printf( "-0/0: %f, sign: %d, isINF: %d, isIND: %d, isNAN: %d, hi: %08x, lo: %08x\n", x, (int)SolarMath::IsSignBitSet( x ), (int)SolarMath::IsINF( x ), (int)SolarMath::IsIND( x ), (int)SolarMath::IsNAN( x ), ((SolarMathDouble*)&x)->w32_parts.msw, ((SolarMathDouble*)&x)->w32_parts.lsw );
	return 0;
}

#endif	// TEST


