#!/bin/sh

# Set kernel parameters that fix clock drift.
# Use for RHEL guests running on KVM.

ARCH=$(uname -m)
GRUB_CONF=/boot/grub/grub.conf
GRUB_CONF_KVM=${GRUB_CONF}.kvm
VENDOR=$(dmidecode -s system-manufacturer)
RELEASE=$(cat /etc/redhat-release | tr -d "[:alpha:] ()")

cat <<EOF
This script will add kernel command line parameters to fix clock drift for kvm 
guests. It will at first make some tests and then it will add the parameters.

The script will not modify "${GRUB_CONF}" directly. It will copy it
to "${GRUB_CONF_KVM}" and modify this file.

When the script completes, please check the this copy to ensure that the 
contents are correct and rename it to "${GRUB_CONF}" to activate the 
changes. It might be good to make a backup copy of the original before 
overwriting it.

Please remember that it is recommended to activate the ntp service.

EOF

echo -n "Do you want to continue? [y/N]: "
read -e ANSWER
if [ "$ANSWER" != "y" ]; then
    echo "Exiting."
    exit 0;
fi
echo

case ${ARCH} in
    x86_64)
	PARAMS="notsc divider=10"
	;;
    i686)
	PARAMS="clock=pmtmr divider=10"
	;;
    *)
	echo "The architecture \"${ARCH}\" is not supported by this script."
	exit 1
	;;
esac

if [ ! -r "${GRUB_CONF}" ]; then
    echo "ERROR: ${GRUB_CONF}: Permission denied."
    exit 1
fi

if [ "${VENDOR}" != "Red Hat" ]; then
    echo "ERROR: This is no kvm guest, exiting."
    exit 1
fi

# This fix should not be needed for 4.9
if [ $(echo "${RELEASE} > 48" | bc -l) -eq 1 ]; then
    echo "ERROR: KVM timer already in use, exiting."
    exit 1
fi

if grep -q -E "^[[:space:]]+kernel.*(clock=)|(divider=)|(notsc)" "${GRUB_CONF}"; then
    echo "ERROR: \"${GRUB_CONF}\" clock source already chosen, not applying time drift fix."
    exit 1
fi

echo "Creating copy \"${GRUB_CONF_KVM}\" .."

cp "${GRUB_CONF}" "${GRUB_CONF_KVM}" || exit 1

echo "Adding \"${PARAMS}\" to the kernel command line(s) .."

sed -i -e "s:^[[:space:]]\+kernel.*:\0 ${PARAMS}:g" "${GRUB_CONF_KVM}"

cat <<EOF

Please check the file "${GRUB_CONF_KVM}" to ensure that the contents 
are correct and rename it to "${GRUB_CONF}".

Added command line parameters: "${PARAMS}"
EOF
