#!/bin/sh

# Sample Postfix installation script. Run this from the top-level
# Postfix source directory.

PATH=/bin:/usr/bin:/usr/sbin:/usr/etc:/sbin:/etc
umask 022

# Workaround, should edit main.cf in place.
trap 'rm -f ./main.cf ./junk; exit 1' 1 2 3 15

cat <<EOF

Warning: this script replaces existing sendmail or Postfix programs.
Make backups if you want to be able to recover.

Otherwise, the script should be safe for installing or upgrading
Postfix.  Existing configuration and queue files are left alone;
existing program files and scripts are replaced, not overwritten.
This is safe unless you are sharing files over NFS.

In particular, this script can be used to change Postfix from using
a world-writable maildrop to a group-writable one.

Before installing files, this script prompts you for some definitions.
You can either edit this script ahead of time, or you can specify
your changes interactively.

    config_directory - directory with Postfix configuration files.
    daemon_directory - directory with Postfix daemon programs.
    command_directory - directory with Postfix administrative commands.
    queue_directory - directory with Postfix queues.

    sendmail_path - full pathname of the Postfix sendmail command.
    newaliases_path - full pathname of the Postfix newaliases command.
    mailq_path - full pathname of the Postfix mailq command.

    owner - owner of Postfix queue files.

    setgid - groupname, e.g., postdrop (default: no). See INSTALL section 12.
    manpages - path to man tree (default: no). Example: /usr/local/man.

EOF

# Default settings, edit to taste or change interactively. Once this
# script has run it saves settings to $config_directory/install.cf.

# *BSD, LINUX
config_directory=/etc/postfix
daemon_directory=/usr/libexec/postfix
command_directory=/usr/sbin
queue_directory=/var/spool/postfix
sendmail_path=/usr/sbin/sendmail
newaliases_path=/usr/bin/newaliases
mailq_path=/usr/bin/mailq
owner=postfix
setgid=no
manpages=no

# Solaris, System V.4
#config_directory=/etc/postfix
#daemon_directory=/usr/lib/postfix
#command_directory=/usr/sbin
#queue_directory=/var/spool/postfix
#sendmail_path=/usr/lib/sendmail
#newaliases_path=/usr/bin/newaliases
#mailq_path=/usr/bin/mailq
#owner=postfix
#setgid=no
#manpages=no

# SunOS 4
#config_directory=/etc/postfix
#daemon_directory=/usr/lib/postfix
#command_directory=/usr/etc
#queue_directory=/var/spool/postfix
#sendmail_path=/usr/lib/sendmail
#newaliases_path=/usr/ucb/newaliases
#mailq_path=/usr/ucb/mailq
#owner=postfix
#setgid=no
#manpages=no

test -f $config_directory/install.cf && . $config_directory/install.cf

# By now, shells must have functions. Ultrix users must use sh5 or lose.

compare_or_replace() {
    cmp $2 $3 >/dev/null 2>&1 || {
	cp $2 junk || exit 1
	mv -f junk $3 || exit 1
	chmod $1 $3 || exit 1
    }
}

compare_or_symlink() {
    cmp $1 $2 >/dev/null 2>&1 || {
	ln -s $1 junk || exit 1
	mv -f junk $2 || exit 1
    }
}

compare_or_move() {
    cmp $2 $3 >/dev/null 2>&1 || {
	mv -f $2 $3 || exit 1
	chmod $1 $3 || exit 1
    }
}

# How to supress newlines in echo

case `echo -n` in
"") n=-n; c=;;
 *) n=; c='\c';;
esac

for name in config_directory daemon_directory command_directory \
    queue_directory sendmail_path newaliases_path mailq_path owner \
    setgid manpages
do
    while :
    do
	eval echo \$n "$name: [\$$name]\  \$c"
	read ans
	case $ans in
	"") break;;
	 *) eval $name=\$ans; break;;
	esac
    done
done

# Sanity checks

for path in $config_directory $daemon_directory $command_directory \
    $queue_directory $sendmail_path $newaliases_path $mailq_path $manpages
do
   case $path in
   /*) ;;
   no) ;;
    *) echo "$path should be an absolute path name" 1>&2; exit 1;;
   esac
done

# Workaround, should edit main.cf in place.
echo "mail_owner = $owner" >./main.cf
echo "myhostname = xx.yy" >>./main.cf

bin/postmap -c . -q "$owner" unix:passwd.byname >/dev/null || {
    echo "$owner needs an entry in the passwd file" 1>&2
    echo "Remember, $owner must have a dedicated user id and group id." 1>&2
    exit 1
}

case $setgid in
no) ;;
 *) bin/postmap -c . -q "$setgid" unix:group.byname >/dev/null || {
	echo "$setgid needs an entry in the group file" 1>&2
	echo "Remember, $setgid must have a dedicated group id." 1>&2
	exit 1
    }
esac

# Create any missing directories.

test -d $config_directory || mkdir -p $config_directory || exit 1
test -d $daemon_directory || mkdir -p $daemon_directory || exit 1
test -d $command_directory || mkdir -p $command_directory || exit 1
test -d $queue_directory || mkdir -p $queue_directory || exit 1

# Save settings.

(echo "# This file was generated by $0"
for name in config_directory daemon_directory command_directory \
    queue_directory sendmail_path newaliases_path mailq_path owner \
    setgid manpages
do
    eval echo $name=\$$name
done) >junk || exit 1
compare_or_move a+x,go-w junk $config_directory/install.cf || exit 1
rm -f junk

# Install files. Be careful to not copy over running programs.

for file in `ls libexec`
do
    compare_or_replace a+x,go-w libexec/$file $daemon_directory/$file || exit 1
done

for file in `ls bin | grep '^post'`
do
    compare_or_replace a+x,go-w bin/$file $command_directory/$file || exit 1
done

test -f bin/sendmail && {
    compare_or_replace a+x,go-w bin/sendmail $sendmail_path || exit 1
    compare_or_symlink $sendmail_path $newaliases_path
    compare_or_symlink $sendmail_path $mailq_path
}

compare_or_replace a+r,go-w conf/LICENSE $config_directory/LICENSE || exit 1

test -f $config_directory/main.cf || {
    cp conf/* $config_directory || exit 1
    chmod a+r,go-w $config_directory/* || exit 1
    sed "
	s;^daemon_directory .*;daemon_directory = $daemon_directory;
	s;^command_directory .*;command_directory = $command_directory;
	s;^queue_directory .*;queue_directory = $queue_directory;
	s;^mail_owner .*;mail_owner = $owner;
    " conf/main.cf >$config_directory/main.cf || exit 1

    echo "Warning: you still need to edit myorigin/mydestination in" 1>&2
    echo "$config_directory/main.cf. See also html/faq.html for dialup" 1>&2
    echo "sites or for sites inside a firewalled network." 1>&2
    echo "" 1>&2
    echo "BTW, Edit your alias database and be sure to set up aliases" 1>&2
    echo "for root and postmaster, then run the newaliases command." 1>&2
}

# Use set-gid privileges instead of writable maildrop (optional).

test -d $queue_directory/maildrop || {
    mkdir -p $queue_directory/maildrop || exit 1
    chown $owner $queue_directory/maildrop || exit 1
}

case $setgid in
no)
    chmod 1733 $queue_directory/maildrop || exit 1
    chmod g-s $command_directory/postdrop || exit 1
    postfix_script=conf/postfix-script-nosgid
    ;;
 *) 
    chgrp $setgid $command_directory/postdrop || exit 1
    chmod g+s $command_directory/postdrop || exit 1
    chgrp $setgid $queue_directory/maildrop || exit 1
    chmod 1730 $queue_directory/maildrop || exit 1
    postfix_script=conf/postfix-script-sgid
    ;;
esac

compare_or_replace a+x,go-w $postfix_script $config_directory/postfix-script ||
    exit 1

# Install manual pages (optional). We just replace whatever is there.

case $manpages in
no) ;;
 *) (
     cd man || exit 1
     for dir in man?
	 do mkdir -p $manpages/$dir || exit 1
     done
     for file in man?/*
     do
	 cmp -s $file $manpages/$file || {
	     rm -f $manpages/$file
	     cp $file $manpages/$file || exit 1
	     chmod 644 $manpages/$file || exit 1
	 }
     done
    )
esac

rm -f ./main.cf ./junk
