/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "Shaders.h"

#include <GL/glew.h>
#include <cstdio>
#include <cstring>

namespace libgltf
{

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               float* fValues)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniform1fv(iLoc, 1, fValues);
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               const float fValue)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniform1fv(iLoc, 1, &fValue);
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               glm::vec2* vVectors)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniform2fv(iLoc, 1, (GLfloat*)vVectors);
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               const glm::vec2 vVector)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniform2fv(iLoc, 1, (GLfloat*)&vVector);
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               glm::vec3* vVectors)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniform3fv(iLoc, 1, (GLfloat*)vVectors);
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               const glm::vec3 vVector)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniform3fv(iLoc, 1, (GLfloat*)&vVector);
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               glm::vec4* vVectors)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniform4fv(iLoc, 1, (GLfloat*)vVectors);
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               const glm::vec4 vVector)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniform4fv(iLoc, 1, (GLfloat*)&vVector);
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               glm::mat3* mMatrices)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniformMatrix3fv(iLoc, 1, false, (GLfloat*)mMatrices);
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               const glm::mat3 mMatrix)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniformMatrix3fv(iLoc, 1, false, (GLfloat*)&mMatrix);
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               glm::mat4* mMatrices)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniformMatrix4fv(iLoc, 1, false, (GLfloat*)mMatrices);
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               const glm::mat4 mMatrix)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniformMatrix4fv(iLoc, 1, false, (GLfloat*)&mMatrix);
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               int* iValues)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniform1iv(iLoc, 1, iValues);
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               const int iValue)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniform1i(iLoc, iValue);
}

unsigned int ShaderProgram::createProgram(const char* pvShader,
                                          size_t ivShaderSize,
                                          const char* pfShader,
                                          size_t ifShaderSize)
{
    unsigned int programId = glCreateProgram();
    if (!loadShader(programId, pvShader, ivShaderSize, GL_VERTEX_SHADER))
    {
        return 0;
    }
    if (!loadShader(programId, pfShader, ifShaderSize, GL_FRAGMENT_SHADER))
    {
        return 0;
    }

    return programId;
}

void ShaderProgram::deleteProgram(unsigned int programId)
{
    glDeleteShader(programId);
}

void ShaderProgram::useProgram(unsigned int programId)
{
    glUseProgram(programId);
}

bool ShaderProgram::loadShader(unsigned int programId, const char* pShader,
                               size_t iSize, int type)
{
    unsigned int shaderId = glCreateShader(type);

    if (!compileShader(pShader, iSize, shaderId))
    {
        fprintf(stderr, "compileShader : compileShader failed.\n");
        return false;
    }

    if (!linkProgram(programId, shaderId))
    {
        fprintf(stderr, "compileShader : linkProgram failed.\n");
        return false;
    }

    deleteShader(shaderId);
    return true;
}

bool ShaderProgram::compileShader(const char* pShader, size_t iSize,
                                  unsigned int shaderId)
{
    GLint iGLSize = static_cast<GLint>(iSize);
    if( strstr(pShader,"#version") == 0 )
    {
        const GLchar* aSources[] = {
            "#version 130\n",
            pShader,
        };

        const GLint aSizes[] = {
            strlen("#version 130\n"),
            iGLSize,
        };

        glShaderSource(shaderId, 2, &aSources[0], &aSizes[0]);
    }
    else
    {
        glShaderSource(shaderId, 1, &pShader, &iGLSize);
    }
    glCompileShader(shaderId);
    int iStatus = 0;
    glGetShaderiv(shaderId, GL_COMPILE_STATUS, &iStatus);
    if (GL_FALSE == iStatus)
    {
        int iLogLength;
        char sInfoLog[1024] = {0};
        glGetShaderInfoLog(shaderId, 1024, &iLogLength, sInfoLog);
        fprintf(stderr, "%s\n", sInfoLog);
        return false;
    }
    return true;
}

bool ShaderProgram::linkProgram(unsigned int programId,
                                unsigned int shaderId)
{
    int iStatus = 0;
    glAttachShader(programId, shaderId);
    glLinkProgram(programId);
    glGetProgramiv(programId, GL_LINK_STATUS, &iStatus);
    if (GL_FALSE == iStatus)
    {
        int iLogLength;
        char sInfoLog[1024] = {0};
        glGetShaderInfoLog(shaderId, 1024, &iLogLength, sInfoLog);
        fprintf(stderr, "%s\n", sInfoLog);
        return false;
    }
    return true;
}

void ShaderProgram::deleteShader(unsigned int shaderId)
{
    glDeleteShader(shaderId);
}

} // namespace libgltf

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
