// |reftest| skip-if(!this.hasOwnProperty('Temporal')) -- Temporal is not enabled unconditionally
// Copyright (C) 2022 André Bargull. All rights reserved.
// This code is governed by the BSD license found in the LICENSE file.

/*---
esid: sec-temporal.timezone.prototype.getnexttransition
description: >
  Compute next transition when seconds resp. nanoseconds are subtracted from the last transition.
features: [Temporal]
---*/

// From <https://github.com/eggert/tz/blob/main/europe>:
//
// # Zone  NAME       STDOFF  RULES FORMAT  [UNTIL]
// Zone  Europe/Paris 0:09:21 -     LMT     1891 Mar 15  0:01
//                    0:09:21 -     PMT     1911 Mar 11  0:01 # Paris MT

let tz = new Temporal.TimeZone("Europe/Paris");

let zdt = new Temporal.PlainDateTime(1800, 1, 1).toZonedDateTime(tz);
assert.sameValue(zdt.toString(), "1800-01-01T00:00:00+00:09[Europe/Paris]");
assert.sameValue(zdt.offsetNanoseconds, (9 * 60 + 21) * 1_000_000_000);

// Ensure the first transition was correctly computed.
let first = tz.getNextTransition(zdt);
assert.sameValue(first.toString(), "1911-03-10T23:50:39Z");
assert.sameValue(new Temporal.ZonedDateTime(first.epochNanoseconds, tz).toString(),
                 "1911-03-10T23:50:39+00:00[Europe/Paris]");

let next;

// Compute the next transition starting from the first transition minus 1s.
let firstMinus1s = first.add({seconds: -1});
assert.sameValue(firstMinus1s.toString(), "1911-03-10T23:50:38Z");
assert.sameValue(new Temporal.ZonedDateTime(firstMinus1s.epochNanoseconds, tz).toString(),
                 "1911-03-10T23:59:59+00:09[Europe/Paris]");
assert.sameValue(new Temporal.ZonedDateTime(firstMinus1s.epochNanoseconds, tz).offsetNanoseconds,
                 (9 * 60 + 21) * 1_000_000_000);

next = tz.getNextTransition(firstMinus1s);
assert.sameValue(next.toString(), "1911-03-10T23:50:39Z");
assert.sameValue(new Temporal.ZonedDateTime(next.epochNanoseconds, tz).toString(),
                 "1911-03-10T23:50:39+00:00[Europe/Paris]");

// Compute the next transition starting from the first transition minus 1ns.
let firstMinus1ns = first.add({nanoseconds: -1});
assert.sameValue(firstMinus1ns.toString(), "1911-03-10T23:50:38.999999999Z");
assert.sameValue(new Temporal.ZonedDateTime(firstMinus1ns.epochNanoseconds, tz).toString(),
                 "1911-03-10T23:59:59.999999999+00:09[Europe/Paris]");
assert.sameValue(new Temporal.ZonedDateTime(firstMinus1ns.epochNanoseconds, tz).offsetNanoseconds,
                 (9 * 60 + 21) * 1_000_000_000);

next = tz.getNextTransition(firstMinus1ns);
assert.sameValue(next.toString(), "1911-03-10T23:50:39Z");
assert.sameValue(new Temporal.ZonedDateTime(next.epochNanoseconds, tz).toString(),
                 "1911-03-10T23:50:39+00:00[Europe/Paris]");

reportCompare(0, 0);
