/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Alex Launi <alex.launi@canonical.com>
 *
 */

using GLib;

namespace Unity.MusicLens {

  public class MusicStoreCollection : Object
  {

    private const string MUSICSTORE_BASE_URI = "http://musicsearch.ubuntu.com/v1/";
    private HashTable<string, string> preview_uri_map; //maps u1ms store uri to details uri used by preview request

    public MusicStoreCollection ()
    {
      preview_uri_map = new HashTable<string, string>(str_hash, str_equal);
    }

    public async void search (DeprecatedScopeSearch search, SearchType search_type,
			owned List<FilterParser> filters, int max_results = -1, GLib.Cancellable cancellable) throws IOError
    {
      string? uri = build_search_uri (search.search_string, filters);

      if (uri == null)
        return;	  

      preview_uri_map.remove_all ();
      var results_model = search.results_model;
      File file = File.new_for_uri (uri);

      yield read_musicstore_search_result_async (file, results_model, cancellable);
    }

    public void get_album_details (string uri, out Album album, out SList<Track> tracks)
    {
      album = new Album ();
      tracks = new SList<Track> ();
      string http_uri = uri.substring (7); // strip off "u1ms://" from the uri
      if (preview_uri_map.contains (http_uri))
      {
        string details_uri = preview_uri_map.get (http_uri);
        debug ("details uri: %s", details_uri);

        var file = File.new_for_uri (details_uri);
        var parser = new Json.Parser ();

        try {
          if (parser.load_from_stream (file.read (null))) //FIXME: make async
          {
            var root_obj = parser.get_root ().get_object ();

            album.title = root_obj.get_string_member ("title");
            album.artwork_path = root_obj.get_string_member ("image");
            album.artist = root_obj.get_string_member ("artist");
            album.uri = http_uri;

            // store the price in the album so that we can get the value easily
            if (root_obj.has_member ("price"))
            {
              album.price = double.parse (root_obj.get_string_member ("price"));
            }

            // musicsearch should give us formatted_price, but fallback to price + currency
            if (root_obj.has_member ("formatted_price"))
            {
              album.formatted_price = root_obj.get_string_member ("formatted_price");
            }
            else if (root_obj.has_member ("price") && root_obj.has_member ("currency"))
            {
              album.formatted_price = root_obj.get_string_member ("price") + " " + root_obj.get_string_member ("currency");
            }

            if (root_obj.has_member ("tracks"))
            {
              var tracks_node = root_obj.get_array_member ("tracks");
              debug ("Album details: '%s', '%s'", uri, details_uri);
              foreach (var track_node in tracks_node.get_elements ())
              {
                var track_obj = track_node.get_object ();
                var track = new Track ();
                track.uri = track_obj.get_string_member ("preview");
                track.title = track_obj.get_string_member ("title");
                track.duration = (int)track_obj.get_member ("duration").get_int ();
                tracks.append (track);
              }
            }
            else // details for single track
            {
              debug ("Single track details: '%s', '%s'", uri, details_uri);
              var track = new Track ();
              track.uri = root_obj.get_string_member ("preview");
              track.title = root_obj.get_string_member ("title");
              track.duration = (int)root_obj.get_member ("duration").get_int ();
              tracks.append (track);
            }

            if (root_obj.has_member ("id"))
            {
                album.purchase_sku = root_obj.get_string_member ("id");
            }
            else
            {
                // U1MS does not allow purchases of an album with no purchase_sku
                album.purchase_sku = "";
                warning ("Json has no purchase_sku in '%s'", details_uri);
            }

          }
          else
          {
            warning ("Can't parse json data for '%s'", details_uri);
          }
        }
        catch (Error e)
        {
          warning ("Error fetching details for '%s': %s", details_uri, e.message);
        }
      }
      else
      {
        warning ("No details uri for '%s'", http_uri);
      }
    }

    private async void read_musicstore_search_result_async (File file, Dee.Model model, GLib.Cancellable cancellable)
    {
      var timer = new Timer ();
      debug ("Searching %s", file.get_uri ());
      var empty_asv = new Variant.array (VariantType.VARDICT.element (), {});

      try {
        var stream = yield file.read_async (Priority.DEFAULT, cancellable);
	var dis = new DataInputStream (stream);
	var parser = new Json.Parser ();
	yield parser.load_from_stream_async (dis, cancellable);
	var root_object = parser.get_root ().get_object ();
	
	Json.Object? results = root_object.get_object_member ("results");
	if (results == null) {
	  warning ("Invalid response from server. No 'results' member.");
	  return;
	}
	
	var albums = results.get_array_member ("album").get_elements ();
	var tracks = results.get_array_member ("track").get_elements ();
	
	debug ("Got %u albums and %u tracks", albums.length (), tracks.length ());
	
	unowned string? uri, details_uri, artwork_path, mimetype, title, artist, dnd_uri;
	
	foreach (var album_node in albums) {
	  var album_obj = album_node.get_object ();
	  
	  uri = album_obj.get_string_member ("purchase_url");
	  details_uri = album_obj.get_string_member ("details");
	  artwork_path = album_obj.get_string_member ("image");

      string icon = artwork_path;
      if (album_obj.has_member ("formatted_price"))
      {
        var icon_obj = new AnnotatedIcon (new FileIcon (File.new_for_uri (artwork_path)));
        icon_obj.category = CategoryType.MUSIC;
        icon_obj.ribbon = album_obj.get_string_member ("formatted_price");
        icon = icon_obj.to_string ();
      }

	  mimetype = "audio-x-generic";
	  title = album_obj.get_string_member ("title");
	  artist = album_obj.get_string_member ("artist");
	  dnd_uri = uri;
	  
	  model.append (uri, icon, Category.PURCHASE,
                        Unity.ResultType.DEFAULT,
	                mimetype, title, artist, dnd_uri,
                        empty_asv);

      preview_uri_map.insert (uri.substring (7), details_uri); // strip off "u1ms://" from the uri
    }
        
	foreach (var track_node in tracks) {
	  var track_obj = track_node.get_object ();
	  
	  uri = track_obj.get_string_member ("purchase_url");
	  details_uri = track_obj.get_string_member ("details");
	  artwork_path = track_obj.get_string_member ("image");

      string icon = artwork_path;
      if (track_obj.has_member ("formatted_price"))
      {
        var icon_obj = new AnnotatedIcon (new FileIcon (File.new_for_uri (artwork_path)));
        icon_obj.category = CategoryType.MUSIC;
        icon_obj.ribbon = track_obj.get_string_member ("formatted_price");
        icon = icon_obj.to_string ();
      }

	  mimetype = "audio-x-generic";
	  title = track_obj.get_string_member ("title");
	  artist = track_obj.get_string_member ("artist");
	  dnd_uri = uri;
          
	  // FIXME drag n drop uri needs to be the u1ms:// link
	  
	  model.append (uri, icon, Category.PURCHASE,
                        Unity.ResultType.DEFAULT,
	                mimetype, title, artist, dnd_uri,
                        empty_asv);
      preview_uri_map.insert (uri.substring (7), details_uri); // strip off "u1ms://" from the uri
    }

	debug ("Retrieved '%s' in %fms", file.get_uri (), timer.elapsed()*1000);
	debug ("Model has %u rows after search", model.get_n_rows ());

      } catch (Error e) {
	warning ("Error reading URL '%s': %s", file.get_uri (), e.message);
      }
    }

    private string? build_search_uri (string query, List<FilterParser> filters)
    {
      if (query.strip() == "")
        return null;
    
      MusicStoreFilterParser store_parser;
      string musicstore_base_uri = MUSICSTORE_BASE_URI;
      if (GLib.Environment.get_variable("MUSICSTORE_URI") != null)
        musicstore_base_uri = GLib.Environment.get_variable("MUSICSTORE_URI");
      debug ("Using base URI of '%s'", musicstore_base_uri);
      StringBuilder uri = new StringBuilder (musicstore_base_uri);
      uri.append ("search?q=");
      uri.append (Uri.escape_string (query, "", false));

      foreach (FilterParser parser in filters) {
	  if (parser is GenreFilterParser)
	    store_parser = new MusicStoreGenreFilterParser (parser as GenreFilterParser);
	  else if (parser is DecadeFilterParser)
	    store_parser = new MusicStoreDecadeFilterParser (parser as DecadeFilterParser);
	  else
	    continue;

	  uri.append (store_parser.parse ());
      }
      
      uri.append ("&pagesize=10");
      uri.append ("&imagesize=100");
      
      // This makes the service return $pagesize results *per content type*.
      // Which we need - as it could otherwise return $pagesize results mixed
      // or artist,album, and track. Since we can't display artists, this can
      // lead to an empty result set in the dash if there is only artists in
      // the response from the webservice
      uri.append ("&grouping=1");

      return uri.str;
    }
  }
}
