#
# Copyright (C) 2022-2023 David Hampton
#
# See the file LICENSE_FSF for licensing information.
#

#
# Debugging
#
# ~~~
# include(CMakePrintSystemInformation)
# include(CMakePrintHelpers)
# ~~~
include(BuildConfigString)
include(SetIfTargetExists)

#
# This module needs pkg-config functionality
#
find_package(PkgConfig REQUIRED)

#
# Does the system have threads?
#
include(FindThreads)

#
# Check libraries
#

# freetype2: fedora:freetype-devel debian:libfreetype-dev
if(ENABLE_FREETYPE)
  # See https://github.com/ImageMagick/freetype/blob/main/docs/VERSIONS.TXT for
  # conversion between freetype release number and the (libtool) number returned
  # by pkg-config.
  pkg_check_modules(FT2 "freetype2>=16.2.10" REQUIRED IMPORTED_TARGET)
  add_build_config(PkgConfig::FT2 "freetype2")
  set(HAVE_FT2 ${FT2_FOUND})
endif()

# fontconfig: fedora:fontconfig-devel debian:libfontconfig-dev
if(ENABLE_FONTCONFIG)
  find_package(Fontconfig REQUIRED) # Only as a module
  add_build_config(Fontconfig::Fontconfig "fontconfig")
  set(HAVE_FONTCONFIG 1)
endif()

# xml2: fedora:libxml2-devel debian:libxml2-dev
if(ENABLE_XML2)
  pkg_check_modules(LibXml2 "libxml-2.0" REQUIRED IMPORTED_TARGET)
  add_build_config(PkgConfig::LibXml2 "libxml2")
  set(HAVE_LIBXML2 1)
endif()

# x264: fedora:x264-devel debian:libx264-dev
find_package(LibX264 0.118)
if(ENABLE_X264 AND LibX264_VERSION)
  add_build_config(LibX264::LibX264 "x264")
  set(CONFIG_LIBX264 ${LibX264_FOUND})
endif()

# x265: fedora:x265-devel debian:libx265-dev
find_package(LibX265)
if(ENABLE_X265 AND LibX265_VERSION)
  add_build_config(LibX265::LibX265 "x265")
  set(CONFIG_LIBX265 ${LibX265_FOUND})
endif()

# xvid: fedora:xvidcore-devel debian:libxvidcore-dev
find_package(LibXvid)
if(ENABLE_XVID AND LibX2vid_VERSION)
  add_build_config(LibXvid::LibXvid "xvid")
  set(CONFIG_LIBXVID ${LibXvid_FOUND})
endif()

# vpx: fedora:libvpx-devel debian:libvpx-dev
# Not used by MythTV code, only by FFmpeg.
if(ENABLE_VPX)
  pkg_check_modules(LIBVPX "vpx" IMPORTED_TARGET)
  add_build_config(PkgConfig::LIBVPX "vpx")
endif()

# mp3lame: fedora:lame-devel debian:libmp3lame-dev
if(ENABLE_MP3LAME AND (NOT MINGW AND NOT MSVC))
  find_package(Lame 3.98.3 MODULE REQUIRED)
  if(Lame_VERSION STREQUAL "unknown")
    message(STATUS "LAME VERSION IS UNKNOWN.")
  endif()
  add_build_config(Lame::Lame "libmp3lame")
  add_library(lame ALIAS Lame::Lame)
  set(CONFIG_LIBMP3LAME TRUE)
endif()

# vdpau: fedora:libvdpau-devel debian:libvdpau-dev
if(ENABLE_VDPAU)
  # Version 0.2 contains VDP_DECODER_PROFILE_MPEG4_PART2_ASP and
  # vdp_device_create_x11
  pkg_check_modules(VDPAU "vdpau>=0.2" IMPORTED_TARGET)
  add_build_config(PkgConfig::VDPAU "vdpau")
  set(CONFIG_VDPAU ${VDPAU_FOUND})
endif()

# vaapi: fedora:libva-devel debian:libva-dev
if(ENABLE_VAAPI)
  pkg_check_modules(VAAPI "libva>=1.2" IMPORTED_TARGET)
  add_build_config(PkgConfig::VAAPI "vaapi")
  if(TARGET PkgConfig::VAAPI)
    set(CONFIG_VAAPI TRUE)

    pkg_check_modules(VAAPI-DRM "libva-drm" IMPORTED_TARGET)
    add_build_config(PkgConfig::VAAPI-DRM "vaapi_drm")
    if(TARGET PkgConfig::VAAPI-DRM)
      set(CONFIG_VAAPI_DRM TRUE)
      target_link_libraries(PkgConfig::VAAPI INTERFACE PkgConfig::VAAPI-DRM)
    endif()

    pkg_check_modules(VAAPI-GLX "libva-glx" IMPORTED_TARGET)
    add_build_config(PkgConfig::VAAPI-GLX "vaapi_glx")
    if(TARGET PkgConfig::VAAPI-GLX)
      set(CONFIG_VAAPI_GLX TRUE)
      target_link_libraries(PkgConfig::VAAPI INTERFACE PkgConfig::VAAPI-GLX)
    endif()

    pkg_check_modules(VAAPI-X11 "libva-x11" IMPORTED_TARGET)
    add_build_config(PkgConfig::VAAPI-X11 "vaapi_x11")
    if(TARGET PkgConfig::VAAPI-X11)
      set(CONFIG_VAAPI_X11 TRUE)
      target_link_libraries(PkgConfig::VAAPI INTERFACE PkgConfig::VAAPI-X11)
    endif()
  endif()
endif()

# ass: fedora:libass-devel debian:libass-dev
if(ENABLE_LIBASS)
  pkg_check_modules(LIBASS "libass>=0.9.10" IMPORTED_TARGET)
  add_build_config(PkgConfig::LIBASS "libass")
  set(CONFIG_LIBASS ${LIBASS_FOUND})
endif()

# udev: fedora:libdav1d-devel debian:libdav1d-dev
if(ENABLE_LIBDAV1D)
  pkg_check_modules(LIBDAV1D "dav1d" IMPORTED_TARGET)
  add_build_config(PkgConfig::LIBDAV1 "dav1d")
endif()

# aom: fedora:libaom-devel debian:libaom-dev
if(ENABLE_LIBAOM)
  pkg_check_modules(LIBAOM "aom" IMPORTED_TARGET)
  add_build_config(PkgConfig::LIBAOM "aom")
endif()

# gnutls: fedora:gnutls-devel debian:libgnutls28-dev
if(ENABLE_GNUTLS)
  pkg_check_modules(GNUTLS "gnutls" IMPORTED_TARGET)
  add_build_config(PkgConfig::GNUTLS "gnutls")
endif()

# sdl2: fedora:SDL2-devel debian:libsdl2-dev
if(ENABLE_SDL2)
  pkg_check_modules(SDL2 "sdl2" IMPORTED_TARGET)
  add_build_config(PkgConfig::SDL2 "sdl2")
endif()

# drm: fedora:libdrm-devel debian:libdrm-dev
if(ENABLE_DRM)
  # ~~~
  # Drop old checks:
  # 2.4.82: first release with DRM_FORMAT_MOD_LINEAR (2017)
  # 2.4.104: first release with hdr_metadata_infoframe (2021)
  # ~~~
  pkg_check_modules(DRM "libdrm>=2.4.104" IMPORTED_TARGET)
  add_build_config(PkgConfig::DRM "drm")
  if(DRM_FOUND)
    set(CONFIG_DRM TRUE)
    set_if_target_exists(CONFIG_DRM_VIDEO Qt${QT_VERSION_MAJOR}::GuiPrivate)
    set(HAVE_STRUCT_HDR_METADATA_INFOFRAME TRUE)
  endif()
endif()

#
# See if there is a system libbluray for FFmpeg.
#
# bluray: fedora:libbluray-devel debian:libbluray-dev
pkg_check_modules(SYSTEM_LIBBLURAY "libbluray>=0.9.3" IMPORTED_TARGET)
add_build_config(PkgConfig::SYSTEM_LIBBLURAY "system_libbluray")
if(TARGET PkgConfig::SYSTEM_LIBBLURAY)
  message(STATUS
    "Found libbluray ${SYSTEM_LIBBLURAY_VERSION} ${SYSTEM_LIBBLURAY_LINK_LIBRARIES}")
endif()

# valgrind - needs valgrind-tools-devel
if(ENABLE_VALGRIND)
  pkg_check_modules(VALGRIND "valgrind" IMPORTED_TARGET)
  set(CONFIG_VALGRIND ${VALGRIND_FOUND})
endif()
