#ifndef H_CDW_FS
#define H_CDW_FS

#include <stddef.h> /* size_t */
#include <stdbool.h>
#include <unistd.h> /* int permissions */
#include <dirent.h>

#include <sys/types.h>
#include <sys/stat.h>

#include "main.h"
#include "cdw_dll.h"
#include "cdw_file.h"


typedef struct {
	bool has_file_over_4GB;
	long long size;
	bool follow_symlinks;
} cdw_fs_visitor_data_t;


#define CDW_FS_CHECK_N_ITEMS   12

enum {
	CDW_FS_CHECK_OK           = 0x00000001,

	CDW_FS_CHECK_IS_FILE      = 0x00000002,
	CDW_FS_CHECK_IS_DIR       = 0x00000004,

	CDW_FS_CHECK_EXISTS       = 0x00000008,
	CDW_FS_CHECK_DOESNT_EXIST = 0x00000010,
	CDW_FS_CHECK_NO_PARENT    = 0x00000020,

	CDW_FS_CHECK_WRONG_PERM   = 0x00000040,
	CDW_FS_CHECK_E_PERM_NO_R  = 0x00000080,
	CDW_FS_CHECK_E_PERM_NO_W  = 0x00000100,
	CDW_FS_CHECK_E_PERM_NO_X  = 0x00000200,

	CDW_FS_CHECK_SYS_ERROR    = 0x00000400,

	CDW_FS_CHECK_GEN_ERROR    = 0x00000800
};




typedef void (*cdw_fs_visitor_t)(const char *fullpath, struct stat *finfo, cdw_fs_visitor_data_t *data);

cdw_rv_t cdw_fs_init(void);
void cdw_fs_clean(void);
const char *cdw_fs_get_tmp_dir_fullpath(void);
const char *cdw_fs_get_home_dir_fullpath(void);
char       *cdw_fs_get_initial_dirpath(void);
char       *cdw_fs_get_home_or_tmp_dirpath(void);

cdw_rv_t cdw_fs_check_device_mounted(const char *device_fullpath);

int cdw_fs_check_existing_path(char *fullpath, int permissions, int filetype);
int cdw_fs_check_fullpath(const char *fullpath, int expected_file_type, int expected_permissions, int expected_new_or_existing);
int cdw_fs_check_dirpath(const char *dirpath);
int cdw_fs_check_selected_file_requirements(int rv, int current_file_type, int expected_file_type, int expected_perms, int expected_new_or_existing);
cdw_rv_t cdw_fs_correct_dir_path_ending(char **path);
ssize_t cdw_fs_get_filename_start(const char *fullpath);
bool cdw_fs_is_hidden(const char *file_name);
char *cdw_fs_get_target_path_simple(const char *fullpath);

cdw_rv_t cdw_fs_traverse_paths(char **pathtable, int n_paths, cdw_fs_visitor_t visitor, cdw_fs_visitor_data_t *visitor_data);
cdw_rv_t cdw_fs_traverse_path(char *fullpath, cdw_fs_visitor_t visitor, cdw_fs_visitor_data_t *visitor_data);
void cdw_fs_visitor(const char *fullpath, struct stat *finfo, cdw_fs_visitor_data_t *data);

char *cdw_fs_shorten_fullpath(cdw_file_t *file);

long long cdw_fs_get_file_size(const char *fullpath);
unsigned long long cdw_fs_copy_dirent_to_list(cdw_dll_item_t **list, const char *dirpath, struct dirent **eps, unsigned long long n_files, bool include_hidden);

int cdw_scandir_filter_one(const struct dirent *entry);
int cdw_scandir_filter_two(const struct dirent *entry);

cdw_rv_t cdw_fs_errno_handler(int e);

const char *cdw_fs_check_get_message(int id);
const char *cdw_fs_check_get_debug_message(int id);

/* for unit tests */
void cdw_fs_run_tests(void);


#endif /* H_CDW_FS */
