------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2000-2006 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-parameters.adb,v 1.17 2008/07/03 06:22:48 vagul Exp $

with
   Ada.Unchecked_Conversion,
   OCI.Thread;

package body OCI.Thick.Parameters is

   use type SWord;
   use type Ub4;
   use type OCIHandle;

   function To_SQL_Type is
     new Ada.Unchecked_Conversion (Ub2, SQL_Type);

   function Get_Attr_Ub2 is new Get_Attr_G (Ub2);
   function Get_Attr_Ub1 is new Get_Attr_G (Ub1);
   function Get_Attr_Sb1 is new Get_Attr_G (Sb1);

   procedure Init
     (Item  : in out Parameter;
      Stmt  : in Statement;
      Index : in Positive);

   ------------
   -- Column --
   ------------

   function Column
     (Stmt : in Statement; Position : in Positive) return Parameter
   is
      Result : Parameter;
   begin
      Init (Result, Stmt, Position);

      return Result;
   end Column;

   ---------------
   -- Data_Size --
   ---------------

   function Data_Size (Param : Parameter) return Natural is
   begin
      return Natural (Get_Attr_Ub2
                        (Param.Handle, OCI_DTYPE_PARAM, OCI_ATTR_DATA_SIZE));
   end Data_Size;

   ---------------
   -- Data_Type --
   ---------------

   function Data_Type (Param : Parameter) return SQL_Type is
   begin
      return To_SQL_Type (Get_Attr_Ub2
                            (Param.Handle,
                             OCI_DTYPE_PARAM,
                             OCI_ATTR_DATA_TYPE));
   end Data_Type;

   -------------
   -- Destroy --
   -------------

   procedure Destroy  (Object : in out Parameter) is
      Rc : SWord;
   begin
      if Object.Handle = Empty_Handle then
         return;
      end if;

      Rc := OCIDescriptorFree
              (Descp => Object.Handle, Dtype => OCI_DTYPE_PARAM);

      Object.Handle := Empty_Handle;

      Check_Error (Rc);
   end Destroy;

   --------------------
   -- Get_Parameters --
   --------------------

   function Get_Parameters (Stmt : Statement) return Parameter_Array is
      Result : Parameter_Array (1 .. Number_Of_Columns (Stmt));
   begin
      for J in Result'Range loop
         Init (Result (J), Stmt, J);
      end loop;

      return Result;
   end Get_Parameters;

   ----------
   -- Init --
   ----------

   procedure Init
     (Item  : in out Parameter;
      Stmt  : in Statement;
      Index : in Positive)
   is
      Result : aliased OCIParam;
      Rc     : constant SWord := OCIParamGet
             (Hndlp   => Handle (Stmt),
              Htype   => OCI_HTYPE_STMT,
              Errhp   => Thread.Error,
              Parmdpp => Result'Access,
              Pos     => Ub4 (Index));
   begin
      --  if Rc = OCI_NO_DATA then
      --     return OCIParam (Empty_Handle);
      --  end if;

      Check_Error (Rc);

      Item.Handle    := OCIHandle (Result);
      Item.Last_Stmt := Stmt;
   end Init;

   -------------
   -- Is_Null --
   -------------

   function Is_Null (Param : Parameter) return Boolean is
      use type Ub1;
   begin
      return Get_Attr_Ub1 (Param.Handle, OCI_DTYPE_PARAM, OCI_ATTR_IS_NULL)
             /= 0;
   end Is_Null;

   ----------
   -- Name --
   ----------

   function Name (Param : Parameter) return String is
      H : constant OCIHandle := Param.Handle;
   begin
      if H = Empty_Handle then
         return "";
      else
         return Get_Attr (H, OCI_DTYPE_PARAM, OCI_ATTR_NAME);
      end if;
   end Name;

   ---------------
   -- Precision --
   ---------------

   function Precision (Param : Parameter) return Integer is
      use System;
      use type Sb2;

      B2 : aliased Sb2 := 0;
      T2 : aliased Sb2;
      T1 : aliased Ub1;
      for T1'Address use T2'Address;
   begin
      Check_Error (OCIAttrGet
                     (Trgthndlp  => Param.Handle,
                      Trghndltyp => OCI_DTYPE_PARAM,
                      Attributep => B2'Address,
                      Sizep      => null,
                      Attrtype   => OCI_ATTR_PRECISION,
                      Errhp      => Thread.Error));

      if Default_Bit_Order = Low_Order_First then
         return Integer (B2);
      end if;

      --  Sb2 result type is for implicit describe by OCIStmtExecute()
      --  Ub1 result for explicit describe with OCIDescribeAny()
      --  Try to detect attribute size.

      T2 := -B2;

      Check_Error (OCIAttrGet
                     (Trgthndlp  => Param.Handle,
                      Trghndltyp => OCI_DTYPE_PARAM,
                      Attributep => T2'Address,
                      Sizep      => null,
                      Attrtype   => OCI_ATTR_PRECISION,
                      Errhp      => Thread.Error));

      if T2 = B2 then
         return Integer (B2);
      else
         return Integer (T1);
      end if;
   end Precision;

   -----------
   -- Scale --
   -----------

   function Scale (Param : Parameter) return Integer is
   begin
      return Integer (Get_Attr_Sb1 (Param.Handle,
                                    OCI_DTYPE_PARAM,
                                    OCI_ATTR_SCALE));
   end Scale;

   -----------------
   -- Schema_Name --
   -----------------

   function Schema_Name (Param : Parameter) return String is
   begin
      return Get_Attr (Param.Handle, OCI_DTYPE_PARAM, OCI_ATTR_SCHEMA_NAME);
   end Schema_Name;

   ---------------
   -- Type_Name --
   ---------------

   function Type_Name (Param : Parameter) return String is
      H : constant OCIHandle := Param.Handle;
   begin
      if H = Empty_Handle then
         return "";
      else
         return Get_Attr (H, OCI_DTYPE_PARAM, OCI_ATTR_TYPE_NAME);
      end if;
   end Type_Name;

end OCI.Thick.Parameters;
