package rc2

import (
	"bytes"
	"testing"
)

// rfc 2268 Sec. 5
var testcases = []struct {
	key    []byte
	bits   uint
	plain  []byte
	cipher []byte
}{
	{
		key:    []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		bits:   63,
		plain:  []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		cipher: []byte{0xeb, 0xb7, 0x73, 0xf9, 0x93, 0x27, 0x8e, 0xff},
	}, {
		key:    []byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
		bits:   64,
		plain:  []byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
		cipher: []byte{0x27, 0x8b, 0x27, 0xe4, 0x2e, 0x2f, 0x0d, 0x49},
	}, {
		key:    []byte{0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		bits:   64,
		plain:  []byte{0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01},
		cipher: []byte{0x30, 0x64, 0x9e, 0xdf, 0x9b, 0xe7, 0xd2, 0xc2},
	}, {
		key:    []byte{0x88},
		bits:   64,
		plain:  []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		cipher: []byte{0x61, 0xa8, 0xa2, 0x44, 0xad, 0xac, 0xcc, 0xf0},
	}, {
		key:    []byte{0x88, 0xbc, 0xa9, 0x0e, 0x90, 0x87, 0x5a},
		bits:   64,
		plain:  []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		cipher: []byte{0x6c, 0xcf, 0x43, 0x08, 0x97, 0x4c, 0x26, 0x7f},
	}, {
		key:	[]byte{0x88, 0xbc, 0xa9, 0x0e, 0x90, 0x87, 0x5a, 0x7f, 
			       0x0f, 0x79, 0xc3, 0x84, 0x62, 0x7b, 0xaf, 0xb2},
		bits:   64,
		plain:  []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0},
		cipher: []byte{0x1a, 0x80, 0x7d, 0x27, 0x2b, 0xbe, 0x5d, 0xb1},
	}, {
		key:    []byte{0x88, 0xbc, 0xa9, 0x0e, 0x90, 0x87, 0x5a, 0x7f, 
			       0x0f, 0x79, 0xc3, 0x84, 0x62, 0x7b, 0xaf, 0xb2},
		bits:   128,
		plain:  []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		cipher: []byte{0x22, 0x69, 0x55, 0x2a, 0xb0, 0xf8, 0x5c, 0xa6},
	}, {
		key:    []byte{0x88, 0xbc, 0xa9, 0x0e, 0x90, 0x87, 0x5a, 0x7f, 
			       0x0f, 0x79, 0xc3, 0x84, 0x62, 0x7b, 0xaf, 0xb2, 
			       0x16, 0xf8, 0x0a, 0x6f, 0x85, 0x92, 0x05, 0x84, 
			       0xc4, 0x2f, 0xce, 0xb0, 0xbe, 0x25, 0x5d, 0xaf, 
			       0x1e},
		bits:   129,
		plain:  []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0},
		cipher: []byte{0x5b, 0x78, 0xd3, 0xa4, 0x3d, 0xff, 0xf1, 0xf1},
	},
}

func TestRC2(t *testing.T) {
	buf := make([]byte, 8)

	for i, tc := range testcases {
		if tc.bits != uint(len(tc.key)<<3) {
			// skip
			continue
		}
		c, err := NewCipher(tc.key)
		if err != nil {
			t.Fatalf("testcase%d NewCipher: %s\n", i, err)
		}
		c.Encrypt(buf, tc.plain)
		if !bytes.Equal(buf, tc.cipher) {
			t.Errorf("testcase%d: encrypt got %x expected %x\n", i, buf, tc.cipher)
		}
		c.Decrypt(buf, tc.cipher)
		if !bytes.Equal(buf, tc.plain) {
			t.Errorf("testcase%d: decrypt got %x expected %x\n", i, buf, tc.plain)
		}
	}
}

func TestRC2Reduced(t *testing.T) {
	buf := make([]byte, 8)

	for i, tc := range testcases {
		c, err := NewCipherReducedStrength(tc.key, tc.bits)
		if err != nil {
			t.Fatalf("testcase%d NewCipherReducedStrength: %s\n", i, err)
		}
		c.Encrypt(buf, tc.plain)
		if !bytes.Equal(buf, tc.cipher) {
			t.Errorf("testcase%d: encrypt got %x expected %x\n", i, buf, tc.cipher)
		}
		c.Decrypt(buf, tc.cipher)
		if !bytes.Equal(buf, tc.plain) {
			t.Errorf("testcase%d: decrypt got %x expected %x\n", i, buf, tc.plain)
		}
	}
}

