//  This software code is made available "AS IS" without warranties of any
//  kind.  You may copy, display, modify and redistribute the software
//  code either by itself or as incorporated into your code; provided that
//  you do not remove any proprietary notices.  Your use of this software
//  code is at your own risk and you waive any claim against Amazon
//  Digital Services, Inc. or its affiliates with respect to your use of
//  this software code. (c) 2006 Amazon Digital Services, Inc. or its
//  affiliates.

package net.noderunner.amazon.s3;

import java.util.Date;

/**
 * A structure representing a single object stored in S3.
 */
public class Entry implements Comparable<Entry> {
	
    private String key;

    private Date lastModified;

    private String eTag;

    private long size;

    private String storageClass;

    private Owner owner;
    
    /**
     * Constructs a new Entry.
     */
    Entry() {
    }

    /**
     * Constructs a new Entry.
     */
    public Entry(String key) {
    	setKey(key);
    }

    /**
	 * Sets lastModified.
	 */
	public void setLastModified(Date lastModified) {
		this.lastModified = lastModified;
	}

    /**
     * The date at which the object was last modified.
     */
	public Date getLastModified() {
		return (Date) lastModified.clone();
	}

	/**
	 * Returns the key
	 */
	@Override
	public String toString() {
        return getKey();
    }

	/**
	 * Sets key.
	 */
	void setKey(String key) {
		this.key = key;
	}

    /**
     * Returns the name of the object
     */
	public String getKey() {
		return key;
	}

	/**
	 * Sets eTag.
	 */
	void setETag(String eTag) {
		this.eTag = eTag;
	}

    /**
     * Returns the object's ETag, which can be used for conditional GETs.
     */
	public String getETag() {
		return eTag;
	}

	/**
	 * Sets owner.
	 */
	public void setOwner(Owner owner) {
		this.owner = owner;
	}

    /**
     * Returns the object's owner
     */
	public Owner getOwner() {
		return owner;
	}

	/**
	 * Sets size.
	 */
	public void setSize(long size) {
		this.size = size;
	}

    /**
     * Returns the size of the object in bytes.
     */
	public long getSize() {
		return size;
	}

	/**
	 * Sets storageClass.
	 */
	void setStorageClass(String storageClass) {
		this.storageClass = storageClass;
	}

    /**
     * The object's storage class
     */
	public String getStorageClass() {
		return storageClass;
	}
	
	/**
	 * Returns true if other is an entry with the same key.
	 */
	@Override
	public boolean equals(Object other) {
		if (!(other instanceof Entry))
			return false;
		Entry entry = (Entry)other;
		return key.equals(entry.key);
	}
	
	/**
	 * Calculates hash using the key.
	 */
	@Override
	public int hashCode() {
		return key.hashCode();
	}

	/**
	 * Compares by key name.
	 */
	public int compareTo(Entry other) {
		return key.compareTo(other.key);
	}
}
