
#include "protoLFSR.h"
#include "protoDebug.h"
#include <stdlib.h>  // for abs()

ProtoLFSR::ProtoLFSR(Polynomial polynomial, 
                     UINT32     initialState,
                     bool       reverse)
 : lfsr_poly((UINT32)polynomial),
   lfsr_state(initialState),
   lfsr_bits(GetPolySize(polynomial)),
   is_mirrored(false), byte_mode(false)
{
    lfsr_mask = ((UINT32)0xffffffff) >> (32 - lfsr_bits);
    lfsr_state &= lfsr_mask;
    if (reverse)
    {
        Mirror();
        is_mirrored = false;
    }
}

unsigned int ProtoLFSR::GetPolySize(Polynomial poly)
{
    unsigned int numBits = 0;
    UINT32 p = (UINT32)poly;
    while (0 != p)
    {
        p >>= 1;
        numBits++;
    }
    return numBits;
}  // end ProtoLFSR::GetPolySize()


const ProtoLFSR::Polynomial ProtoLFSR::POLYNOMIAL_LIST[33] =
{
    PN_NONE,    //  0 bits (invalid)
    PN_NONE,    //  1 bit  (TBD)
    PN3,        //  2 bits
    PN7,        //  3 bits
    PN15,       //  4 bits
    PN31,       //  5 bits
    PN63,       //  5 bits
    PN127,      //  7 bits
    PN255,      //  8 bits
    PN511,      //  9 bits
    PN1023,     // 10 bits
    PN2047,     // 11 bits
    PN4095,     // 12 bits
    PN8191,     // 13 bits
    PN_NONE,    // 14 bits (TBD)
    PN_NONE,    // 15 bits (TBD)
    PN65535,    // 16 bits
    PN_NONE,    // 17 bits (TBD)  
    PN_NONE,    // 18 bits (TBD)  
    PN_NONE,    // 19 bits (TBD)
    PN_NONE,    // 20 bits (TBD)
    PN_NONE,    // 21 bits (TBD) 
    PN_NONE,    // 22 bits (TBD) 
    PN_NONE,    // 23 bits (TBD) 
    PN24BIT,    // 24 bits  
    PN_NONE,    // 25 bits  (TBD)
    PN_NONE,    // 26 bits  (TBD)
    PN_NONE,    // 27 bits  (TBD)
    PN_NONE,    // 28 bits  (TBD)
    PN_NONE,    // 29 bits  (TBD)
    PN_NONE,    // 30 bits  (TBD)
    PN_NONE,    // 31 bits  (TBD)
    PN32BIT     // 32 bits
};

void ProtoLFSR::Seek(int offsetBits)
{
    if (offsetBits < 0)
    {
        if (!IsMirrored()) 
        {
            Mirror(); // reverse here does a -1 offset on its own, so adjust
            //offsetBits++;
        }
    }
    else if (IsMirrored())
    {
        Mirror();  // reverse here does a +1 offset on its own, so adjust
        //offsetBits--;
    }
    Shift(abs(offsetBits));
    byte_mode = false;
}  // end ProtoLFSR::Seek()

void ProtoLFSR::Shift(unsigned int count)
{
    for (unsigned int i = 0; i < count; i++)
    {
        bool bit = (0 != (lfsr_state & 1));
        lfsr_state >>= 1;
        if (bit) lfsr_state ^= lfsr_poly;
    }
}  // end ProtoLFSR::Shift()

// This is used to "reverse load" the shift
// register towards the state it would be
// in to generate the sequence of "bits
// (used by the ProtoLFSR::Sync() method)
void ProtoLFSR::LoadBit(bool bit)
{
    if (bit) lfsr_state ^= lfsr_poly;
    lfsr_state <<= 1;
    lfsr_state &= lfsr_mask;
    if (bit) lfsr_state |= 1;
}  // end ProtoLFSR::LoadBit()

// This sets the LFSR to the state it would be in _before_
// generating the first "lfsr_bits" (register length) of bits 
// provided in the "buffer".  This can be used to "sync" the 
// LFSR to a received sequence of bits.
bool ProtoLFSR::Sync(const char* buffer, unsigned int buflen, unsigned int bitOffset)
{
    if ((buflen << 3) < (lfsr_bits + bitOffset)) return false;
    Reset(0);
    for (int i = bitOffset+lfsr_bits-1; i >= (int)bitOffset; i--)
        LoadBit(GetBit(buffer, i));
    return true;
}  // end ProtoLFSR::SetState()


UINT32 ProtoLFSR::MirrorBits(UINT32 word, unsigned int numBits)
{
    UINT32 bit = 0x00000001 << (numBits - 1);
    UINT32 mbit = 1;
    UINT32 mirrorWord = 0;
    while (0 != bit)
    {
        if (0 != (bit & word))
            mirrorWord |= mbit;
        mbit <<= 1;
        bit >>= 1;
    }
    return mirrorWord;
}  // end ProtoLFSR::MirrorBits()

// Mirror the generator polynomial and state to generate
// the time-reversed version of the sequence
void ProtoLFSR::Mirror()
{
    // 1) "Mirror" the polynomial
    // Starting at most significant bit, mirror
    // all except most significant bit 
    // (which is always set)
    UINT32 mirrorPoly = MirrorBits(lfsr_poly, lfsr_bits - 1);
    // Set the most significant bit
    mirrorPoly |= 0x00000001 << (lfsr_bits - 1);
    lfsr_poly = mirrorPoly;
    // 2) "Mirror" the shift register state
    lfsr_state = MirrorBits(lfsr_state, lfsr_bits);
    is_mirrored = is_mirrored ? false : true;
}  // end ProtoLFSR::Mirror()

bool ProtoLFSR::GetNextBit()
{
    byte_mode = false;
    if (IsMirrored()) 
    {
        Mirror();
        Shift();
    }
    bool bit = (0 != (lfsr_state & 0x00000001));
    Shift();
    return bit;
}  // end ProtoLFSR:::GetNextBit()

UINT8 ProtoLFSR::GetNextByte()
{
    if (IsMirrored())
    {
        Shift();    
        if (byte_mode)
        {
            GetNextBit();
            Shift(7);
        }
    }
    UINT8 nextByte = GetNextBit() ? 0x01 : 0x00;
    for (int i = 1; i < 8; i++)
    {
        nextByte <<= 1;
        if (GetNextBit()) nextByte |= 0x01;
    }
    byte_mode = true;
    return nextByte;
}  // end ProtoLFSR:::GetNextByte()


bool ProtoLFSR::GetPrevBit()
{
    byte_mode = false;
    if (!IsMirrored()) 
    {
        Mirror();
        Shift();
    }
    bool bit = (0 != (lfsr_state & 0x000000001));
    Shift();
    return bit;
}  // end ProtoLFSR::GetPrevBit()

UINT8 ProtoLFSR::GetPrevByte()
{
    if (!IsMirrored())
    {
        Shift();
        if (byte_mode)
        {
            GetPrevBit();
            Shift(7);
        }
    }
    UINT8 prevByte = GetPrevBit() ? 0x80 : 0x00;
    for (int i = 1; i < 8; i++)
    {
       prevByte >>= 1;
       if (GetPrevBit()) prevByte |= 0x80;
    }
    byte_mode = true;
    return prevByte;
}  // end GetPrevByte()

void ProtoLFSR::FillBuffer(char* buffer, unsigned int buflen)
{
    for (unsigned int i = 0; i < buflen; i++)
        buffer[i] = GetNextByte();
}  // end ProtoLFSR::FillBuffer()


// This function searches for an m-sequence generator polynomial 
// for a given shift register size of "m" bits        
UINT32 ProtoLFSR::PolynomialSearch(unsigned int m)
{
    const UINT32 LEN = 0x00000001 << m;
    
    char** seq = new char*[LEN];
    if (NULL == seq)
    {
        PLOG(PL_ERROR, "ProtoLFSR::PolynomialSearch() new 'seq' error: %s", GetErrorString());
        return 0;
    }
    for (UINT32 i = 0; i < LEN; i++)
    {
        seq[i] = new char[LEN >> 3];
        if (NULL == seq[i])
        {
            PLOG(PL_ERROR, "ProtoLFSR::PolynomialSearch() new 'seq[%lu] error: %s", i, GetErrorString());
            for (UINT32 j = 0; j < i; j++)
                delete[] seq[j];
            delete[] seq;
            return 0;
        }
    }
    
    
    UINT32 maxPoly = 0;
    unsigned int maxMin = 0;
    
    // min possible poly has bit "m-1" set
    UINT32 polyMin = LEN >> 1;
    // max possible poly has all "m" bits set
    UINT32 polyMax = 0xffffffff;
    polyMax >>= (32 - m);
    
    for (UINT32 poly = polyMin; poly <= polyMax; poly++)
    {
        for (unsigned int i = 0; i < LEN; i++)
        {
            ProtoLFSR lfsr((ProtoLFSR::Polynomial)poly);
            lfsr.Seek(i);
            lfsr.FillBuffer(seq[i], LEN >> 3);
        }

        unsigned int wtMin = 0xffffffff;
        unsigned int offset = 0;
        for (unsigned int i = 1; i < LEN-1; i++)
        {
            unsigned int wt = 0;
            for (unsigned int j = 0; j < (LEN >> 3); j++)
            {
                unsigned char delta = (unsigned char)(seq[0][j] ^ seq[i][j]);
                wt += ProtoBitmask::GetWeight(delta);
            }
            if (wt < wtMin)
            {
                wtMin = wt;
                offset = i;
            }
        }    
        if (wtMin > maxMin)
        {
            maxMin = wtMin;
            maxPoly = poly;
        }
    }
    for (UINT32 j = 0; j < LEN; j++)
        delete[] seq[j];
    delete[] seq;
    
    return maxPoly;
}  // end ProtoLFSR::PolynomialSearch()
