/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2012-2016 OpenFOAM Foundation
    Modified code Copyright (C) 2015-2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fvMeshTools

Description
    A collection of tools for operating on an fvMesh.

SourceFiles
    fvMeshTools.C

\*---------------------------------------------------------------------------*/

#ifndef fvMeshTools_H
#define fvMeshTools_H

#include "fvMesh.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                         Class fvMeshTools Declaration
\*---------------------------------------------------------------------------*/

class fvMeshTools
{
    template<class GeoField>
    static void addPatchFields
    (
        fvMesh&,
        const dictionary& patchFieldDict,
        const word& defaultPatchFieldType,
        const typename GeoField::value_type& defaultPatchValue
    );

    //- Set patchFields according to dictionary
    template<class GeoField>
    static void setPatchFields
    (
        fvMesh& mesh,
        const label patchi,
        const dictionary& patchFieldDict
    );

    //- Set patchFields to value
    template<class GeoField>
    static void setPatchFields
    (
        fvMesh& mesh,
        const label patchi,
        const typename GeoField::value_type& value
    );

    // Remove last patch fields
    template<class GeoField>
    static void trimPatchFields(fvMesh&, const label nPatches);

    template<class GeoField>
    static void reorderPatchFields(fvMesh&, const labelList& oldToNew);

    // Remove trailing patches
    static void trimPatches(fvMesh&, const label nPatches);


public:

    //- Add patch. Inserts patch before all processor patches.
    //  Supply per field the new patchField per field as a
    //  subdictionary or a default type. If validBoundary call is parallel
    //  synced and all add the same patch with same settings.
    static label addPatch
    (
        fvMesh& mesh,
        const polyPatch& patch,
        const dictionary& patchFieldDict,
        const word& defaultPatchFieldType,
        const bool validBoundary
    );

    //- Change patchField on registered fields according to dictionary
    static void setPatchFields
    (
        fvMesh& mesh,
        const label patchi,
        const dictionary& patchFieldDict
    );

    //- Change patchField to zero on registered fields
    static void zeroPatchFields(fvMesh& mesh, const label patchi);

    //- Reorder and remove trailing patches. If validBoundary call is parallel
    //  synced
    static void reorderPatches
    (
        fvMesh&,
        const labelList& oldToNew,
        const label nPatches,
        const bool validBoundary
    );
    //- Remove zero sized patches. All but processor patches are
    //  assumed to be present on all processors (so size will be reduced
    //  if validBoundary). Return map from new
    //  to old patches
    static labelList removeEmptyPatches(fvMesh&, const bool validBoundary);

    //- Read mesh or create dummy mesh (0 cells, >0 patches). Works in two
    //  modes according to masterOnlyReading:
    //  true : all slaves create dummy mesh
    //  false: checks locally for mesh directories and only creates dummy mesh
    //         if not present
    static autoPtr<fvMesh> newMesh
    (
        const IOobject& io,
        const bool masterOnlyReading
    );

    //- Create additional fv* files
    static void createDummyFvMeshFiles
    (
        const objectRegistry& parent,
        const word& regionName,
        const bool verbose = false
    );
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "fvMeshToolsTemplates.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
