/*	Message

PIRL CVS ID: Message.java,v 1.50 2012/04/16 06:11:36 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Messenger;

import	PIRL.PVL.Parameter;
import	PIRL.PVL.Value;
import	PIRL.PVL.Parser;
import	PIRL.PVL.Lister;
import	PIRL.PVL.Selection;
import	PIRL.PVL.PVL_Exception;
import	PIRL.Utilities.ByteBuffer_InputStream;
import	PIRL.Strings.String_Buffer;

import	java.nio.ByteBuffer;
import	java.io.InputStreamReader;
import	java.io.IOException;
import	java.io.UnsupportedEncodingException;
import	java.util.Iterator;


/**	A <i>Message</i> contains a message and its routing information.
<p>
	A Message is a Parameter Aggregate and the message routing information.
	A Message is typically managed by a Messenger.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.50
	@see		Parameter
	@see		Messenger
*/
public class Message
	extends Parameter
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Message (1.50 2012/04/16 06:11:36)";


//	Common Message content parameters:

/**	A Message parameter specifying the action associated with the message.
*/
public static final String
	ACTION_PARAMETER_NAME		= "Action";

/**	Identify request Message {@link #ACTION_PARAMETER_NAME} value.
*/
public static final String
	IDENTIFY_ACTION				= "Identify";

/**	Identity Message {@link #ACTION_PARAMETER_NAME} value.
*/
public static final String
	IDENTITY_ACTION				= "Identity";

/**	Class ID parameter name.
*/
public static final String
	CLASS_ID_PARAMETER_NAME		= "Class_ID";

/**	Sender name in an {@link #IDENTIFY_ACTION} or {@link #IDENTITY_ACTION}
	action Message.
*/
public static final String
	NAME_PARAMETER_NAME			= "Name";

/**	Acknowledgment Message {@link #ACTION_PARAMETER_NAME} value.
*/
public static final String
	ACK_ACTION					= "ACK";

/**	Negative acknowledgment Message {@link #ACTION_PARAMETER_NAME} value.
*/
public static final String
	NACK_ACTION					= "NACK";

/**	End of messaging - no further messages can be expected - Message
	{@link #ACTION_PARAMETER_NAME} value.
*/
public static final String
	DONE_ACTION					= "Done";


//	Message label parameters:

/**	Synchronization sequence at the beginning of a trasmitted message.
<p>
	@see	#Packet_Label(int)
*/
public static final String
	START_MARKER				= "_Message_";

/**	Separates the label {@link #START_MARKER} from its label
	size value in a transmitted message.
<p>
	@see	#Packet_Label(int)
*/
public static final char
	START_MARKER_DELIMITER		= Parser.PARAMETER_NAME_DELIMITER;

/**	Marks the end of the message label size value and the required
	message transmission start sequence.
<p>
	@see	#Packet_Label(int)
*/
public static final char
	START_MARKER_END			= Parser.STATEMENT_END_DELIMITER;

/**	A label parameter of a transmitted message with a value that is the
	number of message label characters to follow.
<p>
	@see	#Packet_Label(int)
*/
public static final String
	CONTENT_AMOUNT_PARAMETER_NAME	= "Size";

/**	A label parameter of a transmitted message with a value that is the
	Message {@link #Route_To() route-to list}.
<p>
	@see	#Packet_Label(int)
*/
public static final String
	ROUTE_TO_PARAMETER_NAME			= "Route_To";

/**	A label parameter of a transmitted message with a value that is the
	Message {@link #Route_To() route-from list}.
<p>
	@see	#Packet_Label(int)
*/
public static final String
	ROUTE_FROM_PARAMETER_NAME		= "Route_From";


//	The Message routing address lists.
private Value
	Route_To					= null,
	Route_From					= null;


/*	>>> WARNING <<<
	The Parser and Lister objects are not static to prevent collisions
	across threads in the Messenger.
*/
//	PVL Parser for parsing raw byte content.
private final Parser
	PVL_Parser					= PVL_Parser ();

//	PVL Lister for producing raw byte content and String representations.
private final Lister
	PVL_Lister					= PVL_Lister ();

/**	Marks the first character of a Message {@link #PVL() String
	representation} when there was a problem rendering the Message
	content into valid PVL syntax.
*/
public static final char
	ERROR_MESSAGE_MARKER		= Lister.NEW_LINE.charAt (0);

private static String
	NL							= System.getProperty ("line.separator");


// Debug control.
private static final int
	DEBUG_OFF				= 0,
	DEBUG_CONSTRUCTORS		= 1 << 1,
	DEBUG_CONTENT			= 1 << 2,
	DEBUG_ROUTING			= 1 << 3,
	DEBUG_FUNCTIONS			= 1 << 4,
	DEBUG_ALL				= -1,

	DEBUG					= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct an empty Message.
<p>
	The Message will be an empty Group with emtpy {@link #Route_To()
	route-to} and {@link #Route_From() route-from} address lists. The
	Name of the Message will be the {@link Parser#CONTAINER_NAME}.
*/
public Message ()
{
super (Parser.CONTAINER_NAME);
Classification (Parameter.GROUP);
try
	{
	Route_To   = new Value ().Type (Value.SEQUENCE);
	Route_From = new Value ().Type (Value.SEQUENCE);
	}
catch (PVL_Exception exception) {/* The new Values can accept the Type. */}
}

/**	Construct a Message from a token name.
<p>
	A single Token Parameter with a name the same as the token and no
	Value is added to the new empty Message.
<p>
	@param	token	A Token Parameter name.
	@see	#Message()
*/
public Message
	(
	String		token
	)
{
this ();
Add (new Parameter (token));
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">-< Message token:" + NL
		+ this);
}

/**	Construct a copy of another Message.
<p>
	The Message Name, Parameter content and {@link #Route_To()
	route-to} and {@link #Route_From() route-from} address lists are
	copied into this Message.
<p>
	@param	message	The Message to be copied.
	@throws	PVL_Exception	If the message contains an invalid Parameter.
*/
public Message
	(
	Message		message
	)
	throws PVL_Exception
{
super (message);
Route_To   = new Value (message.Route_To);
Route_From = new Value (message.Route_From);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">-< Message copy:" + NL
		+ this);
}

/**	Construct a Message as a copy of a Parameter.
<p>
	The contents of the parameter are copied into an empty Message.
	If the copied Parameter is an Assignment it becomes the sole
	Parameter in this Message. If it is an Aggregate its Parameter list
	becomes the list of Parameters in this Message and its {@link #Name()
	name} becomes the name of this Message.
<p>
	@param	parameter	The Parameter to copy into the new Message.
	@throws	PVL_Exception	If the Parameter is or contains an invalid
		Parameter.
	@see	#Message()
*/
public Message
	(
	Parameter	parameter
	)
	throws PVL_Exception
{
this ();
if (parameter != null)
	{
	if (parameter.Has_List ())
		//	Copy in the Aggregate list.
		Add (new Parameter (parameter).List ())
			.Name (parameter.Name ());
	else
	if (parameter.Is_Assignment ())
		//	Copy in the Assignment.
		Add (new Parameter (parameter));
	}
}

/**	Construct a Message from the contents of a ByteBuffer.
<p>
	@param	content	The ByteBuffer containing the Message content.
	@see	#Content(ByteBuffer)
	@see	#Message()
*/
public Message
	(
	ByteBuffer	content
	)
	throws PVL_Exception
{
this ();
Content (content);
}

/*==============================================================================
	Accessors
*/
/**	Set the Message content.
<p>
	The current Message content is removed and the new content is added
	by {@link Parser#Parser(InputStream) parsing} the ByteBuffer data.
	<b>N.B.</b>: The ByteBuffer must be positioned at the beginning of
	the content data and its limit must be at or beyond the end of the
	content data. The position and limit will be unchanged when this
	method completes, whether normally or by throwing an exception.
<p>
	The ByteBuffer content data must be PVL formatted parameters. All
	the parameters found in the content buffer are added to the Message.
	The Message itself will retain its name.
<p>
	@param	content	The ByteBuffer containing the Message content. If null
		the current content of this Message is simply emptied.
	@return	This Message with the new content.
	@throws	PVL_Exception	If the content can not be parsed by the
		{@link #PVL_Parser() Message PVL parser}.
*/
public Message Content
	(
	ByteBuffer	content
	)
	throws PVL_Exception
{
if ((DEBUG & DEBUG_CONTENT) != 0)
	System.out.println
		(">>> Message.Content: " + content);
Remove_All ();
if (content != null)
	{
	int
		position = content.position ();
	if ((DEBUG & DEBUG_CONTENT) != 0)
		{
		System.out.print
			("    Content " + content + NL
			+">");
		while (content.hasRemaining ())
			System.out.print ((char)content.get ());
		System.out.println ("<");
		content.position (position);
		}

	PVL_Parser.Reset ();
	try
		{
		PVL_Parser.Set_Reader
			(new InputStreamReader
				(new ByteBuffer_InputStream (content),
					Parser.CHARACTER_ENCODING));
		Add (PVL_Parser);
		}
	catch (PVL_Exception exception)
		{
		content.position (position);
		throw new PVL_Exception
			(
			ID, PVL_Exception.FILE_IO,
			"Unable to parse the Message content -" + NL
			+ Content_String (content) + NL
			+ exception.getMessage ()
			);
		}
	catch (UnsupportedEncodingException exception)
		{
		throw new PVL_Exception
			(
			ID, PVL_Exception.FILE_IO,
			"Unable to parse the Message content with the \""
				+ Parser.CHARACTER_ENCODING + "\" character encoding."
			+ ((exception.getMessage () == null) ?
				"" : NL + exception.getMessage ())
			);
		}
	finally
		{
		content.position (position);
		}
	}
if ((DEBUG & DEBUG_CONTENT) != 0)
	System.out.println
		(this.toString () + NL
		+"<<< Message.Content");
return this;
}

/**	Get the String representation of Message content bytes.
<p>
	If the content buffer has an accessible backing array its contents,
	from the array offset to the first byte of the buffer plus the buffer
	position up to the buffer remaining amount, are converted. Otherwise
	the buffer content, from the buffer position up to the buffer
	remaining amount, are copied into a temporary array for conversion.
<p>
	The byte buffer content array is converted to a new String using the
	PVL {@link Parser#CHARACTER_ENCODING}.
<p>
	@param	content	A ByteBuffer containing the remaining bytes starting
		at its current position to be converted to a String.
	@return	A String representation of the buffer content. This will be
		null if the content is null or the content could not be
		converted.
*/
public static String Content_String
	(
	ByteBuffer	content
	)
{
String
	string = null;
if (content != null)
	{
	byte[]
		bytes;
	int
		offset = 0,
		amount = content.remaining ();
	if (content.hasArray ())
		{
		bytes = content.array ();
		offset = content.arrayOffset () + content.position ();
		}
	else
		{
		content = content.slice ();
		bytes = new byte[amount];
		while (content.hasRemaining ())
			bytes[offset++] = content.get ();
		offset = 0;
		}
	try {string =
			new String (bytes, offset, amount, Parser.CHARACTER_ENCODING);}
	catch (UnsupportedEncodingException exception)
		{/* If the Parser encoding is not supported we're SOL */}
	}
return string;
}

/**	Get the Message content as a transmittable packet.
<p>
	A Message packet is a ByteBuffer that contains Message routing
	information and Message content in a form that can be transmitted
	over a stream.
<p>
	The packet begins with a set of {@link #Packet_Label(int) label}
	parameters. The packet label is followed by the Message content as
	{@link #PVL() PVL formatted text}.
<p>
	All content is US_ASCII encoded that ensurces that 7-bit characters,
	one character per byte, will be used for the PVL text.
<p>
	@return	A ByteBuffer containing the Message label and content data.
	@throws	PVL_Exception	If a valid PVL representation can not be
		produced for the Message content.
	@see	#Packet_Label(int)
*/
public ByteBuffer Packet ()
	throws PVL_Exception
{
//	Content description.
String
	content = PVL ();
if (content.length () > 0 &&
	content.charAt (0) == ERROR_MESSAGE_MARKER)
	{
	throw new PVL_Exception
		(ID, PVL_Exception.ILLEGAL_SYNTAX,
		"Unable to generate the Message packet PVL." + NL
		+ content);
	}

//	Combine the Message label and content into a byte array.
byte[]
	bytes;
try {bytes = (Packet_Label (content.length ()) + content)
		.getBytes ("US-ASCII");}
catch (java.io.UnsupportedEncodingException exception)
	{
	throw new PVL_Exception
		(ID,  PVL_Exception.ILLEGAL_SYNTAX,
		"Unable to encode the Message packet PVL as US-ASCII bytes." + NL
		+ ((exception.getMessage () == null) ?
			"" : (exception.getMessage () + NL))
		+ content);
	}
return ByteBuffer.wrap (bytes);
}

/**	Get the message transmission packet label for a Message
<p>
	A Message label section contains PVL with the following parts:
<p>
<dl>
<dt>{@link #START_MARKER}
<dd>Used as a message synchronization marker to locate the beginning of a
	message in a byte stream.

<dt>{@link #START_MARKER_DELIMITER}
<dd>Separates the {@link #START_MARKER} from the label size value that
	immediately follows. There are no spaces around the {@link
	#START_MARKER_DELIMITER} character ('=') that separates the parameter
	name from its value.

<dt>Label size
<dd>The number of Message label characters that follow. <b>N.B.</b> the
	label size does not include the {@link #START_MARKER}, {@link
	#START_MARKER_DELIMITER}, the label size representation characters
	nor the {@link #START_MARKER_END} that immediately follows.

<dt>{@link #START_MARKER_END}
<dd>Marks the end of the label size value representation characters.
	There is no end-of-line sequence following this marker.

<dt>{@link #CONTENT_AMOUNT_PARAMETER_NAME}
<dd>A parameter with the {@link #CONTENT_AMOUNT_PARAMETER_NAME} name and
	the content amount as its assigned value. An {@link Lister#NEW_LINE
	end-of-line sequence} ends the parameter.

<dt>{@link #ROUTE_TO_PARAMETER_NAME}
<dd>A parameter with the {@link #ROUTE_TO_PARAMETER_NAME} name and the
	{@link #Route_To() route-to list} of addresses as its assigned value
	array. An {@link Lister#NEW_LINE end-of-line sequence} ends the
	parameter. If this Message has an empty route-to list this parameter
	is not included in the label.

<dt>{@link #ROUTE_FROM_PARAMETER_NAME}
<dd>A parameter with the {@link #ROUTE_FROM_PARAMETER_NAME} name and the
	{@link #Route_From() route-from list} of addresses as its assigned
	value array. An {@link Lister#NEW_LINE end-of-line sequence} ends the
	parameter. If this Message has an empty route-from list this
	parameter is not included in the label.
</dl>
	<b>N.B.</b>: The message start synchronization sequence is composed
	of the START_MARKER through START_MARKER_END characters. This
	sequence is critical for identifying the start of a message in an
	unreliable transmitted byte stream.
<p>
	@param	content_amount	The value to be assigned to the {@link
		#CONTENT_AMOUNT_PARAMETER_NAME}.
	@return	A String containing the Message label section of a message
		transmission packet.
	@see	#Packet()
*/
protected String Packet_Label
	(
	int		content_amount
	)
{
Message
	label = new Message ()
		.Set (CONTENT_AMOUNT_PARAMETER_NAME, content_amount);
try
	{
	if (Route_To.getChildCount () > 0)
		label.Set (ROUTE_TO_PARAMETER_NAME, Route_To);
	if (Route_From.getChildCount () > 0)
		label.Set (ROUTE_FROM_PARAMETER_NAME, Route_From);
	}
catch (PVL_Exception exception) {/* Route_To and Route_From are valid Values */}
String
	label_string = "";
Iterator
	parameters = label.iterator ();
while (parameters.hasNext ())
	label_string += ((Parameter)parameters.next ()).Description (PVL_Lister)
		+ Lister.NEW_LINE;

return
	START_MARKER +
	START_MARKER_DELIMITER +
	String.valueOf (label_string.length ()) +
	START_MARKER_END +
	label_string;
}

/**	Get the value of the {@link #CONTENT_AMOUNT_PARAMETER_NAME} from the
	Message.
<p>
	This convenience method is used to obtain from a Message label
	the amount of content data that follows the label section.
<p>
	@return	The value of the {@link #CONTENT_AMOUNT_PARAMETER_NAME}.
	@throws	PVL_Exception	If the content amount parameter was not
		found or its Value could not produce an integer.
	@see	#Packet_Label(int)
*/
protected int Content_Amount ()
	throws PVL_Exception
{
Parameter
	parameter = Find (CONTENT_AMOUNT_PARAMETER_NAME, Parameter.ASSIGNMENT);
if (parameter == null)
	throw new PVL_Exception (ID, PVL_Exception.EMPTY_STATEMENT,
		"No content amount " + CONTENT_AMOUNT_PARAMETER_NAME + " parameter -" + NL
		+ toString ());
return (int)(parameter.Value ().long_Data ());
}

/**	Get the Message content as a PVL String representation.
<p>
	The Message's PVL {@link Parameter#Description(Lister) description}
	String is generated using a suitably configured {@link #PVL_Lister()
	PVL Lister}. If the Name of the Message is not the {@link
	Parser#CONTAINER_NAME} it is temporarily changed to this special name
	so a PVL "END" statement will always be provided.
<p>
	<b>N.B.</b>: If the first character of the resultant String is the
	{@link #ERROR_MESSAGE_MARKER} character then a problem was
	encountered while attempting to generate the PVL syntax from the
	Message content. In this case the returned String is an error report.
<p>
	@return	A PVL String representation of the Message content.
	@see	#toString()
*/
public String PVL ()
{
String
	name = null;
if (! Parser.CONTAINER_NAME.equals (Name ()))
	{
	name = Name ();
	Name (Parser.CONTAINER_NAME);
	}
String
	string = Description (PVL_Lister);
if (name != null)
	Name (name);
return string;
}

/**	Get the Message content as a printable PVL String representation.
<p>
	The Message's PVL {@link Parameter#Description() description} String
	is generated in standard format. Any escape sequences are {@link
	#Unescape(String) converted} to their original character values. As a
	result, parameter string values containing new-line sequences, for
	example, will be printed as multiple lines. If the Message {@link
	#Name() name} is not the {@link Parser#CONTAINER_NAME} a top-level
	group with this name will be included in the PVL represenation and no
	PVL "END" statement will be provided; otherwise the top-level
	container group will not be included but the "END" statement will be.
<p>
	<b>N.B.</b>: If the first character of the resultant String is the
	{@link #ERROR_MESSAGE_MARKER} character then a problem was
	encountered while attempting to generate the PVL syntax from the
	Message content. In this case the returned String is an error report.
<p>
	@return	A String representation of the Message content.
	@see	#PVL()
*/
public String toString ()
{return Unescape (Description ());}

/*------------------------------------------------------------------------------
	Parameter Values
*/
/**	Get a named parameter value as a String.
<p>
	The named Parameter must be an Assignment of a single non-Array Value.
<p>
	@param	name	The String naming the parameter to be found. If null,
		null will be returned.
	@return	A String representing the value of the parameter. This will
		be null if the named parameter could not be found, it is not an
		Assignment or its Value is an Array. <b>N.B.</b> The actual
		Parameter Value may not be a String in which case its String
		representation is returned.
	@see	#Value_of(String)
*/
public String Get
	(
	String	name
	)
{
Value
	value = Value_of (name);
if (value != null)
	{
	try {return value.String_Data ();}
	catch (PVL_Exception exception) {/* If the Value is an Array */}
	}
return null;
}

/**	Get the String value of a parameter with a name matching a pattern.
<p>
	The matching Parameter must be an Assignment of a single non-Array Value.
<p>
	@param	pattern	A regular expression pattern String. If null, null
		is returned.
	@return	A String representing the value of the parameter with the
		name that matches the pattern. This will be null if no matching
		parameter could be found, it is not an Assignment or its Value is
		an Array. <b>N.B.</b> The actual Parameter Value may not be a
		String in which case its String representation is returned.
	@see	#Value_of_Matching(String)
*/
public String Matching
	(
	String	pattern
	)
{
Value
	value = Value_of_Matching (pattern);
if (value != null)
	{
	try {return value.String_Data ();}
	catch (PVL_Exception exception) {/* If the Value is an Array */}
	}
return null;
}

/**	Get a named parameter value as a Value.
<p>
	Only an Assignment parameter will be found. The first Assignment
	parameter with the matching name is returned.
<p>
	@param	name	The String naming the parameter to be found. If null,
		null is returned.
	@return	The Value of the Parameter. This will be null if the
		parameter could not be found.
	@see	#Get(String)
*/
public Value Value_of
	(
	String	name
	)
{
if (name != null)
	{
	Parameter
		parameter = Find (name, Parameter.ASSIGNMENT);
	if (parameter != null)
		{
		try {return parameter.Value ();}
		catch (PVL_Exception exception) {/* Shouldn't happen */}
		}
	}
return null;
}

/**	Get the Value of a parameter with a name matching a pattern.
<p>
	Only an Assignment parameter will be found.
<p>
	@param	pattern	A regular expression pattern String. If null, null
		is returned.
	@return	The Value of the Parameter with the name that matches the
		pattern. This will be null if a matching parameter could not be
		found.
*/
public Value Value_of_Matching
	(
	String	pattern
	)
{
if (pattern != null)
	{
	Parameter
		parameter = Find
			(
			new Parameter (pattern)
				.Classification (Parameter.ASSIGNMENT),
			new Selection ()
				.Name (true)
				.Pattern_Match (true)
				.And (true)
				.Classification (true)
			);
	if (parameter != null)
		{
		try {return parameter.Value ();}
		catch (PVL_Exception exception) {/* Shouldn't happen */}
		}
	}
return null;
}

/**	Set a named parameter value at a list index.
<p>
	If a Parameter with the specified name is found its Value is
	set to the specified value; any existing Value is replaced.
	If no Parameter is found a new one with the specified Value
	is added to the parameter list at the specified index.
<p>
	The specified value may be any object suitable for providing the
	{@link Value#Data(Object) data of a Value}. If the specified value is
	null and an existing Parameter is found, the Parameter is removed.
	If, however, no Parameter is found a null value does nothing.
<p>
	@param	name	The name of the Parameter to be assigned the value.
		If null nothing is done.
	@param	value	A Object that contains the value to be applied,
		or null if an existing parameter is to be removed.
	@param	index	The index in the parameter list at which to insert
		a new Parameter. If less than zero a new parameter will be added
		to the beginning of the list (index zero); if greater than or equal
		to the size of the parameter list a new parameter will be added
		to the end of the list (index equal to the list size).
	@return	This Message.
	@throws	PVL_Exception	If the value is not a suitable Object for a
		Value.
*/
public Message Set
	(
	String	name,
	Object	value,
	int		index
	)
	throws PVL_Exception
{
if (name != null)
	{
	Parameter
		parameter = Find (name);
	if (parameter == null)
		{
		if (value != null)
			{
			//	New parameter.
			if (index < 0)
				index = 0;
			else
			if (index > List_Size ())
				index = List_Size ();
			Insert (new Parameter (name).Value (value), index);
			}
		}
	else
	if (! parameter.Is_Aggregate ())
		{
		//	Existing Assignment parameter.
		if (value == null)
			Remove (parameter);
		else
			parameter.Value (value);
		}
	}
return this;
}

/**	Set a named parameter value.
<p>
	If a Parameter with the specified name is found its Value is
	set to the specified value; any existing Value is replaced.
	If no Parameter with the specified name is found a new one with the
	specified value is added to the end of the parameter list.
<p>
	The specified value may be any object suitable for providing the
	{@link Value#Data(Object) data of a Value}. If the specified value is
	null and an existing Parameter is found, the Parameter is removed.
	If, however, no Parameter is found a null value does nothing.
<p>
	@param	name	The name of the Parameter to be assigned the value.
		If null nothing is done.
	@param	value	A Object that contains the value to be assigned,
		or null if an existing parameter is to be removed.
	@return	This Message.
	@throws	PVL_Exception	If the value is not a suitable Object for a
		Value.
	@see	#Set(String, Object, int)
*/
public Message Set
	(
	String	name,
	Object	value
	)
	throws PVL_Exception
{return Set (name, value, List_Size ());}

/**	Set a named parameter value.
<p>
	This is a convenience method to set a String value with no
	possibility of an exception being thrown.
<p>
	@param	name	The name of the Parameter to be assigned the value.
		If null nothing is done.
	@param	value	A String value to be assigned, or null if an existing
		parameter is to be removed.
	@return	This Message.
	@see	#Set(String, Object)
*/
public Message Set
	(
	String	name,
	String	value
	)
{
try {return Set (name, (Object)value);}
catch (PVL_Exception exception) {/* Shouldn't happen */}
return this;
}

/**	Set a named parameter value.
<p>
	This is a convenience method to set a String value with no
	possibility of an exception being thrown.
<p>
	@param	name	The name of the Parameter to be assigned the value.
		If null nothing is done.
	@param	value	A String value to be assigned, or null if an existing
		parameter is to be removed.
	@param	index	The index in the parameter list at which to insert
		a new Parameter. If less than zero a new parameter will be added
		to the beginning of the list (index zero); if greater than or equal
		to the size of the parameter list a new parameter will be added
		to the end of the list (index equal to the list size).
	@return	This Message.
	@see	#Set(String, Object, int)
*/
public Message Set
	(
	String	name,
	String	value,
	int		index
	)
{
try {return Set (name, (Object)value, index);}
catch (PVL_Exception exception) {/* Shouldn't happen */}
return this;
}

/**	Set a named parameter value.
<p>
	This is a convenience method to set an integer value with no
	possibility of an exception being thrown.
<p>
	@param	name	The name of the Parameter to be assigned the value.
		If null nothing is done.
	@param	value	A long integer value to be assigned.
	@return	This Message.
	@see	#Set(String, Object)
*/
public Message Set
	(
	String	name,
	long	value
	)
{
try {return Set (name, new Long (value));}
catch (PVL_Exception exception) {/* Shouldn't happen */}
return this;
}

/**	Set a named parameter value.
<p>
	This is a convenience method to set an integer value with no
	possibility of an exception being thrown.
<p>
	@param	name	The name of the Parameter to be assigned the value.
		If null nothing is done.
	@param	value	A long integer value to be assigned.
	@param	index	The index in the parameter list at which to insert
		a new Parameter. If less than zero a new parameter will be added
		to the beginning of the list (index zero); if greater than or equal
		to the size of the parameter list a new parameter will be added
		to the end of the list (index equal to the list size).
	@return	This Message.
	@see	#Set(String, Object, int)
*/
public Message Set
	(
	String	name,
	long	value,
	int		index
	)
{
try {return Set (name, new Long (value), index);}
catch (PVL_Exception exception) {/* Shouldn't happen */}
return this;
}

/**	Set a named parameter value.
<p>
	This is a convenience method to set a real number value with no
	possibility of an exception being thrown.
<p>
	@param	name	The name of the Parameter to be assigned the value.
		If null nothing is done.
	@param	value	A double value to be assigned.
	@return	This Message.
	@see	#Set(String, Object)
*/
public Message Set
	(
	String	name,
	double	value
	)
{
try {return Set (name, new Double (value));}
catch (PVL_Exception exception) {/* Shouldn't happen */}
return this;
}

/**	Set a named parameter value.
<p>
	This is a convenience method to set a real number value with no
	possibility of an exception being thrown.
<p>
	@param	name	The name of the Parameter to be assigned the value.
		If null nothing is done.
	@param	value	A double value to be assigned.
	@param	index	The index in the parameter list at which to insert
		a new Parameter. If less than zero a new parameter will be added
		to the beginning of the list (index zero); if greater than or equal
		to the size of the parameter list a new parameter will be added
		to the end of the list (index equal to the list size).
	@return	This Message.
	@see	#Set(String, Object, int)
*/
public Message Set
	(
	String	name,
	double	value,
	int		index
	)
{
try {return Set (name, new Double (value), index);}
catch (PVL_Exception exception) {/* Shouldn't happen */}
return this;
}

/**	Set a named parameter token.
<p>
	A parameter token has no value. <b>N.B.</b>: If an existing Parameter
	with the specified name is found it's Value is dropped. If no
	Parameter with the specified name is found a new one is added to the
	end of the parameter list.
<p>
	@param	name	The name of the token Parameter. If null nothing is done.
	@param	index	The index in the parameter list at which to insert
		a new Parameter. If less than zero a new parameter will be added
		to the beginning of the list (index zero); if greater than or equal
		to the size of the parameter list a new parameter will be added
		to the end of the list (index equal to the list size).
	@return	This Message.
*/
public Message Set_Token
	(
	String	name,
	int		index
	)
{
if (name != null)
	{
	Parameter
		parameter = Find (name);
	if (parameter == null)
		{
		//	New parameter.
		if (index < 0)
			index = 0;
		else
		if (index > List_Size ())
			index = List_Size ();
		try {Insert (new Parameter (name), index);}
		catch (PVL_Exception exception)
			{/* Message is an aggregate */}
		}
	else
	if (! parameter.Is_Aggregate ())
		{
		//	Existing Assignment parameter.
		parameter.Classification (Parameter.TOKEN);
		}
	}
return this;
}

/**	Set a named parameter token.
<p>
	The token will be added to the end of the parameter list.
<p>
	@param	name	The name of the token Parameter. If null nothing is done.
	@return	This Message.
	@see	#Set_Token(String, int)
*/
public Message Set_Token
	(
	String	name
	)
{return Set_Token (name, List_Size ());}

/**	Add a parameter to the Message.
<p>
	The parameter is copied. The name of the copy is set to the specified
	name if the specified name is non-null. Then the copy is added to the
	end of the Message parameter list.
<p>
	@param	name	A String that will be the name of the new parameter.
		If null, the existing name will be unchanged.
	@param	parameter	A Parameter a copy of which will be added to
		the end of this Message list. If null, nothing is done.
	@return	This Message.
*/
public Message Add
	(
	String		name,
	Parameter	parameter
	)
{
if (parameter != null)
	{
	try
		{
		//	Copy the Parameter.
		parameter = new Parameter (parameter);
		if (name != null)
			//	Set the name of the copied parameter.
			parameter.Name (name);
		super.Add (parameter);
		}
	catch (PVL_Exception exception) {/* Shouldn't happen */}
	}
return this;
}

/**	Add a Parameter to the Message.
<p>
	The Parameter is added to the end of the Message Parameter list.
	<b>N.B.</b>: The Parameter is not copied.
<p>
	This is a convenience method to add a Parameter with no possibility
	of an exception being thrown.
<p>
	@param	parameter	The Parameter to be added to the end of this
		Message list. If null, nothing is done.
	@return	This Message.
	@see	#Add(String, Parameter)
*/
public Message Add
	(
	Parameter	parameter
	)
{
if (parameter != null)
	{
	try {super.Add (parameter);}
	catch (PVL_Exception exception) {/* Shouldn't happen */}
	}
return this;
}

/**	Remove a named parameter from the Message.
<p>
	Any parameter at the specified pathname - relative or absolute - is
	removed from the Message regardless of its Classification (Token,
	Assignment or Aggregate). Only the first parameter found at the
	pathname is removed.
<p>
	@param	pathname	The pathnname String for the parameter to be removed.
		If null nothing is done.
	@return	The removed Parameter, or null if nothing was removed.
	@see	Parameter#Remove(Parameter)
*/
public Parameter Remove
	(
	String		pathname
	)
{
Parameter
	parameter = null;
if (pathname != null)
	{
	parameter = Find (pathname);
	if (parameter != null)
    	parameter.Parent ().Remove (parameter);
	}
return parameter;
}

/**	Remove all occurances of a named parameter from the Message.
<p>
	All parameters at the specified pathname - relative or absolute - are
	removed from the Message regardless of their Classification (Token,
	Assignment or Aggregate).
<p>
	@param	pathname	The pathnname String for the parameters to be removed.
		If null nothing is done.
	@return	true if at least one parameter was removed; false otherwise.
	@see	Parameter#Remove(Parameter)
*/
public boolean Remove_All
	(
	String		pathname
	)
{
boolean
	removed = false;
if (pathname != null)
	while (Remove (pathname) != null)
		removed = true;
return removed;
}

/**	Get the action associated with this Message.
<p>
	If an {@link #ACTION_PARAMETER_NAME} is found its value is returned.
	Otherwise, if the Message contains at least one parameter and
	the first parameter is a token (a parameter without a value),
	then the token name is returned.
<p>
	@return	A String naming the action associated with this Message.
		This will be null if no action could be identified.
*/
public String Action ()
{
String
	action = Get (ACTION_PARAMETER_NAME);
if (action == null &&
	List_Size () >= 1)
	{
	Parameter
		parameter = null;
	try {parameter = (Parameter)List ().get (0);}
	catch (PVL_Exception exception) {/* Pre-checked */}
	if (parameter.Is_Token ())
		action = parameter.Name ();
	}
return action;
}

/*------------------------------------------------------------------------------
	Routing
*/
/**	Add an address to the {@link #Route_To() route-to} list.
<p>
	@param	to	A {@link Messenger#Address() Messenger address}.
	@return	This Message.
*/
public Message To
	(
	String	to
	)
{Put (to, Route_To); return this;}

/**	Get the next address from the end of the {@link #Route_To() route-to}
	list.
<p>
	The address is not removed from the list.
<p>
	@return	The address on the end of the route-to list, or null
		if the list is empty.
*/
public String To ()
{return Peek (Route_To);}

/**	Get the address from the front of the {@link #Route_To()
	route-to} list.
<p>
	The address is not removed from the list.
<p>
	@return	The address at the front of the route-to list, or null if the
		list is empty. This should be the address of the destination
		Messenger where the Message is to be delivered.
*/
public String Destination_Address ()
{return First (Route_To);}

/**	Get the next address off of the end of the {@link #Route_To()
	route-to} list.
<p>
	The address is removed from the list.
<p>
	@return	The address on the end of the route-to list, or null
		if the list is empty.
*/
public String Pop_To ()
{return Pop (Route_To);}

/**	Get the Message route-to list.
<p>
	The addresses in the list are LIFO ordered: The last entry is the
	address of the first Messenger to which the Message should be sent
	(not the Messenger that sends the Message); the first entry is the
	intended recipient Messenger where the Message is to be delivered. A
	Message with an empty route-to list will be delivered by the
	Messenger that receives it.
<p>
	@return	An Array Value containing zero or more {@link
		Messenger#Address() Messenger addresses}. This is a copy of
		the list owned by the Message.
	@see	#Route_To(Value)
*/
public Value Route_To ()
{
try {return new Value (Route_To);}
catch (PVL_Exception exception) {/* Shouldn't happen */}
return null;
}

/**	Set the Message route-to list.
<p>
	@param	to	An Array Value containing zero or more {@link
		Messenger#Address() Messenger addresses}. The Value will be
		copied into this Message replacing the previous route-to list. If
		the Value is null the current list is emptied.
	@return	This Message.
	@throws	PVL_Exception	If the Value is not an Array of STRING type
		Values.
*/
public Message Route_To
	(
	Value	to
	)
	throws PVL_Exception
{
if (to == null)
	Route_To.Remove_All ();
else
	Route_To = new Value (Valid_Route (to));
return this;
}

/**	Clear the Message route-to list.
<p>
	@return	This Message.
*/
public Message Clear_Route_To ()
{
Route_To.Remove_All ();
return this;
}

/**	Apply the {@link #Route_To() route-to list} from another Message to
	this Message.
<p>
	@param	message	The Message from which to obtain the route-to list.
		If the message is the same as this Message, nothing is done.
	@return	This Message.
*/
public Message Route_To
	(
	Message	message
	)
{
if (message != null &&
	message != this)
	//	Copy the other Route_To to this Route_To.
	Route_To = message.Route_To ();
return this;
}

/**	Get the count of {@link #Route_To() route-to} adresses.
<p>
	@return	The number of addresses on the route-to list.
*/
public int Route_To_Count ()
{return Route_To.Array_Size ();}

/**	Add an address to the {@link #Route_From() route-from} list.
<p>
	@param	from	A {@link Messenger#Address() Messenger address}.
	@return	This Message.
*/
public Message From
	(
	String	from
	)
{Put (from, Route_From); return this;}

/**	Get the next address from the end of the {@link #Route_From() route-from}
	list.
<p>
	The address is not removed from the list.
<p>
	@return	The address on the end of the route-from list, or null
		if the list is empty. This will be the address of the last
		Messenger to have sent the Message.
*/
public String From ()
{return Peek (Route_From);}

/**	Get the address from the front of the {@link #Route_From()
	route-from} list.
<p>
	The address is not removed from the list.
<p>
	@return	The address at the front of the route-from list, or null if
		the list is empty. This should be the address of the source
		Messenger where the Message originated.
*/
public String Source_Address ()
{return First (Route_From);}

/**	Get the next address off of the end of the {@link #Route_From()
	route-from} list.
<p>
	The address is removed from the list.
<p>
	@return	The address on the end of the route-from list, or null
		if the list is empty.
*/
public String Pop_From ()
{return Pop (Route_From);}
	
/**	Get the Message route-from list.
<p>
	Each Messenger that sends a Message adds its adress to the
	end of the list.
<p>
	@return	An Array Value containing zero or more {@link
		Messenger#Address() Messenger addresses}. This is a copy of
		the list owned by the Message.
	@see	#Route_From(Value)
*/
public Value Route_From ()
{
try {return new Value (Route_From);}
catch (PVL_Exception exception) {/* Shouldn't happen */}
return null;
}

/**	Set the Message route-from list.
<p>
	@param	from	An Array Value containing zero or more {@link
		Messenger#Address() Messenger addresses}. The Value will be
		copied into this Message replacing the previous route-to list. If
		the Value is null the current list is emptied.
	@return	This Message.
	@throws	PVL_Exception	If the Value is not an Array of STRING type
		Values.
*/
public Message Route_From
	(
	Value	from
	)
	throws PVL_Exception
{
if (from == null)
	Route_From.Remove_All ();
else
	Route_From = new Value (Valid_Route (from));
return this;
}

/**	Clear the Message route-from list.
<p>
	@return	This Message.
*/
public Message Clear_Route_From ()
{
Route_From.Remove_All ();
return this;
}

/**	Get the count of {@link #Route_From() route-from} adresses.
<p>
	@return	The number of addresses on the route-from list.
*/
public int Route_From_Count ()
{return Route_From.Array_Size ();}

/**	Get the routing description.
<p>
	@return	A String containing a Route To: address list followed by a
		Route From: address list.
*/
public String Routing ()
{
String_Buffer
	string = new String_Buffer ();
return
	"  Route To: " +
		new String_Buffer (Route_To.Description ())
			.replace_span (0, "()", "").toString () + NL +
	"Route From: " +
		new String_Buffer (Route_From.Description ())
			.replace_span (0, "()", "").toString ();
}

/**	Set a Message to be routed back to the sender of another Message.
<p>
	The {@link #Route_To(Value) route-to} list is of this Message is set
	to the {@link #Route_From() route-from} list of the other Message.
	The route-from list of this Message is emptied.
<p>
	@param	message	The Message to receive a reply.
	@return	This Message.
*/
public Message Reply_To
	(
	Message	message
	)
{
if ((DEBUG & DEBUG_ROUTING) != 0)
	System.out.println
		(">>> Message.Reply_To:" + NL
		+"    Route_From " + message.Route_From.Description ());
if (this == message)
	//	Move Route_From to Route_To.
	Route_To = Route_From;
else
	//	Copy the other Route_From to this Route_To.
	Route_To = message.Route_From ();
//	Empty Route_From.
try {Route_From = new Value ().Type (Value.SEQUENCE);}
catch (PVL_Exception exception) {/* Shouldn't happen */}
if ((DEBUG & DEBUG_ROUTING) != 0)
	System.out.println
		("    ->Route_To " + Route_To.Description () + NL
		+"<<< Message.Reply_To");
return this;
}

/**	Put a String Value in an Array.
<p>
	@param	string	The String to be added as a Value of the array.
	@param	array	The Array Value to which to add the string.
*/
private static void Put
	(
	String	string,
	Value	array
	)
{
if (string != null &&
	string.length () != 0)
	{
	try {array.Add (string);}
	catch (PVL_Exception exception) {/* Shouldn't happen */}
	}
}

/**	Peek at the last Value of an Array.
<p>
	The last Value is not removed from the Array.
<p>
	@param	array	The Array Value to examine.
	@return	A String representing the last value of the array. This will
		be null if the array is empty or the last value can not be
		represented as a String.
*/
private static String Peek
	(
	Value	array
	)
{
if (array.getChildCount () > 0)
	{
	try {return array.Get (array.getChildCount () - 1).String_Data ();}
	catch (PVL_Exception exception) {/* Shouldn't happen. */}
	}
return null;
}

/**	Pop the last Value off of an Array.
<p>
	The last Value is removed from the Array.
<p>
	@param	array	The Array Value to examine.
	@return	A String representing the last value of the array. This will
		be null if the array is empty or the last value can not be
		represented as a String.
*/
private static String Pop
	(
	Value	array
	)
{
if (array.getChildCount () > 0)
	{
	try {return array.Remove (array.getChildCount () - 1).String_Data ();}
	catch (PVL_Exception exception) {/* Shouldn't happen. */}
	}
return null;
}

/**	Get the first Value of an Array.
<p>
	The Value is not removed from the Array.
<p>
	@param	array	The Array Value to examine.
	@return	A String representing the first value of the array. This will
		be null if the array is empty or the first value can not be
		represented as a String.
*/
private static String First
	(
	Value	array
	)
{
if (array.getChildCount () > 0)
	{
	try {return array.Get (0).String_Data ();}
	catch (PVL_Exception exception) {/* Shouldn't happen. */}
	}
return null;
}

/**	Check that a Value is a valid routing address list.
<p>
	A valid routing address list is an Array of STRING Type Values.
<p>
	@param	value	The Value to be checked. May not be null, but may
		be an empty Array.
	@throws	PVL_Exception	If the Value is not an Array or one of its
		Value's is not a STRING Type.
*/
public static Value Valid_Route
	(
	Value	value
	)
	throws PVL_Exception
{
if (! value.Is_Array ())
	throw new PVL_Exception (ID, PVL_Exception.INCOMPATIBLE_TYPES,
		"Routing Value is not an Array.");
Iterator
	values = value.iterator ();
while (values.hasNext ())
	if (! ((Value)values.next ()).Is_String ())
		throw new PVL_Exception (ID, PVL_Exception.INCOMPATIBLE_TYPES,
			"Routing Array contains a Value that is not a STRING type.");
return value;
}

/*------------------------------------------------------------------------------
	Lister and Parser
*/
/**	Get a PVL Lister configured for Message content representation.
<p>
	The Lister is configured with {@link Lister#Verbatim_Strings(boolean)
	verbatim strings} enabled, {@link Lister#Indent(boolean) indenting}
	disabled and {@link Lister#Indent_Arrays(boolean) array indenting}
	disabled. The Lister has no writer bound to it.
<p>
	@return	A PVL Lister.
	@see	Lister
*/
public static Lister PVL_Lister ()
{
return
	new Lister ()
		.Verbatim_Strings (true)
		.Indent (false)
		.Indent_Arrays (false);
}

/**	Get a PVL Parser configured for parsing an input stream into Message
	content.
<p>
	The Parser has no source reader bound to it. It has default syntax
	modes set except: {@link Parser#Verbatim_Strings(boolean) verbatim
	strings} is enabled and {@link Parser#String_Continuation(boolean)
	string continuation} and {@link Parser#Filter_Input(boolean) input
	filtering} are disabled.
<p>
	<b>N.B.</b>: The Reader that will be bound to the Parser is
	excpected to employ "US-ASCII" character encoding. Multi-byte
	characters may produce unexpected results.
<p>
	<b>N.B.</b>: If the Parser is to be reused with more than one
	parsing sequence - such as with different input readers - it
	must be {@link Parser#Reset() reset} before each reuse.
<p>
	@return	A PVL Parser suitable for parsing an input stream into
		a Message.
	@see	Parser
	@see	Parser#Set_Reader(Reader, long)
*/
public static Parser PVL_Parser ()
{
Parser
	parser = new Parser ();
parser
	.Verbatim_Strings (true)
	.String_Continuation (false)
	.Filter_Input (false);
return parser;
}

/*==============================================================================
	Common Messages
*/
/**	Get a Message intitialized with an {@link #IDENTIFY_ACTION} {@link
	#ACTION_PARAMETER_NAME}.
<p>
	@return	A Message.
*/
public static Message Identify ()
{return Message.Action (IDENTIFY_ACTION);}

/**	Get a Message intitialized with an {@link #IDENTITY_ACTION} {@link
	#ACTION_PARAMETER_NAME} and a {@link #NAME_PARAMETER_NAME} with a
	specified name value.
<p>
	@param	name	The name String to be assigned as the value of the
		{@link #NAME_PARAMETER_NAME}.
	@return	A Message.
*/
public static Message Identity
	(
	String	name
	)
{return Message.Action (IDENTITY_ACTION).Set (NAME_PARAMETER_NAME, name);}

/**	Get a Message intitialized with an {@link #IDENTITY_ACTION} {@link
	#ACTION_PARAMETER_NAME}.
<p>
	@return	A Message.
*/
public static Message Identity ()
{return Message.Action (IDENTITY_ACTION);}

/**	Get a Message intitialized with a {@link #NACK_ACTION} {@link
	#ACTION_PARAMETER_NAME}.
<p>
	@return	A Message.
*/
public static Message NACK ()
{return Message.Action (NACK_ACTION);}

/**	Get a Message intitialized with a {@link #DONE_ACTION} {@link
	#ACTION_PARAMETER_NAME}.
<p>
	@return	A Message.
*/
public static Message Done ()
{return Message.Action (DONE_ACTION);}

/**	Get a Message intitialized with an {@link #ACTION_PARAMETER_NAME}.
<p>
	@param	action	The String specifying the {@link
		#ACTION_PARAMETER_NAME} value.
	@return	A Message.
*/
public static Message Action
	(
	String	action
	)
{
if (action == null ||
	action.length () == 0)
	return null;
Message
	message = new Message ();
try
	{
	message.Add (new Parameter (ACTION_PARAMETER_NAME).Value (action));
	if ((DEBUG & DEBUG_FUNCTIONS) != 0)
		System.out.println
			(">-< Message.Action: " + action + NL
			+ message);
	}
catch (PVL_Exception exception) {/* Shouldn't happen */}
return message;
}

/*==============================================================================
	Utilities
*/
/**	Replace escape sequences in a String with their unescaped equivalents.
<p>
	When Message content contains parameters with multi-line report text,
	the end-of-line sequences are converted to escape sequences for
	transmision. This applies to other special characters such as horizontal
	tabs. Before printing such report text they should have any escape
	sequences converted back to their original character values.
<p>
	@param	string	A String that may contain escape sequences.
	@return	A copy of the string argument with all escape sequences
		converted to their original character values.
*/
public static String Unescape
	(
	String	string
	)
{return String_Buffer.escape_to_special (string);}


}
