require 'spec_helper'

describe 'swift::ringbuilder' do
  let :facts do
    OSDefaults.get_facts({
      :operatingsystem => 'Ubuntu',
      :osfamily        => 'Debian',
      :processorcount  => 1,
    })
  end
  describe 'when swift class is not included' do
    it 'should fail' do
      expect { catalogue }.to raise_error(Puppet::Error)
    end
  end
  describe 'when swift class is included' do

    let :pre_condition do
      "class { memcached: max_memory => 1}
       class { swift: swift_hash_path_suffix => string }"
    end

    it 'should rebalance the ring for all ring types' do
      is_expected.to contain_swift__ringbuilder__rebalance('object')
      is_expected.to contain_swift__ringbuilder__rebalance('account')
      is_expected.to contain_swift__ringbuilder__rebalance('container')
    end

    describe 'with default parameters' do
      ['object', 'account', 'container'].each do |type|
        it { is_expected.to contain_swift__ringbuilder__create(type).with(
          :part_power     => '18',
          :replicas       => '3',
          :min_part_hours => '24'
        )}
      end
    end

    describe 'with parameter overrides' do

      let :params do
        {:part_power     => '19',
         :replicas       => '3',
         :min_part_hours => '2'
        }
      end

      ['object', 'account', 'container'].each do |type|
        it { is_expected.to contain_swift__ringbuilder__create(type).with(
          :part_power     => '19',
          :replicas       => '3',
          :min_part_hours => '2'
        )}
      end

    end
    describe 'when specifying ring devices' do
      let :pre_condition do
         'class { memcached: max_memory => 1}
          class { swift: swift_hash_path_suffix => string }
          ring_object_device { "127.0.0.1:6000/1":
          zone        => 1,
          weight      => 1,
        }

        ring_container_device { "127.0.0.1:6001/1":
          zone        => 2,
          weight      => 1,
        }

        ring_account_device { "127.0.0.1:6002/1":
          zone        => 3,
          weight      => 1,
        }'
      end

      it 'should set up all of the correct dependencies' do
        is_expected.to contain_swift__ringbuilder__create('object').with(
          {:before => ['Ring_object_device[127.0.0.1:6000/1]']}
        )
        is_expected.to contain_swift__ringbuilder__create('container').with(
        {:before => ['Ring_container_device[127.0.0.1:6001/1]']}
        )
        is_expected.to contain_swift__ringbuilder__create('account').with(
        {:before => ['Ring_account_device[127.0.0.1:6002/1]']}
        )
        is_expected.to contain_ring_object_device('127.0.0.1:6000/1').with(
        {:notify => ['Swift::Ringbuilder::Rebalance[object]']}
        )
        is_expected.to contain_ring_container_device('127.0.0.1:6001/1').with(
        {:notify => ['Swift::Ringbuilder::Rebalance[container]']}
        )
        is_expected.to contain_ring_account_device('127.0.0.1:6002/1').with(
        {:notify => ['Swift::Ringbuilder::Rebalance[account]']}
        )
      end
    end
  end
end
