/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <cutil/global.h>

#ifdef OS_WINDOWS
	#include <winsock2.h>
	#include <windows.h>
#endif

#include <stdlib.h>

//#ifndef CC_MSVC8
//extern "C" {
//#endif
/*#include "glib.h"
#include "libpurple/account.h"
#include "libpurple/blist.h"
#include "libpurple/connection.h"
#include "libpurple/conversation.h"
#include "libpurple/core.h"
#include "libpurple/eventloop.h"
#include "libpurple/internal.h"
#include "libpurple/privacy.h"
#include "libpurple/util.h"*/
#ifdef OS_WINDOWS
//#include "libpurple/win32/win32dep.h"
#endif
#include "libpurple/purple.h"
//#ifndef CC_MSVC8
//}
//#endif

#include "PurpleIMFactory.h"
#include "PurpleAccountMngr.h"
#include "PurpleChatMngr.h"
#include "PurpleConnectMngr.h"
#include "PurpleContactListMngr.h"
#include "PurplePresenceMngr.h"

#include <util/File.h>
#define LOGGER_COMPONENT "Purple"
#include <util/Logger.h>
#include <util/Path.h>

extern PurpleConversationUiOps chat_wg_ops;
extern PurpleBlistUiOps blist_wg_ops;
extern PurpleBlistUiOps null_blist_wg_ops;
extern PurpleConnectionUiOps conn_wg_ops;
extern PurpleConnectionUiOps null_conn_wg_ops;
extern PurpleAccountUiOps acc_wg_ops;
extern PurpleAccountUiOps null_acc_wg_ops;
extern PurplePrivacyUiOps privacy_wg_ops;

/* Static vars */
static GMainLoop * gMainLoop = NULL;


/* ********************* PURPLE CALLBACK ********************* */
#define PURPLE_WG_READ_COND  (G_IO_IN | G_IO_HUP | G_IO_ERR)
#define PURPLE_WG_WRITE_COND (G_IO_OUT | G_IO_HUP | G_IO_ERR | G_IO_NVAL)

extern "C" GIOChannel *wpurple_g_io_channel_win32_new_socket(int socket);

typedef struct _PurpleWgIOClosure {
	PurpleInputFunction function;
	guint result;
	gpointer data;
} PurpleWgIOClosure;

static void purple_wg_io_destroy(gpointer data) {
	g_free(data);
}

static gboolean purple_wg_io_invoke(GIOChannel * source, GIOCondition condition, gpointer data) {

	PurpleInputCondition purple_cond = (PurpleInputCondition) 0;
	if (condition & PURPLE_WG_READ_COND) {
		purple_cond = (PurpleInputCondition)(purple_cond|PURPLE_INPUT_READ);
	}
	if (condition & PURPLE_WG_WRITE_COND) {
		purple_cond = (PurpleInputCondition)(purple_cond|PURPLE_INPUT_WRITE);
	}

#ifdef OS_WINDOWS
	if (!purple_cond) {
		return TRUE;
	}
#endif /* OS_WINDOWS */

	PurpleWgIOClosure * closure = (PurpleWgIOClosure *) data;

	closure->function(closure->data, g_io_channel_unix_get_fd(source), purple_cond);

	return TRUE;
}

static guint purple_wg_input_add(gint fd, PurpleInputCondition condition,
	PurpleInputFunction function, gpointer data) {

	PurpleWgIOClosure * closure = g_new0(PurpleWgIOClosure, 1);

	closure->function = function;
	closure->data = data;

	GIOCondition cond = (GIOCondition) 0;
	if (condition & PURPLE_INPUT_READ) {
		cond = (GIOCondition)(cond|PURPLE_WG_READ_COND);
	}
	if (condition & PURPLE_INPUT_WRITE) {
		cond = (GIOCondition)(cond|PURPLE_WG_WRITE_COND);
	}

	GIOChannel * channel;
#ifdef OS_WINDOWS
	channel = wpurple_g_io_channel_win32_new_socket(fd);
#else
	channel = g_io_channel_unix_new(fd);
#endif
	closure->result = g_io_add_watch_full(channel, G_PRIORITY_DEFAULT, cond,
		purple_wg_io_invoke, closure, purple_wg_io_destroy);

	g_io_channel_unref(channel);
	return closure->result;
}

static void* purple_wg_request_action(const char *title, const char *primary,
		const char *secondary, int default_action,
		PurpleAccount *account, const char *who,
		PurpleConversation *conv, void *user_data, size_t action_count,
		va_list actions) {
	const char *text = va_arg(actions, const char *);
	PurpleRequestActionCb callback = va_arg(actions, PurpleRequestActionCb);

	LOG_WARN("requested action: %s: %s, %s. Default action: %s", title, primary, secondary, text);
	callback(user_data, 0);
        return NULL;
}

gpointer PurpleMainEventLoop(gpointer data) {
	if (gMainLoop) {
		LOG_FATAL("gMainLoop already created");
	}
	gMainLoop = g_main_loop_new(NULL, FALSE);
	LOG_DEBUG("Starting gMainLoop");

	g_main_loop_run(gMainLoop);
	g_main_loop_unref(gMainLoop);
	LOG_DEBUG("gMainLoop stopped");
	gMainLoop = NULL;
	g_thread_exit(NULL);
	return NULL;
}


/* ******************************************************* */

static PurpleCoreUiOps core_wg_ops = {
	NULL,
	NULL,
	PurpleIMFactory::PurpleSetCallbacks,
	PurpleIMFactory::PurpleQuitCallback,
};

static PurpleEventLoopUiOps eventloop_wg_ops = {
	g_timeout_add,
	g_source_remove,
	purple_wg_input_add,
	g_source_remove
};

static PurpleRequestUiOps request_wg_ops = {
	NULL,
	NULL,
	purple_wg_request_action,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

void MyPurpleIMInit(const gchar* pathToProfile) {

	char * home_dir = g_build_filename(pathToProfile, "purple", NULL);
	printf("pathToProfile %s\nEND\n", home_dir);
	// Remove Purple config directory
	File file(home_dir);
	file.remove();

	std::string ssl_certificates;
	std::string ssl_certificates_env("QUTECOM_SSL_CERTIFICATES_DIR=");
#if defined(OS_WINDOWS) or defined(OS_LINUX)
	ssl_certificates = Path::getApplicationDirPath() + Path::getPathSeparator() + "ca-certs";
#elif defined(OS_MACOSX)
	ssl_certificates = Path::getApplicationResourcesDirPath() + Path::getPathSeparator() +  "ca-certs";
#endif

	ssl_certificates_env += ssl_certificates;

	putenv((char*)ssl_certificates_env.c_str());

	File::createPath(home_dir + File::getPathSeparator());
	purple_util_set_user_dir(home_dir);

	if(getenv("OW_PURPLE_DEBUG")) {
		purple_debug_set_enabled(TRUE);
	}

	purple_core_set_ui_ops(&core_wg_ops);
	purple_eventloop_set_ui_ops(&eventloop_wg_ops);

	char * search_path = g_build_filename(Path::getApplicationDirPath().c_str(), "plugins", NULL);
	purple_plugins_add_search_path(search_path);
	purple_plugins_add_search_path("plugins");
	g_free(search_path);

	purple_request_set_ui_ops(&request_wg_ops);

	if (!purple_core_init("QuteCom PURPLE")) {
		LOG_WARN("Initialization of the Purple core failed\n");
	}
}

struct PurpleInitData {
	PurpleInitData(const std::string & _pathToProfile, PurpleIMFactory *_imfactory) :
	    pathToProfile(_pathToProfile), imfactory(_imfactory) { }

	const std::string & pathToProfile;
	PurpleIMFactory *imfactory;
};

static gboolean purple_wg_init_lib(gpointer data) {
	PurpleInitData *initData = static_cast<PurpleInitData *>(data);

	MyPurpleIMInit(initData->pathToProfile.c_str());

	initData->imfactory->purpleStatusChanged(true);

	return false;
}

static gboolean purple_wg_destroy_lib(gpointer data) {
	PurpleIMFactory *imfactory = static_cast<PurpleIMFactory *>(data);

	purple_connections_set_ui_ops(&null_conn_wg_ops);
	purple_accounts_set_ui_ops(&null_acc_wg_ops);
	purple_blist_set_ui_ops(&null_blist_wg_ops);

	purple_core_quit();

	imfactory->purpleStatusChanged(false);

	return false;
}


void PurpleIMFactory::purpleStatusChanged(bool init) {
	g_assert(purple_init != init);
	g_mutex_lock(purple_init_mutex);
	purple_init = init;
	g_cond_signal(purple_init_cond);
	g_mutex_unlock(purple_init_mutex);
}

void PurpleIMFactory::purpleStatusChangeWait(bool init) {
	g_mutex_lock(purple_init_mutex);
	while(purple_init != init) {
		g_cond_wait(purple_init_cond, purple_init_mutex);
	}
	g_mutex_unlock(purple_init_mutex);
}

bool PurpleIMFactory::equals(const IMAccount & imAccount, std::string login, EnumIMProtocol::IMProtocol protocol) {
	return ((imAccount.getLogin() == login) && (imAccount.getProtocol() == protocol));
}

PurpleIMFactory::PurpleIMFactory() {
	AccountMngr = PurpleAccountMngr::getInstance();
	ConnectMngr = PurpleConnectMngr::getInstance();
	PresenceMngr = PurplePresenceMngr::getInstance();
	ChatMngr = PurpleChatMngr::getInstance();
	ContactListMngr = PurpleContactListMngr::getInstance();
//we are using the Qt glib eventloop on linux if Qt is configued with glib option
#if !defined(QT_WITH_GLIB) || !defined(OS_LINUX)
	if (!g_thread_supported()) {
		g_thread_init(NULL);
	}
#endif /* !defined(QT_WITH_GLIB) || !defined(OS_LINUX) */
	purple_init = false;
	purple_init_cond = g_cond_new();
	purple_init_mutex = g_mutex_new();
#if !defined(QT_WITH_GLIB) || !defined(OS_LINUX)
	g_thread_create(PurpleMainEventLoop, NULL, FALSE, NULL);
#endif /* !defined(QT_WITH_GLIB) || !defined(OS_LINUX) */
}

PurpleIMFactory::~PurpleIMFactory() {
#if !defined(QT_WITH_GLIB) || !defined(OS_LINUX)
	if (gMainLoop) {
		LOG_DEBUG("Stopping gMainLoop");
		g_main_loop_quit(gMainLoop);
	} else {
		LOG_ERROR("No gMainLoop created");
	}
#endif /* !defined(QT_WITH_GLIB) || !defined(OS_LINUX) */
	g_mutex_free(purple_init_mutex);
	g_cond_free(purple_init_cond);
}

void PurpleIMFactory::PurpleSetCallbacks() {
	purple_accounts_set_ui_ops(&acc_wg_ops);
	purple_blist_set_ui_ops(&blist_wg_ops);
	purple_privacy_set_ui_ops(&privacy_wg_ops);
	purple_connections_set_ui_ops(&conn_wg_ops);
}

void PurpleIMFactory::PurpleWrapperInit() {
	AccountMngr->Init();
	ConnectMngr->Init();
	ContactListMngr->Init();
	ChatMngr->Init();
	PresenceMngr->Init();
}

void PurpleIMFactory::init(const std::string & pathToProfile) {
	g_assert(!purple_init);
	PurpleInitData initData(pathToProfile, this);
	g_idle_add(purple_wg_init_lib, &initData);

	purpleStatusChangeWait(true);

	PurpleWrapperInit();
}

void PurpleIMFactory::PurpleQuitCallback() {
	LOG_DEBUG("");
	// Don't stop gMainLoop here: this is called when the user logoff, but we
	// want to keep the loop running for the whole time the application is
	// running.
	//PurpleIMFactory::getFactory().terminatedEvent();
}

void PurpleIMFactory::terminate() {
	if (AccountMngr) {
		AccountMngr->reset();
	}

	if(purple_init)
	{
		g_idle_add(purple_wg_destroy_lib, this);
		purpleStatusChangeWait(false);
	}
	
	PurpleIMFactory::getFactory().terminatedEvent();
}

IMConnect * PurpleIMFactory::createIMConnect(IMAccount &account) {
	return ConnectMngr->AddIMConnect(account);
}

IMChat * PurpleIMFactory::createIMChat(IMAccount &account) {
	return ChatMngr->AddIMChat(account);
}

IMPresence * PurpleIMFactory::createIMPresence(IMAccount &account) {
	return PresenceMngr->AddIMPresence(account);
}

IMContactList * PurpleIMFactory::createIMContactList(IMAccount &account) {
	return ContactListMngr->AddIMContactList(account);
}

void PurpleIMFactory::removeIMAccount(IMAccount imAccount) {
	AccountMngr->RemoveIMAccount(imAccount);
}

void PurpleIMFactory::imAccountUpdated(IMAccount imAccount) {
	AccountMngr->UpdateIMAccount(imAccount);
}
