//filter-slfi-static.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2012-2019
 *
 *  This file is part of roard a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include <roaraudio.h>
#include <libroarlight/libroarlight.h>

#define MAX_EVENTS   8
#define MAX_RANGES  32

struct slfi_range {
 ssize_t start;
 ssize_t end;
 uint8_t value;
};

struct slfi_static {
 uint8_t event[MAX_EVENTS];
 size_t event_len;
 struct slfi_range range[MAX_RANGES];
 size_t range_len;
};

static struct slfi_range __parse_range(const char * str) {
 struct slfi_range range = {0, 0, 0};
 char * buf = roar_mm_strdup(str);
 char * value;
 char * end;

 if ( buf == NULL ) {
  ROAR_ERR("__parse_range(*): Can not allocate memory. Bad.");
  return range;
 }

 value = strstr(buf, ":");
 if ( value != NULL ) {
  *value = 0;
  value++;
  range.value = atoi(value);
 }

 end = strstr(buf, "-");
 if ( end != NULL ) {
  *end = 0;
  end++;
 }

 range.start = atoi(buf);
 if ( range.start < 0 )
  range.start = 0;

 range.end = range.start;

 if ( end != NULL ) {
  range.end = atoi(end);
  if ( range.end < range.start )
   range.end = range.start;
 }

 roar_mm_free(buf);
 return range;
}

static int __init(struct roar_slfi_inst * inst, const struct roar_keyval * para, ssize_t paralen) {
 struct slfi_static * self = roar_mm_malloc(sizeof(struct slfi_static));
 const struct roar_keyval * kv;
 ssize_t i;

 if ( self == NULL )
  return -1;

 memset(self, 0, sizeof(*self));
 inst->userdata = self;

 for (i = 0; i < paralen; i++) {
  kv = &(para[i]);
  if ( kv->key == NULL || kv->value == NULL )
   continue;

  if ( !strcmp(kv->key, "event") ) {
   if ( self->event_len == MAX_EVENTS ) {
    ROAR_WARN("__init(*): Can not add (list is full) event: %s", kv->value);
    continue;
   }
   self->event[self->event_len++] = roar_roardmx_str2event(kv->value);
  } else if ( !strcmp(kv->key, "set") || !strcmp(kv->key, "sset") || !strcmp(kv->key, "clear") ||
              !strcmp(kv->key, "rangeset") || !strcmp(kv->key, "rangeclear") ) {
   if ( self->range_len == MAX_RANGES ) {
    ROAR_WARN("__init(*): Can not add (list is full) range: %s", kv->value);
    continue;
   }
   self->range[self->range_len++] = __parse_range(kv->value);
  } else {
   ROAR_WARN("__init(*): Unknown parameter: %s", kv->key);
  }
 }

 return 0;
}

static int __update(struct roar_slfi_inst * inst, uint8_t * universe, ssize_t size_of_universe, int32_t usecspassed, const uint8_t * event, size_t eventlen) {
 struct slfi_static * self = inst->userdata;
 size_t i;
 ssize_t j;

 (void)inst, (void)usecspassed, (void)event, (void)eventlen;

 for (i = 0; i < self->event_len; i++)
  roar_slfi_event_add(inst, self->event[i]);

 for (i = 0; i < self->range_len; i++) {
  for (j = self->range[i].start; j <= self->range[i].end; j++) {
   if ( j >= size_of_universe ) {
    ROAR_WARN("__update(*): Universe too small for filter (range=%lu, channel=%lu).", (unsigned long int)i, (unsigned long int)j);
    break;
   }
   universe[j] = self->range[i].value;
  }
 }

 return 0;
}

static const struct roar_slfi_filter filter[1] = {
 {
  .name = "static",
  .description = "Static SLFI filter",
  .flags = ROAR_SLFI_FLAG_ON_UPDATE,
  .init = __init,
  .uninit = NULL,
  .update = __update,
  .ctl = NULL
 }
};

ROAR_DL_PLUGIN_REG_SLFI(filter);

// This is the plugin control block.
ROAR_DL_PLUGIN_START(filter_slfi_static) {
 // Here we set the name and vendor of our plugin.
 // If you have no Vendor ID you need to use ROAR_DL_PLUGIN_META_PRODUCT_NV().
 ROAR_DL_PLUGIN_META_PRODUCT_NIV("filter-slfi-static", ROAR_VID_ROARAUDIO, ROAR_VNAME_ROARAUDIO);

 // This sets the version of your plugin.
 ROAR_DL_PLUGIN_META_VERSION(ROAR_VERSION_STRING);

 // This sets the license of your plugin.
 // If there is no tag for the license you use you can just
 // use ROAR_DL_PLUGIN_META_LICENSE().
 ROAR_DL_PLUGIN_META_LICENSE_TAG(GPLv3_0);

 // This sets the author and contact infos.
 // There are several other macros to do this with other parameters.
 // See ROAR_DL_PLUGIN_META_CONTACT*() in the header or documentation.
 ROAR_DL_PLUGIN_META_CONTACT_FLNE("Philipp", "Schafft", "ph3-der-loewe", "lion@lion.leolix.org");

 // This sets the description for your plugin.
 ROAR_DL_PLUGIN_META_DESC("This plugin sets static values.");

 // Load filters.
 ROAR_DL_PLUGIN_REG_FNFUNC(ROAR_DL_FN_FILTER);

// This is the end of the control block.
} ROAR_DL_PLUGIN_END

//ll
