-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with Maths, CommandLineData;

separate (Dictionary)
procedure Initialize (Write_To_File : in Boolean) is

   File   : SPARK_IO.File_Type := SPARK_IO.Null_File;
   Status : SPARK_IO.File_Status;

   procedure Assign_Initial_Value
   --# global out Dict;
   --# derives Dict from ;
   is
   begin
      Dynamic_Symbol_Table.Initialize (Dict.Symbols);

      Dict.File_Name       := E_Strings.Empty_String;
      Dict.TemporaryFile   := SPARK_IO.Null_File;
      Dict.Main            :=
        Main_Program_Set'
        (Subprogram     => RawDict.Null_Subprogram_Info_Ref,
         Priority_Given => False,
         The_Priority   => LexTokenManager.Null_String);
      Dict.Packages        :=
        Package_Set'
        (Package_Standard                 => RawDict.Null_Package_Info_Ref,
         Package_ASCII                    => RawDict.Null_Package_Info_Ref,
         Package_Ada                      => RawDict.Null_Package_Info_Ref,
         Package_Ada_Characters           => RawDict.Null_Package_Info_Ref,
         Package_Ada_Characters_Latin1    => RawDict.Null_Package_Info_Ref,
         Package_Real_Time                => RawDict.Null_Package_Info_Ref,
         Package_Synchronous_Task_Control => RawDict.Null_Package_Info_Ref,
         Package_Interrupts               => RawDict.Null_Package_Info_Ref);
      --# accept Flow, 32, Dict.Types, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 31, Dict.Types, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 602, Dict, Dict.Types, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 32, Dict.Subprograms, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 31, Dict.Subprograms, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 602, Dict, Dict.Subprograms, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 32, Dict.Null_Variable, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 31, Dict.Null_Variable, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 602, Dict, Dict.Null_Variable, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 32, Dict.The_Partition, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 31, Dict.The_Partition, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 602, Dict, Dict.The_Partition, "Flow errors due to undefined components of Dict OK";
   end Assign_Initial_Value;

   --------------------------------------------------------------------------------

   procedure Insert_Lex_String (Name  : in     String;
                                Token :    out LexTokenManager.Lex_String)
   --# global in out LexTokenManager.State;
   --# derives LexTokenManager.State,
   --#         Token                 from LexTokenManager.State,
   --#                                    Name;
   is
   begin
      LexTokenManager.Insert_Examiner_String (Str     => E_Strings.Copy_String (Str => Name),
                                              Lex_Str => Token);
   end Insert_Lex_String;

   --------------------------------------------------------------------------------

   procedure Add_Unknown_Type
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --# derives Dict,
   --#         LexTokenManager.State from *,
   --#                                    LexTokenManager.State;
   is
      Token     : LexTokenManager.Lex_String;
      Type_Mark : RawDict.Type_Info_Ref;
   begin
      Insert_Lex_String (Name  => "unknown",
                         Token => Token);
      RawDict.Create_Type
        (Name            => Token,
         The_Declaration => RawDict.Null_Declaration_Info_Ref,
         Is_Private      => False,
         Is_Announcement => False,
         Comp_Unit       => ContextManager.NullUnit,
         Loc             => LexTokenManager.Null_Token_Position,
         Type_Mark       => Type_Mark);
      RawDict.Set_Type_Discriminant (Type_Mark    => Type_Mark,
                                     Discriminant => Unknown_Type_Item);
      RawDict.Set_Type_Static (Type_Mark => Type_Mark,
                               Static    => True);
      Dict.Types.Unknown_Type_Mark := Type_Mark;
   end Add_Unknown_Type;

   --------------------------------------------------------------------------------

   procedure Add_Predefined_Package_Standard
   --# global in out Dict;
   --# derives Dict from *;
   is
      The_Specification : RawDict.Declaration_Info_Ref;
      The_Package       : RawDict.Package_Info_Ref;
   begin
      RawDict.Create_Declaration
        (Context         => ProgramContext,
         Scope           => Set_Visibility (The_Visibility => Visible,
                                            The_Unit       => NullSymbol),
         Comp_Unit       => ContextManager.NullUnit,
         Loc             => LexTokenManager.Null_Token_Position,
         The_Declaration => The_Specification);
      RawDict.Create_Package
        (Name            => LexTokenManager.Standard_Token,
         The_Declaration => The_Specification,
         Comp_Unit       => ContextManager.NullUnit,
         Loc             => LexTokenManager.Null_Token_Position,
         The_Package     => The_Package);
      Dict.Packages.Package_Standard := The_Package;
   end Add_Predefined_Package_Standard;

   --------------------------------------------------------------------------------

   procedure Add_Package_Standard
   --# global in     CommandLineData.Content;
   --#        in out Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         LexTokenManager.State from *,
   --#                                    CommandLineData.Content,
   --#                                    LexTokenManager.State &
   --#         SPARK_IO.File_Sys     from *,
   --#                                    CommandLineData.Content,
   --#                                    Dict,
   --#                                    LexTokenManager.State;
   is

      procedure Add_Predefined_Type
        (Token     : in     LexTokenManager.Lex_String;
         Context   : in     Contexts;
         Type_Mark :    out RawDict.Type_Info_Ref)
      --# global in out Dict;
      --# derives Dict,
      --#         Type_Mark from Context,
      --#                        Dict,
      --#                        Token;
      is
         The_Declaration : RawDict.Declaration_Info_Ref;
      begin
         Add_Declaration
           (Comp_Unit       => ContextManager.NullUnit,
            Loc             => Null_Location,
            Scope           => Predefined_Scope,
            Context         => Context,
            The_Declaration => The_Declaration);
         RawDict.Create_Type
           (Name            => Token,
            The_Declaration => The_Declaration,
            Is_Private      => False,
            Is_Announcement => False,
            Comp_Unit       => ContextManager.NullUnit,
            Loc             => LexTokenManager.Null_Token_Position,
            Type_Mark       => Type_Mark);
      end Add_Predefined_Type;

      --------------------------------------------------------------------------------

      procedure Add_Universal_Types
      --# global in out Dict;
      --# derives Dict from *;
      is

         procedure Add_Universal_Integer_Type
         --# global in out Dict;
         --# derives Dict from *;
         is
            Type_Mark : RawDict.Type_Info_Ref;
         begin
            Add_Predefined_Type
              (Token     => LexTokenManager.Universal_Integer_Token,
               Context   => ProofContext,
               Type_Mark => Type_Mark);
            Add_Integer_Type_Mark
              (Type_Mark => Type_Mark,
               Static    => True,
               Lower     => LexTokenManager.Null_String,
               Upper     => LexTokenManager.Null_String);
            Dict.Types.The_Universal_Types (Universal_Integer_Type) := Type_Mark;
         end Add_Universal_Integer_Type;

         --------------------------------------------------------------------------------

         procedure Add_Universal_Real_Type
         --# global in out Dict;
         --# derives Dict from *;
         is
            Type_Mark : RawDict.Type_Info_Ref;
         begin
            Add_Predefined_Type (Token     => LexTokenManager.Universal_Real_Token,
                                 Context   => ProofContext,
                                 Type_Mark => Type_Mark);
            Add_Floating_Point_Type_Mark (Type_Mark => Type_Mark,
                                          Static    => True);
            Dict.Types.The_Universal_Types (Universal_Real_Type) := Type_Mark;
         end Add_Universal_Real_Type;

         --------------------------------------------------------------------------------

         procedure Add_Universal_Fixed_Type
         --# global in out Dict;
         --# derives Dict from *;
         is
            Type_Mark : RawDict.Type_Info_Ref;
         begin
            Add_Predefined_Type (Token     => LexTokenManager.Universal_Fixed_Token,
                                 Context   => ProofContext,
                                 Type_Mark => Type_Mark);
            Add_Fixed_Point_Type_Mark (Type_Mark => Type_Mark,
                                       Static    => True);
            Dict.Types.The_Universal_Types (Universal_Fixed_Type) := Type_Mark;
         end Add_Universal_Fixed_Type;

      begin -- Add_Universal_Types
         Add_Universal_Integer_Type;
         Add_Universal_Real_Type;
         Add_Universal_Fixed_Type;
      end Add_Universal_Types;

      --------------------------------------------------------------------------------

      procedure Add_Predefined_Type_Marks
      --# global in     CommandLineData.Content;
      --#        in out Dict;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict,
      --#         LexTokenManager.State from *,
      --#                                    CommandLineData.Content,
      --#                                    LexTokenManager.State &
      --#         SPARK_IO.File_Sys     from *,
      --#                                    CommandLineData.Content,
      --#                                    Dict,
      --#                                    LexTokenManager.State;
      is

         procedure Set_Type_Is_Predefined (Type_Mark : in RawDict.Type_Info_Ref)
         --# global in out Dict;
         --# derives Dict from *,
         --#                   Type_Mark;
         is
         begin
            RawDict.Set_Type_Derived (Type_Mark  => Type_Mark,
                                      Is_Derived => False);
         end Set_Type_Is_Predefined;

         --------------------------------------------------------------------------------

         procedure Add_Predefined_Character_Type
         --# global in     CommandLineData.Content;
         --#        in out Dict;
         --#        in out LexTokenManager.State;
         --# derives Dict,
         --#         LexTokenManager.State from *,
         --#                                    CommandLineData.Content,
         --#                                    LexTokenManager.State;
         is
            Type_Mark : RawDict.Type_Info_Ref;
            Store_Rep : LexTokenManager.Lex_String;
         begin
            Add_Predefined_Type (Token     => LexTokenManager.Character_Token,
                                 Context   => ProgramContext,
                                 Type_Mark => Type_Mark);
            Add_Enumeration_Type_Mark (Type_Mark => Type_Mark,
                                       Static    => True);
            -- Set bounds for character type for use with 'first, 'last
            RawDict.Set_Type_Lower (Type_Mark => Type_Mark,
                                    Lower     => LexTokenManager.Zero_Value);

            case CommandLineData.Content.Language_Profile is
               when CommandLineData.SPARK83 =>
                  Maths.StorageRep (Maths.IntegerToValue (127), Store_Rep);
                  RawDict.Set_Type_Upper (Type_Mark => Type_Mark,
                                          Upper     => Store_Rep);

               when CommandLineData.SPARK95_Onwards =>
                  Maths.StorageRep (Maths.IntegerToValue (255), Store_Rep);
                  RawDict.Set_Type_Upper (Type_Mark => Type_Mark,
                                          Upper     => Store_Rep);
            end case;
            Dict.Types.The_Predefined_Types (Predefined_Character_Type) := Type_Mark;
         end Add_Predefined_Character_Type;

         --------------------------------------------------------------------------------

         procedure Add_Predefined_Boolean_Type
         --# global in out Dict;
         --#        in out LexTokenManager.State;
         --# derives Dict,
         --#         LexTokenManager.State from *,
         --#                                    LexTokenManager.State;
         is
            Type_Mark : RawDict.Type_Info_Ref;

            procedure Add_Predefined_Enumeration_Literal
              (Name     : in String;
               Value    : in LexTokenManager.Lex_String;
               The_Type : in RawDict.Type_Info_Ref)
            --# global in out Dict;
            --#        in out LexTokenManager.State;
            --# derives Dict                  from *,
            --#                                    LexTokenManager.State,
            --#                                    Name,
            --#                                    The_Type,
            --#                                    Value &
            --#         LexTokenManager.State from *,
            --#                                    Name;
            is
               Token                   : LexTokenManager.Lex_String;
               The_Enumeration_Literal : RawDict.Enumeration_Literal_Info_Ref;
               pragma Unreferenced (The_Enumeration_Literal);
            begin
               Insert_Lex_String (Name  => Name,
                                  Token => Token);
               --# accept Flow, 10, The_Enumeration_Literal, "Expected ineffective assignment to OK";
               Add_Enumeration_Literal
                 (Name                    => Token,
                  Comp_Unit               => ContextManager.NullUnit,
                  Declaration             => Null_Location,
                  Position                => Value,
                  Type_Mark               => The_Type,
                  The_Enumeration_Literal => The_Enumeration_Literal);
               --# end accept;
               --# accept Flow, 33, The_Enumeration_Literal, "Expected to be neither referenced nor exported";
            end Add_Predefined_Enumeration_Literal;

         begin -- Add_Predefined_Boolean_Type
            Add_Predefined_Type (Token     => LexTokenManager.Boolean_Token,
                                 Context   => ProgramContext,
                                 Type_Mark => Type_Mark);
            Add_Enumeration_Type_Mark (Type_Mark => Type_Mark,
                                       Static    => True);
            Add_Predefined_Enumeration_Literal (Name     => "False",
                                                Value    => LexTokenManager.False_Token,
                                                The_Type => Type_Mark);
            Add_Predefined_Enumeration_Literal (Name     => "True",
                                                Value    => LexTokenManager.True_Token,
                                                The_Type => Type_Mark);
            Dict.Types.The_Predefined_Types (Predefined_Boolean_Type) := Type_Mark;
            Dict.Types.The_Predefined_Literals (Predefined_False)     :=
              RawDict.Get_Type_First_Enumeration_Literal (Type_Mark => Type_Mark);
            Dict.Types.The_Predefined_Literals (Predefined_True)      :=
              RawDict.Get_Type_Last_Enumeration_Literal (Type_Mark => Type_Mark);
         end Add_Predefined_Boolean_Type;

         --------------------------------------------------------------------------------

         procedure Add_Predefined_Integer_Type
         --# global in     CommandLineData.Content;
         --#        in out Dict;
         --#        in out LexTokenManager.State;
         --# derives Dict,
         --#         LexTokenManager.State from *,
         --#                                    CommandLineData.Content,
         --#                                    LexTokenManager.State;
         is
            Token     : LexTokenManager.Lex_String;
            Type_Mark : RawDict.Type_Info_Ref;
         begin
            Add_Predefined_Type (Token     => LexTokenManager.Integer_Token,
                                 Context   => ProgramContext,
                                 Type_Mark => Type_Mark);
            Add_Integer_Type_Mark
              (Type_Mark => Type_Mark,
               Static    => True,
               Lower     => LexTokenManager.Null_String,
               Upper     => LexTokenManager.Null_String);
            Set_Type_Is_Predefined (Type_Mark => Type_Mark);
            Dict.Types.The_Predefined_Types (Predefined_Integer_Type) := Type_Mark;
            if CommandLineData.Content.Target_Data then
               Insert_Lex_String (Name  => "Long_Integer",
                                  Token => Token);
               Add_Predefined_Type (Token     => Token,
                                    Context   => ProgramContext,
                                    Type_Mark => Type_Mark);
               Add_Integer_Type_Mark
                 (Type_Mark => Type_Mark,
                  Static    => True,
                  Lower     => LexTokenManager.Null_String,
                  Upper     => LexTokenManager.Null_String);
               Set_Type_Is_Predefined (Type_Mark => Type_Mark);
               Dict.Types.The_Predefined_Types (Predefined_Long_Integer_Type) := Type_Mark;
            else
               Dict.Types.The_Predefined_Types (Predefined_Long_Integer_Type) := RawDict.Null_Type_Info_Ref;
            end if;
         end Add_Predefined_Integer_Type;

         --------------------------------------------------------------------------------

         procedure Add_Predefined_Float_Type
         --# global in     CommandLineData.Content;
         --#        in out Dict;
         --#        in out LexTokenManager.State;
         --# derives Dict,
         --#         LexTokenManager.State from *,
         --#                                    CommandLineData.Content,
         --#                                    LexTokenManager.State;
         is
            Token     : LexTokenManager.Lex_String;
            Type_Mark : RawDict.Type_Info_Ref;
         begin
            Add_Predefined_Type (Token     => LexTokenManager.Float_Token,
                                 Context   => ProgramContext,
                                 Type_Mark => Type_Mark);
            Add_Floating_Point_Type_Mark (Type_Mark => Type_Mark,
                                          Static    => True);
            Set_Type_Is_Predefined (Type_Mark => Type_Mark);
            Dict.Types.The_Predefined_Types (Predefined_Float_Type) := Type_Mark;
            if CommandLineData.Content.Target_Data then
               Insert_Lex_String (Name  => "Long_Float",
                                  Token => Token);
               Add_Predefined_Type (Token     => Token,
                                    Context   => ProgramContext,
                                    Type_Mark => Type_Mark);
               Add_Floating_Point_Type_Mark (Type_Mark => Type_Mark,
                                             Static    => True);
               Set_Type_Is_Predefined (Type_Mark => Type_Mark);
               Dict.Types.The_Predefined_Types (Predefined_Long_Float_Type) := Type_Mark;
            else
               Dict.Types.The_Predefined_Types (Predefined_Long_Float_Type) := RawDict.Null_Type_Info_Ref;
            end if;
         end Add_Predefined_Float_Type;

         --------------------------------------------------------------------------------

         procedure Add_Duration
         --# global in out Dict;
         --# derives Dict from *;
         is
            Type_Mark : RawDict.Type_Info_Ref;
         begin
            Add_Predefined_Type (Token     => LexTokenManager.Duration_Token,
                                 Context   => ProgramContext,
                                 Type_Mark => Type_Mark);
            Add_Fixed_Point_Type_Mark (Type_Mark => Type_Mark,
                                       Static    => True);
            Dict.Types.The_Predefined_Types (Predefined_Duration_Type) := Type_Mark;
         end Add_Duration;

         --------------------------------------------------------------------------------

         procedure Add_Predefined_String_Type
         --# global in     LexTokenManager.State;
         --#        in out Dict;
         --#        in out SPARK_IO.File_Sys;
         --# derives Dict              from * &
         --#         SPARK_IO.File_Sys from *,
         --#                                Dict,
         --#                                LexTokenManager.State;
         is
            Type_Mark       : RawDict.Type_Info_Ref;
            The_Array_Index : RawDict.Array_Index_Info_Ref;
            pragma Unreferenced (The_Array_Index);
         begin
            Add_Predefined_Type (Token     => LexTokenManager.String_Token,
                                 Context   => ProgramContext,
                                 Type_Mark => Type_Mark);
            Add_Array_Type_Mark (Type_Mark      => Type_Mark,
                                 Component_Type => Get_Predefined_Character_Type,
                                 Static         => False);
            --# accept Flow, 10, The_Array_Index, "Expected ineffective assignment to OK";
            Add_Array_Index
              (The_Array_Type  => Type_Mark,
               Index_Type      => Get_Predefined_Positive_Subtype,
               Comp_Unit       => ContextManager.NullUnit,
               Declaration     => Null_Location,
               The_Array_Index => The_Array_Index);
            --# end accept;
            Dict.Types.The_Predefined_Types (Predefined_String_Type) := Type_Mark;
            --# accept Flow, 33, The_Array_Index, "Expected to be neither referenced nor exported";
         end Add_Predefined_String_Type;

         --------------------------------------------------------------------------------

         procedure Add_Predefined_Natural_Subtype
         --# global in out Dict;
         --# derives Dict from *;
         is
            Type_Mark : RawDict.Type_Info_Ref;
         begin
            Add_Predefined_Type (Token     => LexTokenManager.Natural_Token,
                                 Context   => ProgramContext,
                                 Type_Mark => Type_Mark);
            RawDict.Set_Type_Parent (Type_Mark => Type_Mark,
                                     Parent    => Get_Predefined_Integer_Type);
            Add_Integer_Type_Mark
              (Type_Mark => Type_Mark,
               Static    => True,
               Lower     => LexTokenManager.Zero_Value,
               Upper     => LexTokenManager.Null_String);
            Dict.Types.The_Predefined_Types (Predefined_Natural_Subtype) := Type_Mark;
         end Add_Predefined_Natural_Subtype;

         --------------------------------------------------------------------------------

         procedure Add_Predefined_Positive_Subtype
         --# global in out Dict;
         --# derives Dict from *;
         is
            Type_Mark : RawDict.Type_Info_Ref;
         begin
            Add_Predefined_Type (Token     => LexTokenManager.Positive_Token,
                                 Context   => ProgramContext,
                                 Type_Mark => Type_Mark);
            RawDict.Set_Type_Parent (Type_Mark => Type_Mark,
                                     Parent    => Get_Predefined_Integer_Type);
            Add_Integer_Type_Mark
              (Type_Mark => Type_Mark,
               Static    => True,
               Lower     => LexTokenManager.One_Value,
               Upper     => LexTokenManager.Null_String);
            Dict.Types.The_Predefined_Types (Predefined_Positive_Subtype) := Type_Mark;
         end Add_Predefined_Positive_Subtype;

      begin -- Add_Predefined_Type_Marks
         Add_Predefined_Boolean_Type;
         Add_Predefined_Integer_Type;
         Add_Predefined_Float_Type;
         Add_Predefined_Character_Type;
         Add_Predefined_Natural_Subtype;
         Add_Predefined_Positive_Subtype;
         Add_Predefined_String_Type;
         if not CommandLineData.Content.No_Duration then
            Add_Duration;
         end if;
      end Add_Predefined_Type_Marks;

      --------------------------------------------------------------------------------

      procedure Add_Null_Own_Variable
      --# global in out Dict;
      --#        in out LexTokenManager.State;
      --# derives Dict,
      --#         LexTokenManager.State from *,
      --#                                    LexTokenManager.State;
      is
         The_Variable     : RawDict.Variable_Info_Ref;
         The_Own_Variable : RawDict.Own_Variable_Info_Ref;
         Null_Str         : LexTokenManager.Lex_String;
      begin
         Insert_Lex_String (Name  => "null",
                            Token => Null_Str);
         RawDict.Create_Variable
           (Name         => Null_Str,
            Comp_Unit    => ContextManager.NullUnit,
            Loc          => LexTokenManager.Null_Token_Position,
            The_Variable => The_Variable);
         RawDict.Create_Own_Variable
           (Variable         => The_Variable,
            Owner            => RawDict.Get_Package_Symbol (Get_Predefined_Package_Standard),
            Comp_Unit        => ContextManager.NullUnit,
            Loc              => LexTokenManager.Null_Token_Position,
            The_Own_Variable => The_Own_Variable);
         RawDict.Set_Variable_Own_Variable (The_Variable => The_Variable,
                                            Own_Variable => The_Own_Variable);
         RawDict.Set_Package_Own_Variables (The_Package   => Get_Predefined_Package_Standard,
                                            Own_Variables => The_Own_Variable);
         RawDict.Set_Own_Variable_Mode (The_Own_Variable => The_Own_Variable,
                                        Mode             => OutMode);
         Dict.Null_Variable := The_Variable;
      end Add_Null_Own_Variable;

   begin -- Add_Package_Standard
      Add_Predefined_Package_Standard;
      Add_Universal_Types;
      Add_Predefined_Type_Marks;
      -- package standard has a mode out own variable called "Null" which is used
      -- as a "data sink" for operations that affect only things outside the SPARK
      -- boundary such as delay statements or calls to procedures which derive null
      -- from something.
      Add_Null_Own_Variable;
   end Add_Package_Standard;

   --------------------------------------------------------------------------------

   procedure Add_Predefined_Constant (Name        : in String;
                                      Asc_Code    : in Natural;
                                      The_Package : in RawDict.Package_Info_Ref)
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --# derives Dict                  from *,
   --#                                    Asc_Code,
   --#                                    LexTokenManager.State,
   --#                                    Name,
   --#                                    The_Package &
   --#         LexTokenManager.State from *,
   --#                                    Asc_Code,
   --#                                    Name;
   is
      Token, Store_Rep : LexTokenManager.Lex_String;
      The_Constant     : RawDict.Constant_Info_Ref;
      The_Declaration  : RawDict.Declaration_Info_Ref;
   begin
      Add_Declaration
        (Comp_Unit       => ContextManager.NullUnit,
         Loc             => Null_Location,
         Scope           => Set_Visibility (The_Visibility => Visible,
                                            The_Unit       => RawDict.Get_Package_Symbol (The_Package)),
         Context         => ProgramContext,
         The_Declaration => The_Declaration);
      Insert_Lex_String (Name  => Name,
                         Token => Token);
      RawDict.Create_Constant
        (Name            => Token,
         Type_Mark       => Get_Predefined_Character_Type,
         Static          => True,
         The_Declaration => The_Declaration,
         Is_Deferred     => False,
         Comp_Unit       => ContextManager.NullUnit,
         Loc             => LexTokenManager.Null_Token_Position,
         The_Constant    => The_Constant);
      -- Add values for predefined constants
      Maths.StorageRep (Maths.IntegerToValue (Asc_Code), Store_Rep);
      RawDict.Set_Constant_Value (The_Constant => The_Constant,
                                  Value        => Store_Rep);
   end Add_Predefined_Constant;

   -------------------------------------

   -- Adds Character constants which are common to both
   -- SPARK83 and SPARK95 _AND_ have the same identifier - this
   -- is a subset of the first 127 Characters.
   procedure Add_Common_Characters (To_The_Package : in RawDict.Package_Info_Ref)
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --# derives Dict                  from *,
   --#                                    LexTokenManager.State,
   --#                                    To_The_Package &
   --#         LexTokenManager.State from *;
   is
   begin
      Add_Predefined_Constant (Name        => "NUL",
                               Asc_Code    => 0,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "SOH",
                               Asc_Code    => 1,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "STX",
                               Asc_Code    => 2,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "ETX",
                               Asc_Code    => 3,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "EOT",
                               Asc_Code    => 4,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "ENQ",
                               Asc_Code    => 5,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "ACK",
                               Asc_Code    => 6,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "BEL",
                               Asc_Code    => 7,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "BS",
                               Asc_Code    => 8,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "HT",
                               Asc_Code    => 9,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LF",
                               Asc_Code    => 10,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "VT",
                               Asc_Code    => 11,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "FF",
                               Asc_Code    => 12,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "CR",
                               Asc_Code    => 13,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "SO",
                               Asc_Code    => 14,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "SI",
                               Asc_Code    => 15,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "DLE",
                               Asc_Code    => 16,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "DC1",
                               Asc_Code    => 17,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "DC2",
                               Asc_Code    => 18,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "DC3",
                               Asc_Code    => 19,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "DC4",
                               Asc_Code    => 20,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "NAK",
                               Asc_Code    => 21,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "SYN",
                               Asc_Code    => 22,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "ETB",
                               Asc_Code    => 23,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "CAN",
                               Asc_Code    => 24,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "EM",
                               Asc_Code    => 25,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "SUB",
                               Asc_Code    => 26,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "ESC",
                               Asc_Code    => 27,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "FS",
                               Asc_Code    => 28,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "GS",
                               Asc_Code    => 29,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "RS",
                               Asc_Code    => 30,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "US",
                               Asc_Code    => 31,
                               The_Package => To_The_Package);

      --# assert True;

      Add_Predefined_Constant (Name        => "LC_A",
                               Asc_Code    => 97,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_B",
                               Asc_Code    => 98,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_C",
                               Asc_Code    => 99,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_D",
                               Asc_Code    => 100,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_E",
                               Asc_Code    => 101,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_F",
                               Asc_Code    => 102,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_G",
                               Asc_Code    => 103,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_H",
                               Asc_Code    => 104,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_I",
                               Asc_Code    => 105,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_J",
                               Asc_Code    => 106,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_K",
                               Asc_Code    => 107,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_L",
                               Asc_Code    => 108,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_M",
                               Asc_Code    => 109,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_N",
                               Asc_Code    => 110,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_O",
                               Asc_Code    => 111,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_P",
                               Asc_Code    => 112,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_Q",
                               Asc_Code    => 113,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_R",
                               Asc_Code    => 114,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_S",
                               Asc_Code    => 115,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_T",
                               Asc_Code    => 116,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_U",
                               Asc_Code    => 117,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_V",
                               Asc_Code    => 118,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_W",
                               Asc_Code    => 119,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_X",
                               Asc_Code    => 120,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_Y",
                               Asc_Code    => 121,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_Z",
                               Asc_Code    => 122,
                               The_Package => To_The_Package);

      Add_Predefined_Constant (Name        => "DEL",
                               Asc_Code    => 127,
                               The_Package => To_The_Package);
   end Add_Common_Characters;

   --------------------------------------------------------------------------------

   procedure Add_Package_ASCII
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         LexTokenManager.State from *,
   --#                                    LexTokenManager.State &
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Dict,
   --#                                    LexTokenManager.State;
   is
      Token         : LexTokenManager.Lex_String;
      Package_ASCII : RawDict.Package_Info_Ref;
   begin
      Insert_Lex_String (Name  => "ASCII",
                         Token => Token);
      Add_New_Package
        (Name        => Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => Predefined_Scope,
         The_Parent  => RawDict.Null_Package_Info_Ref,
         Child_Sort  => Public,
         The_Package => Package_ASCII);
      Dict.Packages.Package_ASCII := Package_ASCII;

      Add_Common_Characters (To_The_Package => Package_ASCII);

      -- Ada83 Characters whose identiifers differ from those used in Ada95.
      Add_Predefined_Constant (Name        => "Exclam",
                               Asc_Code    => 33,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Quotation",
                               Asc_Code    => 34,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Sharp",
                               Asc_Code    => 35,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Dollar",
                               Asc_Code    => 36,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Percent",
                               Asc_Code    => 37,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Ampersand",
                               Asc_Code    => 38,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Colon",
                               Asc_Code    => 58,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Semicolon",
                               Asc_Code    => 59,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Query",
                               Asc_Code    => 63,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "At_Sign",
                               Asc_Code    => 64,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "L_Bracket",
                               Asc_Code    => 91,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Back_Slash",
                               Asc_Code    => 92,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "R_Bracket",
                               Asc_Code    => 93,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Circumflex",
                               Asc_Code    => 94,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Underline",
                               Asc_Code    => 95,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Grave",
                               Asc_Code    => 96,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "L_Brace",
                               Asc_Code    => 123,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Bar",
                               Asc_Code    => 124,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "R_Brace",
                               Asc_Code    => 125,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Tilde",
                               Asc_Code    => 126,
                               The_Package => Package_ASCII);
   end Add_Package_ASCII;

   --------------------------------------------------------------------------------

   procedure Add_Package_Ada
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         LexTokenManager.State from *,
   --#                                    LexTokenManager.State &
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Dict,
   --#                                    LexTokenManager.State;
   is
      Token                                           : LexTokenManager.Lex_String;
      Package_Ada, Package_Characters, Package_Latin1 : RawDict.Package_Info_Ref;

      procedure Add_Lower_Half_Control_And_Graphic_Characters (Package_Latin1 : in RawDict.Package_Info_Ref)
      --# global in out Dict;
      --#        in out LexTokenManager.State;
      --# derives Dict                  from *,
      --#                                    LexTokenManager.State,
      --#                                    Package_Latin1 &
      --#         LexTokenManager.State from *;
      is
      begin
         --ISO 6429 control characters
         Add_Predefined_Constant (Name        => "IS4",
                                  Asc_Code    => 28,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "IS3",
                                  Asc_Code    => 29,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "IS2",
                                  Asc_Code    => 30,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "IS1",
                                  Asc_Code    => 31,
                                  The_Package => Package_Latin1);

         --ISO 646 graphic characters
         Add_Predefined_Constant (Name        => "Space",
                                  Asc_Code    => 32,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Exclamation",
                                  Asc_Code    => 33,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Quotation",
                                  Asc_Code    => 34,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Number_Sign",
                                  Asc_Code    => 35,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Dollar_Sign",
                                  Asc_Code    => 36,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Percent_Sign",
                                  Asc_Code    => 37,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Ampersand",
                                  Asc_Code    => 38,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Apostrophe",
                                  Asc_Code    => 39,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Left_Parenthesis",
                                  Asc_Code    => 40,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Right_Parenthesis",
                                  Asc_Code    => 41,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Asterisk",
                                  Asc_Code    => 42,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Plus_Sign",
                                  Asc_Code    => 43,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Comma",
                                  Asc_Code    => 44,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Hyphen",
                                  Asc_Code    => 45,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Minus_Sign",
                                  Asc_Code    => 45,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Full_Stop",
                                  Asc_Code    => 46,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Solidus",
                                  Asc_Code    => 47,
                                  The_Package => Package_Latin1);

         --Decimal digits '0' through '9 are at positions 48 through 57

         Add_Predefined_Constant (Name        => "Colon",
                                  Asc_Code    => 58,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Semicolon",
                                  Asc_Code    => 59,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Less_Than_Sign",
                                  Asc_Code    => 60,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Equals_Sign",
                                  Asc_Code    => 61,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Greater_Than_Sign",
                                  Asc_Code    => 62,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Question",
                                  Asc_Code    => 63,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Commercial_At",
                                  Asc_Code    => 64,
                                  The_Package => Package_Latin1);

         --Letters 'A' through 'Z' are at positions 65 through 90

         Add_Predefined_Constant (Name        => "Left_Square_Bracket",
                                  Asc_Code    => 91,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Reverse_Solidus",
                                  Asc_Code    => 92,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Right_Square_Bracket",
                                  Asc_Code    => 93,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Circumflex",
                                  Asc_Code    => 94,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Low_Line",
                                  Asc_Code    => 95,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Grave",
                                  Asc_Code    => 96,
                                  The_Package => Package_Latin1);

         Add_Predefined_Constant (Name        => "Left_Curly_Bracket",
                                  Asc_Code    => 123,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Vertical_Line",
                                  Asc_Code    => 124,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Right_Curly_Bracket",
                                  Asc_Code    => 125,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Tilde",
                                  Asc_Code    => 126,
                                  The_Package => Package_Latin1);

      end Add_Lower_Half_Control_And_Graphic_Characters;

      --------------------------------------------------------------------------------

      procedure Add_Upper_Half_Ada95_Characters (Package_Latin1 : in RawDict.Package_Info_Ref)
      --# global in out Dict;
      --#        in out LexTokenManager.State;
      --# derives Dict                  from *,
      --#                                    LexTokenManager.State,
      --#                                    Package_Latin1 &
      --#         LexTokenManager.State from *;
      is
      begin
         Add_Predefined_Constant (Name        => "Reserved_128",
                                  Asc_Code    => 128,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Reserved_129",
                                  Asc_Code    => 129,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "BPH",
                                  Asc_Code    => 130,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "NBH",
                                  Asc_Code    => 131,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Reserved_132",
                                  Asc_Code    => 132,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "NEL",
                                  Asc_Code    => 133,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "SSA",
                                  Asc_Code    => 134,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "ESA",
                                  Asc_Code    => 135,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "HTS",
                                  Asc_Code    => 136,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "HTJ",
                                  Asc_Code    => 137,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "VTS",
                                  Asc_Code    => 138,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "PLD",
                                  Asc_Code    => 139,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "PLU",
                                  Asc_Code    => 140,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "RI",
                                  Asc_Code    => 141,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "SS2",
                                  Asc_Code    => 142,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "SS3",
                                  Asc_Code    => 143,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "DCS",
                                  Asc_Code    => 144,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "PU1",
                                  Asc_Code    => 145,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "PU2",
                                  Asc_Code    => 146,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "STS",
                                  Asc_Code    => 147,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "CCH",
                                  Asc_Code    => 148,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "MW",
                                  Asc_Code    => 149,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "SPA",
                                  Asc_Code    => 150,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "EPA",
                                  Asc_Code    => 151,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "SOS",
                                  Asc_Code    => 152,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Reserved_153",
                                  Asc_Code    => 153,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "SCI",
                                  Asc_Code    => 154,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "CSI",
                                  Asc_Code    => 155,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "ST",
                                  Asc_Code    => 156,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "OSC",
                                  Asc_Code    => 157,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "PM",
                                  Asc_Code    => 158,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "APC",
                                  Asc_Code    => 159,
                                  The_Package => Package_Latin1);

         --# assert True;

         --other graphic characters
         --character positions 160 .. 175
         Add_Predefined_Constant (Name        => "No_Break_Space",
                                  Asc_Code    => 160,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "NBSP",
                                  Asc_Code    => 160,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Inverted_Exclamation",
                                  Asc_Code    => 161,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Cent_Sign",
                                  Asc_Code    => 162,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Pound_Sign",
                                  Asc_Code    => 163,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Currency_Sign",
                                  Asc_Code    => 164,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Yen_Sign",
                                  Asc_Code    => 165,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Broken_Bar",
                                  Asc_Code    => 166,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Section_Sign",
                                  Asc_Code    => 167,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Diaeresis",
                                  Asc_Code    => 168,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Copyright_Sign",
                                  Asc_Code    => 169,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Feminine_Ordinal_Indicator",
                                  Asc_Code    => 170,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Left_Angle_Quotation",
                                  Asc_Code    => 171,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Not_Sign",
                                  Asc_Code    => 172,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Soft_Hyphen",
                                  Asc_Code    => 173,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Registered_Trade_Mark_Sign",
                                  Asc_Code    => 174,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Macron",
                                  Asc_Code    => 175,
                                  The_Package => Package_Latin1);

         --# assert True;

         --character positions 176 .. 191
         Add_Predefined_Constant (Name        => "Degree_Sign",
                                  Asc_Code    => 176,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Ring_Above",
                                  Asc_Code    => 177,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Plus_Minus_Sign",
                                  Asc_Code    => 177,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Superscript_Two",
                                  Asc_Code    => 178,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Superscript_Three",
                                  Asc_Code    => 179,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Acute",
                                  Asc_Code    => 180,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Micro_Sign",
                                  Asc_Code    => 181,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Pilcrow_Sign",
                                  Asc_Code    => 182,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Paragraph_Sign",
                                  Asc_Code    => 182,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Middle_Dot",
                                  Asc_Code    => 183,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Cedilla",
                                  Asc_Code    => 184,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Superscript_One",
                                  Asc_Code    => 185,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Masculine_Ordinal_Indicator",
                                  Asc_Code    => 186,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Right_Angle_Quotation",
                                  Asc_Code    => 187,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Fraction_One_Quarter",
                                  Asc_Code    => 188,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Fraction_One_Half",
                                  Asc_Code    => 189,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Fraction_Three_Quarters",
                                  Asc_Code    => 190,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Inverted_Question",
                                  Asc_Code    => 191,
                                  The_Package => Package_Latin1);

         --# assert True;

         --character positions 192 .. 207
         Add_Predefined_Constant (Name        => "UC_A_Grave",
                                  Asc_Code    => 192,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_A_Acute",
                                  Asc_Code    => 193,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_A_Circumflex",
                                  Asc_Code    => 194,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_A_Tilde",
                                  Asc_Code    => 195,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_A_Diaeresis",
                                  Asc_Code    => 196,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_A_Ring",
                                  Asc_Code    => 197,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_AE_Diphthong",
                                  Asc_Code    => 198,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_C_Cedilla",
                                  Asc_Code    => 199,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_E_Grave",
                                  Asc_Code    => 200,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_E_Acute",
                                  Asc_Code    => 201,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_E_Circumflex",
                                  Asc_Code    => 202,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_E_Diaeresis",
                                  Asc_Code    => 203,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_I_Grave",
                                  Asc_Code    => 204,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_I_Acute",
                                  Asc_Code    => 205,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_I_Circumflex",
                                  Asc_Code    => 206,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_I_Diaeresis",
                                  Asc_Code    => 207,
                                  The_Package => Package_Latin1);

         --# assert True;

         --character positions 208 .. 223
         Add_Predefined_Constant (Name        => "UC_Icelandic_Eth",
                                  Asc_Code    => 208,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_N_Tilde",
                                  Asc_Code    => 209,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_O_Grave",
                                  Asc_Code    => 210,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_O_Acute",
                                  Asc_Code    => 211,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_O_Circumflex",
                                  Asc_Code    => 212,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_O_Tilde",
                                  Asc_Code    => 213,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_O_Diaeresis",
                                  Asc_Code    => 214,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Multiplication_Sign",
                                  Asc_Code    => 215,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_O_Oblique_Stroke",
                                  Asc_Code    => 216,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_U_Grave",
                                  Asc_Code    => 217,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_U_Acute",
                                  Asc_Code    => 218,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_U_Circumflex",
                                  Asc_Code    => 219,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_U_Diaeresis",
                                  Asc_Code    => 220,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_Y_Acute",
                                  Asc_Code    => 221,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_Icelandic_Thorn",
                                  Asc_Code    => 222,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_German_Sharp_S",
                                  Asc_Code    => 223,
                                  The_Package => Package_Latin1);

         --# assert True;

         --character positions 224 .. 239
         Add_Predefined_Constant (Name        => "LC_A_Grave",
                                  Asc_Code    => 224,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_A_Acute",
                                  Asc_Code    => 225,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_A_Circumflex",
                                  Asc_Code    => 226,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_A_Tilde",
                                  Asc_Code    => 227,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_A_Diaeresis",
                                  Asc_Code    => 228,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_A_Ring",
                                  Asc_Code    => 229,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_AE_Diphthong",
                                  Asc_Code    => 230,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_C_Cedilla",
                                  Asc_Code    => 231,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_E_Grave",
                                  Asc_Code    => 232,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_E_Acute",
                                  Asc_Code    => 233,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_E_Circumflex",
                                  Asc_Code    => 234,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_E_Diaeresis",
                                  Asc_Code    => 235,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_I_Grave",
                                  Asc_Code    => 236,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_I_Acute",
                                  Asc_Code    => 237,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_I_Circumflex",
                                  Asc_Code    => 238,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_I_Diaeresis",
                                  Asc_Code    => 239,
                                  The_Package => Package_Latin1);

         --# assert True;

         --character positions 240 .. 255
         Add_Predefined_Constant (Name        => "LC_Icelandic_Eth",
                                  Asc_Code    => 240,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_N_Tilde",
                                  Asc_Code    => 241,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_O_Grave",
                                  Asc_Code    => 242,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_O_Acute",
                                  Asc_Code    => 243,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_O_Circumflex",
                                  Asc_Code    => 244,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_O_Tilde",
                                  Asc_Code    => 245,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_O_Diaeresis",
                                  Asc_Code    => 246,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Division_Sign",
                                  Asc_Code    => 247,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_O_Oblique_Stroke",
                                  Asc_Code    => 248,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_U_Grave",
                                  Asc_Code    => 249,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_U_Acute",
                                  Asc_Code    => 250,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_U_Circumflex",
                                  Asc_Code    => 251,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_U_Diaeresis",
                                  Asc_Code    => 252,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_Y_Acute",
                                  Asc_Code    => 253,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_Icelandic_Thorn",
                                  Asc_Code    => 254,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_Y_Diaeresis",
                                  Asc_Code    => 255,
                                  The_Package => Package_Latin1);
      end Add_Upper_Half_Ada95_Characters;

   begin -- Add_Package_Ada
      Insert_Lex_String (Name  => "Ada",
                         Token => Token);
      Add_New_Package
        (Name        => Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => GlobalScope,
         The_Parent  => RawDict.Null_Package_Info_Ref,
         Child_Sort  => Public,
         The_Package => Package_Ada);
      Dict.Packages.Package_Ada := Package_Ada;
      Insert_Lex_String (Name  => "Characters",
                         Token => Token);
      Add_New_Package
        (Name        => Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => GlobalScope,
         The_Parent  => Package_Ada,
         Child_Sort  => Public,
         The_Package => Package_Characters);
      Dict.Packages.Package_Ada_Characters := Package_Characters;
      Insert_Lex_String (Name  => "Latin_1",
                         Token => Token);
      Add_New_Package
        (Name        => Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => GlobalScope,
         The_Parent  => Package_Characters,
         Child_Sort  => Public,
         The_Package => Package_Latin1);
      Dict.Packages.Package_Ada_Characters_Latin1 := Package_Latin1;
      --control characters

      Add_Common_Characters (To_The_Package => Package_Latin1);

      Add_Lower_Half_Control_And_Graphic_Characters (Package_Latin1 => Package_Latin1);

      Add_Upper_Half_Ada95_Characters (Package_Latin1 => Package_Latin1);
   end Add_Package_Ada;

   --------------------------------------------------------------------------------

   procedure Add_Package_Real_Time
   --# global in     CommandLineData.Content;
   --#        in out Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    CommandLineData.Content,
   --#                                    Dict,
   --#                                    LexTokenManager.State &
   --#         LexTokenManager.State from *,
   --#                                    CommandLineData.Content;
   is
      Token, SC_Token, TS_Token                : LexTokenManager.Lex_String;
      T_Param, SC_Param, TS_Param              : Symbol;
      Time_Type, Time_Span_Type, Seconds_Count : RawDict.Type_Info_Ref;
      Clock_Time                               : RawDict.Variable_Info_Ref;
      The_Subprogram                           : RawDict.Subprogram_Info_Ref;
      Package_Real_Time                        : RawDict.Package_Info_Ref;
      The_Constant                             : RawDict.Constant_Info_Ref;
      The_Global_Variable                      : RawDict.Global_Variable_Info_Ref;
      Real_Time_Scope                          : Scopes;
      Arg                                      : Iterator;

      -- Adds a function with pattern
      --   function XXX (YYY : in Integer) return Time_Span
      -- where XXX and YYY are the two formal parameters given
      procedure Add_Time_Span_Constructor_Function
        (Function_Name  : in String;
         Parameter_Name : in String)
      --# global in     Real_Time_Scope;
      --#        in     Time_Span_Type;
      --#        in out LexTokenManager.State;
      --#        in out Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dict from *,
      --#                   Function_Name,
      --#                   Parameter_Name,
      --#                   Real_Time_Scope,
      --#                   Time_Span_Type,
      --#                   LexTokenManager.State &
      --#         LexTokenManager.State from *,
      --#                   Function_Name,
      --#                   Parameter_Name &
      --#         SPARK_IO.File_Sys from *,
      --#                   Dict,
      --#                   Function_Name,
      --#                   Parameter_Name,
      --#                   Real_Time_Scope,
      --#                   Time_Span_Type,
      --#                   LexTokenManager.State;
      is
         The_Function        : RawDict.Subprogram_Info_Ref;
         The_Parameter_Token : LexTokenManager.Lex_String;
         The_Function_Token  : LexTokenManager.Lex_String;
      begin
         Insert_Lex_String (Name  => Function_Name,
                            Token => The_Function_Token);
         Add_Subprogram
           (Name           => The_Function_Token,
            Comp_Unit      => ContextManager.NullUnit,
            Specification  => Null_Location,
            Scope          => Real_Time_Scope,
            Context        => ProgramContext,
            The_Subprogram => The_Function);
         Insert_Lex_String (Name  => Parameter_Name,
                            Token => The_Parameter_Token);
         Add_Subprogram_Parameter
           (Name           => The_Parameter_Token,
            The_Subprogram => The_Function,
            Type_Mark      => Get_Predefined_Integer_Type,
            Type_Reference => Null_Location,
            Mode           => InMode,
            Comp_Unit      => ContextManager.NullUnit,
            Specification  => Null_Location);
         Add_Return_Type
           (The_Function   => The_Function,
            Type_Mark      => Time_Span_Type,
            Comp_Unit      => ContextManager.NullUnit,
            Type_Reference => Null_Location);
      end Add_Time_Span_Constructor_Function;

   begin -- Add_Package_Real_Time

      -- Create the package as a child of package Ada
      Add_New_Package
        (Name        => LexTokenManager.Real_Time_Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => GlobalScope,
         The_Parent  => Get_Predefined_Package_Ada,
         Child_Sort  => Public,
         The_Package => Package_Real_Time);
      Dict.Packages.Package_Real_Time := Package_Real_Time;
      Real_Time_Scope                 :=
        Set_Visibility (The_Visibility => Visible,
                        The_Unit       => RawDict.Get_Package_Symbol (Package_Real_Time));

      -- Add type definitions for private types Time and Time_Span
      Insert_Lex_String (Name  => "Time",
                         Token => Token);
      Add_Private_Type_Local
        (Name           => Token,
         Comp_Unit      => ContextManager.NullUnit,
         Declaration    => Null_Location,
         The_Package    => Package_Real_Time,
         Is_Limited     => False,
         Is_Tagged_Type => False,
         Extends        => RawDict.Null_Type_Info_Ref,
         Type_Mark      => Time_Type);
      Dict.Types.The_Predefined_Types (Predefined_Time_Type) := Time_Type;
      Insert_Lex_String (Name  => "Time_Span",
                         Token => Token);
      Add_Private_Type_Local
        (Name           => Token,
         Comp_Unit      => ContextManager.NullUnit,
         Declaration    => Null_Location,
         The_Package    => Package_Real_Time,
         Is_Limited     => False,
         Is_Tagged_Type => False,
         Extends        => RawDict.Null_Type_Info_Ref,
         Type_Mark      => Time_Span_Type);
      Dict.Types.The_Predefined_Types (Predefined_Time_Span_Type) := Time_Span_Type;

      -- Create annotation "own protected in ClockTime : Time" for external time returned by Clock.
      --
      -- Note that the type-announcement is needed here so that the signature
      -- of the proof function associated with function Clock is well-formed
      -- in FDL.
      Insert_Lex_String (Name  => "ClockTime",
                         Token => Token);
      Add_Own_Variable_Local
        (Name                => Token,
         The_Package         => Package_Real_Time,
         Mode                => InMode,
         Is_Protected        => True,
         Is_Interrupt_Stream => False,
         Comp_Unit           => ContextManager.NullUnit,
         Declaration         => Null_Location,
         The_Variable        => Clock_Time);

      Add_Own_Variable_Type (Own_Variable   => Clock_Time,
                             Type_Mark      => Time_Type,
                             Type_Reference => Null_Location);

      -- Add the seven deferred constants. Note that Time_Unit is not supported.
      Insert_Lex_String (Name  => "Time_First",
                         Token => Token);
      --# accept Flow, 10, The_Constant, "Expected ineffective assignment to OK";
      Add_Deferred_Constant_Local
        (Name           => Token,
         Type_Mark      => Time_Type,
         Type_Reference => Null_Location,
         Comp_Unit      => ContextManager.NullUnit,
         Declaration    => Null_Location,
         The_Package    => Package_Real_Time,
         The_Constant   => The_Constant);
      --# end accept;
      Insert_Lex_String (Name  => "Time_Last",
                         Token => Token);
      --# accept Flow, 10, The_Constant, "Expected ineffective assignment to OK";
      Add_Deferred_Constant_Local
        (Name           => Token,
         Type_Mark      => Time_Type,
         Type_Reference => Null_Location,
         Comp_Unit      => ContextManager.NullUnit,
         Declaration    => Null_Location,
         The_Package    => Package_Real_Time,
         The_Constant   => The_Constant);
      --# end accept;
      Insert_Lex_String (Name  => "Time_Span_First",
                         Token => Token);
      --# accept Flow, 10, The_Constant, "Expected ineffective assignment to OK";
      Add_Deferred_Constant_Local
        (Name           => Token,
         Type_Mark      => Time_Span_Type,
         Type_Reference => Null_Location,
         Comp_Unit      => ContextManager.NullUnit,
         Declaration    => Null_Location,
         The_Package    => Package_Real_Time,
         The_Constant   => The_Constant);
      --# end accept;
      Insert_Lex_String (Name  => "Time_Span_Last",
                         Token => Token);
      --# accept Flow, 10, The_Constant, "Expected ineffective assignment to OK";
      Add_Deferred_Constant_Local
        (Name           => Token,
         Type_Mark      => Time_Span_Type,
         Type_Reference => Null_Location,
         Comp_Unit      => ContextManager.NullUnit,
         Declaration    => Null_Location,
         The_Package    => Package_Real_Time,
         The_Constant   => The_Constant);
      --# end accept;
      Insert_Lex_String (Name  => "Time_Span_Zero",
                         Token => Token);
      --# accept Flow, 10, The_Constant, "Expected ineffective assignment to OK";
      Add_Deferred_Constant_Local
        (Name           => Token,
         Type_Mark      => Time_Span_Type,
         Type_Reference => Null_Location,
         Comp_Unit      => ContextManager.NullUnit,
         Declaration    => Null_Location,
         The_Package    => Package_Real_Time,
         The_Constant   => The_Constant);
      --# end accept;
      Insert_Lex_String (Name  => "Time_Span_Unit",
                         Token => Token);
      --# accept Flow, 10, The_Constant, "Expected ineffective assignment to OK";
      Add_Deferred_Constant_Local
        (Name           => Token,
         Type_Mark      => Time_Span_Type,
         Type_Reference => Null_Location,
         Comp_Unit      => ContextManager.NullUnit,
         Declaration    => Null_Location,
         The_Package    => Package_Real_Time,
         The_Constant   => The_Constant);
      --# end accept;
      Insert_Lex_String (Name  => "Tick",
                         Token => Token);
      --# accept Flow, 10, The_Constant, "Expected ineffective assignment to OK";
      Add_Deferred_Constant_Local
        (Name           => Token,
         Type_Mark      => Time_Span_Type,
         Type_Reference => Null_Location,
         Comp_Unit      => ContextManager.NullUnit,
         Declaration    => Null_Location,
         The_Package    => Package_Real_Time,
         The_Constant   => The_Constant);
      --# end accept;
      -- Add function Clock return Time.
      Insert_Lex_String (Name  => "Clock",
                         Token => Token);
      Add_Subprogram
        (Name           => Token,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location,
         Scope          => Real_Time_Scope,
         Context        => ProgramContext,
         The_Subprogram => The_Subprogram);
      Add_Return_Type
        (The_Function   => The_Subprogram,
         Type_Mark      => Time_Type,
         Comp_Unit      => ContextManager.NullUnit,
         Type_Reference => Null_Location);
      Dict.Subprograms.Ada_Real_Time_Clock := The_Subprogram;
      -- Clock requires annotation "global in ClockTime"
      --# accept Flow, 10, The_Global_Variable, "Expected ineffective assignment to OK";
      Add_Subprogram_Global_Variable
        (Abstraction         => IsAbstract,
         The_Subprogram      => The_Subprogram,
         The_Variable        => Clock_Time,
         Mode                => InMode,
         Prefix_Needed       => False,
         Comp_Unit           => ContextManager.NullUnit,
         Variable_Reference  => Null_Location,
         The_Global_Variable => The_Global_Variable);
      --# end accept;
      -- The arithmetic and relational operators are not in the dictionary.
      -- Functions To_Duration and To_Time_Span are not supported.
      --
      -- Add functions Nanoseconds / Microseconds / Milliseconds. These take a
      -- parameter of type Integer and return Time_Span.
      Add_Time_Span_Constructor_Function (Function_Name  => "Nanoseconds",
                                          Parameter_Name => "NS");
      Add_Time_Span_Constructor_Function (Function_Name  => "Microseconds",
                                          Parameter_Name => "US");
      Add_Time_Span_Constructor_Function (Function_Name  => "Milliseconds",
                                          Parameter_Name => "MS");

      -- Ada2005 additionally defines functions "Seconds" and "Minutes", so
      if CommandLineData.Content.Language_Profile in CommandLineData.SPARK2005_Profiles then
         Add_Time_Span_Constructor_Function (Function_Name  => "Seconds",
                                             Parameter_Name => "S");
         Add_Time_Span_Constructor_Function (Function_Name  => "Minutes",
                                             Parameter_Name => "M");
      end if;

      -- Add type Seconds_Count. This is an integer type with implementation-
      -- defined range. The range is left blank at this stage, and may be
      -- overridden by values supplied in the configuration file.
      Add_Integer_Type_Local
        (Name        => LexTokenManager.Seconds_Count_Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Lower       => LexTokenManager.Null_String,
         Upper       => LexTokenManager.Null_String,
         Scope       => Real_Time_Scope,
         Context     => ProgramContext,
         Type_Mark   => Seconds_Count);
      Insert_Lex_String (Name  => "Split",
                         Token => Token);
      Add_Subprogram
        (Name           => Token,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location,
         Scope          => Real_Time_Scope,
         Context        => ProgramContext,
         The_Subprogram => The_Subprogram);
      Insert_Lex_String (Name  => "T",
                         Token => Token);
      Add_Subprogram_Parameter
        (Name           => Token,
         The_Subprogram => The_Subprogram,
         Type_Mark      => Time_Type,
         Type_Reference => Null_Location,
         Mode           => InMode,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location);
      Insert_Lex_String (Name  => "SC",
                         Token => SC_Token); -- Remember the token for further use of "SC"
      Add_Subprogram_Parameter
        (Name           => SC_Token,
         The_Subprogram => The_Subprogram,
         Type_Mark      => Seconds_Count,
         Type_Reference => Null_Location,
         Mode           => OutMode,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location);
      Insert_Lex_String (Name  => "TS",
                         Token => TS_Token); -- Remember the token for further use of "TS"
      Add_Subprogram_Parameter
        (Name           => TS_Token,
         The_Subprogram => The_Subprogram,
         Type_Mark      => Time_Span_Type,
         Type_Reference => Null_Location,
         Mode           => OutMode,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location);

      -- Split requires annotation "derives SC, TS from T", so we need the
      -- symbols for the three parameters.
      Arg      := First_Ada_Subprogram_Parameter (The_Subprogram => The_Subprogram);
      T_Param  := CurrentSymbol (Arg);
      Arg      := NextSymbol (Arg);
      SC_Param := CurrentSymbol (Arg);
      Arg      := NextSymbol (Arg);
      TS_Param := CurrentSymbol (Arg);

      Add_Subprogram_Export_Parameter
        (The_Subprogram   => The_Subprogram,
         Abstraction      => IsAbstract,
         The_Export       => RawDict.Get_Subprogram_Parameter_Info_Ref (SC_Param),
         Export_Reference => Null_Location,
         Annotation       => Null_Location);
      Add_Subprogram_Export_Parameter
        (The_Subprogram   => The_Subprogram,
         Abstraction      => IsAbstract,
         The_Export       => RawDict.Get_Subprogram_Parameter_Info_Ref (TS_Param),
         Export_Reference => Null_Location,
         Annotation       => Null_Location);
      Add_Subprogram_Dependency
        (Abstraction      => IsAbstract,
         Comp_Unit        => ContextManager.NullUnit,
         The_Subprogram   => The_Subprogram,
         The_Export       => SC_Param,
         The_Import       => T_Param,
         Import_Reference => Null_Location);
      Add_Subprogram_Dependency
        (Abstraction      => IsAbstract,
         Comp_Unit        => ContextManager.NullUnit,
         The_Subprogram   => The_Subprogram,
         The_Export       => TS_Param,
         The_Import       => T_Param,
         Import_Reference => Null_Location);
      Insert_Lex_String (Name  => "Time_Of",
                         Token => Token);
      -- Mark the subprogram as having a derives annotation so that calls to
      -- it from other subprograms with derives annotations are legal with flow=auto
      RawDict.Set_Subprogram_Has_Derives_Annotation (The_Subprogram => The_Subprogram);

      Add_Subprogram
        (Name           => Token,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location,
         Scope          => Real_Time_Scope,
         Context        => ProgramContext,
         The_Subprogram => The_Subprogram);
      Add_Subprogram_Parameter
        (Name           => SC_Token,
         The_Subprogram => The_Subprogram,
         Type_Mark      => Seconds_Count,
         Type_Reference => Null_Location,
         Mode           => InMode,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location);
      Add_Subprogram_Parameter
        (Name           => TS_Token,
         The_Subprogram => The_Subprogram,
         Type_Mark      => Time_Span_Type,
         Type_Reference => Null_Location,
         Mode           => InMode,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location);
      Add_Return_Type
        (The_Function   => The_Subprogram,
         Type_Mark      => Time_Type,
         Comp_Unit      => ContextManager.NullUnit,
         Type_Reference => Null_Location);
      --# accept Flow, 33, The_Constant, "Expected to be neither referenced nor exported" &
      --#        Flow, 33, The_Global_Variable, "Expected to be neither referenced nor exported";
   end Add_Package_Real_Time;

   --------------------------------------------------------------------------------

   procedure Add_Package_Synchronous_Task_Control
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Dict,
   --#                                    LexTokenManager.State &
   --#         LexTokenManager.State from *;
   is
      Token, S_Token                   : LexTokenManager.Lex_String;
      S_Param                          : Symbol;
      The_Subprogram                   : RawDict.Subprogram_Info_Ref;
      Package_Synchronous_Task_Control : RawDict.Package_Info_Ref;
      Suspension_Object_Type           : RawDict.Type_Info_Ref;
      Synchronous_Task_Control_Scope   : Scopes;
      Arg                              : Iterator;
   begin
      -- Create the package as a child of package Ada
      Add_New_Package
        (Name        => LexTokenManager.Synchronous_Task_Control_Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => GlobalScope,
         The_Parent  => Get_Predefined_Package_Ada,
         Child_Sort  => Public,
         The_Package => Package_Synchronous_Task_Control);
      Dict.Packages.Package_Synchronous_Task_Control := Package_Synchronous_Task_Control;
      Synchronous_Task_Control_Scope                 :=
        Set_Visibility (The_Visibility => Visible,
                        The_Unit       => RawDict.Get_Package_Symbol (Package_Synchronous_Task_Control));

      -- Add type definitions for limited private type Suspension_Object
      Insert_Lex_String (Name  => "Suspension_Object",
                         Token => Token);
      Add_Private_Type_Local
        (Name           => Token,
         Comp_Unit      => ContextManager.NullUnit,
         Declaration    => Null_Location,
         The_Package    => Package_Synchronous_Task_Control,
         Is_Limited     => True,
         Is_Tagged_Type => False,
         Extends        => RawDict.Null_Type_Info_Ref,
         Type_Mark      => Suspension_Object_Type);
      Dict.Types.The_Predefined_Types (Predefined_Suspension_Object_Type) := Suspension_Object_Type;

      -- Mark the type as Atomic
      RawDict.Set_Type_Atomic (Type_Mark => Suspension_Object_Type);

      Insert_Lex_String (Name  => "Set_True",
                         Token => Token);
      Add_Subprogram
        (Name           => Token,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location,
         Scope          => Synchronous_Task_Control_Scope,
         Context        => ProgramContext,
         The_Subprogram => The_Subprogram);
      Insert_Lex_String (Name  => "S",
                         Token => S_Token); -- used in all subroutine definitions for this package.
      Add_Subprogram_Parameter
        (Name           => S_Token,
         The_Subprogram => The_Subprogram,
         Type_Mark      => Suspension_Object_Type,
         Type_Reference => Null_Location,
         Mode           => InOutMode,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location);

      -- The annotation
      Arg     := First_Ada_Subprogram_Parameter (The_Subprogram => The_Subprogram);
      S_Param := CurrentSymbol (Arg);

      Add_Subprogram_Export_Parameter
        (The_Subprogram   => The_Subprogram,
         Abstraction      => IsAbstract,
         The_Export       => RawDict.Get_Subprogram_Parameter_Info_Ref (S_Param),
         Export_Reference => Null_Location,
         Annotation       => Null_Location);

      -- Mark the subprogram as having a derives annotation so that calls to
      -- it from other subprograms with derives annotations are legal with flow=auto
      RawDict.Set_Subprogram_Has_Derives_Annotation (The_Subprogram => The_Subprogram);

      Insert_Lex_String (Name  => "Set_False",
                         Token => Token);
      Add_Subprogram
        (Name           => Token,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location,
         Scope          => Synchronous_Task_Control_Scope,
         Context        => ProgramContext,
         The_Subprogram => The_Subprogram);
      Add_Subprogram_Parameter
        (Name           => S_Token,
         The_Subprogram => The_Subprogram,
         Type_Mark      => Suspension_Object_Type,
         Type_Reference => Null_Location,
         Mode           => InOutMode,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location);
      -- The annotation
      Arg     := First_Ada_Subprogram_Parameter (The_Subprogram => The_Subprogram);
      S_Param := CurrentSymbol (Arg);

      Add_Subprogram_Export_Parameter
        (The_Subprogram   => The_Subprogram,
         Abstraction      => IsAbstract,
         The_Export       => RawDict.Get_Subprogram_Parameter_Info_Ref (S_Param),
         Export_Reference => Null_Location,
         Annotation       => Null_Location);

      -- Mark the subprogram as having a derives annotation so that calls to
      -- it from other subprograms with derives annotations are legal with flow=auto
      RawDict.Set_Subprogram_Has_Derives_Annotation (The_Subprogram => The_Subprogram);

      Insert_Lex_String (Name  => "Suspend_Until_True",
                         Token => Token);
      Add_Subprogram
        (Name           => Token,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location,
         Scope          => Synchronous_Task_Control_Scope,
         Context        => ProgramContext,
         The_Subprogram => The_Subprogram);

      -- Record the symbol for later retrieval by wellformedness checker.
      Dict.Subprograms.STC_Suspend_Until_True := The_Subprogram;

      Add_Subprogram_Parameter
        (Name           => S_Token,
         The_Subprogram => The_Subprogram,
         Type_Mark      => Suspension_Object_Type,
         Type_Reference => Null_Location,
         Mode           => InOutMode,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location);
      -- The annotations
      Arg     := First_Ada_Subprogram_Parameter (The_Subprogram => The_Subprogram);
      S_Param := CurrentSymbol (Arg);

      Add_Subprogram_Export_Parameter
        (The_Subprogram   => The_Subprogram,
         Abstraction      => IsAbstract,
         The_Export       => RawDict.Get_Subprogram_Parameter_Info_Ref (S_Param),
         Export_Reference => Null_Location,
         Annotation       => Null_Location);

      -- Mark the subprogram as having a derives annotation so that calls to
      -- it from other subprograms with derives annotations are legal with flow=auto
      RawDict.Set_Subprogram_Has_Derives_Annotation (The_Subprogram => The_Subprogram);
   end Add_Package_Synchronous_Task_Control;

   --------------------------------------------------------------------------------

   procedure Add_Package_Interrupts
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.State;
   is
      Type_Mark          : RawDict.Type_Info_Ref;
      Package_Interrupts : RawDict.Package_Info_Ref;
      Interrupts_Scope   : Scopes;
      pragma Unreferenced (Type_Mark);
   begin
      -- Create the package as a child of package Ada
      Add_New_Package
        (Name        => LexTokenManager.Interrupts_Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => GlobalScope,
         The_Parent  => Get_Predefined_Package_Ada,
         Child_Sort  => Public,
         The_Package => Package_Interrupts);
      Dict.Packages.Package_Interrupts := Package_Interrupts;
      Interrupts_Scope                 :=
        Set_Visibility (The_Visibility => Visible,
                        The_Unit       => RawDict.Get_Package_Symbol (Package_Interrupts));

      -- Add type Interrupt_ID. This is an integer type with implementation-
      -- defined range. The range is left blank at this stage, and may be
      -- overridden by values supplied in the configuration file.

      --# accept Flow, 10, Type_Mark, "Expected ineffective assignment to Type_Mark";
      Add_Integer_Type_Local
        (Name        => LexTokenManager.Interrupt_ID_Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Lower       => LexTokenManager.Null_String,
         Upper       => LexTokenManager.Null_String,
         Scope       => Interrupts_Scope,
         Context     => ProgramContext,
         Type_Mark   => Type_Mark);
      --# end accept;
      --# accept Flow, 33, Type_Mark, "Expected to be neither referenced nor exported";
   end Add_Package_Interrupts;

   --------------------------------------------------------------------------------

   -- This procedure creates a subprogram record with which we can associate globals
   -- and dependencies from the partition annotation
   procedure Add_The_Partition
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Dict,
   --#                                    LexTokenManager.State &
   --#         LexTokenManager.State from *;

   is
      Token          : LexTokenManager.Lex_String;
      The_Subprogram : RawDict.Subprogram_Info_Ref;
   begin
      Insert_Lex_String (Name  => "main_program",
                         Token => Token); -- suitable reserved name for the subprog
      Add_Subprogram
        (Name           => Token,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location,
         Scope          => GlobalScope,
         Context        => ProgramContext,
         The_Subprogram => The_Subprogram);
      Dict.The_Partition := The_Subprogram;
   end Add_The_Partition;

   --------------------------------------------------------------------------------

   procedure Add_Unchecked_Conversion
   --# global in     CommandLineData.Content;
   --#        in out Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    CommandLineData.Content,
   --#                                    Dict,
   --#                                    LexTokenManager.State &
   --#         LexTokenManager.State from *;
   is
      Subprogram_Token, Source_Token, Target_Token, S_Token : LexTokenManager.Lex_String;
      The_Generic_Unit                                      : RawDict.Generic_Unit_Info_Ref;
      The_Subprogram                                        : RawDict.Subprogram_Info_Ref;
      The_Generic_Parameter                                 : RawDict.Generic_Parameter_Info_Ref;
      The_Declaration                                       : RawDict.Declaration_Info_Ref;
      Type_Mark                                             : RawDict.Type_Info_Ref;
   begin
      -- add the function itself
      Insert_Lex_String (Name  => "Unchecked_Conversion",
                         Token => Subprogram_Token);
      Add_Subprogram
        (Name           => Subprogram_Token,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location,
         Scope          => GlobalScope,
         Context        => ProgramContext,
         The_Subprogram => The_Subprogram);
      -- mark predefined generic unit as having a body
      Add_Subprogram_Body
        (The_Subprogram => The_Subprogram,
         Comp_Unit      => ContextManager.NullUnit,
         The_Body       => Null_Location,
         Hidden         => True);

      -- also add a declaration so that Ada.Unchecked_Conversion can be looked up
      case CommandLineData.Content.Language_Profile is
         when CommandLineData.SPARK83 =>
            null;
         when CommandLineData.SPARK95_Onwards =>
            Add_Declaration
              (Comp_Unit       => ContextManager.NullUnit,
               Loc             => Null_Location,
               Scope           => Set_Visibility (The_Visibility => Visible,
                                                  The_Unit       => RawDict.Get_Package_Symbol (Get_Predefined_Package_Ada)),
               Context         => ProgramContext,
               The_Declaration => The_Declaration);
            RawDict.Set_Declaration_Item
              (The_Declaration => The_Declaration,
               Item            => RawDict.Get_Subprogram_Symbol (The_Subprogram));
      end case;

      RawDict.Create_Generic_Unit
        (Kind             => Generic_Of_Subprogram,
         Scope            => GlobalScope,
         Comp_Unit        => ContextManager.NullUnit,
         Loc              => LexTokenManager.Null_Token_Position,
         The_Generic_Unit => The_Generic_Unit);

      -- make it generic
      RawDict.Set_Subprogram_Generic_Unit (The_Subprogram   => The_Subprogram,
                                           The_Generic_Unit => The_Generic_Unit);
      RawDict.Set_Generic_Unit_Owning_Subprogram (The_Generic_Unit => The_Generic_Unit,
                                                  The_Subprogram   => The_Subprogram);

      -- create generic parameters
      Insert_Lex_String (Name  => "Source",
                         Token => Source_Token);
      Add_Generic_Type_Local
        (Name        => Source_Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => Set_Visibility (The_Visibility => Visible,
                                        The_Unit       => RawDict.Get_Generic_Unit_Symbol (The_Generic_Unit)),
         Type_Mark   => Type_Mark);
      Set_Generic_Private_Type_Local (Type_Mark  => Type_Mark,
                                      Is_Limited => False);
      Add_Generic_Formal_Parameter_Local
        (Comp_Unit             => ContextManager.NullUnit,
         Declaration           => Null_Location,
         The_Generic_Unit      => The_Generic_Unit,
         Type_Mark             => Type_Mark,
         The_Object            => RawDict.Null_Constant_Info_Ref,
         The_Generic_Parameter => The_Generic_Parameter);

      -- add subprogram parameter
      Insert_Lex_String (Name  => "S",
                         Token => S_Token);
      Add_Subprogram_Parameter
        (Name           => S_Token,
         The_Subprogram => The_Subprogram,
         Type_Mark      => RawDict.Get_Generic_Parameter_Type (The_Generic_Parameter => The_Generic_Parameter),
         Type_Reference => Null_Location,
         Mode           => InMode,
         Comp_Unit      => ContextManager.NullUnit,
         Specification  => Null_Location);

      -- create generic parameters
      Insert_Lex_String (Name  => "Target",
                         Token => Target_Token);
      Add_Generic_Type_Local
        (Name        => Target_Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => Set_Visibility (The_Visibility => Visible,
                                        The_Unit       => RawDict.Get_Generic_Unit_Symbol (The_Generic_Unit)),
         Type_Mark   => Type_Mark);
      Set_Generic_Private_Type_Local (Type_Mark  => Type_Mark,
                                      Is_Limited => False);
      Add_Generic_Formal_Parameter_Local
        (Comp_Unit             => ContextManager.NullUnit,
         Declaration           => Null_Location,
         The_Generic_Unit      => The_Generic_Unit,
         Type_Mark             => Type_Mark,
         The_Object            => RawDict.Null_Constant_Info_Ref,
         The_Generic_Parameter => The_Generic_Parameter);

      -- add return type
      Add_Return_Type
        (The_Function   => The_Subprogram,
         Type_Mark      => RawDict.Get_Generic_Parameter_Type (The_Generic_Parameter => The_Generic_Parameter),
         Comp_Unit      => ContextManager.NullUnit,
         Type_Reference => Null_Location);

      Dict.Subprograms.Unchecked_Conversion := The_Subprogram;
   end Add_Unchecked_Conversion;

begin -- Initialize
   Assign_Initial_Value;
   Add_Unknown_Type;
   Add_Package_Standard;
   Add_Package_ASCII;

   case CommandLineData.Content.Language_Profile is
      when CommandLineData.SPARK83 =>
         Dict.Packages.Package_Ada := RawDict.Null_Package_Info_Ref;
      when CommandLineData.SPARK95_Onwards =>
         Add_Package_Ada;
   end case;

   Add_Unchecked_Conversion;
   if CommandLineData.Ravenscar_Selected then
      Add_Package_Real_Time;
      Add_Package_Synchronous_Task_Control;
      Add_Package_Interrupts;
      Add_The_Partition;
   else
      Dict.Packages.Package_Real_Time                := RawDict.Null_Package_Info_Ref;
      Dict.Packages.Package_Synchronous_Task_Control := RawDict.Null_Package_Info_Ref;
      Dict.Packages.Package_Interrupts               := RawDict.Null_Package_Info_Ref;
      Dict.The_Partition                             := RawDict.Null_Subprogram_Info_Ref;
   end if;
   if Write_To_File then
      SPARK_IO.Create (File, 0, "", "", Status); -- Expected flow err
      if Status = SPARK_IO.Ok then
         Dict.TemporaryFile := File;
      else
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Init,
                                   Msg     => "in Dictionary.Initialize");
         Dict.TemporaryFile := SPARK_IO.Null_File;
      end if;
   else
      Dict.TemporaryFile := SPARK_IO.Null_File;
   end if;
   LexTokenManager.Set_Last_Token;
end Initialize;
