// SPDX-License-Identifier: GPL-2.0+

#include <kunit/test.h>

#include <drm/drm_fixed.h>
#include <drm/drm_fourcc.h>

#include "../../drm_crtc_internal.h"

#include "../vkms_formats.h"

#define TEST_BUFF_SIZE 50

MODULE_IMPORT_NS("EXPORTED_FOR_KUNIT_TESTING");

/**
 * struct pixel_yuv_u8 - Internal representation of a pixel color.
 * @y: Luma value, stored in 8 bits, without padding, using
 *     machine endianness
 * @u: Blue difference chroma value, stored in 8 bits, without padding, using
 *     machine endianness
 * @v: Red difference chroma value, stored in 8 bits, without padding, using
 *     machine endianness
 */
struct pixel_yuv_u8 {
	u8 y, u, v;
};

/*
 * struct yuv_u8_to_argb_u16_case - Reference values to test the color
 * conversions in VKMS between YUV to ARGB
 *
 * @encoding: Encoding used to convert RGB to YUV
 * @range: Range used to convert RGB to YUV
 * @n_colors: Count of test colors in this case
 * @format_pair.name: Name used for this color conversion, used to
 *                    clarify the test results
 * @format_pair.rgb: RGB color tested
 * @format_pair.yuv: Same color as @format_pair.rgb, but converted to
 *                   YUV using @encoding and @range.
 */
struct yuv_u8_to_argb_u16_case {
	enum drm_color_encoding encoding;
	enum drm_color_range range;
	size_t n_colors;
	struct format_pair {
		char *name;
		struct pixel_yuv_u8 yuv;
		struct pixel_argb_u16 argb;
	} colors[TEST_BUFF_SIZE];
};

/*
 * The YUV color representation were acquired via the colour python framework.
 * Below are the function calls used for generating each case.
 *
 * For more information got to the docs:
 * https://colour.readthedocs.io/en/master/generated/colour.RGB_to_YCbCr.html
 */
static struct yuv_u8_to_argb_u16_case yuv_u8_to_argb_u16_cases[] = {
	/*
	 * colour.RGB_to_YCbCr(<rgb color in 16 bit form>,
	 *                     K=colour.WEIGHTS_YCBCR["ITU-R BT.601"],
	 *                     in_bits = 16,
	 *                     in_legal = False,
	 *                     in_int = True,
	 *                     out_bits = 8,
	 *                     out_legal = False,
	 *                     out_int = True)
	 *
	 * Tests cases for color conversion generated by converting RGB
	 * values to YUV BT601 full range using the ITU-R BT.601 weights.
	 */
	{
		.encoding = DRM_COLOR_YCBCR_BT601,
		.range = DRM_COLOR_YCBCR_FULL_RANGE,
		.n_colors = 6,
		.colors = {
			{ "white", { 0xff, 0x80, 0x80 }, { 0xffff, 0xffff, 0xffff, 0xffff }},
			{ "gray",  { 0x80, 0x80, 0x80 }, { 0xffff, 0x8080, 0x8080, 0x8080 }},
			{ "black", { 0x00, 0x80, 0x80 }, { 0xffff, 0x0000, 0x0000, 0x0000 }},
			{ "red",   { 0x4c, 0x55, 0xff }, { 0xffff, 0xffff, 0x0000, 0x0000 }},
			{ "green", { 0x96, 0x2c, 0x15 }, { 0xffff, 0x0000, 0xffff, 0x0000 }},
			{ "blue",  { 0x1d, 0xff, 0x6b }, { 0xffff, 0x0000, 0x0000, 0xffff }},
		},
	},
	/*
	 * colour.RGB_to_YCbCr(<rgb color in 16 bit form>,
	 *                     K=colour.WEIGHTS_YCBCR["ITU-R BT.601"],
	 *                     in_bits = 16,
	 *                     in_legal = False,
	 *                     in_int = True,
	 *                     out_bits = 8,
	 *                     out_legal = True,
	 *                     out_int = True)
	 * Tests cases for color conversion generated by converting RGB
	 * values to YUV BT601 limited range using the ITU-R BT.601 weights.
	 */
	{
		.encoding = DRM_COLOR_YCBCR_BT601,
		.range = DRM_COLOR_YCBCR_LIMITED_RANGE,
		.n_colors = 6,
		.colors = {
			{ "white", { 0xeb, 0x80, 0x80 }, { 0xffff, 0xffff, 0xffff, 0xffff }},
			{ "gray",  { 0x7e, 0x80, 0x80 }, { 0xffff, 0x8080, 0x8080, 0x8080 }},
			{ "black", { 0x10, 0x80, 0x80 }, { 0xffff, 0x0000, 0x0000, 0x0000 }},
			{ "red",   { 0x51, 0x5a, 0xf0 }, { 0xffff, 0xffff, 0x0000, 0x0000 }},
			{ "green", { 0x91, 0x36, 0x22 }, { 0xffff, 0x0000, 0xffff, 0x0000 }},
			{ "blue",  { 0x29, 0xf0, 0x6e }, { 0xffff, 0x0000, 0x0000, 0xffff }},
		},
	},
	/*
	 * colour.RGB_to_YCbCr(<rgb color in 16 bit form>,
	 *                     K=colour.WEIGHTS_YCBCR["ITU-R BT.709"],
	 *                     in_bits = 16,
	 *                     in_legal = False,
	 *                     in_int = True,
	 *                     out_bits = 8,
	 *                     out_legal = False,
	 *                     out_int = True)
	 * Tests cases for color conversion generated by converting RGB
	 * values to YUV BT709 full range using the ITU-R BT.709 weights.
	 */
	{
		.encoding = DRM_COLOR_YCBCR_BT709,
		.range = DRM_COLOR_YCBCR_FULL_RANGE,
		.n_colors = 6,
		.colors = {
			{ "white", { 0xff, 0x80, 0x80 }, { 0xffff, 0xffff, 0xffff, 0xffff }},
			{ "gray",  { 0x80, 0x80, 0x80 }, { 0xffff, 0x8080, 0x8080, 0x8080 }},
			{ "black", { 0x00, 0x80, 0x80 }, { 0xffff, 0x0000, 0x0000, 0x0000 }},
			{ "red",   { 0x36, 0x63, 0xff }, { 0xffff, 0xffff, 0x0000, 0x0000 }},
			{ "green", { 0xb6, 0x1e, 0x0c }, { 0xffff, 0x0000, 0xffff, 0x0000 }},
			{ "blue",  { 0x12, 0xff, 0x74 }, { 0xffff, 0x0000, 0x0000, 0xffff }},
		},
	},
	/*
	 * colour.RGB_to_YCbCr(<rgb color in 16 bit form>,
	 *                     K=colour.WEIGHTS_YCBCR["ITU-R BT.709"],
	 *                     in_bits = 16,
	 *                     int_legal = False,
	 *                     in_int = True,
	 *                     out_bits = 8,
	 *                     out_legal = True,
	 *                     out_int = True)
	 * Tests cases for color conversion generated by converting RGB
	 * values to YUV BT709 limited range using the ITU-R BT.709 weights.
	 */
	{
		.encoding = DRM_COLOR_YCBCR_BT709,
		.range = DRM_COLOR_YCBCR_LIMITED_RANGE,
		.n_colors = 6,
		.colors = {
			{ "white", { 0xeb, 0x80, 0x80 }, { 0xffff, 0xffff, 0xffff, 0xffff }},
			{ "gray",  { 0x7e, 0x80, 0x80 }, { 0xffff, 0x8080, 0x8080, 0x8080 }},
			{ "black", { 0x10, 0x80, 0x80 }, { 0xffff, 0x0000, 0x0000, 0x0000 }},
			{ "red",   { 0x3f, 0x66, 0xf0 }, { 0xffff, 0xffff, 0x0000, 0x0000 }},
			{ "green", { 0xad, 0x2a, 0x1a }, { 0xffff, 0x0000, 0xffff, 0x0000 }},
			{ "blue",  { 0x20, 0xf0, 0x76 }, { 0xffff, 0x0000, 0x0000, 0xffff }},
		},
	},
	/*
	 * colour.RGB_to_YCbCr(<rgb color in 16 bit form>,
	 *                     K=colour.WEIGHTS_YCBCR["ITU-R BT.2020"],
	 *                     in_bits = 16,
	 *                     in_legal = False,
	 *                     in_int = True,
	 *                     out_bits = 8,
	 *                     out_legal = False,
	 *                     out_int = True)
	 * Tests cases for color conversion generated by converting RGB
	 * values to YUV BT2020 full range using the ITU-R BT.2020 weights.
	 */
	{
		.encoding = DRM_COLOR_YCBCR_BT2020,
		.range = DRM_COLOR_YCBCR_FULL_RANGE,
		.n_colors = 6,
		.colors = {
			{ "white", { 0xff, 0x80, 0x80 }, { 0xffff, 0xffff, 0xffff, 0xffff }},
			{ "gray",  { 0x80, 0x80, 0x80 }, { 0xffff, 0x8080, 0x8080, 0x8080 }},
			{ "black", { 0x00, 0x80, 0x80 }, { 0xffff, 0x0000, 0x0000, 0x0000 }},
			{ "red",   { 0x43, 0x5c, 0xff }, { 0xffff, 0xffff, 0x0000, 0x0000 }},
			{ "green", { 0xad, 0x24, 0x0b }, { 0xffff, 0x0000, 0xffff, 0x0000 }},
			{ "blue",  { 0x0f, 0xff, 0x76 }, { 0xffff, 0x0000, 0x0000, 0xffff }},
		},
	},
	/*
	 * colour.RGB_to_YCbCr(<rgb color in 16 bit form>,
	 *                     K=colour.WEIGHTS_YCBCR["ITU-R BT.2020"],
	 *                     in_bits = 16,
	 *                     in_legal = False,
	 *                     in_int = True,
	 *                     out_bits = 8,
	 *                     out_legal = True,
	 *                     out_int = True)
	 * Tests cases for color conversion generated by converting RGB
	 * values to YUV BT2020 limited range using the ITU-R BT.2020 weights.
	 */
	{
		.encoding = DRM_COLOR_YCBCR_BT2020,
		.range = DRM_COLOR_YCBCR_LIMITED_RANGE,
		.n_colors = 6,
		.colors = {
			{ "white", { 0xeb, 0x80, 0x80 }, { 0xffff, 0xffff, 0xffff, 0xffff }},
			{ "gray",  { 0x7e, 0x80, 0x80 }, { 0xffff, 0x8080, 0x8080, 0x8080 }},
			{ "black", { 0x10, 0x80, 0x80 }, { 0xffff, 0x0000, 0x0000, 0x0000 }},
			{ "red",   { 0x4a, 0x61, 0xf0 }, { 0xffff, 0xffff, 0x0000, 0x0000 }},
			{ "green", { 0xa4, 0x2f, 0x19 }, { 0xffff, 0x0000, 0xffff, 0x0000 }},
			{ "blue",  { 0x1d, 0xf0, 0x77 }, { 0xffff, 0x0000, 0x0000, 0xffff }},
		},
	},
};

/*
 * vkms_format_test_yuv_u8_to_argb_u16 - Testing the conversion between YUV
 * colors to ARGB colors in VKMS
 *
 * This test will use the functions get_conversion_matrix_to_argb_u16 and
 * argb_u16_from_yuv888 to convert YUV colors (stored in
 * yuv_u8_to_argb_u16_cases) into ARGB colors.
 *
 * The conversion between YUV and RGB is not totally reversible, so there may be
 * some difference between the expected value and the result.
 * In addition, there may be some rounding error as the input color is 8 bits
 * and output color is 16 bits.
 */
static void vkms_format_test_yuv_u8_to_argb_u16(struct kunit *test)
{
	const struct yuv_u8_to_argb_u16_case *param = test->param_value;
	struct pixel_argb_u16 argb;

	for (size_t i = 0; i < param->n_colors; i++) {
		const struct format_pair *color = &param->colors[i];
		struct conversion_matrix matrix;

		get_conversion_matrix_to_argb_u16
			(DRM_FORMAT_NV12, param->encoding, param->range, &matrix);

		argb = argb_u16_from_yuv888(color->yuv.y, color->yuv.u, color->yuv.v, &matrix);

		KUNIT_EXPECT_LE_MSG(test, abs_diff(argb.a, color->argb.a), 0x1ff,
				    "On the A channel of the color %s expected 0x%04x, got 0x%04x",
				    color->name, color->argb.a, argb.a);
		KUNIT_EXPECT_LE_MSG(test, abs_diff(argb.r, color->argb.r), 0x1ff,
				    "On the R channel of the color %s expected 0x%04x, got 0x%04x",
				    color->name, color->argb.r, argb.r);
		KUNIT_EXPECT_LE_MSG(test, abs_diff(argb.g, color->argb.g), 0x1ff,
				    "On the G channel of the color %s expected 0x%04x, got 0x%04x",
				    color->name, color->argb.g, argb.g);
		KUNIT_EXPECT_LE_MSG(test, abs_diff(argb.b, color->argb.b), 0x1ff,
				    "On the B channel of the color %s expected 0x%04x, got 0x%04x",
				    color->name, color->argb.b, argb.b);
	}
}

static void vkms_format_test_yuv_u8_to_argb_u16_case_desc(struct yuv_u8_to_argb_u16_case *t,
							  char *desc)
{
	snprintf(desc, KUNIT_PARAM_DESC_SIZE, "%s - %s",
		 drm_get_color_encoding_name(t->encoding), drm_get_color_range_name(t->range));
}

KUNIT_ARRAY_PARAM(yuv_u8_to_argb_u16, yuv_u8_to_argb_u16_cases,
		  vkms_format_test_yuv_u8_to_argb_u16_case_desc
);

static struct kunit_case vkms_format_test_cases[] = {
	KUNIT_CASE_PARAM(vkms_format_test_yuv_u8_to_argb_u16, yuv_u8_to_argb_u16_gen_params),
	{}
};

static struct kunit_suite vkms_format_test_suite = {
	.name = "vkms-format",
	.test_cases = vkms_format_test_cases,
};

kunit_test_suite(vkms_format_test_suite);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Kunit test for vkms format conversion");
