// SPDX-License-Identifier: GPL-2.0-only
/*
 * Samsung AMS561RA01 panel with S6E8AA5X01 controller.
 *
 * Copyright (C) 2025 Kaustabh Chakraborty <kauschluss@disroot.org>
 */

#include <linux/backlight.h>
#include <linux/gpio/consumer.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/regulator/consumer.h>

#include <drm/drm_mipi_dsi.h>
#include <drm/drm_modes.h>
#include <drm/drm_panel.h>
#include <drm/drm_probe_helper.h>

/* Manufacturer Command Set */
#define MCS_AIDCTL		0xb2
#define MCS_ADAPTIVECTL		0xb5
#define MCS_ELVSS		0xb6
#define MCS_TEMPERCTL		0xb8
#define MCS_PENTILE		0xc0
#define MCS_GAMMACTL		0xca
#define MCS_LTPSCTL		0xcb
#define MCS_PCD			0xcc
#define MCS_ERRFLAG		0xe7
#define MCS_ACCESSPROT		0xf0
#define MCS_DISPCTL		0xf2
#define MCS_GAMMAUPD		0xf7

#define GAMMA_CMD_LEN	34
#define AID_CMD_LEN	3

static const struct {
	u8 gamma[GAMMA_CMD_LEN];
	u8 aid[AID_CMD_LEN];
} s6e8aa5x01_ams561ra01_cmds[] = {
	{
		/* 5 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x94,
		  0x88, 0x89, 0x8a, 0x87, 0x87, 0x89,
		  0x8d, 0x8c, 0x8d, 0x89, 0x8c, 0x8e,
		  0x8e, 0x8f, 0x90, 0xa3, 0xa2, 0x9a,
		  0xcf, 0xca, 0x9f, 0xe6, 0xff, 0xb4,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x05, 0xa5 },
	}, {
		/* 6 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x95,
		  0x88, 0x89, 0x8b, 0x87, 0x87, 0x89,
		  0x8c, 0x8a, 0x8c, 0x85, 0x88, 0x8c,
		  0x8b, 0x8c, 0x8e, 0xa2, 0xa2, 0x9a,
		  0xd0, 0xcc, 0xa2, 0xed, 0xff, 0xb7,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x05, 0x95 },
	}, {
		/* 7 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x95,
		  0x88, 0x89, 0x8b, 0x87, 0x87, 0x89,
		  0x8c, 0x8a, 0x8c, 0x85, 0x88, 0x8c,
		  0x8b, 0x8c, 0x8e, 0xa2, 0xa2, 0x99,
		  0xc8, 0xc4, 0x9d, 0xed, 0xff, 0xb7,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x05, 0x89 },
	}, {
		/* 8 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x96,
		  0x88, 0x89, 0x8a, 0x87, 0x87, 0x89,
		  0x8a, 0x88, 0x8b, 0x83, 0x86, 0x8b,
		  0x8c, 0x8b, 0x8d, 0x9d, 0x9f, 0x97,
		  0xc7, 0xc3, 0x9c, 0xf5, 0xff, 0xbb,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x05, 0x7e },
	}, {
		/* 9 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x96,
		  0x88, 0x89, 0x8a, 0x87, 0x87, 0x89,
		  0x89, 0x86, 0x8a, 0x82, 0x84, 0x88,
		  0x90, 0x8f, 0x91, 0x95, 0x97, 0x94,
		  0xc6, 0xc2, 0x9d, 0xf5, 0xff, 0xbb,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x05, 0x73 },
	}, {
		/* 10 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x96,
		  0x88, 0x89, 0x8a, 0x87, 0x87, 0x89,
		  0x89, 0x86, 0x8a, 0x82, 0x84, 0x88,
		  0x90, 0x8f, 0x91, 0x94, 0x97, 0x93,
		  0xc6, 0xc2, 0x9e, 0xec, 0xff, 0xb7,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x05, 0x67 },
	}, {
		/* 11 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x96,
		  0x88, 0x89, 0x8a, 0x87, 0x87, 0x89,
		  0x89, 0x86, 0x8a, 0x82, 0x84, 0x88,
		  0x8b, 0x8b, 0x8d, 0x90, 0x93, 0x92,
		  0xc5, 0xc1, 0x9c, 0xf5, 0xff, 0xbb,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x05, 0x56 },
	}, {
		/* 12 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x96,
		  0x88, 0x89, 0x8b, 0x87, 0x87, 0x89,
		  0x89, 0x86, 0x89, 0x82, 0x84, 0x88,
		  0x87, 0x86, 0x8a, 0x8c, 0x90, 0x8f,
		  0xcd, 0xc9, 0xa1, 0xec, 0xff, 0xb7,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x05, 0x4a },
	}, {
		/* 13 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x96,
		  0x88, 0x89, 0x8b, 0x87, 0x87, 0x89,
		  0x89, 0x86, 0x89, 0x82, 0x84, 0x88,
		  0x87, 0x86, 0x8a, 0x8c, 0x90, 0x8e,
		  0xc4, 0xbf, 0x9c, 0xf5, 0xff, 0xbb,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x05, 0x3b },
	}, {
		/* 14 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x96,
		  0x88, 0x89, 0x8b, 0x87, 0x87, 0x89,
		  0x89, 0x86, 0x89, 0x82, 0x84, 0x88,
		  0x87, 0x86, 0x89, 0x8c, 0x90, 0x8f,
		  0xc2, 0xbf, 0x9c, 0xec, 0xff, 0xb7,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x05, 0x35 },
	}, {
		/* 15 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x96,
		  0x88, 0x89, 0x8b, 0x87, 0x87, 0x89,
		  0x89, 0x86, 0x89, 0x82, 0x84, 0x88,
		  0x87, 0x86, 0x89, 0x8c, 0x90, 0x8f,
		  0xb7, 0xb6, 0x96, 0xec, 0xff, 0xb7,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x05, 0x25 },
	}, {
		/* 16 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x96,
		  0x88, 0x89, 0x8b, 0x87, 0x87, 0x89,
		  0x89, 0x86, 0x89, 0x82, 0x84, 0x88,
		  0x88, 0x86, 0x89, 0x8c, 0x90, 0x8f,
		  0xb7, 0xb6, 0x96, 0xec, 0xff, 0xb7,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x05, 0x20 },
	}, {
		/* 17 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x96,
		  0x88, 0x89, 0x8b, 0x87, 0x87, 0x89,
		  0x89, 0x86, 0x89, 0x7f, 0x80, 0x86,
		  0x86, 0x85, 0x89, 0x88, 0x8c, 0x8e,
		  0xbf, 0xbe, 0x9c, 0xec, 0xff, 0xb7,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x05, 0x11 },
	}, {
		/* 19 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x96,
		  0x88, 0x89, 0x8b, 0x87, 0x87, 0x89,
		  0x89, 0x86, 0x89, 0x7f, 0x80, 0x86,
		  0x87, 0x85, 0x89, 0x88, 0x8c, 0x8e,
		  0xb3, 0xb4, 0x97, 0xeb, 0xff, 0xb7,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x04, 0xf2 },
	}, {
		/* 20 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x95,
		  0x88, 0x89, 0x8b, 0x87, 0x87, 0x89,
		  0x89, 0x86, 0x89, 0x7f, 0x80, 0x86,
		  0x87, 0x85, 0x89, 0x89, 0x8c, 0x8e,
		  0xb3, 0xb4, 0x97, 0xeb, 0xff, 0xb7,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x04, 0xe4 },
	}, {
		/* 21 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x96,
		  0x88, 0x89, 0x8b, 0x87, 0x87, 0x89,
		  0x8a, 0x88, 0x8b, 0x7d, 0x7e, 0x84,
		  0x8c, 0x8a, 0x8c, 0x8e, 0x90, 0x8f,
		  0xb6, 0xb6, 0x97, 0xe3, 0xff, 0xb3,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x04, 0xd5 },
	}, {
		/* 22 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x97,
		  0x88, 0x89, 0x8b, 0x87, 0x87, 0x89,
		  0x8a, 0x88, 0x8b, 0x81, 0x82, 0x86,
		  0x87, 0x86, 0x88, 0x8e, 0x90, 0x8f,
		  0xb6, 0xb6, 0x95, 0xe3, 0xff, 0xb3,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x04, 0xc5 },
	}, {
		/* 24 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x97,
		  0x88, 0x89, 0x8b, 0x88, 0x88, 0x8a,
		  0x8a, 0x87, 0x8a, 0x81, 0x82, 0x86,
		  0x87, 0x86, 0x88, 0x8e, 0x90, 0x8f,
		  0xb6, 0xb6, 0x94, 0xe3, 0xff, 0xb3,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x04, 0xa7 },
	}, {
		/* 25 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x98,
		  0x88, 0x89, 0x8b, 0x88, 0x88, 0x8a,
		  0x8a, 0x87, 0x8a, 0x81, 0x82, 0x86,
		  0x87, 0x86, 0x87, 0x8e, 0x90, 0x8f,
		  0xbf, 0xbf, 0x9a, 0xda, 0xfa, 0xaf,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x04, 0x95 },
	}, {
		/* 27 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x99,
		  0x88, 0x89, 0x8b, 0x88, 0x88, 0x8a,
		  0x8a, 0x87, 0x8a, 0x83, 0x86, 0x8a,
		  0x88, 0x87, 0x87, 0x88, 0x8b, 0x8c,
		  0xbf, 0xbf, 0x9a, 0xda, 0xfa, 0xaf,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x04, 0x76 },
	}, {
		/* 29 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x99,
		  0x88, 0x89, 0x8b, 0x88, 0x88, 0x8a,
		  0x8a, 0x87, 0x8b, 0x83, 0x86, 0x89,
		  0x88, 0x87, 0x88, 0x88, 0x8b, 0x8b,
		  0xbf, 0xbf, 0x9a, 0xda, 0xfa, 0xaf,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x04, 0x54 },
	}, {
		/* 30 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x9a,
		  0x88, 0x89, 0x8b, 0x88, 0x88, 0x8a,
		  0x8a, 0x87, 0x8a, 0x84, 0x86, 0x8a,
		  0x87, 0x87, 0x87, 0x88, 0x8b, 0x8b,
		  0xbf, 0xbf, 0x99, 0xda, 0xfa, 0xaf,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x04, 0x44 },
	}, {
		/* 32 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x9a,
		  0x89, 0x89, 0x8c, 0x88, 0x88, 0x8a,
		  0x89, 0x87, 0x8a, 0x84, 0x86, 0x8a,
		  0x87, 0x87, 0x87, 0x89, 0x8b, 0x8b,
		  0xbf, 0xbf, 0x98, 0xd2, 0xf2, 0xac,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x04, 0x1f },
	}, {
		/* 34 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x9b,
		  0x88, 0x89, 0x8b, 0x88, 0x88, 0x8a,
		  0x8b, 0x87, 0x8b, 0x83, 0x86, 0x89,
		  0x87, 0x87, 0x88, 0x88, 0x8b, 0x8a,
		  0xbf, 0xbf, 0x98, 0xd2, 0xf2, 0xac,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x03, 0xff },
	}, {
		/* 37 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x9b,
		  0x89, 0x89, 0x8c, 0x88, 0x88, 0x8a,
		  0x8a, 0x87, 0x8a, 0x81, 0x82, 0x86,
		  0x86, 0x86, 0x86, 0x8d, 0x90, 0x8d,
		  0xc0, 0xbf, 0x9a, 0xd2, 0xf2, 0xac,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x03, 0xd3 },
	}, {
		/* 39 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x9b,
		  0x89, 0x89, 0x8c, 0x88, 0x88, 0x8a,
		  0x8a, 0x87, 0x8a, 0x81, 0x82, 0x86,
		  0x87, 0x86, 0x87, 0x8d, 0x90, 0x8d,
		  0xb6, 0xb6, 0x93, 0xda, 0xf9, 0xaf,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x03, 0xb3 },
	}, {
		/* 41 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x9b,
		  0x89, 0x89, 0x8c, 0x88, 0x88, 0x8a,
		  0x8a, 0x87, 0x8b, 0x81, 0x82, 0x85,
		  0x87, 0x86, 0x87, 0x8d, 0x90, 0x8d,
		  0xb6, 0xb6, 0x94, 0xda, 0xf9, 0xaf,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x03, 0x93 },
	}, {
		/* 44 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x9b,
		  0x89, 0x89, 0x8c, 0x88, 0x88, 0x8a,
		  0x8a, 0x87, 0x8b, 0x81, 0x82, 0x86,
		  0x87, 0x86, 0x86, 0x85, 0x87, 0x8a,
		  0xbe, 0xbe, 0x99, 0xda, 0xf9, 0xaf,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x03, 0x66 },
	}, {
		/* 47 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x9b,
		  0x89, 0x89, 0x8c, 0x88, 0x88, 0x8a,
		  0x8a, 0x87, 0x8b, 0x81, 0x82, 0x86,
		  0x88, 0x86, 0x87, 0x84, 0x87, 0x89,
		  0xb4, 0xb4, 0x94, 0xe2, 0xff, 0xb3,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x03, 0x40 },
	}, {
		/* 50 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x9c,
		  0x89, 0x89, 0x8b, 0x88, 0x88, 0x8a,
		  0x8a, 0x87, 0x8b, 0x81, 0x82, 0x86,
		  0x88, 0x86, 0x87, 0x84, 0x87, 0x89,
		  0xb4, 0xb4, 0x95, 0xe2, 0xff, 0xb3,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x03, 0x0e },
	}, {
		/* 53 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x9c,
		  0x89, 0x89, 0x8b, 0x88, 0x88, 0x8a,
		  0x8a, 0x87, 0x8b, 0x81, 0x82, 0x86,
		  0x88, 0x86, 0x87, 0x85, 0x87, 0x8a,
		  0xb4, 0xb4, 0x96, 0xe2, 0xff, 0xb3,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x02, 0xe2 },
	}, {
		/* 56 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x9c,
		  0x89, 0x89, 0x8b, 0x88, 0x88, 0x8a,
		  0x8a, 0x87, 0x8b, 0x81, 0x82, 0x86,
		  0x88, 0x86, 0x87, 0x85, 0x87, 0x8a,
		  0xab, 0xab, 0x90, 0xdd, 0xf7, 0xaf,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x02, 0xb5 },
	}, {
		/* 60 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x9c,
		  0x89, 0x89, 0x8b, 0x88, 0x88, 0x8a,
		  0x8a, 0x87, 0x8b, 0x82, 0x82, 0x87,
		  0x83, 0x81, 0x84, 0x81, 0x84, 0x88,
		  0xb3, 0xb3, 0x96, 0xcf, 0xe5, 0xa8,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x02, 0x77 },
	}, {
		/* 64 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x98, 0x00, 0xa4, 0x00, 0x9c,
		  0x89, 0x89, 0x8b, 0x88, 0x88, 0x8a,
		  0x8a, 0x87, 0x8b, 0x82, 0x82, 0x87,
		  0x83, 0x81, 0x84, 0x82, 0x84, 0x88,
		  0xb2, 0xb3, 0x97, 0xcf, 0xe5, 0xa8,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x02, 0x36 },
	}, {
		/* 68 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x9b, 0x00, 0xa6, 0x00, 0x9d,
		  0x88, 0x88, 0x89, 0x89, 0x89, 0x8b,
		  0x8a, 0x88, 0x8b, 0x7f, 0x80, 0x86,
		  0x88, 0x86, 0x87, 0x7d, 0x7f, 0x85,
		  0xb2, 0xb3, 0x97, 0xcf, 0xe5, 0xa8,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x02, 0x15 },
	}, {
		/* 72 nits */
		{ MCS_GAMMACTL,
		  0x00, 0x9c, 0x00, 0xa9, 0x00, 0xa0,
		  0x88, 0x88, 0x89, 0x88, 0x88, 0x8a,
		  0x8c, 0x8a, 0x8d, 0x7f, 0x81, 0x85,
		  0x84, 0x82, 0x84, 0x85, 0x87, 0x8a,
		  0xaa, 0xab, 0x93, 0xcf, 0xe5, 0xa8,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x02, 0x15 },
	}, {
		/* 77 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xa1, 0x00, 0xad, 0x00, 0xa5,
		  0x89, 0x89, 0x8a, 0x88, 0x87, 0x89,
		  0x8c, 0x89, 0x8d, 0x7f, 0x81, 0x85,
		  0x84, 0x83, 0x84, 0x81, 0x83, 0x86,
		  0xaa, 0xab, 0x93, 0xc0, 0xd3, 0xa1,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x02, 0x15 },
	}, {
		/* 82 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xa5, 0x00, 0xb0, 0x00, 0xa9,
		  0x88, 0x89, 0x89, 0x85, 0x86, 0x89,
		  0x8a, 0x88, 0x8b, 0x82, 0x82, 0x87,
		  0x81, 0x80, 0x82, 0x89, 0x8b, 0x8b,
		  0xa2, 0xa3, 0x8e, 0xc0, 0xd3, 0xa1,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x02, 0x15 },
	}, {
		/* 87 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xab, 0x00, 0xb4, 0x00, 0xad,
		  0x88, 0x89, 0x8a, 0x84, 0x86, 0x88,
		  0x8a, 0x88, 0x8b, 0x7f, 0x7f, 0x84,
		  0x86, 0x84, 0x85, 0x85, 0x86, 0x88,
		  0xa2, 0xa3, 0x8f, 0xc0, 0xd3, 0xa1,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x02, 0x15 },
	}, {
		/* 93 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xaf, 0x00, 0xb9, 0x00, 0xb1,
		  0x88, 0x89, 0x8a, 0x84, 0x85, 0x87,
		  0x8a, 0x89, 0x8b, 0x7e, 0x7e, 0x83,
		  0x87, 0x86, 0x86, 0x88, 0x8a, 0x89,
		  0x9c, 0x9c, 0x8b, 0xc0, 0xd3, 0xa1,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x02, 0x15 },
	}, {
		/* 98 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xb3, 0x00, 0xbc, 0x00, 0xb5,
		  0x88, 0x88, 0x88, 0x84, 0x84, 0x86,
		  0x8a, 0x88, 0x8a, 0x7f, 0x7f, 0x84,
		  0x84, 0x83, 0x84, 0x88, 0x8a, 0x89,
		  0x9c, 0x9c, 0x8b, 0xc0, 0xd3, 0xa1,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x02, 0x15 },
	}, {
		/* 105 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xb7, 0x00, 0xc0, 0x00, 0xba,
		  0x87, 0x87, 0x88, 0x85, 0x85, 0x87,
		  0x89, 0x88, 0x89, 0x7f, 0x7f, 0x83,
		  0x81, 0x80, 0x82, 0x88, 0x8a, 0x89,
		  0x9c, 0x9c, 0x8c, 0xb2, 0xc2, 0x9a,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x02, 0x15 },
	}, {
		/* 111 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xbb, 0x00, 0xc3, 0x00, 0xbe,
		  0x87, 0x87, 0x88, 0x85, 0x85, 0x88,
		  0x88, 0x87, 0x89, 0x80, 0x80, 0x84,
		  0x81, 0x81, 0x82, 0x85, 0x86, 0x87,
		  0x9c, 0x9c, 0x8b, 0xb2, 0xc2, 0x9a,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x02, 0x15 },
	}, {
		/* 119 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xc0, 0x00, 0xc8, 0x00, 0xc4,
		  0x87, 0x87, 0x88, 0x82, 0x84, 0x86,
		  0x87, 0x85, 0x87, 0x82, 0x81, 0x84,
		  0x83, 0x82, 0x83, 0x80, 0x81, 0x84,
		  0x9c, 0x9c, 0x8c, 0xb2, 0xc2, 0x9a,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x02, 0x14 },
	}, {
		/* 126 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xc0, 0x00, 0xc8, 0x00, 0xc4,
		  0x87, 0x87, 0x88, 0x82, 0x84, 0x86,
		  0x87, 0x85, 0x87, 0x82, 0x81, 0x84,
		  0x83, 0x82, 0x83, 0x80, 0x81, 0x84,
		  0x9c, 0x9c, 0x8d, 0xb2, 0xc2, 0x9a,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x01, 0xde },
	}, {
		/* 134 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xc0, 0x00, 0xc8, 0x00, 0xc4,
		  0x87, 0x87, 0x88, 0x82, 0x84, 0x86,
		  0x87, 0x85, 0x87, 0x82, 0x81, 0x84,
		  0x83, 0x82, 0x83, 0x80, 0x81, 0x84,
		  0x9c, 0x9c, 0x8d, 0xa4, 0xb0, 0x92,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x01, 0x94 },
	}, {
		/* 143 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xc0, 0x00, 0xc8, 0x00, 0xc3,
		  0x87, 0x87, 0x88, 0x82, 0x84, 0x86,
		  0x87, 0x85, 0x87, 0x82, 0x81, 0x85,
		  0x83, 0x82, 0x83, 0x80, 0x81, 0x84,
		  0x92, 0x92, 0x89, 0xab, 0xb6, 0x96,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x01, 0x46 },
	}, {
		/* 152 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xc0, 0x00, 0xc8, 0x00, 0xc3,
		  0x87, 0x87, 0x88, 0x83, 0x84, 0x86,
		  0x87, 0x85, 0x87, 0x81, 0x81, 0x85,
		  0x84, 0x82, 0x83, 0x80, 0x81, 0x83,
		  0x92, 0x92, 0x8b, 0xab, 0xb6, 0x96,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0xfa },
	}, {
		/* 162 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xc0, 0x00, 0xc8, 0x00, 0xc3,
		  0x87, 0x87, 0x88, 0x83, 0x84, 0x86,
		  0x87, 0x85, 0x87, 0x81, 0x81, 0x84,
		  0x84, 0x82, 0x84, 0x80, 0x81, 0x83,
		  0x92, 0x92, 0x8b, 0x9d, 0xa4, 0x8e,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0xac },
	}, {
		/* 172 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xc0, 0x00, 0xc8, 0x00, 0xc3,
		  0x87, 0x87, 0x88, 0x83, 0x84, 0x86,
		  0x87, 0x85, 0x87, 0x81, 0x81, 0x84,
		  0x84, 0x82, 0x83, 0x80, 0x81, 0x84,
		  0x93, 0x92, 0x8c, 0x9d, 0xa4, 0x8e,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x57 },
	}, {
		/* 183 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xc2, 0x00, 0xca, 0x00, 0xc5,
		  0x86, 0x86, 0x87, 0x85, 0x84, 0x87,
		  0x87, 0x86, 0x88, 0x7e, 0x80, 0x83,
		  0x84, 0x82, 0x83, 0x80, 0x81, 0x83,
		  0x93, 0x92, 0x8c, 0x9d, 0xa4, 0x8e,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 195 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xc7, 0x00, 0xce, 0x00, 0xc9,
		  0x86, 0x87, 0x86, 0x83, 0x83, 0x85,
		  0x85, 0x84, 0x86, 0x82, 0x82, 0x85,
		  0x80, 0x80, 0x81, 0x81, 0x81, 0x84,
		  0x93, 0x92, 0x8c, 0x9d, 0xa4, 0x8e,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 207 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xcc, 0x00, 0xd2, 0x00, 0xce,
		  0x86, 0x86, 0x87, 0x81, 0x83, 0x84,
		  0x84, 0x82, 0x84, 0x83, 0x83, 0x85,
		  0x81, 0x81, 0x82, 0x7c, 0x7d, 0x81,
		  0x93, 0x92, 0x8c, 0x9d, 0xa4, 0x8e,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 220 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xd1, 0x00, 0xd6, 0x00, 0xd3,
		  0x86, 0x86, 0x86, 0x81, 0x83, 0x84,
		  0x84, 0x82, 0x84, 0x80, 0x80, 0x83,
		  0x81, 0x81, 0x82, 0x7c, 0x7d, 0x81,
		  0x93, 0x92, 0x8c, 0x9d, 0xa4, 0x8e,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 234 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xd6, 0x00, 0xdb, 0x00, 0xd8,
		  0x85, 0x85, 0x85, 0x81, 0x83, 0x84,
		  0x83, 0x82, 0x83, 0x80, 0x80, 0x82,
		  0x84, 0x82, 0x83, 0x79, 0x79, 0x7e,
		  0x93, 0x92, 0x8d, 0x9d, 0xa4, 0x8e,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 249 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xdc, 0x00, 0xe0, 0x00, 0xdd,
		  0x84, 0x84, 0x84, 0x81, 0x82, 0x83,
		  0x84, 0x82, 0x84, 0x7f, 0x7f, 0x82,
		  0x81, 0x80, 0x81, 0x80, 0x81, 0x82,
		  0x8c, 0x8c, 0x86, 0x9d, 0xa4, 0x8e,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 265 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xe2, 0x00, 0xe5, 0x00, 0xe3,
		  0x83, 0x83, 0x83, 0x81, 0x82, 0x83,
		  0x82, 0x82, 0x83, 0x82, 0x81, 0x83,
		  0x7f, 0x7e, 0x80, 0x7c, 0x7d, 0x80,
		  0x8c, 0x8c, 0x86, 0x8e, 0x92, 0x87,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 282 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xe8, 0x00, 0xea, 0x00, 0xe9,
		  0x83, 0x83, 0x83, 0x80, 0x82, 0x82,
		  0x81, 0x82, 0x82, 0x82, 0x81, 0x82,
		  0x81, 0x80, 0x81, 0x80, 0x80, 0x81,
		  0x85, 0x85, 0x83, 0x8e, 0x92, 0x87,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 300 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xed, 0x00, 0xef, 0x00, 0xed,
		  0x81, 0x82, 0x81, 0x81, 0x81, 0x82,
		  0x82, 0x82, 0x83, 0x80, 0x80, 0x81,
		  0x81, 0x81, 0x82, 0x83, 0x83, 0x83,
		  0x80, 0x80, 0x7f, 0x8e, 0x92, 0x87,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 316 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xf3, 0x00, 0xf4, 0x00, 0xf3,
		  0x80, 0x81, 0x80, 0x81, 0x81, 0x81,
		  0x82, 0x82, 0x82, 0x81, 0x80, 0x81,
		  0x82, 0x82, 0x83, 0x80, 0x80, 0x80,
		  0x80, 0x80, 0x7f, 0x80, 0x80, 0x80,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 333 nits */
		{ MCS_GAMMACTL,
		  0x00, 0xf8, 0x00, 0xf8, 0x00, 0xf8,
		  0x80, 0x81, 0x80, 0x81, 0x80, 0x81,
		  0x81, 0x82, 0x82, 0x81, 0x80, 0x81,
		  0x83, 0x83, 0x83, 0x7e, 0x7d, 0x7e,
		  0x80, 0x80, 0x7f, 0x80, 0x80, 0x80,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 360 nits */
		{ MCS_GAMMACTL,
		  0x01, 0x00, 0x01, 0x00, 0x01, 0x00,
		  0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
		  0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
		  0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
		  0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 378 nits */
		{ MCS_GAMMACTL,
		  0x01, 0x04, 0x01, 0x03, 0x01, 0x04,
		  0x7f, 0x7f, 0x80, 0x7f, 0x7f, 0x7f,
		  0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f,
		  0x80, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f,
		  0x80, 0x80, 0x7f, 0x7f, 0x7f, 0x7f,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 395 nits */
		{ MCS_GAMMACTL,
		  0x01, 0x09, 0x01, 0x07, 0x01, 0x08,
		  0x7e, 0x7f, 0x80, 0x7f, 0x7f, 0x7f,
		  0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f,
		  0x80, 0x7f, 0x7f, 0x7e, 0x7e, 0x7e,
		  0x80, 0x80, 0x7f, 0x7e, 0x7e, 0x7f,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 413 nits */
		{ MCS_GAMMACTL,
		  0x01, 0x0e, 0x01, 0x0b, 0x01, 0x0c,
		  0x7e, 0x7f, 0x80, 0x7e, 0x7e, 0x7e,
		  0x7e, 0x7e, 0x7e, 0x7f, 0x7f, 0x7f,
		  0x80, 0x7f, 0x7f, 0x7d, 0x7d, 0x7d,
		  0x80, 0x80, 0x7f, 0x7d, 0x7e, 0x7e,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 430 nits */
		{ MCS_GAMMACTL,
		  0x01, 0x13, 0x01, 0x0f, 0x01, 0x10,
		  0x7d, 0x7f, 0x80, 0x7e, 0x7e, 0x7e,
		  0x7e, 0x7e, 0x7e, 0x7f, 0x7f, 0x7f,
		  0x80, 0x7f, 0x7f, 0x7d, 0x7d, 0x7d,
		  0x80, 0x80, 0x7f, 0x7c, 0x7d, 0x7e,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 448 nits */
		{ MCS_GAMMACTL,
		  0x01, 0x18, 0x01, 0x13, 0x01, 0x14,
		  0x7c, 0x7e, 0x80, 0x7e, 0x7e, 0x7e,
		  0x7e, 0x7e, 0x7d, 0x7e, 0x7f, 0x7e,
		  0x80, 0x7f, 0x7f, 0x7c, 0x7c, 0x7c,
		  0x80, 0x80, 0x7e, 0x7b, 0x7c, 0x7d,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 465 nits */
		{ MCS_GAMMACTL,
		  0x01, 0x1d, 0x01, 0x17, 0x01, 0x18,
		  0x7c, 0x7e, 0x80, 0x7d, 0x7d, 0x7d,
		  0x7d, 0x7d, 0x7d, 0x7e, 0x7f, 0x7e,
		  0x80, 0x7f, 0x7f, 0x7b, 0x7b, 0x7b,
		  0x80, 0x80, 0x7e, 0x7a, 0x7c, 0x7d,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 483 nits */
		{ MCS_GAMMACTL,
		  0x01, 0x22, 0x01, 0x1b, 0x01, 0x1c,
		  0x7b, 0x7e, 0x80, 0x7d, 0x7d, 0x7d,
		  0x7d, 0x7d, 0x7c, 0x7e, 0x7f, 0x7e,
		  0x80, 0x7f, 0x7f, 0x7a, 0x7a, 0x7a,
		  0x80, 0x80, 0x7e, 0x79, 0x7b, 0x7c,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	}, {
		/* 500 nits */
		{ MCS_GAMMACTL,
		  0x01, 0x27, 0x01, 0x1f, 0x01, 0x20,
		  0x7b, 0x7e, 0x80, 0x7d, 0x7d, 0x7d,
		  0x7d, 0x7d, 0x7c, 0x7e, 0x7f, 0x7e,
		  0x80, 0x7f, 0x7f, 0x7a, 0x7a, 0x7a,
		  0x81, 0x80, 0x7e, 0x79, 0x7b, 0x7c,
		  0x00, 0x00, 0x00, },
		{ MCS_AIDCTL, 0x00, 0x10 },
	},
};

struct s6e8aa5x01_ams561ra01_ctx {
	struct drm_panel panel;
	struct mipi_dsi_device *dsi;
	struct backlight_device *bl;
	struct gpio_desc *reset_gpio;
	struct regulator_bulk_data *supplies;
	u32 nr_supplies;
};

static const struct regulator_bulk_data s6e8aa5x01_ams561ra01_supplies[] = {
	{ .supply = "vdd" },
	{ .supply = "vci" },
};

static inline struct s6e8aa5x01_ams561ra01_ctx *to_ctx(struct drm_panel *panel)
{
	return container_of(panel, struct s6e8aa5x01_ams561ra01_ctx, panel);
}

static int s6e8aa5x01_ams561ra01_update_status(struct backlight_device *bl)
{
	struct s6e8aa5x01_ams561ra01_ctx *ctx = bl_get_data(bl);
	struct mipi_dsi_multi_context dsi = { .dsi = ctx->dsi };
	u16 lvl = backlight_get_brightness(bl);

	if (!ctx->panel.enabled)
		return 0;

	mipi_dsi_dcs_write_seq_multi(&dsi, MCS_ACCESSPROT, 0x5a, 0x5a);

	mipi_dsi_dcs_write_buffer_multi(&dsi,
					s6e8aa5x01_ams561ra01_cmds[lvl].gamma,
					GAMMA_CMD_LEN);
	mipi_dsi_dcs_write_buffer_multi(&dsi,
					s6e8aa5x01_ams561ra01_cmds[lvl].aid,
					AID_CMD_LEN);
	mipi_dsi_dcs_write_seq_multi(&dsi, MCS_GAMMAUPD, 0x03);

	mipi_dsi_dcs_write_seq_multi(&dsi, MCS_ACCESSPROT, 0xa5, 0xa5);

	return dsi.accum_err;
}

static int s6e8aa5x01_ams561ra01_prepare(struct drm_panel *panel)
{
	struct s6e8aa5x01_ams561ra01_ctx *ctx = to_ctx(panel);
	struct device *dev = &ctx->dsi->dev;
	int ret;

	ret = regulator_bulk_enable(ctx->nr_supplies, ctx->supplies);
	if (ret < 0) {
		dev_err(dev, "failed to enable regulators: %d\n", ret);
		return ret;
	}

	gpiod_set_value_cansleep(ctx->reset_gpio, 0);
	usleep_range(5000, 6000);
	gpiod_set_value_cansleep(ctx->reset_gpio, 1);
	usleep_range(5000, 6000);
	gpiod_set_value_cansleep(ctx->reset_gpio, 0);
	usleep_range(10000, 11000);

	return 0;
}

static int s6e8aa5x01_ams561ra01_unprepare(struct drm_panel *panel)
{
	struct s6e8aa5x01_ams561ra01_ctx *ctx = to_ctx(panel);

	gpiod_set_value_cansleep(ctx->reset_gpio, 1);
	usleep_range(5000, 6000);

	regulator_bulk_disable(ctx->nr_supplies, ctx->supplies);

	return 0;
}

static int s6e8aa5x01_ams561ra01_enable(struct drm_panel *panel)
{
	struct s6e8aa5x01_ams561ra01_ctx *ctx = to_ctx(panel);
	struct mipi_dsi_multi_context dsi = { .dsi = ctx->dsi };

	mipi_dsi_dcs_exit_sleep_mode_multi(&dsi);
	mipi_dsi_msleep(&dsi, 100);

	mipi_dsi_dcs_write_seq_multi(&dsi, MCS_ACCESSPROT, 0x5a, 0x5a);

	mipi_dsi_dcs_write_seq_multi(&dsi, MCS_PENTILE, 0xd8, 0xd8, 0x00);
	mipi_dsi_dcs_write_seq_multi(&dsi, MCS_PCD, 0x5c);
	mipi_dsi_dcs_write_seq_multi(&dsi, MCS_ERRFLAG, 0xed, 0xc7, 0x23, 0x67);
	mipi_dsi_dcs_write_seq_multi(&dsi, MCS_DISPCTL, 0x0c, 0x0c, 0xb9, 0x01);
	mipi_dsi_dcs_write_seq_multi(&dsi, MCS_LTPSCTL,
				0x00, 0x45, 0x10, 0x10, 0x08, 0x32, 0x54, 0x00,
				0x00, 0x00, 0x00, 0x07, 0x06, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x48, 0x5e, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x03, 0x00, 0x00, 0x00, 0xad, 0x00, 0x00,
				0x08, 0x05, 0x2a, 0x54, 0x03, 0xcc, 0x00, 0xff,
				0xfb, 0x03, 0x0d, 0x00, 0x11, 0x0f, 0x02, 0x03,
				0x0b, 0x0c, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13,
				0x13, 0x13, 0x13, 0x13, 0x00, 0x02, 0x03, 0x0b,
				0x0c, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13,
				0x13, 0x13);

	mipi_dsi_dcs_write_seq_multi(&dsi, MCS_ACCESSPROT, 0xa5, 0xa5);

	mipi_dsi_dcs_set_display_on_multi(&dsi);

	return dsi.accum_err;
}

static int s6e8aa5x01_ams561ra01_disable(struct drm_panel *panel)
{
	struct s6e8aa5x01_ams561ra01_ctx *ctx = to_ctx(panel);
	struct mipi_dsi_multi_context dsi = { .dsi = ctx->dsi };

	mipi_dsi_dcs_set_display_off_multi(&dsi);
	mipi_dsi_msleep(&dsi, 100);

	mipi_dsi_dcs_enter_sleep_mode_multi(&dsi);
	mipi_dsi_msleep(&dsi, 150);

	return dsi.accum_err;
}

static const struct drm_display_mode s6e8aa5x01_ams561ra01_mode = {
	.clock = (720 + 62 + 2 + 26) * (1480 + 12 + 2 + 10) * 60 / 1000,
	.hdisplay = 720,
	.hsync_start = 720 + 62,
	.hsync_end = 720 + 62 + 2,
	.htotal = 720 + 62 + 2 + 26,
	.vdisplay = 1480,
	.vsync_start = 1480 + 12,
	.vsync_end = 1480 + 12 + 2,
	.vtotal = 1480 + 12 + 2 + 10,
	.width_mm = 62,
	.height_mm = 128,
	.type = DRM_MODE_TYPE_DRIVER | DRM_MODE_TYPE_PREFERRED,
};

static int s6e8aa5x01_ams561ra01_get_modes(struct drm_panel *panel,
					   struct drm_connector *connector)
{
	return drm_connector_helper_get_modes_fixed(connector,
						    &s6e8aa5x01_ams561ra01_mode);
}

static const struct backlight_ops s6e8aa5x01_ams561ra01_bl_ops = {
	.update_status = s6e8aa5x01_ams561ra01_update_status,
};

static const struct drm_panel_funcs s6e8aa5x01_ams561ra01_panel_funcs = {
	.prepare = s6e8aa5x01_ams561ra01_prepare,
	.unprepare = s6e8aa5x01_ams561ra01_unprepare,
	.enable = s6e8aa5x01_ams561ra01_enable,
	.disable = s6e8aa5x01_ams561ra01_disable,
	.get_modes = s6e8aa5x01_ams561ra01_get_modes,
};

static int s6e8aa5x01_ams561ra01_probe(struct mipi_dsi_device *dsi)
{
	struct device *dev = &dsi->dev;
	struct s6e8aa5x01_ams561ra01_ctx *ctx;
	int ret;

	ctx = devm_drm_panel_alloc(dev, struct s6e8aa5x01_ams561ra01_ctx, panel,
				   &s6e8aa5x01_ams561ra01_panel_funcs,
				   DRM_MODE_CONNECTOR_DSI);
	if (IS_ERR(ctx))
		return PTR_ERR(ctx);

	ctx->dsi = dsi;
	mipi_dsi_set_drvdata(dsi, ctx);

	ctx->nr_supplies = ARRAY_SIZE(s6e8aa5x01_ams561ra01_supplies);
	ret = devm_regulator_bulk_get_const(dev, ctx->nr_supplies,
					    s6e8aa5x01_ams561ra01_supplies,
					    &ctx->supplies);
	if (ret < 0)
		return dev_err_probe(dev, ret, "failed to get regulators\n");

	ctx->reset_gpio = devm_gpiod_get_optional(dev, "reset", GPIOD_ASIS);
	if (IS_ERR(ctx->reset_gpio))
		return dev_err_probe(dev, PTR_ERR(ctx->reset_gpio),
				     "failed to get reset-gpios\n");

	ctx->bl = devm_backlight_device_register(dev, dev_name(dev), dev, ctx,
						 &s6e8aa5x01_ams561ra01_bl_ops,
						 NULL);
	if (IS_ERR(ctx->bl))
		return dev_err_probe(dev, PTR_ERR(ctx->bl),
				     "failed to register backlight device\n");

	ctx->bl->props.type = BACKLIGHT_PLATFORM;
	ctx->bl->props.brightness = ARRAY_SIZE(s6e8aa5x01_ams561ra01_cmds) - 1;
	ctx->bl->props.max_brightness = ctx->bl->props.brightness;

	dsi->lanes = 4;
	dsi->format = MIPI_DSI_FMT_RGB888;
	dsi->mode_flags = MIPI_DSI_MODE_VIDEO | MIPI_DSI_MODE_VIDEO_BURST |
			  MIPI_DSI_MODE_VIDEO_NO_HFP;

	ctx->panel.prepare_prev_first = true;
	drm_panel_add(&ctx->panel);

	ret = devm_mipi_dsi_attach(dev, dsi);
	if (ret < 0) {
		drm_panel_remove(&ctx->panel);
		return dev_err_probe(dev, ret, "failed to attach to DSI host\n");
	}

	return 0;
}

static void s6e8aa5x01_ams561ra01_remove(struct mipi_dsi_device *dsi)
{
	struct s6e8aa5x01_ams561ra01_ctx *ctx = mipi_dsi_get_drvdata(dsi);

	drm_panel_remove(&ctx->panel);
}

static const struct of_device_id s6e8aa5x01_ams561ra01_of_device_id[] = {
	{ .compatible = "samsung,s6e8aa5x01-ams561ra01" },
	{ }
};
MODULE_DEVICE_TABLE(of, s6e8aa5x01_ams561ra01_of_device_id);

static struct mipi_dsi_driver s6e8aa5x01_ams561ra01_dsi_driver = {
	.probe = s6e8aa5x01_ams561ra01_probe,
	.remove = s6e8aa5x01_ams561ra01_remove,
	.driver = {
		.name = "panel-samsung-s6e8aa5x01-ams561ra01",
		.of_match_table = s6e8aa5x01_ams561ra01_of_device_id,
	},
};
module_mipi_dsi_driver(s6e8aa5x01_ams561ra01_dsi_driver);

MODULE_AUTHOR("Kaustabh Chakraborty <kauschluss@disroot.org>");
MODULE_DESCRIPTION("Samsung AMS561RA01 Panel with S6E8AA5X01 Controller");
MODULE_LICENSE("GPL");
