/*
 * $Id: floppy_gen_floppydrive.c,v 1.73 2012-02-22 09:27:20 siflkres Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG_CONTROL_FLOW	0

#include "config.h"

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include "fixme.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "glue.h"

#include "floppy_gen_floppydrive.h"

#define COMP "floppy_gen_floppydrive"
#define COMP_(x) floppy_gen_floppydrive_ ## x

struct cpssp {
	/*
	 * Config
	 */
	char name[1024];
	unsigned int model; /* FIXME */
	unsigned int unit; /* FIXME */
#if 0
	unsigned int type;
#endif

	/*
	 * Ports
	 */
	struct sig_shugart_bus *shugart_bus;
	struct sig_boolean_or *shugart_bus_trk0;
	struct sig_boolean_or *shugart_bus_dskchg;
	struct sig_boolean *port_opt_busy_led;
	struct sig_floppy *port_media;

	/*
	 * State
	 */
	unsigned int select;	/* Is drive selected? */
	unsigned int motor;	/* Is motor spinning? */
	unsigned int hds;	/* Selected head */
	unsigned int inserted;	/* Disk inserted? */
	unsigned int changed;	/* Disk changed? */
	unsigned int wp;	/* Write protected? */

	unsigned char heads;	/* number of sides */
	unsigned char tracks;	/* tracks per side */
	unsigned char sectors;	/* sectors per track */
	
	enum {
		DRIVE_NONE,
		DRIVE_READ,
		DRIVE_READ_DONE,
		DRIVE_WRITE,
		DRIVE_WRITE_DONE,
	} op;			/* Operation */
	unsigned char track;	/* track position */
	unsigned int angle;	/*
				 * 0: ID sector 1 (Index hole = 1)
				 * 1: Data sector 1
				 * 2: ID sector 2
				 * 2: Data sector 2
				 * ...
				 * 2*N+0: ID sector N+1
				 * 2*N+1: Data sector N+1
				 */

	unsigned char buffer[16*1024];
	int bufsize;
};

static int
COMP_(media_wp)(struct cpssp *cpssp)
{
	int ret;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s\n", __FUNCTION__);
	}

	ret = sig_floppy_wp(cpssp->port_media, cpssp);

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: ret=%d\n", __FUNCTION__, ret);
	}

	return ret;
}

static int
COMP_(media_read)(
	struct cpssp *cpssp,
	unsigned int blk,
	uint8_t *data,
	uint8_t *trackp,
	uint8_t *sectorp
)
{
	int ret;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: blk=%u\n", __FUNCTION__, blk);
	}

	ret = sig_floppy_read(cpssp->port_media, cpssp,
			blk, data, trackp, sectorp);

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: ret=%d\n", __FUNCTION__, ret);
	}

	return ret;
}

static int
COMP_(media_write)(
	struct cpssp *cpssp,
	unsigned int blk,
	const uint8_t *data,
	uint8_t track,
	uint8_t sector
)
{
	int ret;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: blk=%u\n", __FUNCTION__, blk);
	}

	ret = sig_floppy_write(cpssp->port_media, cpssp,
			blk, data, track, sector);

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: ret=%d\n", __FUNCTION__, ret);
	}

	return ret;
}

/*forward*/ static void
floppy_gen_floppydrive_process(struct cpssp *cpssp);

static void
floppy_gen_floppydrive_step(struct cpssp *cpssp)
{
again:	;
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: angle=%u op=%u, bufsize=%u\n",
				__FUNCTION__,
				cpssp->angle, cpssp->op, cpssp->bufsize);
	}

	sig_shugart_bus_index_set(cpssp->shugart_bus, cpssp, cpssp->angle == 0);

	switch (cpssp->op) {
	case DRIVE_READ:
		if (cpssp->angle % 2 == 0) {
			/* Read ID. */
			/*
			 * This really will read the sector.
			 * But we need to know whether the sector exists.
			 */
			floppy_gen_floppydrive_process(cpssp);
			return;

		} else {
			/* Read data. */
			static unsigned char zero[512] = { 0, 0 /* ... */ };

			cpssp->angle = (cpssp->angle + 1) % 36;
			cpssp->op = DRIVE_READ_DONE;

			if (cpssp->bufsize < 0) {
				/* Read error. */
				assert(0);
			} else if (cpssp->bufsize == 0) {
				/* No media. */
				sig_shugart_bus_readdata(cpssp->shugart_bus, cpssp,
						zero, sizeof(zero));
			} else {
				/* Data read. */
				sig_shugart_bus_readdata(cpssp->shugart_bus, cpssp,
						cpssp->buffer, cpssp->bufsize);
			}
		}
		break;

	case DRIVE_WRITE:
		if (cpssp->angle % 2 == 0) {
			/* Write ID. */
			/*
			 * We don't write any IDs.
			 */
			/* Nothing to do... */

			cpssp->angle = (cpssp->angle + 1) % 36;
			cpssp->op = DRIVE_WRITE_DONE;

		} else {
			/* Write data. */
			floppy_gen_floppydrive_process(cpssp);
			return;
		}
		break;

	default:
		assert(0);
	}

	switch (cpssp->op) {
	case DRIVE_READ_DONE:
	case DRIVE_WRITE_DONE:
		cpssp->op = DRIVE_NONE;
		break;
	case DRIVE_READ:
	case DRIVE_WRITE:
		goto again;
	default:
		assert(0);
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: done.\n", __FUNCTION__);
	}
}

static void
floppy_gen_floppydrive_interrupt(struct cpssp *cpssp, unsigned int retval)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: retval=%u\n", __FUNCTION__, retval);
	}

	switch (cpssp->op) {
	case DRIVE_READ:
		cpssp->op = DRIVE_READ_DONE;
		cpssp->angle = (cpssp->angle + 1) % 36;

		cpssp->bufsize = retval;

		if (cpssp->bufsize < 0) {
			/* Read error. */
			assert(0);
		} else if (cpssp->bufsize == 0) {
			/* No media. */
			sig_shugart_bus_readid(cpssp->shugart_bus, cpssp,
					0, 0);
		} else {
			/* ID read. */
			sig_shugart_bus_readid(cpssp->shugart_bus, cpssp,
					cpssp->track, cpssp->angle / 2 + 1);
		}
		break;

	case DRIVE_WRITE:
		/* No feedback. */
		cpssp->op = DRIVE_WRITE_DONE;
		cpssp->angle = (cpssp->angle + 1) % 36;
		break;

	default:
		assert(0); /* Cannot happen. */
	}

	switch (cpssp->op) {
	case DRIVE_READ_DONE:
	case DRIVE_WRITE_DONE:
		cpssp->op = DRIVE_NONE;
		break;
	case DRIVE_READ:
	case DRIVE_WRITE:
		floppy_gen_floppydrive_step(cpssp);
		break;
	default:
		assert(0);
	}
}

static void
floppy_gen_floppydrive_change(void *_cpssp, const char *media)
{
	struct cpssp *cpssp = _cpssp;
	char name[1024];

	cpssp->changed = 1;
	sig_boolean_or_set(cpssp->shugart_bus_dskchg, cpssp,
			cpssp->changed & cpssp->select);

	/* Remove old media (if any). */
	system_port_disconnect(cpssp->name, "media");
	cpssp->inserted = 0;

	if (*media == '\0') {
		return;
	}

	/* Insert new media (if any). */
	name[0] = ':';
	strcpy(&name[1], media);
	system_port_connect(cpssp->name, "media", name, "connect");
	cpssp->inserted = 1;

	cpssp->wp = COMP_(media_wp)(cpssp);
	if (cpssp->select) {
		sig_shugart_bus_wp_set(cpssp->shugart_bus, cpssp, cpssp->wp);
	}

	cpssp->angle = 0;
	cpssp->op = DRIVE_NONE;
}

static void
floppy_gen_floppydrive_select_set(void *s, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) s;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: %u\n", __FUNCTION__, val);
	}

	cpssp->select = val;

	sig_boolean_or_set(cpssp->shugart_bus_trk0, cpssp,
			val & (cpssp->track == 0));
	sig_boolean_or_set(cpssp->shugart_bus_dskchg, cpssp,
			val & cpssp->changed);
	if (cpssp->select) {
		sig_shugart_bus_wp_set(cpssp->shugart_bus, cpssp, cpssp->wp);
	}
}

static void
floppy_gen_floppydrive_motor_set(void *s, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) s;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: %u\n", __FUNCTION__, val);
	}

	cpssp->motor = val;

	sig_boolean_set(cpssp->port_opt_busy_led, cpssp, val);
}

static void
floppy_gen_floppydrive_hds_set(void *s, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) s;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: %u\n", __FUNCTION__, val);
	}

	cpssp->hds = val;
}

static void
floppy_gen_floppydrive_step_in(void *s)
{
	struct cpssp *cpssp = (struct cpssp *) s;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s\n", __FUNCTION__);
	}

	if (! cpssp->select) {
		return;
	}

	if (cpssp->track < 79) {
		cpssp->track++;
	}

	if (cpssp->changed
	 && cpssp->inserted) {
		cpssp->changed = 0;
		sig_boolean_or_set(cpssp->shugart_bus_dskchg, cpssp, 0);
	}
	sig_boolean_or_set(cpssp->shugart_bus_trk0, cpssp, cpssp->track == 0);
}

static void
floppy_gen_floppydrive_step_out(void *s)
{
	struct cpssp *cpssp = (struct cpssp *) s;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s\n", __FUNCTION__);
	}

	if (! cpssp->select) {
		return;
	}

	if (0 < cpssp->track) {
		cpssp->track--;
	}

	if (cpssp->changed
	 && cpssp->inserted) {
		cpssp->changed = 0;
		sig_boolean_or_set(cpssp->shugart_bus_dskchg, cpssp, 0);
	}
	sig_boolean_or_set(cpssp->shugart_bus_trk0, cpssp, cpssp->track == 0);
}

static void
floppy_gen_floppydrive_read_start(void *s)
{
	struct cpssp *cpssp = (struct cpssp *) s;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s\n", __FUNCTION__);
	}

	if (! cpssp->select) {
		return;
	}

	cpssp->bufsize = 512; /* FIXME VOSSI */
	switch (cpssp->op) {
	case DRIVE_NONE:
		cpssp->op = DRIVE_READ;
		floppy_gen_floppydrive_step(cpssp);
		break;
	case DRIVE_READ_DONE:
	case DRIVE_WRITE_DONE:
		cpssp->op = DRIVE_READ;
		break;
	default:
		assert(0);
	}
}

static void
floppy_gen_floppydrive_writedata(
	void *s,
	unsigned char *buf,
	unsigned int bufsize
)
{
	struct cpssp *cpssp = (struct cpssp *) s;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s\n", __FUNCTION__);
	}

	if (! cpssp->select) {
		return;
	}

	memcpy(cpssp->buffer, buf, bufsize);
	cpssp->bufsize = bufsize;
	switch (cpssp->op) {
	case DRIVE_NONE:
		cpssp->op = DRIVE_WRITE;
		floppy_gen_floppydrive_step(cpssp);
		break;
	case DRIVE_READ_DONE:
	case DRIVE_WRITE_DONE:
		cpssp->op = DRIVE_WRITE;
		break;
	default:
		assert(0);
	}
}

static void
floppy_gen_floppydrive_process(struct cpssp *cpssp)
{
	unsigned int blk;
	unsigned int out;
	uint8_t track;
	uint8_t sector;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: track=%u, hds=%u, sectors=%u, angle=%u, bufsize=%u\n",
				__FUNCTION__,
				cpssp->track, cpssp->hds, cpssp->sectors,
				cpssp->angle, cpssp->bufsize);
	}

	blk = ((cpssp->track * 2) + cpssp->hds) * cpssp->sectors + cpssp->angle / 2;

	switch (cpssp->op) {
	case DRIVE_READ:
		if (COMP_(media_read)(cpssp, blk, cpssp->buffer,
				&track, &sector) == 1) {
			out = cpssp->bufsize;
		} else {
			out = 0;
		}
		break;
	case DRIVE_WRITE:
		if (cpssp->wp) {
			out = cpssp->bufsize;
		} else {
			COMP_(media_write)(cpssp, blk, cpssp->buffer,
					cpssp->track, cpssp->angle / 2);
			out = cpssp->bufsize;
		}
		break;
	default:
		assert(0);
	}

	floppy_gen_floppydrive_interrupt(cpssp, out);
}

void *
floppy_gen_floppydrive_create(
	const char *name,
	const char *model,
	const char *unit,
	struct sig_manage *port_manage,
	struct sig_power_device *port_power,
	struct sig_shugart_bus *port_shugart,
	struct sig_boolean *port_opt_busy_led,
	struct sig_floppy *port_media,
	struct sig_string *port_change
)
{
	static const struct sig_shugart_bus_funcs shugart_funcs = {
		.motor_set = floppy_gen_floppydrive_motor_set,
		.select_set = floppy_gen_floppydrive_select_set,
		.hds_set = floppy_gen_floppydrive_hds_set,
		.step_in = floppy_gen_floppydrive_step_in,
		.step_out = floppy_gen_floppydrive_step_out,
		.read_start = floppy_gen_floppydrive_read_start,
		.writedata = floppy_gen_floppydrive_writedata,
	};
	static const struct sig_string_funcs change_funcs = {
		.set = floppy_gen_floppydrive_change,
	};
	struct cpssp *cpssp;

	if (! model
	 || atoi(model) == -1) model = "4";

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	system_name_push(name);

	strcpy(cpssp->name, system_path());

	cpssp->model = strtoul(model, NULL, 0);

	assert(unit);
	cpssp->unit = strtoul(unit, NULL, 0);

	cpssp->op = DRIVE_NONE;

	cpssp->hds = 0;
	cpssp->track = 0;
	cpssp->angle = 0;
	cpssp->inserted = 0;
	cpssp->changed = 1;
	cpssp->wp = 0;

	cpssp->heads = 2;
	cpssp->tracks = 80;
	cpssp->sectors = 18;

	cpssp->shugart_bus = port_shugart;
	sig_shugart_bus_connect_drive(cpssp->shugart_bus, cpssp,
			cpssp->unit, &shugart_funcs);
	cpssp->shugart_bus_trk0 = port_shugart->trk0;
	sig_boolean_or_connect_out(cpssp->shugart_bus_trk0, cpssp, 0);
	cpssp->shugart_bus_dskchg = port_shugart->dskchg;
	sig_boolean_or_connect_out(cpssp->shugart_bus_dskchg, cpssp, 0);

	cpssp->port_opt_busy_led = port_opt_busy_led;
	sig_boolean_connect_out(port_opt_busy_led, cpssp, 0);

	cpssp->port_media = port_media;

	sig_string_connect(port_change, cpssp, &change_funcs);

	system_name_pop();

	return cpssp;
}

void
floppy_gen_floppydrive_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
floppy_gen_floppydrive_suspend(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
floppy_gen_floppydrive_resume(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
