/***************************************************************************
 *   Copyright (C) 2008-2011 by Marcel Hasler                              *
 *   mahasler@gmail.com                                                    *
 *                                                                         *
 *   This file is part of KOSD.                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation, either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the          *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program. If not, see <http://www.gnu.org/licenses/>.  *
 ***************************************************************************/

#include "compactosd.h"
#include "configuration.h"
#include "kosdd.h"
#include "standardosd.h"

#include "kosd/kosd.h"

#include <KIcon>
#include <KLocale>
#include <KPluginFactory>
#include <Plasma/Theme>

#include <QMessageBox>

K_PLUGIN_FACTORY(KOsdDFactory, registerPlugin<KOsdD>();)
K_EXPORT_PLUGIN(KOsdDFactory("kosd"))

static int clamp(int value, int low, int high)
{
    if (value < low)
        return low;
    else if (value > high)
        return high;
    return value;
}

class KOsdD::KOsdDPrivate
{
public:
    enum VolumePixmaps
    {
        VolumePixmapMuted,
        VolumePixmapLow,
        VolumePixmapMedium,
        VolumePixmapHigh,
        
        VolumePixmapCount
    };

    enum BatteryPixmaps
    {
        BatteryPixmap100,
        BatteryPixmap080,
        BatteryPixmap060,
        BatteryPixmap040,
        BatteryPixmap020,
        BatteryPixmap000,

        BatteryPixmapCharging100,
        BatteryPixmapCharging080,
        BatteryPixmapCharging060,
        BatteryPixmapCharging040,
        BatteryPixmapCharging020,
        BatteryPixmapCharging000,

        BatteryPixmapCount
    };

    Osd *osd;
    Plasma::Svg *svg;

    QPixmap *volumePixmaps[VolumePixmapCount];
    QPixmap *batteryPixmaps[BatteryPixmapCount];
    QPixmap *brightnessPixmap;

    static QString volumeElements[VolumePixmapCount];
    static QString batteryFillElements[BatteryPixmapCount][2];
};


QString KOsdD::KOsdDPrivate::volumeElements[KOsdD::KOsdDPrivate::VolumePixmapCount] =
{
    "audio-volume-muted",
    "audio-volume-low",
    "audio-volume-medium",
    "audio-volume-high"
};


QString KOsdD::KOsdDPrivate::batteryFillElements[KOsdD::KOsdDPrivate::BatteryPixmapCount][2] =
{
    { "Fill100", "" },
    { "Fill80",  "" },
    { "Fill60",  "" },
    { "Fill40",  "" },
    { "Fill20",  "" },
    { "",        "" },

    { "Fill100", "AcAdapter" },
    { "Fill80",  "AcAdapter" },
    { "Fill60",  "AcAdapter" },
    { "Fill40",  "AcAdapter" },
    { "Fill20",  "AcAdapter" },
    { "",        "AcAdapter" }
};


KOsdD::KOsdD(QObject* parent, const QList<QVariant>&)
    : KDEDModule(parent),
      d(new KOsdDPrivate)
{
    d->osd = 0;
    d->svg = new Plasma::Svg();
    d->svg->setContainsMultipleImages(true);

    for (int i = 0; i < KOsdDPrivate::BatteryPixmapCount; ++i)
        d->batteryPixmaps[i] = 0;

    for (int i = 0; i < KOsdDPrivate::VolumePixmapCount; ++i)
        d->volumePixmaps[i] = 0;

    d->brightnessPixmap = 0;

    connect(Plasma::Theme::defaultTheme(), SIGNAL(themeChanged()), this, SLOT(parseConfiguration()));
    parseConfiguration();
}


KOsdD::~KOsdD()
{
    delete d->osd;
    delete d->svg;

    for (int i = 0; i < KOsdDPrivate::BatteryPixmapCount; ++i)
        delete d->batteryPixmaps[i];

    for (int i = 0; i < KOsdDPrivate::VolumePixmapCount; ++i)
        delete d->volumePixmaps[i];

    delete d->brightnessPixmap;

    delete d;
}


void KOsdD::showBattery(int percent, bool plugged)
{
    QPixmap *pixmap = d->batteryPixmaps[KOsdDPrivate::BatteryPixmap000];
    percent = clamp(percent, 0, 100);

    if (plugged)
    {
        if (percent > 95)
            pixmap = d->batteryPixmaps[KOsdDPrivate::BatteryPixmapCharging100];
        else if (percent > 80)
            pixmap = d->batteryPixmaps[KOsdDPrivate::BatteryPixmapCharging080];
        else if (percent > 50)
            pixmap = d->batteryPixmaps[KOsdDPrivate::BatteryPixmapCharging060];
        else if (percent > 20)
            pixmap = d->batteryPixmaps[KOsdDPrivate::BatteryPixmapCharging040];
        else if (percent > 10)
            pixmap = d->batteryPixmaps[KOsdDPrivate::BatteryPixmapCharging020];
        else
            pixmap = d->batteryPixmaps[KOsdDPrivate::BatteryPixmapCharging000];
    }
    else
    {        
        if (percent > 95)
            pixmap = d->batteryPixmaps[KOsdDPrivate::BatteryPixmap100];
        else if (percent > 80)
            pixmap = d->batteryPixmaps[KOsdDPrivate::BatteryPixmap080];
        else if (percent > 50)
            pixmap = d->batteryPixmaps[KOsdDPrivate::BatteryPixmap060];
        else if (percent > 20)
            pixmap = d->batteryPixmaps[KOsdDPrivate::BatteryPixmap040];
        else if (percent > 10)
            pixmap = d->batteryPixmaps[KOsdDPrivate::BatteryPixmap020];
        else
            pixmap = d->batteryPixmaps[KOsdDPrivate::BatteryPixmap000];
    }

    d->osd->show(*pixmap, i18n("Battery Charge<br>%1%", percent), percent);
}


void KOsdD::showBrightness(int percent)
{
    percent = clamp(percent, 0, 100);
    d->osd->show(*d->brightnessPixmap, i18n("Brightness<br>%1%", percent), percent);
}


void KOsdD::showVolume(int percent, bool muted)
{
    QPixmap *pixmap = d->volumePixmaps[KOsdDPrivate::VolumePixmapMuted];
    percent = clamp(percent, 0, 100);

    if (muted)
        d->osd->show(*pixmap, i18n("Volume Muted"), 0);
    else
    {        
        if (percent >= 75)
            pixmap = d->volumePixmaps[KOsdDPrivate::VolumePixmapHigh];
        else if (percent >= 25)
            pixmap = d->volumePixmaps[KOsdDPrivate::VolumePixmapMedium];
        else
            pixmap = d->volumePixmaps[KOsdDPrivate::VolumePixmapLow];

        d->osd->show(*pixmap, i18n("Volume<br>%1%", percent), percent);
    }
}


void KOsdD::showCustomSvg(const QString &svg, const QStringList &elements, const QString &text, int percent)
{
    QPixmap pixmap(d->osd->pixmapSize());
    paintSvg(svg, elements, &pixmap);
    d->osd->show(pixmap, text, clamp(percent, 0, 100));
}


void KOsdD::showCustomPixmap(const QString &pixmap, const QString &text, int percent)
{
    QPixmap p = KIcon(pixmap).pixmap(d->osd->pixmapSize());
    d->osd->show(p, text, clamp(percent, 0, 100));
}


void KOsdD::parseConfiguration()
{
    if (d->osd)
        delete d->osd;

    Configuration::self()->readConfig();
    if (Configuration::self()->type() == 0)
        d->osd = new StandardOsd();
    else
        d->osd = new CompactOsd();

    d->osd->setOpacity(Configuration::self()->opacity());
    d->osd->setPosition(Configuration::self()->hPos(), Configuration::self()->vPos());
    d->osd->setPrimaryScreenLock(Configuration::self()->primaryScreen());
    d->osd->setTimeout(1000*Configuration::self()->timeout());

    d->svg->resize(d->osd->pixmapSize());

    QStringList elements;

    for (int i = 0; i < KOsdDPrivate::BatteryPixmapCount; ++i)
    {
        if (d->batteryPixmaps[i])
            delete d->batteryPixmaps[i];

        elements << "Battery";
        elements << KOsdDPrivate::batteryFillElements[i][0];
        elements << KOsdDPrivate::batteryFillElements[i][1];
        d->batteryPixmaps[i] = createPixmap("icons/battery", elements);
        elements.clear();
    }

    for (int i = 0; i < KOsdDPrivate::VolumePixmapCount; ++i)
    {
        if (d->volumePixmaps[i])
            delete d->volumePixmaps[i];
    
        elements << KOsdDPrivate::volumeElements[i];
        d->volumePixmaps[i] = createPixmap("icons/audio", elements);
        elements.clear();
    }

    if (d->brightnessPixmap)
        delete d->brightnessPixmap;

    // There's no themed brightness icon (yet?)
    d->brightnessPixmap = createPixmap("icons/preferences", elements);
}


QPixmap* KOsdD::createPixmap(const QString &svg, const QStringList &elements)
{
    QPixmap *pixmap = new QPixmap(d->osd->pixmapSize());
    paintSvg(svg, elements, pixmap);
    return pixmap;
}


void KOsdD::paintSvg(const QString &svg, const QStringList &elements, QPixmap *pixmap)
{
    pixmap->fill(Qt::transparent);

    QPainter painter(pixmap);
    d->svg->setImagePath(svg);

    if (elements.isEmpty())
        d->svg->paint(&painter, 0, 0);
    else
    {
        QString element;
        foreach (element, elements)
        {
            if (d->svg->hasElement(element))
                d->svg->paint(&painter, 0, 0, element);
        }
    }
}

#include "kosdd.moc"
