package latexDraw.ui.dialog;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import latexDraw.figures.*;
import latexDraw.figures.properties.Arrowable;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.DrawPanel;
import latexDraw.ui.components.ButtonIcon;
import latexDraw.ui.components.ColorButton;
import latexDraw.ui.components.LaTeXDrawComboBox;
import latexDraw.util.LaTeXDrawPoint2D;
import latexDraw.util.LaTeXDrawResources;
import latexDraw.util.UndoRedoManager;


/** 
 * This class defines the pattern for the frames which allow to change the parameters of the figures.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE.  See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public abstract class AbstractParametersFrame extends JDialog implements ActionListener, ItemListener, KeyListener, ChangeListener
{
	private static final long	serialVersionUID	= 1L;

	protected boolean deleteOnCancel;
	
	/** The panel containing the drawing */
	protected DrawPanel drawPanel;
	
	/** The figure to modify */
	protected Figure figure;
	
	/** The title of the frame */
	protected String title;
	
	/** The width of many panels. */
	public static final int WIDTH_PANELS = 430;
	
	/** The label of the panel : Position/dimensions. */
	public static final String LABEL_POSITION_DIMENSIONS = LaTeXDrawLang.getString1_7("AbstractParametersFrame.0"); //$NON-NLS-1$
	
	/** The name of the button ok */
	public static final String NAME_BUTTON_OK = "BUTTON_OK";//$NON-NLS-1$
	
	/** The name of the button cancel */
	public static final String NAME_BUTTON_CANCEL = "BUTTON_CANCEL";//$NON-NLS-1$
	
	/** The name of the button default */
	public static final String NAME_BUTTON_DEFAULT = "BUTTON_DEFAULT";//$NON-NLS-1$
	
	/** Allows to know if the frame has been launched by the main frame:
	 * if true, the undo/redo manager will be used. */
	protected boolean attachToMainFrame;
	
	protected JSpinner dotSizeNum;
	
	protected JSpinner dotSizeDim;
	
	protected JSpinner bracketNum;
	
	protected JSpinner rbracketNum;
	
	protected JSpinner tbarsizeNum;
	
	protected JSpinner tbarsizeDim;
	
	protected JSpinner arrowSizeDim;
	
	protected JSpinner arrowSizeNum;
	
	protected JSpinner arrowLength;
	
	protected JSpinner arrowInset;
	
	public static final String LABEL_DOTSIZENUM = "DotSizeNum";//$NON-NLS-1$
	
	public static final String LABEL_DOTSIZEDIM = "DotSizeDim";//$NON-NLS-1$
	
	public static final String LABEL_BRACKETNUM = "BracketNum";//$NON-NLS-1$
	
	public static final String LABEL_RBRACKETNUM = "rBracketNum";//$NON-NLS-1$
	
	public static final String LABEL_TBARSIZENUM = "tBarSizeNum"; //$NON-NLS-1$
	
	public static final String LABEL_TBARSIZEDIM = "tBarSizeDim"; //$NON-NLS-1$
	
	public static final String LABEL_ARROWLENGTH = "arrowLength";//$NON-NLS-1$
	
	public static final String LABEL_ARROWINSET = "arrowInset"; //$NON-NLS-1$
	
	public static final String LABEL_ARROWSIZENUM = "arrowSizeNum";//$NON-NLS-1$
	
	public static final String LABEL_ARROWSIZEDIM = "arrowSizeDim"; //$NON-NLS-1$
	
	/** Allows to change the mid point of the gradient. */
	protected JSpinner gradientMidPtField;
	
	/** Allows to change the angle of the gradient. */
	protected JSpinner gradientAngleField;
	
	/** Allows to change the colour of the shadow. */
	protected ColorButton gradientStartColButton;
	
	/** Allows to change the colour of the shadow. */
	protected ColorButton gradientEndColButton;
	
	/** Allows to change the size of the shadow. */
	protected JSpinner shadowSizeField;
	
	/** Allows to change the colour of the shadow. */
	protected ColorButton shadowColorButton;
	
	/** Allows to change the angle of the shadow. */
	protected JSpinner shadowAngleField;
	
	/** Allows to set if the figure has a shadow or not. */
	protected JCheckBox shadowCheckBox;
	
	/** Allows to change the thickness of the figure */
	protected JSpinner thicknessField;
	
	/** Allows to change the colour of the borders of the figure */
	protected ColorButton bordersButton;
	
	/** Allows to change the colour of the hatch */
	protected ColorButton hatchButton;
	
	/** Allows to set if the figure is filled */
	protected JCheckBox filledCheckBox;
	
	/** Allows to change the colour of the interior of the figure */
	protected ColorButton interiorButton;
	
	/** Allows to change the colour of the double boundary of the figure */
	protected ColorButton doublecolorButton;
	
	/** Allows to change the kind of the line */
	protected LaTeXDrawComboBox lineChoice;
	
	/** Allows to change the kind of hatch */
	protected LaTeXDrawComboBox hatchChoice;
	
	/** Allows to change the position of the borders */
	protected LaTeXDrawComboBox bordersPositionChoice;
	
	/** Allows to change the angle of rotation of the figure */
	protected JSpinner hatchSepField;

	protected JSpinner hatchAngleField;
	
	/** Allows to change the angle of rotation of the figure */
	protected JSpinner rotationField;
	
	/** Allows to change the width of the hatch */
	protected JSpinner hatchWidthField;
	
	/** Allows to set the separation between the double line */
	protected JSpinner doubleSep;
	
	/** Allows to set if the figure must have double boundary or not */
	protected JCheckBox doubleLine;
	
	/** The button ok */
	protected JButton buttonOk;
	
	/** The panel where the figure is displayed */
	protected GlimpsePanel glimpsePanel;
	
	/** This scrollpane contains the glimpsepanel */
	protected JPanel glimpsePanelZoom;
	
	/** The scroll pane of the frame to see the figure. */
	protected JScrollPane glimpseScrollPane;
	
	/** The first arrow choice */
	protected LaTeXDrawComboBox arrow1Choice;
	
	/** The second arrow choice */
	protected LaTeXDrawComboBox arrow2Choice;
	
	/** Defines if the shape has been modified with the frame. @since 2.0.0 */
	protected boolean isModified;
	
	/** The label of the check box which set if the figure has a shadow or not. */
	public static final String LABEL_SHADOW_CB = LaTeXDrawResources.LABEL_SHADOW_CHECKBOX;
	
	/** The label of the button which set the angle of the shadow. */
	public static final String LABEL_SHADOW_ANGLE = LaTeXDrawLang.getString1_7("AbstractParametersFrame.1"); //$NON-NLS-1$
	
	/** The label of the button which set the size of the shadow. */
	public static final String LABEL_SHADOW_SIZE = LaTeXDrawLang.getString1_7("AbstractParametersFrame.2"); //$NON-NLS-1$
	
	/** The label of the button which set the angle of the gradient. */
	public static final String LABEL_GRADIENT_ANGLE = LaTeXDrawLang.getString1_7("AbstractParametersFrame.3"); //$NON-NLS-1$

	public static final String LABEL_HATCHINGS_ANGLE = "hatchAngle";
	
	/** The label of the button which set the middle point of the gradient. */
	public static final String LABEL_GRADIENT_MID_PT = LaTeXDrawLang.getString1_7("AbstractParametersFrame.4"); //$NON-NLS-1$
	
	public static final String TITLE_TABBED_PANE_GENERAL = LaTeXDrawLang.getDialogFrameString("PreferencesFrame.general"); //$NON-NLS-1$
	
	public static final String TITLE_TABBED_PANE_OTHERS = LaTeXDrawLang.getString1_7("AbstractParametersFrame.6"); //$NON-NLS-1$
	
	public static final String LABEL_BUTTON_BOUNDARY = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.0"); //$NON-NLS-1$
	
	public static final String LABEL_DOUBLELINE = LaTeXDrawLang.getString1_7("AbstractParametersFrame.9"); //$NON-NLS-1$
	
	public static final String LABEL_DOUBLESEP = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.2"); //$NON-NLS-1$
	
	public static final String LABEL_HATCH_SEP = LaTeXDrawLang.getString1_8("AbstractParametersFrame.0"); //$NON-NLS-1$
	
	public static final String TITLE_GENERAL = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.3"); //$NON-NLS-1$
	
	public static final String LABEL_ARROWS = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.38"); //$NON-NLS-1$
	
	public static final String LABEL_OK = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.4"); //$NON-NLS-1$
	
	public static final String LABEL_DEFAULT = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.5"); //$NON-NLS-1$
	
	public static final String LABEL_CANCEL = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.6"); //$NON-NLS-1$
	
	public static final String LABEL_ROTATION = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.7"); //$NON-NLS-1$
	
	public static final String LABEL_THICKNESS = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.8"); //$NON-NLS-1$
	
	public static final String LABEL_GENERAL = LaTeXDrawLang.getDialogFrameString("PreferencesFrame.general"); //$NON-NLS-1$
	
	public static final String LABEL_TYPE_LINE = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.10"); //$NON-NLS-1$
	
	public static final String LABEL_HATCH_WIDTH = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.11"); //$NON-NLS-1$
	
	public static final String LABEL_LEFT_ARROW = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.12"); //$NON-NLS-1$
	
	public static final String LABEL_RIGHT_ARROW = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.13"); //$NON-NLS-1$
	
	public static final String NAME_TABBEDPANE_ARROW = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.14"); //$NON-NLS-1$
	
	/** The label of the showPointsCB JCheckBox */
	public static final String LABEL_SHOWPOINTCB = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.15"); //$NON-NLS-1$
	
	/** The label of the isRound checkbox */
	public static final String LABEL_ISROUND = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.16"); //$NON-NLS-1$
	
	/** The label of the field frameArcField */
	public static final String LABEL_ROUND_CORNER = LaTeXDrawLang.getString1_7("AbstractParametersFrame.7"); //$NON-NLS-1$
	
	/** The label of the X-centre field */
	public static final String LABEL_CENTER_X = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.17"); //$NON-NLS-1$
	
	/** The label of the Y-centre field */
	public static final String LABEL_CENTER_Y = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.18"); //$NON-NLS-1$
	
	/** The label of the radius field */
	public static final String LABEL_RADIUS = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.19"); //$NON-NLS-1$
	
	public static final String LABEL_DOTSIZE = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.20"); //$NON-NLS-1$
	
	public static final String LABEL_TBARSIZE = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.21"); //$NON-NLS-1$
	
	public static final String LABEL_BRACKET_LGTH = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.22"); //$NON-NLS-1$
	
	public static final String LABEL_RBRACKET_LGTH  = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.23"); //$NON-NLS-1$
	
	public static final String LABEL_CROSS_THICKNESS = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.24"); //$NON-NLS-1$
	
	public static final String LABEL_CROSS_WIDTH = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.25"); //$NON-NLS-1$
	
	public static final String LABEL_ARROW_LENGTH = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.26"); //$NON-NLS-1$
	
	public static final String LABEL_ARROW_INSET = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.27"); //$NON-NLS-1$
	
	public static final String LABEL_ARROW_WIDTH = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.28"); //$NON-NLS-1$
	
	public static final String LABEL_X = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.29"); //$NON-NLS-1$
	
	public static final String LABEL_Y = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.30"); //$NON-NLS-1$
	
	/** The label of the width field */
	public static final String LABEL_WIDTH = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.31"); //$NON-NLS-1$
	
	/** The label of the height field */
	public static final String LABEL_HEIGHT = LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.32"); //$NON-NLS-1$
	
	
	
	 /**
	 * 01/20/06<br>
	 * @version 1.9<br>
	 */
	protected static class GlimpsePanel extends JPanel implements Scrollable, MouseListener, MouseMotionListener
	{
		private static final long serialVersionUID = 1L;
		
		/** The figure to display in the panel */
		protected Figure glimpseFigure;
		
		/** The zoom by default (100%) */
		public static final double DEFAULT_ZOOM = 1.;
		
		/** The biggest zoom we can do */
		public static final double MAX_ZOOM = 4.5;
		
		/** The smallest zoom we can do */
		public static final double MIN_ZOOM = 0.25;	
		   
		/** The value we add to the zoom when to user click on a zoom button */
		public static final double DEFAULT_ADD_ZOOM = 0.25;
		
		/** The zoom : MIN_ZOOM<=zoom<=MAX_ZOOM */
		protected double zoom;
		
		private transient Cursor formerCursor;

		private transient Point formerPoint;
		
		private transient double tx;
		
		private transient double ty;
		
		
		/**
		 * The constructor by default
		 * @param f The figure to display (can be null)
		 */
		public GlimpsePanel(Figure f)
		{
			if(f!=null)
				f.setSelected(false);
			glimpseFigure = f;
			zoom = DEFAULT_ZOOM;
			formerCursor = null;
			formerPoint = null;
			tx = ty = 0.;
			addMouseListener(this);
			addMouseMotionListener(this);
		}
		
		
		/**
		 * Allows to zoom out the drawing (up to 25%).
		 */
		public void zoomOut()
		{
			if(zoom-DEFAULT_ADD_ZOOM>=MIN_ZOOM && glimpseFigure!=null)
			{
				zoom-=DEFAULT_ADD_ZOOM;
				updateTxTy();
				repaint();
				updatePreferredSize();
				revalidate();
			}
		}
		
		
		/**
		 * Allows to set the zoom to default. In fact the zoom is not set at ZOOM_DEFAULT, it's set to see the whole figure.
		 */
		public void zoomDefault()
		{
			if(glimpseFigure!=null)
			{
				LaTeXDrawPoint2D nw = glimpseFigure.getTheNWBoundPoint();
				LaTeXDrawPoint2D se = glimpseFigure.getTheSEBoundPoint();
				double figWidth  = se.x-nw.x;
				double figHeight = se.y-nw.y;
				int height = getParent().getParent().getHeight();
				int width  = getParent().getParent().getWidth();
				zoom = DEFAULT_ZOOM;
				
				while(((figHeight*zoom)>height || (figWidth*zoom)>width) && zoom>MIN_ZOOM)
				{
					zoom-=DEFAULT_ADD_ZOOM;
				}
				
				if(zoom<MIN_ZOOM)
					zoom = MIN_ZOOM;
				
				updateTxTy();
				repaint();
				updatePreferredSize();
				revalidate();
			}
		}
		
		
		/**
		 * Allows to zoom in the drawing (up to 450%)
		 */
		public void zoomIn()
		{
			if(zoom+DEFAULT_ADD_ZOOM<=MAX_ZOOM && glimpseFigure!=null)
			{
				zoom+=DEFAULT_ADD_ZOOM;
				updateTxTy();
				updatePreferredSize();
				repaint();
				revalidate(); 
			}
		}
		
		
		
		/**
		 * Updates the fields tx and ty.
		 * @since 1.9.2
		 */
		public void updateTxTy()
		{
			LaTeXDrawPoint2D nw = glimpseFigure.getTheNWBoundPoint();
			
			tx = getParent().getParent().getWidth()/2.-glimpseFigure.getGravityCenter().x*zoom;
			ty = getParent().getParent().getHeight()/2.-glimpseFigure.getGravityCenter().y*zoom;
			
			if(nw.x*zoom+tx<0)
				tx = Math.min(-nw.x*zoom+30, -30);
			
			if(nw.y*zoom+ty<0)
				ty = Math.min(-nw.y*zoom+30, -30);
		}
		
		
		
		/**
		 * Allows to set the figure to display
		 * @param f The new figure to display
		 */
		public void setGlimpseFigure(Figure f)
		{
			glimpseFigure = f;
			
			if(f!=null)
			{
				f.setSelected(false);
				zoomDefault();
			}
			else
				tx = ty = 0.;
		}
		
		
		
		/**
		 * Allows to get the figure to display.
		 * @return The glimpse figure.
		 */
		public Figure getGlimpseFigure()
		{
			return glimpseFigure;
		}
		
		
		
		@Override
		public void paintComponent(Graphics g)
		{
	    	if(g==null || glimpseFigure==null) return;

			Graphics2D g2 = (Graphics2D) g;
	    	g2.setColor(Color.WHITE);
	    	g2.fillRect(0, 0, getWidth(), getHeight());
			g2.translate(tx, ty);
			g2.scale(zoom, zoom);

			g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
			g2.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
			g2.setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_QUALITY);
			g2.setRenderingHint(RenderingHints.KEY_COLOR_RENDERING, RenderingHints.VALUE_COLOR_RENDER_QUALITY);
			g2.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
			
			glimpseFigure.draw(g2, RenderingHints.VALUE_ANTIALIAS_ON, RenderingHints.VALUE_RENDER_QUALITY, 
								RenderingHints.VALUE_ALPHA_INTERPOLATION_QUALITY, RenderingHints.VALUE_COLOR_RENDER_QUALITY);
		}


		
		
		/**
	     * Allows to update the dimensions of the drawing (needed for the scrollers)
	     */
		public void updatePreferredSize()
		{ 
			if(glimpseFigure==null)
				setPreferredSize(new Dimension(0,0));
			else
			{
				LaTeXDrawPoint2D SE = glimpseFigure.getTheSEBoundPoint();
				setPreferredSize(new Dimension((int)(SE.x*zoom+tx), (int)(SE.y*zoom+ty)));
			}
		}
		

		
		@Override
		public void repaint()
		{
			updatePreferredSize();
			revalidate();
			super.repaint();		
		}
		
		
		
		/**
		 * Useful for scrollbar
		 */
		public Dimension getPreferredScrollableViewportSize()
		{	return new Dimension(-100, 100);  }
		
		

		/**
		 * Useful for scrollbar
		 */
		public int getScrollableBlockIncrement(Rectangle visibleRect, int orientation, int direction) 
		{
	        return (orientation == SwingConstants.VERTICAL) ? visibleRect.height :
	            visibleRect.width;
		}
		
		

		/**
		 * Useful for scrollbar
		 */
		public boolean getScrollableTracksViewportHeight()
		{
			if(this.getParent() instanceof JViewport) 
		    	return (((JViewport)getParent()).getHeight() > getPreferredSize().height);
			return false;
		}
		

		
		
		/**
		 * Useful for scrollbar
		 */
		public boolean getScrollableTracksViewportWidth()
		{
			if(getParent() instanceof JViewport) 
		    	return (((JViewport)getParent()).getWidth() > getPreferredSize().width);
			return false;
		}
		

		
		/**
		 * Useful for scrollbar
		 */
		public int getScrollableUnitIncrement(Rectangle visibleRect, int orientation, int direction)  
		{
			return 4;
		}


		public void mouseClicked(MouseEvent e)
		{
			// Nothing to do.
		}


		public void mouseEntered(MouseEvent e)
		{
			formerCursor = getCursor();
			setCursor(Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
		}


		public void mouseExited(MouseEvent e)
		{
			if(formerCursor!=null)
			{
				setCursor(formerCursor);
				formerCursor = null;
			}
		}


		public void mousePressed(MouseEvent e)
		{
			// Nothing to do.
		}


		public void mouseReleased(MouseEvent e)
		{
			formerPoint = null;
		}


		public void mouseDragged(MouseEvent e)
		{
			if(formerPoint!=null)
			{
				tx += e.getPoint().x-formerPoint.x;
				ty += e.getPoint().y-formerPoint.y;
				repaint();
			}
			
			formerPoint = e.getPoint();
		}


		public void mouseMoved(MouseEvent e)
		{
			// Nothing to do.
		}
	}
	
	
	
	/**
	 * The abstract constructor
	 * @param d The drawpanel
	 */
	protected AbstractParametersFrame(Frame parent, DrawPanel d, boolean attachToMainFrame)
	{
		super(parent, true);
		try
		{	
			setModified(false);
			this.attachToMainFrame = attachToMainFrame;
			drawPanel = d;
			glimpsePanel = new GlimpsePanel(null);
			glimpseScrollPane = new JScrollPane(glimpsePanel);
			glimpsePanel.updatePreferredSize();
			glimpsePanel.revalidate();
	     	glimpsePanelZoom = new JPanel(new BorderLayout());
	     	glimpsePanelZoom.add(glimpseScrollPane, BorderLayout.CENTER);
	     	glimpsePanelZoom.add(createZoomPanel(), BorderLayout.EAST);
	     	glimpsePanelZoom.setBorder(new CompoundBorder(new TitledBorder(null, "", //$NON-NLS-1$
					  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
	     	glimpsePanelZoom.setPreferredSize(new Dimension(420, 190));
	     	glimpsePanelZoom.setMaximumSize(new Dimension(700, 240));
	     	createRotationField();

	     	setResizable(true);
	     	
	     	addWindowListener(
	                new WindowAdapter() 
	                {
						@Override
						public void windowClosing(WindowEvent e) 
	                    {
							if(deleteOnCancel)
							{
								drawPanel.getDraw().deleteFigure(figure, null);
								deleteOnCancel = false;
							}
	                    }
	                }); 
	     	
		}catch(Exception e)
		{
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	
	
	
	
	/**
	 * Allows to create a panel containing buttons to zoom the 'glimpse figure'.
	 * @return The created panel.
	 */
	public JPanel createZoomPanel()
	{
		JPanel panel = new JPanel();
		JPanel panel2 = new JPanel();
		panel.setLayout(new BoxLayout(panel, BoxLayout.X_AXIS));
		panel2.setLayout(new BoxLayout(panel2, BoxLayout.Y_AXIS));
		
		JButton zoomIn = new JButton(LaTeXDrawResources.zoomInIcon);
		zoomIn.setActionCommand(LaTeXDrawResources.LABEL_ZOOM_IN);
		zoomIn.addActionListener(this);
		
		JButton zoomOut = new JButton(LaTeXDrawResources.zoomOutIcon);
		zoomOut.setActionCommand(LaTeXDrawResources.LABEL_ZOOM_OUT);
		zoomOut.addActionListener(this);
		
		JButton zoomDefault = new JButton(LaTeXDrawResources.zoomDefaultIcon);
		zoomDefault.setActionCommand(LaTeXDrawResources.LABEL_ZOOM_DEFAULT);
		zoomDefault.addActionListener(this);
		
		panel2.add(zoomDefault);
		panel2.add(zoomIn);
		panel2.add(zoomOut);
		panel.add(panel2);
		
		return panel;
	}
	
	
	
	
	/**
	 * Allows to set the title of the frame
	 */
	public void setTitle()
	{
		setTitle(LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.33")+ " #" +figure.getNumber()); //$NON-NLS-1$ //$NON-NLS-2$
	}
	
	
	
	
	/**
	 * Allows to save the values of the fields in the attributes of the figure.
	 */
	public void saveParameters()
	{
		try
		{
			if(attachToMainFrame && !deleteOnCancel)
			{
				drawPanel.getFrameParent().getUndoManager().add(UndoRedoManager.LABEL_CHANGE_PARAM, 
						drawPanel.getDraw().getFigures().indexOf(figure), figure.clone());
				drawPanel.getFrameParent().updateUndoRedo();
			}
			
			if(figure.canHaveArrow())
			{
				ArrowHead leftA  = ((Arrowable)figure).getArrowHead1();
				ArrowHead rightA = ((Arrowable)figure).getArrowHead2();
				
				JLabel lab = (JLabel)arrow1Choice.getSelectedItem();
				((Arrowable)figure).setArrow1Style(lab.getText());
				lab = (JLabel)arrow2Choice.getSelectedItem();
				((Arrowable)figure).setArrow2Style(lab.getText());
				
				leftA.setBracketNum(Double.valueOf(bracketNum.getValue().toString()).doubleValue());
				rightA.setBracketNum(Double.valueOf(bracketNum.getValue().toString()).doubleValue());
				leftA.setDotSizeDim(Double.valueOf(dotSizeDim.getValue().toString()).doubleValue());
				rightA.setDotSizeDim(Double.valueOf(dotSizeDim.getValue().toString()).doubleValue());
				leftA.setDotSizeNum(Double.valueOf(dotSizeNum.getValue().toString()).doubleValue());
				rightA.setDotSizeNum(Double.valueOf(dotSizeNum.getValue().toString()).doubleValue());
				leftA.setRBracketNum(Double.valueOf(rbracketNum.getValue().toString()).doubleValue());
				rightA.setRBracketNum(Double.valueOf(rbracketNum.getValue().toString()).doubleValue());
				leftA.setTBarSizeDim(Double.valueOf(tbarsizeDim.getValue().toString()).doubleValue());
				rightA.setTBarSizeDim(Double.valueOf(tbarsizeDim.getValue().toString()).doubleValue());
				leftA.setTBarSizeNum(Double.valueOf(tbarsizeNum.getValue().toString()).doubleValue());
				rightA.setTBarSizeNum(Double.valueOf(tbarsizeNum.getValue().toString()).doubleValue());		
				leftA.setArrowSizeNum(Double.valueOf(arrowSizeNum.getValue().toString()).doubleValue());
				rightA.setArrowSizeNum(Double.valueOf(arrowSizeNum.getValue().toString()).doubleValue());
				leftA.setArrowSizeDim(Double.valueOf(arrowSizeDim.getValue().toString()).doubleValue());
				rightA.setArrowSizeDim(Double.valueOf(arrowSizeDim.getValue().toString()).doubleValue());
				leftA.setArrowLength(Double.valueOf(arrowLength.getValue().toString()).doubleValue());
				rightA.setArrowLength(Double.valueOf(arrowLength.getValue().toString()).doubleValue());
				leftA.setArrowInset(Double.valueOf(arrowInset.getValue().toString()).doubleValue());
				rightA.setArrowInset(Double.valueOf(arrowInset.getValue().toString()).doubleValue());
			}
			
			if(doublecolorButton!=null)
				figure.setDoubleColor(doublecolorButton.getColor());
			
			if(bordersPositionChoice!=null)
			{
				JLabel l = (JLabel) bordersPositionChoice.getSelectedItem();
				figure.setBordersPosition(l.getText());
			}
			
			if(doubleLine!=null)
			{
				figure.setHasDoubleBoundary(doubleLine.isSelected());
				figure.setDoubleSep(Double.valueOf(doubleSep.getValue().toString()));
			}
			
			if(lineChoice!=null)
			{
				JLabel l = (JLabel) lineChoice.getSelectedItem();
				figure.setLineStyle(l.getText());
			}
			
			if(thicknessField!=null)
				figure.setThickness(Float.valueOf(thicknessField.getValue().toString()));
			
			if(bordersButton!=null)
				figure.setLinesColor(bordersButton.getColor());
			
			if(interiorButton!=null)
				figure.setInteriorColor(interiorButton.getColor());
			
			if(figure.canHaveShadow())
			{
				figure.setHasShadow(shadowCheckBox.isSelected());
				figure.setShadowSize(Double.valueOf(shadowSizeField.getValue().toString()).doubleValue());
				figure.setShadowAngle(Math.toRadians(Double.valueOf(shadowAngleField.getValue().toString()).doubleValue()));
				figure.setShadowColor(shadowColorButton.getColor());
			}
			
			if(filledCheckBox!=null)
				figure.setIsFilled(filledCheckBox.isSelected());
			
			if(figure.canBeHatched())
			{
				figure.setHatchingAngle(Math.toRadians(Double.valueOf(hatchAngleField.getValue().toString()).doubleValue()));
				figure.setHatchingSep(Float.valueOf(hatchSepField.getValue().toString()));
				figure.setHatchingWidth(Float.valueOf(hatchWidthField.getValue().toString()));
				figure.setHatchingColor(hatchButton.getColor());
				JLabel l = (JLabel) hatchChoice.getSelectedItem();
				figure.setHatchingStyle(l.getText());
				figure.setGradientAngle(Math.toRadians(Double.valueOf(gradientAngleField.getValue().toString()).doubleValue()));
				figure.setGradientEndColor(((ButtonIcon)gradientEndColButton.getIcon()).getColor());
				figure.setGradientStartColor(((ButtonIcon)gradientStartColButton.getIcon()).getColor());
				figure.setGradientMidPoint(Double.valueOf(gradientMidPtField.getValue().toString()).doubleValue());
			}
			
			if(rotationField!=null)
				figure.setRotationAngle(Math.toRadians(
						Double.valueOf(rotationField.getValue().toString()).doubleValue()));
			
			drawPanel.getDraw().updateBorders();
			drawPanel.updateDraw(true);
			
		}catch(Exception e)
		{
			e.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	
	
	

	public void keyTyped(KeyEvent e) 
	{
		/* No action to do */
	}


	public void keyReleased(KeyEvent e) 
	{
		/* No action to do */
	}


	public void keyPressed(KeyEvent e) 
	{
		/* No action to do */
	}


	
	
	public void actionPerformed(ActionEvent e) 
	{
		Object o = e.getSource();
		
		try
		{
			if(o instanceof JButton || o instanceof JCheckBox)
			{
				String label = ((AbstractButton)o).getActionCommand();
				Figure f = glimpsePanel.getGlimpseFigure();
				
	    		if(label.equals(LaTeXDrawResources.LABEL_ZOOM_IN))
	    		{
	    			glimpsePanel.zoomIn();
	    			return;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_ZOOM_OUT))
	    		{
	    			glimpsePanel.zoomOut();
	    			return;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_ZOOM_DEFAULT))
	    		{
	    			glimpsePanel.zoomDefault();
	    			return;
	    		}
				
				if(label.equals(LABEL_DOUBLELINE))
				{
					setModified(true);
					f.setHasDoubleBoundary(doubleLine.isSelected());
					doubleSep.setEnabled(doubleLine.isSelected());
					doublecolorButton.setEnabled(doubleLine.isSelected());
					glimpsePanel.repaint();
					return ;
				}
				
				if(label.equals(NAME_BUTTON_CANCEL))
				{
					if(deleteOnCancel)
					{
						drawPanel.getDraw().deleteFigure(figure, null);
						deleteOnCancel = false;
					}
					super.setVisible(false);
					return;
				}
				
				if(label.equals(NAME_BUTTON_DEFAULT))
				{					
					setDefaultsValues();
					glimpsePanel.repaint();
					return;
				}
				
				if(label.equals(NAME_BUTTON_OK))
				{
					saveParameters();
					
					drawPanel.getFrameParent().updateFrameFields();
					
					if(isModified())
						drawPanel.setIsModified(true);
					
					setVisible(false);
					return;
				}
				
				
				if(label.equals(LaTeXDrawResources.LABEL_FILLED_CHECKBOX))
	    		{
					if(interiorButton!=null)
					{
						setModified(true);
						f.setIsFilled(filledCheckBox.isSelected());
						glimpsePanel.repaint();
					}
	    			return;
	    		}
				
				if(label.equals(LaTeXDrawResources.LABEL_SHADOW_CHECKBOX))
	    		{
					if(shadowColorButton!=null)
					{
						setModified(true);
						shadowColorButton.setEnabled(shadowCheckBox.isSelected());
						shadowAngleField.setEnabled(shadowCheckBox.isSelected());
						shadowSizeField.setEnabled(shadowCheckBox.isSelected());
						f.setHasShadow(shadowCheckBox.isSelected());
						updateFilledCB();
						glimpsePanel.repaint();
					}
	    			return;
	    		}
				
				if(label.equals(LaTeXDrawResources.LABEL_COLOR_GRADIENT_FIRST))
				{
					Color c = JColorChooser.showDialog((Component) e.getSource(), 
							LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.34"),  //$NON-NLS-1$
							((ColorButton)o).getColor());
						
					if(c!=null)
					{
						setModified(true);
						((ColorButton)o).setColor(c);
						f.setGradientStartColor(c);
						glimpsePanel.repaint();
					}
					return ;
				}
				
				if(label.equals(LaTeXDrawResources.LABEL_COLOR_GRADIENT_SECOND))
				{
					Color c = JColorChooser.showDialog((Component) e.getSource(), 
							LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.34"),  //$NON-NLS-1$
							((ColorButton)o).getColor());
						
					if(c!=null)
					{
						setModified(true);
						((ColorButton)o).setColor(c);
						f.setGradientEndColor(c);
						glimpsePanel.repaint();
					}
					return ;
				}
				
				if(label.equals(LaTeXDrawResources.LABEL_COLOR_HATCH))
				{
					Color c = JColorChooser.showDialog((Component) e.getSource(), 
							LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.34"),  //$NON-NLS-1$
							((ColorButton)o).getColor());
						
					if(c!=null)
					{
						setModified(true);
						((ColorButton)o).setColor(c);
						f.setHatchingColor(c);
						glimpsePanel.repaint();
					}
					return ;
				}
				
				if(label.equals(LaTeXDrawResources.NAME_COLOR_SHADOW))
				{
					Color c = JColorChooser.showDialog((Component) e.getSource(), 
							LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.34"),  //$NON-NLS-1$
							((ColorButton)o).getColor());
						
					if(c!=null)
					{
						setModified(true);
						((ColorButton)o).setColor(c);
						f.setShadowColor(c);
						glimpsePanel.repaint();
					}
					return ;
				}
				
				if(label.equals(LABEL_BUTTON_BOUNDARY))
				{
					Color c = JColorChooser.showDialog((Component) e.getSource(), 
							LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.35"),  //$NON-NLS-1$
							((ColorButton)o).getColor());
						
					if(c!=null)
					{
						setModified(true);
						((ColorButton)o).setColor(c);
						f.setDoubleColor(c);
						glimpsePanel.repaint();
					}
					return ;
				}
				
				if(label.equals(LaTeXDrawResources.LABEL_COLOR_BUTTON))
				{
					Color c = JColorChooser.showDialog((Component) e.getSource(), 
							LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.36"),  //$NON-NLS-1$
							((ColorButton)o).getColor());
						
					if(c!=null)
					{
						setModified(true);
						((ColorButton)o).setColor(c);
						f.setLinesColor(c);
						glimpsePanel.repaint();
					}
					return ;
				}
				
				if(label.equals(LaTeXDrawResources.LABEL_COLOR_INTERIOR_BUTTON))
				{ 
					Color c = JColorChooser.showDialog((Component) e.getSource(), 
					LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.37"),  //$NON-NLS-1$
					((ColorButton)o).getColor());
				
					if(c!=null)
					{
						setModified(true);
						((ColorButton)o).setColor(c);
						f.setInteriorColor(c);
						glimpsePanel.repaint();
					}
					return;
				}
			} // if(o instanceof JButton)
		}catch(Exception ex) 
		{ 
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
	
	
	
	
	/**
	 * Allows to set the value of the specific fields of the frame
	 * from the attributes of the figure
	 */
	public abstract void setFigureFrameField();
	
	
	
	
	/**
	 * Initialises some fields of the frame and set visible or not it.
	 * @param visible True: the frame must be visible.
	 * @param selected The selected figure.
	 * @param deleteOnCancel True: the figure must be deleted if the user click on the button cancel.
	 */
	public void setVisible(boolean visible, Figure selected, boolean deleteOnCancel, boolean isFramedBox)
	{		
		try
		{
			if(visible)
			{
				setModified(false);
				this.deleteOnCancel = deleteOnCancel;
				figure = selected;
				setTitle();

				glimpsePanel.setGlimpseFigure((Figure) selected.clone());
				glimpsePanel.updatePreferredSize();
				glimpsePanel.revalidate();
				glimpsePanel.repaint();
				
				setFigureFrameField();
				
				if(figure.canHaveArrow())
				{
					arrow1Choice.setSelectedItem(((Arrowable)figure).getArrow1Style());
					arrow2Choice.setSelectedItem(((Arrowable)figure).getArrow2Style());
					
					ArrowHead rightA = ((Arrowable)figure).getArrowHead2();
					
					arrowInset.setValue(rightA.getArrowInset());
					arrowLength.setValue(rightA.getArrowLength());
					arrowSizeDim.setValue(rightA.getArrowSizeDim());
					arrowSizeNum.setValue(rightA.getArrowSizeNum());
					bracketNum.setValue(rightA.getBracketNum());
					dotSizeDim.setValue(rightA.getDotSizeDim());
					dotSizeNum.setValue(rightA.getDotSizeNum());
					rbracketNum.setValue(rightA.getRBracketNum());
					tbarsizeDim.setValue(rightA.getTBarSizeDim());
					tbarsizeNum.setValue(rightA.getTBarSizeNum());

					updateEnableParamsArrow();
				}
				
				if(doublecolorButton!=null)
				{
					doublecolorButton.setEnabled(figure.hasDoubleBoundary());
					doublecolorButton.setColor(figure.getDoubleColor());
				}
				
				if(bordersPositionChoice!=null)
				{
					bordersPositionChoice.setEnabled(figure.isBordersMovable() && !isFramedBox);
					bordersPositionChoice.setSelectedItem(figure.getBordersPosition());
				}
				
				if(doubleLine!=null)
				{
					doubleLine.setSelected(figure.hasDoubleBoundary());
					doubleSep.setEnabled(doubleLine.isSelected());
					doubleSep.setValue(figure.getDoubleSep());
				}
				
				if(rotationField!=null)
				{
					rotationField.setEnabled(!isFramedBox);
					rotationField.setValue(Math.toDegrees(selected.getRotationAngle()));
				}
				
				if(lineChoice!=null)
					lineChoice.setSelectedItem(selected.getLineStyle());
				
				if(thicknessField!=null)
					thicknessField.setValue((double)selected.getThickness());
				
				if(bordersButton!=null)
					bordersButton.setColor(selected.getLinesColor());
				
				String txt = selected.getHatchingStyle();
				int num=0;
				if(txt.equals(PSTricksConstants.TOKEN_FILL_GRADIENT))
					num=1;
				else if(txt.equals(PSTricksConstants.TOKEN_FILL_NONE) ||
						txt.equals(PSTricksConstants.TOKEN_FILL_SOLID))
						num=2;
				
				if(selected.canBeHatched())
				{
					hatchAngleField.setEnabled(num==0);
					hatchAngleField.setValue(Math.toDegrees(selected.getHatchingAngle()));
					hatchChoice.setSelectedItem(selected.getHatchingStyle());
					hatchButton.setEnabled(num==0);
					hatchWidthField.setValue((double)selected.getHatchingWidth());
					hatchSepField.setValue(selected.getHatchingSep());
					hatchButton.setColor(selected.getHatchingColor());
					hatchWidthField.setEnabled(num==0);
					hatchSepField.setEnabled(num==0);
					gradientAngleField.setEnabled(num==1);
					gradientAngleField.setValue(Math.toDegrees(selected.getGradientAngle()));
					gradientEndColButton.setEnabled(num==1);
					gradientMidPtField.setEnabled(num==1);
					gradientMidPtField.setValue(selected.getGradientMidPoint());
					gradientStartColButton.setEnabled(num==1);
					updateFilledCB();
					filledCheckBox.setSelected(selected.isFilled() && selected.canBeFilled());
					gradientEndColButton.setColor(selected.getGradientEndColor());
					gradientStartColButton.setColor(selected.getGradientStartColor());
				}
				
				if(interiorButton!=null)
					interiorButton.setColor(selected.getInteriorColor());
				
				if(selected.canHaveShadow())
				{
					shadowSizeField.setValue(figure.getShadowSize());
					shadowAngleField.setValue(Math.toDegrees(figure.getShadowAngle()));
					shadowColorButton.setEnabled(figure.hasShadow());
					shadowColorButton.setColor(figure.getShadowColor());
					shadowCheckBox.setSelected(figure.hasShadow());
					shadowColorButton.setEnabled(shadowCheckBox.isSelected());
					shadowAngleField.setEnabled(shadowCheckBox.isSelected());
					shadowSizeField.setEnabled(shadowCheckBox.isSelected());
				}
			}
			updateFilledCB();
		
		}catch(Exception e)
		{
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
		super.setVisible(visible);
	}
	
	
	
	
	/**
	 * @return A panel with components to change the parameters of the hatching of the figure.
	 */
	public JPanel createHatchingPanel()
	{
  		GridBagConstraints constraint = new GridBagConstraints();
    	constraint.gridx = 0;  		constraint.gridy = 0;
     	constraint.gridwidth = 1; 	constraint.gridheight = 1;
     	constraint.weightx = 0.1; 	constraint.weighty = 0.1;
     	constraint.fill = GridBagConstraints.NONE;
     	constraint.anchor = GridBagConstraints.WEST;
     	
     	JPanel hatchingP = new JPanel(new GridBagLayout());
     	
     	hatchButton = new ColorButton(LaTeXDrawResources.LABEL_COLOR_HATCH, new ButtonIcon(Figure.DEFAULT_BORDERS_COL));
     	hatchButton.setActionCommand(LaTeXDrawResources.LABEL_COLOR_HATCH);
     	hatchButton.addActionListener(this);
     	hatchingP.add(hatchButton, constraint);      	
     	
     	constraint.gridx++;	
     	hatchingP.add(new JLabel(LABEL_HATCH_WIDTH), constraint);
     	
     	constraint.gridx++;
     	SpinnerModel model = new SpinnerNumberModel(Figure.DEFAULT_HATCH_WIDTH, 0.1, 100,0.1);
     	hatchWidthField = new JSpinner(model);
     	hatchWidthField.setEditor(new JSpinner.NumberEditor(hatchWidthField, "0.0"));//$NON-NLS-1$
     	hatchWidthField.addChangeListener(this);
     	hatchWidthField.setName(Figure.LABEL_HATCH_WIDTH);
     	hatchingP.add(hatchWidthField, constraint);	
     	
     	constraint.gridx++;
     	hatchingP.add(new JLabel(LABEL_HATCH_SEP), constraint);
     	
     	constraint.gridx++;
     	model = new SpinnerNumberModel(Figure.DEFAULT_HATCH_SEP, 0.01, 1000,1);
     	hatchSepField = new JSpinner(model);
     	hatchSepField.setEditor(new JSpinner.NumberEditor(hatchSepField, "0.00"));//$NON-NLS-1$
     	hatchSepField.addChangeListener(this);
     	hatchSepField.setName(LABEL_HATCH_SEP);
     	hatchingP.add(hatchSepField, constraint);	
     	
     	constraint.gridy++;
     	constraint.gridx=1;
     	hatchingP.add(new JLabel("angle"), constraint);
     	
     	constraint.gridx++;
     	model = new SpinnerNumberModel(0., -1000., 1000., 1.);
     	hatchAngleField = new JSpinner(model);
     	hatchAngleField.setEditor(new JSpinner.NumberEditor(hatchAngleField, "0.00"));//$NON-NLS-1$
     	hatchAngleField.addChangeListener(this);
     	hatchAngleField.setName(LABEL_HATCHINGS_ANGLE);
     	hatchingP.add(hatchAngleField, constraint);	
     	
     	hatchingP.setBorder(new CompoundBorder(new TitledBorder(null, LaTeXDrawLang.getString1_7("AbstractParametersFrame.8"),  //$NON-NLS-1$
				  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
     	
     	return hatchingP;
	}
	
	
	
	
	/**
	 * @return A panel with components to change the parameters of the double borders of the figure.
	 */
	public JPanel createDoubleLinePanel()
	{
     	JPanel fillingP = new JPanel(new FlowLayout(FlowLayout.LEFT));
     	
     	doubleLine = new JCheckBox(LABEL_DOUBLELINE);
     	doubleLine.setActionCommand(LABEL_DOUBLELINE);
     	doubleLine.addActionListener(this);
     	fillingP.add(doubleLine);
     	
     	fillingP.add(new JLabel(LABEL_DOUBLESEP));	     	
     	
     	SpinnerModel model = new SpinnerNumberModel(Figure.DEFAULT_DOUBLESEP, 1, 1000,1);
     	doubleSep = new JSpinner(model);
     	doubleSep.setEditor(new JSpinner.NumberEditor(doubleSep, "0"));	   //$NON-NLS-1$  	
     	doubleSep.setName(LABEL_DOUBLESEP);
     	doubleSep.addChangeListener(this);
     	doubleSep.setEnabled(doubleLine.isSelected());
     	fillingP.add(doubleSep);
     	
     	doublecolorButton = new ColorButton(LABEL_BUTTON_BOUNDARY, new ButtonIcon(Figure.DEFAULT_DOUBLE_COLOR));
     	doublecolorButton.setActionCommand(LABEL_BUTTON_BOUNDARY);
     	doublecolorButton.addActionListener(this);
     	fillingP.add(doublecolorButton);
     	fillingP.setBorder(new CompoundBorder(new TitledBorder(null, LaTeXDrawLang.getString1_7("AbstractParametersFrame.9"),  //$NON-NLS-1$
				  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
     	
     	return fillingP;
	}
	
	
	
	
	/**
	 * @return A panel with components to change the parameters of the borders of the figure.
	 */
	public JPanel createBordersPanel(boolean movable)
	{
     	JPanel borderP = new JPanel(new FlowLayout(FlowLayout.LEFT));
    	borderP.add(new JLabel(LABEL_THICKNESS));
     	
     	SpinnerModel model = new SpinnerNumberModel(Figure.DEFAULT_THICKNESS, 0.1, 100,0.1);
		thicknessField = new JSpinner(model);
		thicknessField.setName(LaTeXDrawResources.NAME_THICKNESS_FIELD);
		thicknessField.setEditor(new JSpinner.NumberEditor(thicknessField, "0.0"));//$NON-NLS-1$
     	thicknessField.addChangeListener(this);
     	borderP.add(thicknessField);
     	
     	bordersButton = new ColorButton(LaTeXDrawResources.LABEL_COLOR_BUTTON, new ButtonIcon(Figure.DEFAULT_BORDERS_COL));
     	bordersButton.setActionCommand(LaTeXDrawResources.LABEL_COLOR_BUTTON);
     	bordersButton.addActionListener(this);
     	borderP.add(bordersButton);
     	borderP.add(new JLabel(LABEL_TYPE_LINE));
     	
     	lineChoice = createLineChoice(this);
     	borderP.add(lineChoice);
     	
     	if(movable)
     	{
	     	bordersPositionChoice = Figure.createBordersPositionChoice();
	     	bordersPositionChoice.addItemListener(this);
	     	borderP.add(bordersPositionChoice);
     	}
     	
     	borderP.setBorder(new CompoundBorder(new TitledBorder(null, LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.47"),  //$NON-NLS-1$
				  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
     	
     	return borderP;
	}
	
	
	
	
	/**
	 * @return A panel with components to change the parameters of the filling of the figure.
	 */
	public JPanel createFillPanel()
	{
     	JPanel fillingP = new JPanel(new FlowLayout(FlowLayout.LEFT));
     	
     	filledCheckBox = new JCheckBox(LaTeXDrawResources.LABEL_FILLED_CHECKBOX);
     	filledCheckBox.setActionCommand(LaTeXDrawResources.LABEL_FILLED_CHECKBOX);
     	filledCheckBox.addActionListener(this);
     	fillingP.add(filledCheckBox);
     		     		     	
     	interiorButton = new ColorButton(LaTeXDrawResources.LABEL_COLOR_INTERIOR_BUTTON, new ButtonIcon(Figure.DEFAULT_INTERIOR_COL));
     	interiorButton.setActionCommand(LaTeXDrawResources.LABEL_COLOR_INTERIOR_BUTTON);
     	interiorButton.addActionListener(this);
     	fillingP.add(interiorButton);
 	
     	hatchChoice = Figure.createFillChoice();
     	hatchChoice.setPreferredSize(new Dimension(60,25));
     	hatchChoice.addItemListener(this);
     	fillingP.add(hatchChoice); 
     	
     	fillingP.setBorder(new CompoundBorder(new TitledBorder(null, LaTeXDrawLang.getString1_7("AbstractParametersFrame.11"),  //$NON-NLS-1$
				  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
     	
     	return fillingP;
	}
	
	
	
	
	/**
	 * @return A panel with components to change the parameters of the shadow of the figure.
	 */
	public JPanel createGradientPanel()
	{
  		GridBagConstraints constraint = new GridBagConstraints();
    	constraint.gridx = 0;  		constraint.gridy = 0;
     	constraint.gridwidth = 1; 	constraint.gridheight = 1;
     	constraint.weightx = 0.1; 	constraint.weighty = 0.1;
     	constraint.fill = GridBagConstraints.NONE;
     	constraint.anchor = GridBagConstraints.WEST;
     	
     	JPanel gradientP = new JPanel(new GridBagLayout());
     	
     	gradientStartColButton = new ColorButton(LaTeXDrawResources.LABEL_COLOR_GRADIENT_FIRST, new ButtonIcon(PSTricksConstants.DEFAULT_GRADIENT_START_COLOR));
     	gradientStartColButton.setActionCommand(LaTeXDrawResources.LABEL_COLOR_GRADIENT_FIRST);
     	gradientStartColButton.addActionListener(this);
     	gradientP.add(gradientStartColButton, constraint);
     	
     	gradientEndColButton = new ColorButton(LaTeXDrawResources.LABEL_COLOR_GRADIENT_SECOND, new ButtonIcon(PSTricksConstants.DEFAULT_GRADIENT_END_COLOR));
     	gradientEndColButton.setActionCommand(LaTeXDrawResources.LABEL_COLOR_GRADIENT_SECOND);
     	gradientEndColButton.addActionListener(this);
     	constraint.gridx++;	
     	gradientP.add(gradientEndColButton, constraint);
     	
     	constraint.gridy++;		constraint.gridx=0;	
     	gradientP.add(new JLabel(LABEL_GRADIENT_MID_PT), constraint);
     	
     	constraint.gridx++;	
     	SpinnerModel model = new SpinnerNumberModel(Figure.DEFAULT_GRADIENT_MID_POINT,0,1,0.01);
     	gradientMidPtField = new JSpinner(model);
     	gradientMidPtField.setEditor(new JSpinner.NumberEditor(gradientMidPtField, "0.000"));//$NON-NLS-1$
     	gradientMidPtField.addChangeListener(this);
     	gradientMidPtField.setName(LABEL_GRADIENT_MID_PT);
     	gradientP.add(gradientMidPtField, constraint);	  
     	
     	constraint.gridx++;	
     	gradientP.add(new JLabel(LABEL_GRADIENT_ANGLE), constraint);
     	
     	constraint.gridx++;	
     	model = new SpinnerNumberModel(Math.toDegrees(Figure.DEFAULT_GRADIENT_ANGLE), -360, 360,0.5);
     	gradientAngleField = new JSpinner(model);
     	gradientAngleField.setEditor(new JSpinner.NumberEditor(gradientAngleField, "0.0"));//$NON-NLS-1$
     	gradientAngleField.addChangeListener(this);
     	gradientAngleField.setName(LABEL_GRADIENT_ANGLE);
     	constraint.gridx++;	
     	gradientP.add(gradientAngleField, constraint);	
     	
     	gradientP.setBorder(new CompoundBorder(new TitledBorder(null, LaTeXDrawLang.getString1_7("AbstractParametersFrame.12"),  //$NON-NLS-1$
				  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
     	
     	return gradientP;
	}
	
	
	
	/**
	 * @return A panel with components to change the parameters of the shadow of the figure.
	 */
	public JPanel createShadowPanel()
	{
  		GridBagConstraints constraint = new GridBagConstraints();
    	constraint.gridx = 0;  		constraint.gridy = 0;
     	constraint.gridwidth = 1; 	constraint.gridheight = 1;
     	constraint.weightx = 0.1; 	constraint.weighty = 0.1;
     	constraint.fill = GridBagConstraints.NONE;
     	constraint.anchor = GridBagConstraints.CENTER;
     	
     	JPanel shadowP = new JPanel(new GridBagLayout());
     	
     	constraint.gridy=0;		constraint.gridx=0;	
     	shadowCheckBox = new JCheckBox(LaTeXDrawResources.LABEL_SHADOW_CHECKBOX);
     	shadowCheckBox.setActionCommand(LaTeXDrawResources.LABEL_SHADOW_CHECKBOX);
     	shadowCheckBox.addActionListener(this);
     	constraint.gridx=0;	constraint.gridy=0;	
     	shadowP.add(shadowCheckBox, constraint);
     	
     	shadowColorButton = new ColorButton(LaTeXDrawResources.LABEL_COLOR_SHADOW, new ButtonIcon(Figure.DEFAULT_SHADOW_COLOR));
     	shadowColorButton.setActionCommand(LaTeXDrawResources.NAME_COLOR_SHADOW);
     	shadowColorButton.addActionListener(this);
     	constraint.gridx++;	
     	shadowP.add(shadowColorButton, constraint);
     	
     	constraint.gridy++;		constraint.gridx=0;	
     	shadowP.add(new JLabel(LABEL_SHADOW_SIZE), constraint);
     	
     	constraint.gridx++;	
     	SpinnerModel model = new SpinnerNumberModel(Figure.DEFAULT_SHADOW_SIZE,0.01,1000,1);
     	shadowSizeField = new JSpinner(model);
     	shadowSizeField.setEditor(new JSpinner.NumberEditor(shadowSizeField, "0.00"));//$NON-NLS-1$
     	shadowSizeField.addChangeListener(this);
     	shadowSizeField.setName(LABEL_SHADOW_SIZE);
     	shadowP.add(shadowSizeField, constraint);	  
     	
     	constraint.gridx++;	
     	shadowP.add(new JLabel(LABEL_SHADOW_ANGLE), constraint);
     	
     	constraint.gridx++;	
     	model = new SpinnerNumberModel(Math.toDegrees(Figure.DEFAULT_SHADOW_ANGLE), -360, 360,0.5);
     	shadowAngleField = new JSpinner(model);
     	shadowAngleField.setEditor(new JSpinner.NumberEditor(shadowAngleField, "0.0"));//$NON-NLS-1$
     	shadowAngleField.addChangeListener(this);
     	shadowAngleField.setName(LABEL_SHADOW_ANGLE);
     	constraint.gridx++;	
     	shadowP.add(shadowAngleField, constraint);	
     	
     	shadowP.setBorder(new CompoundBorder(new TitledBorder(null, LaTeXDrawLang.getString1_7("LaTeXDrawFrame.0"),  //$NON-NLS-1$
				  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
     	
     	return shadowP;
	}
	
	
	
	
	/**
	 * Allows to create a panel with defaults buttons (ok, cancel, default)
	 * @param frame The frame which ask the panel
	 * @return The created panel
	 */
	public JPanel createButtonsPanel(AbstractParametersFrame frame)
	{
		JPanel panel = new JPanel();
		
		buttonOk = new JButton(LABEL_OK);
		JButton buttonCancel  = new JButton(LABEL_CANCEL),
				buttonDefault = new JButton(LABEL_DEFAULT);  		
		
  		buttonOk.setActionCommand(NAME_BUTTON_OK);
  		buttonCancel.setActionCommand(NAME_BUTTON_CANCEL);
  		buttonDefault.setActionCommand(NAME_BUTTON_DEFAULT);
  		
  		panel.add(buttonOk);
  		panel.add(buttonCancel);
  		panel.add(buttonDefault);
  		panel.setPreferredSize(new Dimension(280, 40));
  		panel.setMaximumSize(new Dimension(280, 40));
  		buttonOk.addActionListener(frame);
  		buttonCancel.addActionListener(frame);
  		buttonDefault.addActionListener(frame);

  		return panel;
	}
	
	
	
	
	/**
	 * Allows to set by default some fields of the frame
	 */
	public void setDefaultsValues()
	{
		Figure f = glimpsePanel.getGlimpseFigure();
		setModified(true);
		
		if(f.canHaveShadow())
		{
			f.setHasShadow(PSTricksConstants.DEFAULT_SHADOW);
			shadowAngleField.setValue(Math.toDegrees(Figure.DEFAULT_SHADOW_ANGLE));
			shadowSizeField.setValue(Figure.DEFAULT_SHADOW_SIZE);
			shadowColorButton.setEnabled(Figure.DEFAULT_SHADOW_HAS);
			shadowColorButton.setColor(Figure.DEFAULT_SHADOW_COLOR);
			f.setShadowColor(Figure.DEFAULT_SHADOW_COLOR);
			glimpsePanel.glimpseFigure.setDoubleColor(Figure.DEFAULT_SHADOW_COLOR);
			shadowCheckBox.setSelected(Figure.DEFAULT_SHADOW_HAS);
			shadowSizeField.setEnabled(shadowCheckBox.isSelected());
			shadowAngleField.setEnabled(shadowCheckBox.isSelected());
		}
		
		if(f.canHaveArrow())
		{
			((Arrowable)f).setArrow1Style(ArrowHead.DEFAULT_STYLE);
			((Arrowable)f).setArrow2Style(ArrowHead.DEFAULT_STYLE);
			arrow1Choice.setSelectedItem(ArrowHead.DEFAULT_STYLE);
			arrow2Choice.setSelectedItem(ArrowHead.DEFAULT_STYLE);
			
			bracketNum.setValue(ArrowHead.DEFAULT_BRACKET_NUM);
			dotSizeDim.setValue(ArrowHead.DEFAULT_DOT_SIZE_DIM);
			dotSizeNum.setValue(ArrowHead.DEFAULT_ARROW_SIZE_NUM);
			rbracketNum.setValue(ArrowHead.DEFAULT_RBRACKET_NUM);
			tbarsizeDim.setValue(ArrowHead.DEFAULT_TBAR_SIZE_DIM);
			tbarsizeNum.setValue(ArrowHead.DEFAULT_TBAR_SIZE_NUM);
			arrowInset.setValue(ArrowHead.DEFAULT_ARROW_INSET);
			arrowLength.setValue(ArrowHead.DEFAULT_ARROW_LGTH);
			arrowSizeDim.setValue(ArrowHead.DEFAULT_ARROW_SIZE_DIM);
			arrowSizeNum.setValue(ArrowHead.DEFAULT_ARROW_SIZE_NUM);
			
			ArrowHead leftA = ((Arrowable)f).getArrowHead1();
			ArrowHead rightA = ((Arrowable)f).getArrowHead2();
			
			leftA.setBracketNum(ArrowHead.DEFAULT_BRACKET_NUM);
			rightA.setBracketNum(ArrowHead.DEFAULT_BRACKET_NUM);
			leftA.setDotSizeDim(ArrowHead.DEFAULT_DOT_SIZE_DIM);
			rightA.setDotSizeDim(ArrowHead.DEFAULT_DOT_SIZE_DIM);
			leftA.setDotSizeNum(ArrowHead.DEFAULT_ARROW_SIZE_NUM);
			rightA.setDotSizeNum(ArrowHead.DEFAULT_ARROW_SIZE_NUM);
			leftA.setRBracketNum(ArrowHead.DEFAULT_RBRACKET_NUM);
			rightA.setRBracketNum(ArrowHead.DEFAULT_RBRACKET_NUM);
			leftA.setTBarSizeDim(ArrowHead.DEFAULT_TBAR_SIZE_DIM);
			rightA.setTBarSizeDim(ArrowHead.DEFAULT_TBAR_SIZE_DIM);
			leftA.setTBarSizeNum(ArrowHead.DEFAULT_TBAR_SIZE_NUM);
			rightA.setTBarSizeNum(ArrowHead.DEFAULT_TBAR_SIZE_NUM);
			
			leftA.setArrowSizeNum(ArrowHead.DEFAULT_ARROW_SIZE_NUM);
			rightA.setArrowSizeNum(ArrowHead.DEFAULT_ARROW_SIZE_NUM);
			leftA.setArrowSizeDim(ArrowHead.DEFAULT_ARROW_SIZE_DIM);
			rightA.setArrowSizeDim(ArrowHead.DEFAULT_ARROW_SIZE_DIM);
			leftA.setArrowLength(ArrowHead.DEFAULT_ARROW_LGTH);
			rightA.setArrowLength(ArrowHead.DEFAULT_ARROW_LGTH);
			leftA.setArrowInset(ArrowHead.DEFAULT_ARROW_INSET);
			rightA.setArrowInset(ArrowHead.DEFAULT_ARROW_INSET);
			
			updateEnableParamsArrow();
		}
		
		if(doublecolorButton!=null)
		{
			doublecolorButton.setEnabled(Figure.DEFAULT_DOUBLELINE);
			doublecolorButton.setColor(Figure.DEFAULT_DOUBLE_COLOR);
			glimpsePanel.glimpseFigure.setDoubleColor(Figure.DEFAULT_DOUBLE_COLOR);
		}
		
		if(bordersPositionChoice!=null)
		{
			bordersPositionChoice.setSelectedItem(Figure.DEFAULT_BORDERS_POSITION);
			glimpsePanel.glimpseFigure.setBordersPosition(Figure.DEFAULT_BORDERS_POSITION);
		}
		
		if(doubleLine!=null)
		{
			glimpsePanel.glimpseFigure.setHasDoubleBoundary(Figure.DEFAULT_DOUBLELINE);
			glimpsePanel.glimpseFigure.setDoubleSep(Figure.DEFAULT_DOUBLESEP);
			doubleLine.setSelected(Figure.DEFAULT_DOUBLELINE);
			doubleSep.setValue(Figure.DEFAULT_DOUBLESEP);
		}		
		
		if(lineChoice!=null)
			lineChoice.setSelectedItem(Figure.DEFAULT_LINE_STYLE);

		if(f.canBeHatched())
		{
			hatchButton.setColor(Figure.DEFAULT_HATCH_COL);
			glimpsePanel.getGlimpseFigure().setHatchingColor(Figure.DEFAULT_HATCH_COL);
			hatchChoice.setSelectedItem(Figure.DEFAULT_HATCH_STYLE);
			hatchWidthField.setValue((double)Figure.DEFAULT_HATCH_WIDTH);	
			hatchSepField.setValue(Figure.DEFAULT_HATCH_SEP);
			hatchAngleField.setValue(0.);
		}
		
		if(bordersButton!=null)
		{
			bordersButton.setColor(Figure.DEFAULT_BORDERS_COL);
			glimpsePanel.getGlimpseFigure().setLinesColor(Figure.DEFAULT_BORDERS_COL);
		}
		
		if(interiorButton!=null)
		{
			interiorButton.setColor(Figure.DEFAULT_INTERIOR_COL);
			glimpsePanel.getGlimpseFigure().setInteriorColor(Figure.DEFAULT_INTERIOR_COL);
		}
		
		if(filledCheckBox!=null)
			filledCheckBox.setSelected(Figure.DEFAULT_IS_FILLED);
		
		if(f.canBeHatched())
		{
			gradientAngleField.setValue(Math.toDegrees(Figure.DEFAULT_GRADIENT_ANGLE));
			gradientMidPtField.setValue(Figure.DEFAULT_GRADIENT_MID_POINT);
			gradientEndColButton.setColor(PSTricksConstants.DEFAULT_GRADIENT_END_COLOR);
			boolean ok = Figure.DEFAULT_HATCH_STYLE.equals(PSTricksConstants.TOKEN_FILL_GRADIENT);
			gradientEndColButton.setEnabled(ok);
			gradientStartColButton.setColor(PSTricksConstants.DEFAULT_GRADIENT_START_COLOR);
			gradientStartColButton.setEnabled(ok);
			gradientMidPtField.setEnabled(ok);
			gradientAngleField.setEnabled(ok);
		}
		
		if(rotationField!=null)
			rotationField.setValue(Figure.DEFAULT_ROTATION_ANGLE);
		
		if(thicknessField!=null)
			thicknessField.setValue((double)Figure.DEFAULT_THICKNESS);
		
		glimpsePanel.repaint();
	}
	
	
	
	
	/**
	 * Allows to create the list of the different styles of line.
	 * @param frame The frame which ask the Choice.
	 * @return The created Choice.
	 */
	public LaTeXDrawComboBox createLineChoice(AbstractParametersFrame frame)
	{
		LaTeXDrawComboBox choice = Figure.createStyleLineChoice();
		choice.setActionCommand(LaTeXDrawResources.CHOICE_LINE_NAME);
		choice.setName(LaTeXDrawResources.CHOICE_LINE_NAME);
		choice.addItemListener(frame);
     	
     	return choice;
	}
	
	
	
	
	public void itemStateChanged(ItemEvent e) 
	{
		try
		{
			Object o = e.getSource();
			Figure f = glimpsePanel.getGlimpseFigure();

			if(o instanceof LaTeXDrawComboBox)
			{
				String ac = ((LaTeXDrawComboBox)o).getActionCommand();
				
				if(ac.equals(LaTeXDrawResources.CHOICE_LINE_NAME))
				{
					JLabel l = (JLabel) lineChoice.getSelectedItem();
					
					setModified(true);
					f.setLineStyle(l.getText());
					glimpsePanel.repaint();
					return ;
				}
				
				if(ac.equals(Figure.LABEL_BORDERS_POSITION_CHOICE))
				{
					JLabel l = (JLabel) bordersPositionChoice.getSelectedItem();
					
					setModified(true);
					f.setBordersPosition(l.getText());
					glimpsePanel.repaint();
					return ;
				}
				
				if(ac.equals(Figure.LABEL_HATCH_CHOICE))
				{
					String txt = ((JLabel)hatchChoice.getSelectedItem()).getText();
					int num=0;
					
					if(txt.equals(PSTricksConstants.TOKEN_FILL_GRADIENT))
						num=1;
					else if(txt.equals(PSTricksConstants.TOKEN_FILL_NONE) || txt.equals(PSTricksConstants.TOKEN_FILL_SOLID))
							num=2;
					
					setModified(true);
					f.setHatchingStyle(txt);
					hatchButton.setEnabled(num==0);
					hatchWidthField.setEnabled(num==0);
					hatchAngleField.setEnabled(num==0);
					hatchSepField.setEnabled(num==0);
					gradientAngleField.setEnabled(num==1);
					gradientEndColButton.setEnabled(num==1);
					gradientMidPtField.setEnabled(num==1);
					gradientStartColButton.setEnabled(num==1);
					updateFilledCB();
					glimpsePanel.repaint();
					return ;
				}
				
				if(ac.equals(LaTeXDrawResources.CHOICE_ARROW_LEFT_NAME))
				{
					JLabel label = (JLabel) arrow1Choice.getSelectedItem();

					setModified(true);
					((Arrowable)f).setArrow1Style(label.getText());
					updateEnableParamsArrow();	
					glimpsePanel.repaint();
					return ;
				}
				
				if(ac.equals(LaTeXDrawResources.CHOICE_ARROW_RIGHT_NAME))
				{
					JLabel label = (JLabel) arrow2Choice.getSelectedItem();

					setModified(true);
					((Arrowable)f).setArrow2Style(label.getText());
					updateEnableParamsArrow();
					glimpsePanel.repaint();
					return ;
				}
			}
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
	
	
	

	/**
	 * Defines actions to do when a item like JSpinner has changed
	 */
	public void stateChanged(ChangeEvent e)
	{
		try
		{
			Object o = e.getSource();
			
			if(o instanceof JSpinner)
			{
				String name = ((JSpinner)o).getName();
				Figure f = glimpsePanel.getGlimpseFigure();
				double v = Double.valueOf(((JSpinner)o).getValue().toString()).doubleValue();
				
				if(name.equals(LaTeXDrawResources.NAME_THICKNESS_FIELD))
				{
					if(f instanceof Dot)
						((Dot)f).setWidth((float)v);
					else 
						f.setThickness((float)v);
					
					setModified(true);
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_DOUBLESEP))
				{
					setModified(true);
					f.setDoubleSep(v);
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_HATCH_SEP))
				{
					setModified(true);
					f.setHatchingSep(v);
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_HATCHINGS_ANGLE))
				{
					setModified(true);
					f.setHatchingAngle(Math.toRadians(v));
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_GRADIENT_ANGLE))
				{
					setModified(true);
					f.setGradientAngle(Math.toRadians(v));
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_GRADIENT_MID_PT))
				{
					setModified(true);
					f.setGradientMidPoint(v);				
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_SHADOW_ANGLE))
				{
					setModified(true);
					f.setShadowAngle(Math.toRadians(v));
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_SHADOW_SIZE))
				{
					setModified(true);
					f.setShadowSize(v);				
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(Figure.LABEL_HATCH_WIDTH))
				{
					setModified(true);
					f.setHatchingWidth((float)v);				
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LaTeXDrawResources.LABEL_ROTATE))
				{
					setModified(true);
					f.setRotationAngle(Math.toRadians(v));
					glimpsePanel.repaint();
					return;
				}

				if(name.equals(LABEL_BRACKETNUM))
				{
					setModified(true);
					((Arrowable)f).getArrowHead2().setBracketNum(v);
					((Arrowable)f).getArrowHead1().setBracketNum(v);
					glimpsePanel.repaint();
					return;
				}

				if(name.equals(LABEL_DOTSIZEDIM))
				{
					setModified(true);
					((Arrowable)f).getArrowHead1().setDotSizeDim(v);
					((Arrowable)f).getArrowHead2().setDotSizeDim(v);
					glimpsePanel.repaint();
					return;
				}

				if(name.equals(LABEL_DOTSIZENUM))
				{
					setModified(true);
					((Arrowable)f).getArrowHead1().setDotSizeNum(v);
					((Arrowable)f).getArrowHead2().setDotSizeNum(v);
					glimpsePanel.repaint();
					return;
				}

				if(name.equals(LABEL_RBRACKETNUM))
				{
					setModified(true);
					((Arrowable)f).getArrowHead1().setRBracketNum(v);
					((Arrowable)f).getArrowHead2().setRBracketNum(v);
					glimpsePanel.repaint();
					return;
				}

				if(name.equals(LABEL_TBARSIZEDIM))
				{
					setModified(true);
					((Arrowable)f).getArrowHead1().setTBarSizeDim(v);
					((Arrowable)f).getArrowHead2().setTBarSizeDim(v);
					glimpsePanel.repaint();
					return;
				}

				if(name.equals(LABEL_TBARSIZENUM))
				{
					setModified(true);
					((Arrowable)f).getArrowHead1().setTBarSizeNum(v);
					((Arrowable)f).getArrowHead2().setTBarSizeNum(v);
					glimpsePanel.repaint();
					return;
				}

				if(name.equals(LABEL_ARROWSIZENUM))
				{
					setModified(true);
					((Arrowable)f).getArrowHead1().setArrowSizeNum(v);
					((Arrowable)f).getArrowHead2().setArrowSizeNum(v);
					glimpsePanel.repaint();
					return;
				}
				
				if(name.equals(LABEL_ARROWSIZEDIM))
				{
					setModified(true);
					((Arrowable)f).getArrowHead1().setArrowSizeDim(v);
					((Arrowable)f).getArrowHead2().setArrowSizeDim(v);
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_ARROWLENGTH))
				{
					setModified(true);
					((Arrowable)f).getArrowHead1().setArrowLength(v);
					((Arrowable)f).getArrowHead2().setArrowLength(v);
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_ARROWINSET))
				{
					setModified(true);
					((Arrowable)f).getArrowHead1().setArrowInset(v);
					((Arrowable)f).getArrowHead2().setArrowInset(v);
					glimpsePanel.repaint();
					return ;
				}
			}
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
	
	
	
	/**
	 * Allows to set the field rotationField.
	 */
	public void createRotationField()
	{
		SpinnerModel model = new SpinnerNumberModel(Figure.DEFAULT_ROTATION_ANGLE, -360, 360,0.1);
     	rotationField = new JSpinner(model);
     	rotationField.setEditor(new JSpinner.NumberEditor(rotationField, "0.0"));	//$NON-NLS-1$     	
     	rotationField.setName(LaTeXDrawResources.LABEL_ROTATE);
     	rotationField.addChangeListener(this);
     	rotationField.setName(LaTeXDrawResources.LABEL_ROTATE);
	}
	
	
	
	/**
	 * @return Create a panel with fields to modify the arrows.
	 * @param twoLeftArrows True if the figure has two 'left' arrows like arc and not a right arrow and a left arrow like line.
	 */
	public JPanel createArrowPanel(boolean twoLeftArrows, boolean withFillButton)
	{
		JPanel arrowP = new JPanel(new GridBagLayout());
     	GridBagConstraints constraint = new GridBagConstraints();
    	constraint.gridx = 0;
     	constraint.gridy = 0;
     	constraint.gridwidth = 1;
     	constraint.gridheight = 1;
     	constraint.weightx = 0.1;
     	constraint.weighty = 0.1;
     	constraint.fill = GridBagConstraints.NONE;
     	constraint.anchor = GridBagConstraints.WEST;
     	
     	arrowP.add(new JLabel(LABEL_LEFT_ARROW), constraint);
     	
    	arrow1Choice = ArrowHead.createLeftArrowStyleList();
     	arrow1Choice.setName(LaTeXDrawResources.CHOICE_ARROW_LEFT_NAME);
     	arrow1Choice.setActionCommand(LaTeXDrawResources.CHOICE_ARROW_LEFT_NAME);
     	arrow1Choice.setSelectedItem(PSTricksConstants.NONEARROW_STYLE);
     	arrow1Choice.addItemListener(this);
     	constraint.gridx++;
     	arrowP.add(arrow1Choice, constraint);
     	
    	constraint.gridx++;
    	arrowP.add(new JLabel(LABEL_RIGHT_ARROW), constraint);	     	
     	
    	if(twoLeftArrows)
    		arrow2Choice = ArrowHead.createLeftArrowStyleList();
    	else
    		arrow2Choice = ArrowHead.createRightArrowStyleList();
     	arrow2Choice.setName(LaTeXDrawResources.CHOICE_ARROW_RIGHT_NAME);
     	arrow2Choice.setActionCommand(LaTeXDrawResources.CHOICE_ARROW_RIGHT_NAME);
     	arrow2Choice.setSelectedItem(PSTricksConstants.NONEARROW_STYLE);
     	arrow2Choice.addItemListener(this);
     	constraint.gridx++;
     	arrowP.add(arrow2Choice, constraint);
     	
     	constraint.gridx=0;	constraint.gridy++;
     	arrowP.add(new JLabel(LABEL_DOTSIZE), constraint);
     	
     	SpinnerNumberModel model = new SpinnerNumberModel(ArrowHead.DEFAULT_DOT_SIZE_NUM, 0.1, 100,0.1);
     	dotSizeNum = new JSpinner(model);
     	dotSizeNum.setName(LABEL_DOTSIZENUM);
     	dotSizeNum.setEditor(new JSpinner.NumberEditor(dotSizeNum, "0.0000"));//$NON-NLS-1$
     	dotSizeNum.addChangeListener(this);
     	constraint.gridx++;
     	arrowP.add(dotSizeNum, constraint);

     	constraint.gridx++;
     	arrowP.add(new JLabel(LABEL_CROSS_THICKNESS), constraint);     	
     	
     	model = new SpinnerNumberModel(ArrowHead.DEFAULT_DOT_SIZE_DIM, 0, 1000,1);
     	dotSizeDim = new JSpinner(model);
     	dotSizeDim.setName(LABEL_DOTSIZEDIM);
     	dotSizeDim.setEditor(new JSpinner.NumberEditor(dotSizeDim, "0.00"));//$NON-NLS-1$
     	dotSizeDim.addChangeListener(this);
     	constraint.gridx++;
     	arrowP.add(dotSizeDim, constraint);	     	
     	
     	constraint.gridx = 0;
     	constraint.gridy++;
     	arrowP.add(new JLabel(LABEL_TBARSIZE), constraint);
     	
     	model = new SpinnerNumberModel(ArrowHead.DEFAULT_TBAR_SIZE_NUM,0.1, 100,0.1);
     	tbarsizeNum = new JSpinner(model);
     	tbarsizeNum.setName(LABEL_TBARSIZENUM);
     	tbarsizeNum.setEditor(new JSpinner.NumberEditor(tbarsizeNum, "0.0000"));//$NON-NLS-1$
     	tbarsizeNum.addChangeListener(this);
     	constraint.gridx++;
     	arrowP.add(tbarsizeNum, constraint);

     	constraint.gridx++;
     	arrowP.add(new JLabel(LABEL_CROSS_THICKNESS), constraint);     	
     	
     	model = new SpinnerNumberModel(ArrowHead.DEFAULT_TBAR_SIZE_DIM, 0, 1000,1);
     	tbarsizeDim = new JSpinner(model);
     	tbarsizeDim.setName(LABEL_TBARSIZEDIM);
     	tbarsizeDim.setEditor(new JSpinner.NumberEditor(tbarsizeDim, "0.00"));//$NON-NLS-1$
     	tbarsizeDim.addChangeListener(this);
     	constraint.gridx++;
     	arrowP.add(tbarsizeDim, constraint);	     		     	
     	
     	constraint.gridx = 0;
     	constraint.gridy++;
     	arrowP.add(new JLabel(LABEL_BRACKET_LGTH), constraint);
     	
     	model = new SpinnerNumberModel(ArrowHead.DEFAULT_BRACKET_NUM, 0.001, 100,0.01);
     	bracketNum = new JSpinner(model);
     	bracketNum.setName(LABEL_BRACKETNUM);
     	bracketNum.setEditor(new JSpinner.NumberEditor(bracketNum, "0.0000"));//$NON-NLS-1$
     	bracketNum.addChangeListener(this);
     	constraint.gridx++;
     	arrowP.add(bracketNum, constraint);

     	constraint.gridx++;
     	arrowP.add(new JLabel(LABEL_CROSS_WIDTH), constraint);
     	
     	constraint.gridx = 0;
     	constraint.gridy++;
     	arrowP.add(new JLabel(LABEL_RBRACKET_LGTH), constraint);
     	
     	model = new SpinnerNumberModel(ArrowHead.DEFAULT_RBRACKET_NUM, 0.001, 100,0.01);
     	rbracketNum = new JSpinner(model);
     	rbracketNum.setName(LABEL_RBRACKETNUM);
     	rbracketNum.setEditor(new JSpinner.NumberEditor(rbracketNum, "0.0000"));//$NON-NLS-1$
     	rbracketNum.addChangeListener(this);
     	constraint.gridx++;
     	arrowP.add(rbracketNum, constraint);

     	constraint.gridx++;
     	arrowP.add(new JLabel(LABEL_CROSS_WIDTH), constraint); 
     	
     	constraint.gridx=0;
     	constraint.gridy++;
     	arrowP.add(new JLabel(LABEL_ARROW_WIDTH), constraint); 	     	
     	
     	model = new SpinnerNumberModel(ArrowHead.DEFAULT_ARROW_SIZE_NUM, 0.001, 100,0.01);
     	arrowSizeNum = new JSpinner(model);
     	arrowSizeNum.setName(LABEL_ARROWSIZENUM);
     	arrowSizeNum.setEditor(new JSpinner.NumberEditor(arrowSizeNum, "0.0000"));//$NON-NLS-1$
     	arrowSizeNum.addChangeListener(this);
     	constraint.gridx++;
     	arrowP.add(arrowSizeNum, constraint);

     	constraint.gridx++;
     	arrowP.add(new JLabel(LABEL_CROSS_THICKNESS), constraint);     	
     	
     	model = new SpinnerNumberModel(ArrowHead.DEFAULT_ARROW_SIZE_DIM, 0, 1000, 1);
     	arrowSizeDim = new JSpinner(model);
     	arrowSizeDim.setName(LABEL_ARROWSIZEDIM);
     	arrowSizeDim.setEditor(new JSpinner.NumberEditor(arrowSizeDim, "0.00"));//$NON-NLS-1$
     	arrowSizeDim.addChangeListener(this);
     	constraint.gridx++;
     	arrowP.add(arrowSizeDim, constraint);	   
     	
     	constraint.gridx=0;
     	constraint.gridy++;
     	arrowP.add(new JLabel(LABEL_ARROW_LENGTH), constraint); 
     	
     	model = new SpinnerNumberModel(ArrowHead.DEFAULT_ARROW_LGTH, 0, 100,0.01);
     	arrowLength = new JSpinner(model);
     	arrowLength.setName(LABEL_ARROWLENGTH);
     	arrowLength.setEditor(new JSpinner.NumberEditor(arrowLength, "0.0000"));//$NON-NLS-1$
     	arrowLength.addChangeListener(this);
     	constraint.gridx++;
     	arrowP.add(arrowLength, constraint);
     	
     	constraint.gridx=0;
     	constraint.gridy++;
     	arrowP.add(new JLabel(LABEL_ARROW_INSET), constraint); 
     	
     	model = new SpinnerNumberModel(ArrowHead.DEFAULT_ARROW_INSET, 0, 100,0.01);
     	arrowInset = new JSpinner(model);
     	arrowInset.setName(LABEL_ARROWINSET);
     	arrowInset.setEditor(new JSpinner.NumberEditor(arrowInset, "0.0000"));//$NON-NLS-1$
     	arrowInset.addChangeListener(this);
     	constraint.gridx++;
     	arrowP.add(arrowInset, constraint);	     	
     	
     	if(withFillButton)
     	{
	     	interiorButton = new ColorButton(LaTeXDrawResources.LABEL_COLOR_INTERIOR_BUTTON, new ButtonIcon(Figure.DEFAULT_INTERIOR_COL));
	     	interiorButton.setActionCommand(LaTeXDrawResources.LABEL_COLOR_INTERIOR_BUTTON);
	     	interiorButton.addActionListener(this);
	     	constraint.gridx++;
	     	arrowP.add(interiorButton, constraint);
     	}
     	
     	arrowP.setBorder(new CompoundBorder(new TitledBorder(null, LABEL_ARROWS, 
						  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
		return arrowP;
	}
	
	
	
	/**
	 * Allows to set enable or disable the parameters of the left arrow
	 */
	public void updateEnableParamsArrow()
	{
		JLabel label1 = (JLabel) arrow2Choice.getSelectedItem();
		JLabel label2 = (JLabel) arrow1Choice.getSelectedItem();
		String text1 = label1.getText();
		String text2 = label2.getText();
		
		boolean ok = text1.equals(PSTricksConstants.BAREND_STYLE) ||
					text1.equals(PSTricksConstants.BARIN_STYLE) ||
					text1.equals(PSTricksConstants.LSBRACKET_STYLE) ||
					text1.equals(PSTricksConstants.RSBRACKET_STYLE) ||
					text1.equals(PSTricksConstants.RRBRACKET_STYLE) ||
					text1.equals(PSTricksConstants.LRBRACKET_STYLE) || 
					text2.equals(PSTricksConstants.BAREND_STYLE) ||
					text2.equals(PSTricksConstants.BARIN_STYLE) ||
					text2.equals(PSTricksConstants.LSBRACKET_STYLE) ||
					text2.equals(PSTricksConstants.RSBRACKET_STYLE) ||
					text2.equals(PSTricksConstants.RRBRACKET_STYLE) ||
					text2.equals(PSTricksConstants.LRBRACKET_STYLE) ;
		
		tbarsizeDim.setEnabled(ok);
		tbarsizeNum.setEnabled(ok);
		bracketNum.setEnabled(text1.equals(PSTricksConstants.RSBRACKET_STYLE) ||
								text1.equals(PSTricksConstants.LSBRACKET_STYLE) ||
								text2.equals(PSTricksConstants.RSBRACKET_STYLE) ||
								text2.equals(PSTricksConstants.LSBRACKET_STYLE));
		
		rbracketNum.setEnabled(text1.equals(PSTricksConstants.RRBRACKET_STYLE) ||
								text1.equals(PSTricksConstants.LRBRACKET_STYLE) ||
								text2.equals(PSTricksConstants.RRBRACKET_STYLE) ||
								text2.equals(PSTricksConstants.LRBRACKET_STYLE));
		
		ok = text1.equals(PSTricksConstants.CIRCLEEND_STYLE) ||
			text1.equals(PSTricksConstants.CIRCLEIN_STYLE) ||
			text1.equals(PSTricksConstants.DISKEND_STYLE) ||
			text1.equals(PSTricksConstants.DISKIN_STYLE) ||
			text2.equals(PSTricksConstants.CIRCLEEND_STYLE) ||
			text2.equals(PSTricksConstants.CIRCLEIN_STYLE) ||
			text2.equals(PSTricksConstants.DISKEND_STYLE) ||
			text2.equals(PSTricksConstants.DISKIN_STYLE);
		
		dotSizeDim.setEnabled(ok);
		dotSizeNum.setEnabled(ok);
		
		ok = (text1.equals(PSTricksConstants.DLARROW_STYLE) ||
			text1.equals(PSTricksConstants.LARROW_STYLE) ||
			text1.equals(PSTricksConstants.DRARROW_STYLE) ||
			text1.equals(PSTricksConstants.RARROW_STYLE) ||
			text2.equals(PSTricksConstants.DLARROW_STYLE) ||
			text2.equals(PSTricksConstants.LARROW_STYLE) ||
			text2.equals(PSTricksConstants.DRARROW_STYLE) ||
			text2.equals(PSTricksConstants.RARROW_STYLE));
		
		arrowInset.setEnabled(ok);
		arrowLength.setEnabled(ok);
		arrowSizeDim.setEnabled(ok);
		arrowSizeNum.setEnabled(ok);
	}
	
	
	
	
	/**
	 * Defines if the checkbox "filledCB" must be enable or not.
	 * @since 1.9
	 */
	protected void updateFilledCB()
	{
		Figure f = glimpsePanel.getGlimpseFigure();
		
		if(!f.canBeFilled() || filledCheckBox==null)
			return ;
		
		String txt = f.getHatchingStyle();
		
		if(txt.equals(PSTricksConstants.TOKEN_FILL_GRADIENT))
		{
			filledCheckBox.setEnabled(false);
			return ;
		}
		
		if(f instanceof JoinedLines || f instanceof BezierCurve || f instanceof AkinPoints)
		{
			filledCheckBox.setEnabled(!(shadowCheckBox.isSelected() && f.canHaveShadow() &&
					(txt.equals(PSTricksConstants.TOKEN_FILL_CROSSHATCH) ||
					txt.equals(PSTricksConstants.TOKEN_FILL_CROSSHATCH_F) ||
					txt.equals(PSTricksConstants.TOKEN_FILL_HLINES) ||
					txt.equals(PSTricksConstants.TOKEN_FILL_HLINES_F) ||
					txt.equals(PSTricksConstants.TOKEN_FILL_VLINES) ||
					txt.equals(PSTricksConstants.TOKEN_FILL_VLINES_F))));
		}
		else
			filledCheckBox.setEnabled(!f.canHaveShadow() || !shadowCheckBox.isSelected());
	}



	/**
	 * @return the isModified.
	 * @since 2.0.0
	 */
	public synchronized boolean isModified()
	{
		return isModified;
	}


	/**
	 * @param isModified the isModified to set.
	 * @since 2.0.0
	 */
	public synchronized void setModified(boolean isModified)
	{
		this.isModified = isModified;
	}
}
