/* ***************************************************************************
 *
 * Pico Technology USB Device Driver
 *
 *//**
 * \file      readingbuff.cpp 
 * \brief     Circular reading buffer class
 **//*
 *
 * Copyright (c) 2007, Pico Technology.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * The name of Pico Technology may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Version $Id: readingbuff.cpp,v 1.2 2007/07/30 16:29:09 douglas Exp $
 *
 *************************************************************************** */

//  Class to handle a reading buffer .

#include <string.h>
#include <assert.h>

#include "readingbuff.h"

//////////////////////////////////////////////////////////////////////////
//
// Constructor
//
//////////////////////////////////////////////////////////////////////////

ReadingBuffer::ReadingBuffer () {
  LastValueTime = 0;
  SamplingInterval = 0;

  ValueBuffer = NULL;
  OverflowFlagBuffer = NULL;
  ValueBufferCount = 0;
  BufferFull = false;
  
	pthread_mutex_init(&m_LockBuffer,NULL);
  }

//////////////////////////////////////////////////////////////////////////
//
// Destructor
//
//////////////////////////////////////////////////////////////////////////

ReadingBuffer::~ReadingBuffer ()
  {
  if (ValueBuffer)
    delete [] ValueBuffer;
  if (OverflowFlagBuffer)
    delete [] OverflowFlagBuffer;
  }

//////////////////////////////////////////////////////////////////////////
//
// Allocate memory for the Value Buffer
//
//////////////////////////////////////////////////////////////////////////

bool ReadingBuffer::set_ReadingBufferSize (long lCount)
  {
  if (!m_GetLock())
    return false;

  if (ValueBuffer)
    delete [] ValueBuffer;
  if (OverflowFlagBuffer)
    delete [] OverflowFlagBuffer;

  ValueBufferCount = lCount;
  LastValuePosition = ValueBufferCount - 1; 
 
  ValueBuffer = new float [ValueBufferCount];
  OverflowFlagBuffer = new unsigned char [ValueBufferCount];
  
  // Release the lock before resetting the buffer
  //  since ResetBuffers() also uses the lock
  //
  m_ReleaseLock();

  ResetBuffers();

  return true;
  }

//////////////////////////////////////////////////////////////////////////
//
// Reset all buffers
//
//////////////////////////////////////////////////////////////////////////

bool ReadingBuffer::ResetBuffers(void)
  {
  if (!m_GetLock())
    return false;

  memset(ValueBuffer,0, (size_t)ValueBufferCount * sizeof(ValueBuffer[0]));
  memset(OverflowFlagBuffer,0, (size_t)ValueBufferCount * sizeof(OverflowFlagBuffer[0]));
  
  // So the last read position is effectively 
  //  1 less than the current read cursor
  LastValuePosition = ValueBufferCount - 1;
  
  CurrentValuePosition = ValueBufferCount - 1;
  LastValueTime = 0;
  BufferFull = false;

  m_ReleaseLock();
  return true;
  }

//////////////////////////////////////////////////////////////////////////
//
// Set the sampling interval
//  A time offset is provided to allow for multiplexed channels
//
//////////////////////////////////////////////////////////////////////////

bool ReadingBuffer::set_TimeInfo (long lInterval_ms)
  {
  if (!m_GetLock())
    return false;

  SamplingInterval = lInterval_ms;

  m_ReleaseLock();
  return true;
  }

//////////////////////////////////////////////////////////////////////////
//
// Get a sequence of times and values from the value array
//  this function copes with the wrapping of a cyclic buffer
//
//////////////////////////////////////////////////////////////////////////

long ReadingBuffer::get_Readings(float * Readings, bool * overflow, long lArrayCount, long * firsttime)
  {
  long  lReadings2Copy;
  long  lAvailableCount;
  long  lFirstCopyCount;
  bool  fWrapped;


  if (!m_GetLock())
    return -1;

  if (!Readings)
    {
    m_ReleaseLock();
    return -1;
    }

  lAvailableCount = get_ReadingCount(&fWrapped);

  // check the number of readings that the caller wants to collect
  if (lArrayCount < lAvailableCount)
    lReadings2Copy = lArrayCount;
  else
    lReadings2Copy = lAvailableCount;

  // Copy the readings across at high speed with memcpy()
  //  if the buffer is wrapped then we may need to do this in 2 passes
  //
  //  If any of the readings overflowed, report it back
  //
  if (overflow)
    *overflow = false;

  if (fWrapped)
    {
    lFirstCopyCount = ((ValueBufferCount - 1) - LastValuePosition); // number of values left to the end of the buffer
    if (lReadings2Copy <= lFirstCopyCount)
      {
      memcpy(Readings, &ValueBuffer[m_FirstAvailable()], lReadings2Copy * sizeof(ValueBuffer[0]));
      for (int i = m_FirstAvailable(); i < m_FirstAvailable() + lReadings2Copy; i++)
        {
        if (OverflowFlagBuffer[i])
          {
          if (overflow)
            *overflow |= true;
          }
        }

      LastValuePosition += lReadings2Copy; // update the cursor
      }
    else
      {
      memcpy(Readings, &ValueBuffer[m_FirstAvailable()], lFirstCopyCount * sizeof(ValueBuffer[0]));
      for (int i = m_FirstAvailable(); i < (m_FirstAvailable() + lFirstCopyCount); i++)
        {
        if (OverflowFlagBuffer[i])
          {
          if (overflow)
            *overflow |= true;
          }
        }

      memcpy(Readings + lFirstCopyCount, &ValueBuffer[0], (lReadings2Copy - lFirstCopyCount) * sizeof(ValueBuffer[0]));
      for (int i = 0; i < (lReadings2Copy - lFirstCopyCount); i++)
        {
        if (OverflowFlagBuffer[i])
          {
          if (overflow)
            *overflow |= true;
          }
        }
      
      LastValuePosition = ((lReadings2Copy - 1) - lFirstCopyCount); // update the cursor (-1 since the array is zero based)
      }
    }
  else
    {
    memcpy(Readings, &ValueBuffer[m_FirstAvailable()], lReadings2Copy * sizeof(ValueBuffer[0])); 
    for (int i = m_FirstAvailable(); i < (m_FirstAvailable() + lReadings2Copy); i++)
      {
      if (OverflowFlagBuffer[i])
        {
        if (overflow)
          *overflow |= true;
        }
      }

    LastValuePosition += lReadings2Copy; // update the cursor
    }

  
  ///////////////////////////////////////////////////////////////////////////
  // Get the time of the first reading (the caller can work out the rest)
  if (firsttime)
    *firsttime = LastValueTime;

  // start back from zero
  //  if the long variable wrapped around
  if ((LastValueTime + (lReadings2Copy * SamplingInterval)) < 0)
    LastValueTime = 0; 
  else
    LastValueTime += (lReadings2Copy * SamplingInterval);

  BufferFull = false;

  m_ReleaseLock();
  return lReadings2Copy;
  }

///////////////////////////////////////////////////////////////////////////
//
// Overload for getting readings without wanting the overflow flag
//
///////////////////////////////////////////////////////////////////////////
long ReadingBuffer::get_Readings (float * readings, long arraysize, long * firsttime)
  {
  return get_Readings(readings, NULL, arraysize, firsttime);
  }

//////////////////////////////////////////////////////////////////////////
//
// Get the LastValuePosition + 1 ... synchronised with the array
//
//////////////////////////////////////////////////////////////////////////

inline long ReadingBuffer::m_FirstAvailable(void)
  {
  if ((LastValuePosition + 1) == ValueBufferCount)
    return 0; // wrap back to the first buffer element
  else
    return LastValuePosition + 1;
  }

///////////////////////////////////////////////////////////////////////////////
//
// 2 x functions to Get and Release a buffer lock, this serialises functions
//  with write access to class members, thereby making the class thread-safe
//
///////////////////////////////////////////////////////////////////////////////

//
// Wait for the lock to be released
//
bool ReadingBuffer::m_GetLock (void)
  {
       pthread_mutex_lock(&m_LockBuffer);
	   return true;
  }

//
// Release the lock
//
void ReadingBuffer::m_ReleaseLock (void)
  {
	pthread_mutex_unlock(&m_LockBuffer);
  }

//////////////////////////////////////////////////////////////////////////
//
// Let the caller know if the buffer is full
//
//////////////////////////////////////////////////////////////////////////

bool ReadingBuffer::BufferIsFull (void)
  {
  return BufferFull;
  }

//////////////////////////////////////////////////////////////////////////
//
// Get the number of readings available
//
//////////////////////////////////////////////////////////////////////////

long ReadingBuffer::get_ReadingCount (bool * pWrapped)
  {
  // determine whether the buffer is wrapped, and whether there are values to collect
  if ((LastValuePosition == CurrentValuePosition) && !BufferFull)
    {
    if (pWrapped)
      *pWrapped = false;
    return 0; // no readings
    }
  else if (BufferFull || (LastValuePosition > CurrentValuePosition))
    {
    if (pWrapped)
      *pWrapped = true;
    return ((ValueBufferCount - LastValuePosition) + CurrentValuePosition);
    }
  else
    {
    if (pWrapped)
      *pWrapped = false;
    return (CurrentValuePosition - LastValuePosition);
    }
  }

//////////////////////////////////////////////////////////////////////////
//
// Add a new reading to the value buffer
//
//////////////////////////////////////////////////////////////////////////

bool ReadingBuffer::AddReading (float Reading, bool overflow)
  {
  if (!m_GetLock())
    return false;

  m_IncrementValueBuffer();
  ValueBuffer[CurrentValuePosition] = Reading;
  OverflowFlagBuffer[CurrentValuePosition] = (unsigned char)overflow;

  m_ReleaseLock();
  return true;
  }

bool ReadingBuffer::AddReading (float Reading)
  {
  return AddReading(Reading, false);
  }

//////////////////////////////////////////////////////////////////////////
//
// A quick way to pad out the buffer with QNaN floats
//
//////////////////////////////////////////////////////////////////////////

bool ReadingBuffer::PadQNaN (long ReadingCount)
  {
  long lReading;

  if (!ValueBuffer || !OverflowFlagBuffer)
    return false;

  if (!m_GetLock())
    return false;

  for (lReading = 0; lReading < ReadingCount; ++lReading)
    {
    m_IncrementValueBuffer();
    ValueBuffer[CurrentValuePosition] = std::numeric_limits<float>::quiet_NaN();
    OverflowFlagBuffer[CurrentValuePosition] = true;
    }

  m_ReleaseLock();
  return true;
  }

//////////////////////////////////////////////////////////////////////////
//
// Increment the cyclic value array and flag if its full
//
//////////////////////////////////////////////////////////////////////////

inline void ReadingBuffer::m_IncrementValueBuffer (void)
  {
  // update the current value cursor
  if (CurrentValuePosition == (ValueBufferCount - 1))
    CurrentValuePosition = 0; // back to the beginning
  else
    ++CurrentValuePosition;

  // update the last value cursor and flag a full buffer if necessary
  if (BufferFull)
    {
    LastValuePosition = CurrentValuePosition; // same as doing ++LastValuePosition
    LastValueTime += SamplingInterval;        // we've overwritten a reading so update the time
    }
  else if (CurrentValuePosition == LastValuePosition)
    BufferFull = true;
  }
