"""General utilities."""

import os
from urllib.parse import urlsplit

class Unspecified(object):
    pass

unspecified = Unspecified()

def ident(x):
    return x

def map_option(f, opt):
    if opt is None: return None
    return f(opt)

class Result(object):
    pass

class ResultOk(Result):
    def __init__(self, value):
        Result.__init__(self)
        self.value = value

    def is_ok(self):
        return True

    def is_error(self):
        return False

    def get(self):
        return self.value

class ResultError(Result):
    def __init__(self, error):
        Result.__init__(self)
        self.error = error

    def is_ok(self):
        return False

    def is_error(self):
        return True

    def get(self):
        raise self.error

    def __str__(self):
        return 'ResultError: %s' % self.error

class lazy_property(object):
    def __init__(self, func):
        self._func = func

    def __get__(self, other, _=None):
        if other is None:
            return self
        else:
            value = self._func(other)
            setattr(other, self._func.__name__, value)
            return value

def lazy_staticmethod(thunk):
    def g(cls):
        # pylint: disable=protected-access
        if not hasattr(cls, '_lazy_static'):
            cls._lazy_static = {}
        h = hash(thunk)
        if not h in cls._lazy_static:
            cls._lazy_static[h] = thunk()
        return cls._lazy_static[h]
    return classmethod(g)

class GetattrProxy(object):
    def __init__(self, *objs):
        self.objs = objs

    def __getitem__(self, k):
        for obj in self.objs:
            try:
                return getattr(obj, k)
            except AttributeError:
                pass

class Statdict(object):
    def __init__(self):
        self._d = {}
    def __getitem__(self, k):
        return self._d.get(k, 0)
    def __setitem__(self, k, v):
        self._d.__setitem__(k, v)
    def items(self):
        return self._d.items()
    def __bool__(self):
        return bool(self._d)
    def __len__(self):
        return len(self._d)

def options_of_spec(shortopts, longopts):
    def fix_longopt(longopt):
        if longopt.endswith('='):
            return '--' + longopt[0:len(longopt)-1]
        else:
            return '--' + longopt
    return ['-' + opt for opt in shortopts if opt != ':'] \
         + list(map(fix_longopt, longopts))

def counted_noun(count, sing_word, pl_word = None):
    if count == 1:
        return '%d %s'%(count, sing_word)
    else:
        return '%d %s'%(count, pl_word or sing_word + 's')

def counted_adjectives(cws, if_empty = ''):
    return ', '.join(map(lambda cw: '%d %s' % cw,
                         filter(lambda cw: cw[0] != 0, cws))) or if_empty

def nth(n):
    if n % 100 > 3 and n % 100 < 21:
        return str(n) + 'th'
    else:
        return str(n) + {1: 'st', 2: 'nd', 3: 'rd'}.get(n % 10, 'th')

def file_contents(fp):
    try:
        fd = open(fp)
        c = fd.read()
        fd.close()
        return c
    except IOError:
        return None

def exited_with(sc):
    if os.WIFEXITED(sc):
        return 'exited with %d' % os.WEXITSTATUS(sc)
    elif os.WIFSIGNALED(sc):
        return 'terminated by signal %d' % os.WTERMSIG(sc)
    elif os.WIFCONTINUED(sc):
        return 'continued'
    elif os.WIFSTOPPED(sc):
        return 'stopped'
    return 'hit a rock'

def host_of_uri(jobid):
    netloc = urlsplit(jobid)[1]
    if ':' in netloc:
        return netloc.split(':', 1)[0]
    else:
        return netloc

def log_process_error(log, exn, synopsis = None, prefix = None):
    if exn.output:
        hint = 'stderr follows'
    else:
        hint = 'no stderr'
    if synopsis:
        log.error('%s: %s (%s)', synopsis.capitalize(), str(exn), hint)
    else:
        log.error('%s (%s)', str(exn), hint)
    if exn.output:
        if prefix: prefix = '[%s] ' % prefix
        else:      prefix = ''

        # exn.output may be str or bytes depending on how the subprocess
        # function is invoked.
        if isinstance(exn.output, str):
            output = exn.output
        else:
            output = exn.output.decode('utf-8')

        for ln in output.strip().split('\n'):
            log.error(prefix + ln)
