(**************************************************************************)
(*                                                                        *)
(*                                 OCaml                                  *)
(*                                                                        *)
(*              Damien Doligez, projet Para, INRIA Rocquencourt           *)
(*                                                                        *)
(*   Copyright 1998 Institut National de Recherche en Informatique et     *)
(*     en Automatique.                                                    *)
(*                                                                        *)
(*   All rights reserved.  This file is distributed under the terms of    *)
(*   the GNU Lesser General Public License version 2.1, with the          *)
(*   special exception on linking described in the file LICENSE.          *)
(*                                                                        *)
(**************************************************************************)

let mk_a f =
  "-a", Arg.Unit f, " Build a library"

let mk_alert f =
  "-alert", Arg.String f,
  Printf.sprintf
    "<list>  Enable or disable alerts according to <list>:\n\
    \        +<alertname>  enable alert <alertname>\n\
    \        -<alertname>  disable alert <alertname>\n\
    \        ++<alertname> treat <alertname> as fatal error\n\
    \        --<alertname> treat <alertname> as non-fatal\n\
    \        @<alertname>  enable <alertname> and treat it as fatal error\n\
    \    <alertname> can be 'all' to refer to all alert names";;

let mk_absname f =
  "-absname", Arg.Unit f, " Show absolute filenames in error messages"

let mk_no_absname f =
    "-no-absname", Arg.Unit f,
    " Do not try to show absolute filenames in error messages (default)"

let mk_annot f =
  "-annot", Arg.Unit f, " (deprecated) Save information in <filename>.annot"

let mk_binannot f =
  "-bin-annot", Arg.Unit f, " Save typedtree in <filename>.cmt"

let mk_binannot_occurrences f =
  "-bin-annot-occurrences", Arg.Unit f,
  " Store every occurrence of a bound name in the .cmt file.\n\
    This information can be used by external tools to provide\n\
    features such as project-wide occurrences. This flag has\n\
    no effect in the absence of '-bin-annot'."

let mk_c f =
  "-c", Arg.Unit f, " Compile only (do not link)"

let mk_cc f =
  "-cc", Arg.String f, "<command>  Use <command> as the C compiler and linker"

let mk_cclib f =
  "-cclib", Arg.String f, "<opt>  Pass option <opt> to the C linker"

let mk_ccopt f =
  "-ccopt", Arg.String f,
  "<opt>  Pass option <opt> to the C compiler and linker"

let mk_clambda_checks f =
  "-clambda-checks", Arg.Unit f, " Instrument clambda code with closure and \
    field access checks (for debugging the compiler)"

let mk_cmi_file f =
  "-cmi-file", Arg.String f,
    "<file>  Use the <file> interface file to type-check"

let mk_compact f =
  "-compact", Arg.Unit f, " Optimize code size rather than speed"

let mk_compat_32 f =
  "-compat-32", Arg.Unit f,
  " Check that generated bytecode can run on 32-bit platforms"

let mk_config f =
  "-config", Arg.Unit f, " Print configuration values and exit"

let mk_config_var f =
  "-config-var", Arg.String f,
  " Print the value of a configuration variable, without a newline, and exit\n\
\    (print nothing and exit with error value if the variable does not exist)"

let mk_custom f =
  "-custom", Arg.Unit f, " Link in custom mode"

let mk_dllib f =
  "-dllib", Arg.String f, "<lib>  Use the dynamically-loaded library <lib>"

let mk_dllpath f =
  "-dllpath", Arg.String f,
  "<dir>  Add <dir> to the run-time search path for shared libraries"

let mk_eval f =
  "-e", Arg.String f,
  "<script>  Evaluate given script"

let mk_function_sections f =
  if Config.function_sections then
    "-function-sections",  Arg.Unit f,
    " Generate each function in a separate section if target supports it"
  else
    let err () =
      raise (Arg.Bad "OCaml has been configured without support for \
                      -function-sections")
    in
    "-function-sections", Arg.Unit err, " (option not available)"

let mk_stop_after ~native f =
  let pass_names = Clflags.Compiler_pass.available_pass_names
                     ~filter:(fun _ -> true)
                     ~native
  in
  "-stop-after", Arg.Symbol (pass_names, f),
  " Stop after the given compilation pass."

let mk_save_ir_after ~native f =
  let pass_names =
    Clflags.Compiler_pass.(available_pass_names
                             ~filter:can_save_ir_after
                             ~native)
  in
  "-save-ir-after", Arg.Symbol (pass_names, f),
  " Save intermediate representation after the given compilation pass\
    (may be specified more than once)."

let mk_dtypes f =
  "-dtypes", Arg.Unit f, " (deprecated) same as -annot"

let mk_for_pack_byt f =
  "-for-pack", Arg.String f,
  "<ident>  Generate code that can later be `packed' with\n\
  \     ocamlc -pack -o <ident>.cmo"

let mk_for_pack_opt f =
  "-for-pack", Arg.String f,
  "<ident>  Generate code that can later be `packed' with\n\
  \     ocamlopt -pack -o <ident>.cmx"

let mk_g_byt f =
  "-g", Arg.Unit f, " Save debugging information"

let mk_g_opt f =
  "-g", Arg.Unit f, " Record debugging information for exception backtrace"

let mk_no_g f =
  "-no-g", Arg.Unit f,
  " Do not record debugging information (default)"

let mk_i f =
  "-i", Arg.Unit f, " Print inferred interface"

let mk_i_variance f =
  "-i-variance", Arg.Unit f, " Print inferred variances"

let mk_I f =
  "-I", Arg.String f, "<dir>  Add <dir> to the list of include directories"

let mk_H f =
  "-H", Arg.String f,
  "<dir>  Add <dir> to the list of \"hidden\" include directories\n\
 \     (Like -I, but the program can not directly reference these dependencies)"

let mk_impl f =
  "-impl", Arg.String f, "<file>  Compile <file> as a .ml file"

let mk_init f =
  "-init", Arg.String f, "<file>  Load <file> instead of default init file"

let mk_inline f =
  "-inline", Arg.String f,
    Printf.sprintf "<n>|<round>=<n>[,...]  Aggressiveness of inlining \
        (default %.02f, higher numbers mean more aggressive)"
      Clflags.default_inline_threshold

let mk_inline_toplevel f =
  "-inline-toplevel", Arg.String f,
    Printf.sprintf "<n>|<round>=<n>[,...]  Aggressiveness of inlining at \
      toplevel (higher numbers mean more aggressive)"

let mk_inlining_report f =
  "-inlining-report", Arg.Unit f, " Emit `.<round>.inlining' file(s) (one per \
      round) showing the inliner's decisions"

let mk_dump_pass f =
  "-dump-pass", Arg.String f,
  Format.asprintf
    " @[<4>Record transformations performed by these passes:@ @[%a@]@]"
    (Format.pp_print_list
       ~pp_sep:Format.pp_print_space
       Format.pp_print_string)
    !Clflags.all_passes

let mk_o2 f =
  "-O2", Arg.Unit f, " Apply increased optimization for speed"

let mk_o3 f =
  "-O3", Arg.Unit f, " Apply aggressive optimization for speed (may \
    significantly increase code size and compilation time)"

let mk_rounds f =
  "-rounds", Arg.Int f,
    Printf.sprintf "<n>  Repeat tree optimization and inlining phases this \
        many times (default %d).  Rounds are numbered starting from zero."
      !Clflags.default_simplify_rounds

let mk_inline_max_unroll f =
  "-inline-max-unroll", Arg.String f,
    Printf.sprintf "<n>|<round>=<n>[,...]  Unroll recursive functions at most \
      this many times (default %d)"
      Clflags.default_inline_max_unroll

let mk_classic_inlining f =
  "-Oclassic", Arg.Unit f, " Make inlining decisions at function definition \
     time rather than at the call site (replicates previous behaviour of the \
     compiler)"

let mk_inline_cost arg descr default f =
  Printf.sprintf "-inline-%s-cost" arg,
  Arg.String f,
  Printf.sprintf "<n>|<round>=<n>[,...]  The cost of not removing %s during \
      inlining (default %d, higher numbers more costly)"
    descr
    default

let mk_inline_call_cost =
  mk_inline_cost "call" "a call" Clflags.default_inline_call_cost
let mk_inline_alloc_cost =
  mk_inline_cost "alloc" "an allocation" Clflags.default_inline_alloc_cost
let mk_inline_prim_cost =
  mk_inline_cost "prim" "a primitive" Clflags.default_inline_prim_cost
let mk_inline_branch_cost =
  mk_inline_cost "branch" "a conditional" Clflags.default_inline_branch_cost
let mk_inline_indirect_cost =
  mk_inline_cost "indirect" "an indirect call"
    Clflags.default_inline_indirect_cost

let mk_inline_lifting_benefit f =
  "-inline-lifting-benefit",
  Arg.String f,
  Printf.sprintf "<n>|<round>=<n>[,...]  The benefit of lifting definitions \
    to toplevel during inlining (default %d, higher numbers more beneficial)"
    Clflags.default_inline_lifting_benefit

let mk_inline_branch_factor f =
  "-inline-branch-factor", Arg.String f,
    Printf.sprintf "<n>|<round>=<n>[,...]  Estimate the probability of a \
        branch being cold as 1/(1+n) (used for inlining) (default %.2f)"
    Clflags.default_inline_branch_factor

let mk_intf f =
  "-intf", Arg.String f, "<file>  Compile <file> as a .mli file"

let mk_intf_suffix f =
  "-intf-suffix", Arg.String f,
  "<string>  Suffix for interface files (default: .mli)"

let mk_intf_suffix_2 f =
  "-intf_suffix", Arg.String f, "<string>  (deprecated) same as -intf-suffix"

let mk_insn_sched f =
  "-insn-sched", Arg.Unit f,
  Printf.sprintf " Run the instruction scheduling pass%s"
    (if Clflags.insn_sched_default then " (default)" else "")

let mk_no_insn_sched f =
  "-no-insn-sched", Arg.Unit f,
  Printf.sprintf " Do not run the instruction scheduling pass%s"
    (if not Clflags.insn_sched_default then " (default)" else "")

let mk_keep_docs f =
  "-keep-docs", Arg.Unit f, " Keep documentation strings in .cmi files"

let mk_no_keep_docs f =
  "-no-keep-docs", Arg.Unit f,
  " Do not keep documentation strings in .cmi files (default)"

let mk_keep_locs f =
  "-keep-locs", Arg.Unit f, " Keep locations in .cmi files (default)"

let mk_no_keep_locs f =
  "-no-keep-locs", Arg.Unit f, " Do not keep locations in .cmi files"

let mk_labels f =
  "-labels", Arg.Unit f, " Use commuting label mode"

let mk_linkall f =
  "-linkall", Arg.Unit f, " Link all modules, even unused ones"

let mk_linscan f =
  "-linscan", Arg.Unit f, " Use the linear scan register allocator"

let mk_make_runtime f =
  "-make-runtime", Arg.Unit f,
  " Build a runtime system with given C objects and libraries"

let mk_make_runtime_2 f =
  "-make_runtime", Arg.Unit f, " (deprecated) same as -make-runtime"

let mk_inline_max_depth f =
  "-inline-max-depth", Arg.String f,
    Printf.sprintf "<n>|<round>=<n>[,...]  Maximum depth of search for \
      inlining opportunities inside inlined functions (default %d)"
      Clflags.default_inline_max_depth

let mk_modern f =
  "-modern", Arg.Unit f, " (deprecated) same as -labels"

let mk_alias_deps f =
  "-alias-deps", Arg.Unit f,
  " Do record dependencies for module aliases"

let mk_no_alias_deps f =
  "-no-alias-deps", Arg.Unit f,
  " Do not record dependencies for module aliases"

let mk_app_funct f =
  "-app-funct", Arg.Unit f, " Activate applicative functors"

let mk_no_app_funct f =
  "-no-app-funct", Arg.Unit f, " Deactivate applicative functors"

let mk_no_check_prims f =
  "-no-check-prims", Arg.Unit f, " Do not check runtime for primitives"

let mk_no_float_const_prop f =
  "-no-float-const-prop", Arg.Unit f,
  " Deactivate constant propagation for floating-point operations"

let mk_noassert f =
  "-noassert", Arg.Unit f, " Do not compile assertion checks"

let mk_noautolink_byt f =
  "-noautolink", Arg.Unit f,
  " Do not automatically link C libraries specified in .cma files"

let mk_noautolink_opt f =
  "-noautolink", Arg.Unit f,
  " Do not automatically link C libraries specified in .cmxa files"

let mk_nodynlink f =
  "-nodynlink", Arg.Unit f,
  " Enable optimizations for code that will not be dynlinked"

let mk_noinit f =
  "-noinit", Arg.Unit f,
  " Do not load any init file"

let mk_nolabels f =
  "-nolabels", Arg.Unit f, " Ignore non-optional labels in types"

let mk_prompt f =
  "-prompt", Arg.Unit f, " Output prompts (default)"

let mk_noprompt f =
  "-noprompt", Arg.Unit f, " Suppress all prompts"

let mk_nopromptcont f =
  "-nopromptcont", Arg.Unit f,
  " Suppress prompts for continuation lines of multi-line inputs"

let mk_nostdlib f =
  "-nostdlib", Arg.Unit f,
  " Do not add default directory to the list of include directories"

let mk_nocwd f =
  "-nocwd", Arg.Unit f,
  " Do not implicitly add the current directory to the load path"

let mk_no_unbox_free_vars_of_closures f =
  "-no-unbox-free-vars-of-closures", Arg.Unit f,
  " Do not unbox variables that will appear inside function closures"

let mk_no_unbox_specialised_args f =
  "-no-unbox-specialised-args", Arg.Unit f,
  " Do not unbox arguments to which functions have been specialised"

let mk_o f =
  "-o", Arg.String f, "<file>  Set output file name to <file>"

let mk_open f =
  "-open", Arg.String f, "<module>  Opens the module <module> before typing"

let mk_output_obj f =
  "-output-obj", Arg.Unit f, " Output an object file instead of an executable"

let mk_output_complete_obj f =
  "-output-complete-obj", Arg.Unit f,
  " Output an object file, including runtime, instead of an executable"

let mk_output_complete_exe f =
  "-output-complete-exe", Arg.Unit f,
  " Output a self-contained executable, including runtime and C stubs"

let mk_p f =
  "-p", Arg.Unit f, " (no longer supported)"

let mk_pack_byt f =
  "-pack", Arg.Unit f, " Package the given .cmo files into one .cmo"

let mk_pack_opt f =
  "-pack", Arg.Unit f, " Package the given .cmx files into one .cmx"

let mk_pp f =
  "-pp", Arg.String f, "<command>  Pipe sources through preprocessor <command>"

let mk_ppx f =
  "-ppx", Arg.String f,
  "<command>  Pipe abstract syntax trees through preprocessor <command>"

let mk_keywords f =
  "-keywords", Arg.String f,
  "<version+list>  set keywords following the <version+list> spec:\n
  \                -<version> if present specifies the base set of keywords\n
  \                  (if absent the current set of keywords is used)
  \                -<list> is a \"+\"-separated list of keywords to add to\n
  \                  the base set of keywords.
  "

let mk_plugin f =
  "-plugin", Arg.String f,
  "<plugin>  (no longer supported)"

let mk_principal f =
  "-principal", Arg.Unit f, " Check principality of type inference"

let mk_no_principal f =
  "-no-principal", Arg.Unit f,
  " Do not check principality of type inference (default)"

let mk_rectypes f =
  "-rectypes", Arg.Unit f, " Allow arbitrary recursive types"

let mk_no_rectypes f =
  "-no-rectypes", Arg.Unit f,
  " Do not allow arbitrary recursive types (default)"

let mk_remove_unused_arguments f =
  "-remove-unused-arguments", Arg.Unit f,
  " Remove unused function arguments"

let mk_runtime_variant f =
  "-runtime-variant", Arg.String f,
  "<str>  Use the <str> variant of the run-time system"

let mk_with_runtime f =
  "-with-runtime", Arg.Unit f,
  "Include the runtime system in the generated program (default)"

let mk_without_runtime f =
  "-without-runtime", Arg.Unit f,
  "Do not include the runtime system in the generated program."

let mk_S f =
  "-S", Arg.Unit f, " Keep intermediate assembly file"

let mk_safe_string =
  "-safe-string", Arg.Unit (fun () -> ()),
  " (default unconditionally since 5.0)"

let mk_safer_matching f =
  "-safer-matching", Arg.Unit f,
  " Do not use type information to optimize pattern-matching"

let mk_shared f =
  "-shared", Arg.Unit f, " Produce a dynlinkable plugin"

let mk_short_paths f =
  "-short-paths", Arg.Unit f, " Shorten paths in types"

let mk_stdin f =
  "-stdin", Arg.Unit f, " Read script from standard input"

let mk_no_strict_sequence f =
  "-no-strict-sequence", Arg.Unit f,
  " Left-hand part of a sequence need not have type unit (default)"

let mk_strict_sequence f =
  "-strict-sequence", Arg.Unit f,
  " Left-hand part of a sequence must have type unit"

let mk_thread f =
  "-thread", Arg.Unit f,
  " (deprecated) same as -I +threads"

let mk_dtimings f =
  "-dtimings", Arg.Unit f, " Print timings information for each pass"

let mk_dprofile f =
  "-dprofile", Arg.Unit f, Profile.options_doc

let mk_unbox_closures f =
  "-unbox-closures", Arg.Unit f,
  " Pass free variables via specialised arguments rather than closures"

let mk_unbox_closures_factor f =
  "-unbox-closures-factor", Arg.Int f,
  Printf.sprintf "<n > 0>  Scale the size threshold above which \
      unbox-closures will slow down indirect calls rather than duplicating a \
      function (default %d)"
    Clflags.default_unbox_closures_factor

let mk_unboxed_types f =
  "-unboxed-types", Arg.Unit f,
  " unannotated unboxable types will be unboxed"

let mk_no_unboxed_types f =
  "-no-unboxed-types", Arg.Unit f,
  " unannotated unboxable types will not be unboxed (default)"

let mk_unsafe f =
  "-unsafe", Arg.Unit f,
  " Do not compile bounds checking on array and string access"

let mk_unsafe_string =
 let err () =
   raise (Arg.Bad "-unsafe-string is not available since OCaml 5.0")
 in
 "-unsafe-string", Arg.Unit err, " (option not available)"

let mk_use_runtime f =
  "-use-runtime", Arg.String f,
  "<file>  Generate bytecode for the given runtime system"

let mk_use_runtime_2 f =
  "-use_runtime", Arg.String f,
  "<file>  (deprecated) same as -use-runtime"

let mk_v f =
  "-v", Arg.Unit f,
  " Print compiler version and location of standard library and exit"

let mk_verbose f =
  "-verbose", Arg.Unit f, " Print calls to external commands"

let mk_version f =
  "-version", Arg.Unit f, " Print version and exit"

let mk__version f =
  "--version", Arg.Unit f, " Print version and exit"

let mk_no_version f =
  "-no-version", Arg.Unit f, " Do not print version at startup"

let mk_vmthread f =
  "-vmthread", Arg.Unit f,
  "  (no longer supported)"

let mk_vnum f =
  "-vnum", Arg.Unit f, " Print version number and exit"

let mk_w f =
  "-w", Arg.String f,
  Printf.sprintf
  "<list>  Enable or disable warnings according to <list>:\n\
  \        +<spec>   enable warnings in <spec>\n\
  \        -<spec>   disable warnings in <spec>\n\
  \        @<spec>   enable warnings in <spec> and treat them as errors\n\
  \     <spec> can be:\n\
  \        <num>             a single warning number\n\
  \        <num1>..<num2>    a range of consecutive warning numbers\n\
  \        <letter>          a predefined set\n\
  \     default setting is %S" Warnings.defaults_w

let mk_warn_error f =
  "-warn-error", Arg.String f,
  Printf.sprintf
  "<list>  Enable or disable error status for warnings according\n\
  \     to <list>.  See option -w for the syntax of <list>.\n\
  \     Default setting is %S" Warnings.defaults_warn_error

let mk_warn_help f =
  "-warn-help", Arg.Unit f, " Show description of warning numbers"

let mk_color f =
  "-color", Arg.Symbol (["auto"; "always"; "never"], f),
  Printf.sprintf
  "  Enable or disable colors in compiler messages\n\
  \    The following settings are supported:\n\
  \      auto    use heuristics to enable colors only if supported\n\
  \      always  enable colors\n\
  \      never   disable colors\n\
  \    The default setting is 'auto', and the current heuristic\n\
  \    checks that the TERM environment variable exists and is\n\
  \    not empty or \"dumb\", and that isatty(stderr) holds.\n\
  \  If the option is not specified, these setting can alternatively\n\
  \  be set through the OCAML_COLOR environment variable."

let mk_error_style f =
  "-error-style", Arg.Symbol (["contextual"; "short"], f),
  Printf.sprintf
    "  Control the way error messages and warnings are printed\n\
    \    The following settings are supported:\n\
    \      short       only print the error and its location\n\
    \      contextual  like \"short\", but also display the source code\n\
    \                  snippet corresponding to the location of the error\n\
    \    The default setting is 'contextual'.\n\
    \  If the option is not specified, these setting can alternatively\n\
    \  be set through the OCAML_ERROR_STYLE environment variable."

let mk_where f =
  "-where", Arg.Unit f, " Print location of standard library and exit"

let mk_nopervasives f =
  "-nopervasives", Arg.Unit f, " (undocumented)"

let mk_match_context_rows f =
  "-match-context-rows", Arg.Int f,
  let[@manual.ref "s:comp-options"] chapter, section = 13, 2 in
  Printf.sprintf
  "<n>  (advanced, see manual section %d.%d.)" chapter section

let mk_use_prims f =
  "-use-prims", Arg.String f, "<file>  (undocumented)"

let mk_dump_into_file f =
  "-dump-into-file", Arg.Unit f, " dump output like -dlambda into <target>.dump"

let mk_dump_dir f =
  "-dump-dir", Arg.String f,
  "<dir> dump output like -dlambda into <dir>/<target>.dump"

let mk_dparsetree f =
  "-dparsetree", Arg.Unit f, " (undocumented)"

let mk_dtypedtree f =
  "-dtypedtree", Arg.Unit f, " (undocumented)"

let mk_dshape f =
  "-dshape", Arg.Unit f, " (undocumented)"

let mk_dmatchcomp f =
  "-dmatchcomp", Arg.Unit f, " (undocumented)"

let mk_drawlambda f =
  "-drawlambda", Arg.Unit f, " (undocumented)"

let mk_dno_unique_ids f =
  "-dno-unique-ids", Arg.Unit f, " (undocumented)"

let mk_dunique_ids f =
  "-dunique-ids", Arg.Unit f, " (undocumented)"

let mk_dno_canonical_ids f =
  "-dno-canonical-ids", Arg.Unit f, " (undocumented)"

let mk_dcanonical_ids f =
  "-dcanonical-ids", Arg.Unit f, " (undocumented)"

let mk_dno_locations f =
  "-dno-locations", Arg.Unit f, " (undocumented)"

let mk_dlocations f =
  "-dlocations", Arg.Unit f, " (undocumented)"

let mk_dsource f =
  "-dsource", Arg.Unit f, " (undocumented)"

let mk_dlambda f =
  "-dlambda", Arg.Unit f, " (undocumented)"

let mk_drawclambda f =
  "-drawclambda", Arg.Unit f, " (undocumented)"

let mk_dclambda f =
  "-dclambda", Arg.Unit f, " (undocumented)"

let mk_dflambda f =
  "-dflambda", Arg.Unit f, " Print Flambda terms"

let mk_drawflambda f =
  "-drawflambda", Arg.Unit f, " Print Flambda terms after closure conversion"

let mk_dflambda_invariants f =
  "-dflambda-invariants", Arg.Unit f, " Check Flambda invariants \
      around each pass"

let mk_dflambda_no_invariants f =
  "-dflambda-no-invariants", Arg.Unit f, " Do not Check Flambda invariants \
      around each pass"

let mk_dflambda_let f =
  "-dflambda-let", Arg.Int f, "<stamp>  Print when the given Flambda [Let] \
      is created"

let mk_dflambda_verbose f =
  "-dflambda-verbose", Arg.Unit f, " Print Flambda terms including around \
      each pass"

let mk_dinstr f =
  "-dinstr", Arg.Unit f, " (undocumented)"

let mk_dcamlprimc f =
  "-dcamlprimc", Arg.Unit f, " (undocumented)"

let mk_dcmm_invariants f =
  "-dcmm-invariants", Arg.Unit f, " Extra sanity checks on Cmm"

let mk_dcmm f =
  "-dcmm", Arg.Unit f, " (undocumented)"

let mk_dsel f =
  "-dsel", Arg.Unit f, " (undocumented)"

let mk_dcombine f =
  "-dcombine", Arg.Unit f, " (undocumented)"

let mk_dcse f =
  "-dcse", Arg.Unit f, " (undocumented)"

let mk_dlive f =
  "-dlive", Arg.Unit f, " (undocumented)"

let mk_dspill f =
  "-dspill", Arg.Unit f, " (undocumented)"

let mk_dsplit f =
  "-dsplit", Arg.Unit f, " (undocumented)"

let mk_dinterf f =
  "-dinterf", Arg.Unit f, " (undocumented)"

let mk_dprefer f =
  "-dprefer", Arg.Unit f, " (undocumented)"

let mk_dalloc f =
  "-dalloc", Arg.Unit f, " (undocumented)"

let mk_dreload f =
  "-dreload", Arg.Unit f, " (undocumented)"

let mk_dscheduling f =
  "-dscheduling", Arg.Unit f, " (undocumented)"

let mk_dlinear f =
  "-dlinear", Arg.Unit f, " (undocumented)"

let mk_dinterval f =
  "-dinterval", Arg.Unit f, " (undocumented)"

let mk_dparsetree_loc_ghost_invariants f =
  "-dparsetree-loc-ghost-invariants", Arg.Unit f, " (undocumented)"

let mk_dstartup f =
  "-dstartup", Arg.Unit f, " (undocumented)"

let mk_opaque f =
  "-opaque", Arg.Unit f,
  " Does not generate cross-module optimization information\n\
  \     (reduces necessary recompilation on module change)"

let mk_strict_formats f =
  "-strict-formats", Arg.Unit f,
  " Reject invalid formats accepted by legacy implementations (default)\n\
  \     (Warning: Invalid formats may behave differently from\n\
  \      previous OCaml versions, and will become always-rejected\n\
  \      in future OCaml versions. You should always use this flag\n\
  \      to detect invalid formats so you can fix them.)"

let mk_no_strict_formats f =
  "-no-strict-formats", Arg.Unit f,
  " Accept invalid formats accepted by legacy implementations\n\
  \     (Warning: Invalid formats may behave differently from\n\
  \      previous OCaml versions, and will become always-rejected\n\
  \      in future OCaml versions. You should never use this flag\n\
  \      and instead fix invalid formats.)"

let mk_args f =
  "-args", Arg.Expand f,
  "<file> Read additional newline-terminated command line arguments\n\
  \      from <file>"

let mk_args0 f =
  "-args0", Arg.Expand f,
  "<file> Read additional null character terminated command line arguments\n\
          from <file>"

let mk_afl_instrument f =
  "-afl-instrument", Arg.Unit f, "Enable instrumentation for afl-fuzz"

let mk_afl_inst_ratio f =
  "-afl-inst-ratio", Arg.Int f,
  "Configure percentage of branches instrumented\n\
  \     (advanced, see afl-fuzz docs for AFL_INST_RATIO)"

let mk__ f =
  "-", Arg.String f,
  "<file>  Treat <file> as a file name (even if it starts with `-')"

module type Common_options = sig
  val _absname : unit -> unit
  val _no_absname : unit -> unit
  val _alert : string -> unit
  val _i_variance : unit -> unit
  val _I : string -> unit
  val _H : string -> unit
  val _labels : unit -> unit
  val _alias_deps : unit -> unit
  val _no_alias_deps : unit -> unit
  val _app_funct : unit -> unit
  val _no_app_funct : unit -> unit
  val _noassert : unit -> unit
  val _nolabels : unit -> unit
  val _nostdlib : unit -> unit
  val _nocwd : unit -> unit
  val _open : string -> unit
  val _ppx : string -> unit
  val _keywords: string -> unit
  val _principal : unit -> unit
  val _no_principal : unit -> unit
  val _rectypes : unit -> unit
  val _no_rectypes : unit -> unit
  val _safer_matching : unit -> unit
  val _short_paths : unit -> unit
  val _strict_sequence : unit -> unit
  val _no_strict_sequence : unit -> unit
  val _strict_formats : unit -> unit
  val _no_strict_formats : unit -> unit
  val _unboxed_types : unit -> unit
  val _no_unboxed_types : unit -> unit
  val _version : unit -> unit
  val _vnum : unit -> unit
  val _w : string -> unit

  val anonymous : string -> unit
end

module type Core_options = sig
  include Common_options

  val _nopervasives : unit -> unit
  val _unsafe : unit -> unit
  val _warn_error : string -> unit
  val _warn_help : unit -> unit

  val _dno_unique_ids : unit -> unit
  val _dunique_ids : unit -> unit
  val _dno_canonical_ids : unit -> unit
  val _dcanonical_ids : unit -> unit
  val _dno_locations : unit -> unit
  val _dlocations : unit -> unit

  val _dsource : unit -> unit
  val _dparsetree : unit -> unit
  val _dparsetree_loc_ghost_invariants : unit -> unit
  val _dtypedtree : unit -> unit
  val _dshape : unit -> unit
  val _dmatchcomp : unit -> unit
  val _drawlambda : unit -> unit
  val _dlambda : unit -> unit

end

module type Compiler_options = sig
  val _a : unit -> unit
  val _annot : unit -> unit
  val _binannot : unit -> unit
  val _binannot_occurrences : unit -> unit
  val _c : unit -> unit
  val _cc : string -> unit
  val _cclib : string -> unit
  val _ccopt : string -> unit
  val _cmi_file : string -> unit
  val _config : unit -> unit
  val _config_var : string -> unit
  val _for_pack : string -> unit
  val _g : unit -> unit
  val _no_g : unit -> unit
  val _stop_after : string -> unit
  val _i : unit -> unit
  val _i_variance : unit -> unit
  val _impl : string -> unit
  val _intf : string -> unit
  val _intf_suffix : string -> unit
  val _keep_docs : unit -> unit
  val _no_keep_docs : unit -> unit
  val _keep_locs : unit -> unit
  val _no_keep_locs : unit -> unit
  val _linkall : unit -> unit
  val _noautolink : unit -> unit
  val _o : string -> unit
  val _opaque :  unit -> unit
  val _output_obj : unit -> unit
  val _output_complete_obj : unit -> unit
  val _pack : unit -> unit
  val _plugin : string -> unit
  val _pp : string -> unit
  val _principal : unit -> unit
  val _no_principal : unit -> unit
  val _rectypes : unit -> unit
  val _runtime_variant : string -> unit
  val _with_runtime : unit -> unit
  val _without_runtime : unit -> unit
  val _short_paths : unit -> unit
  val _thread : unit -> unit
  val _v : unit -> unit
  val _verbose : unit -> unit
  val _where : unit -> unit
  val _color : string -> unit
  val _error_style : string -> unit
  val _match_context_rows : int -> unit
  val _dtimings : unit -> unit
  val _dprofile : unit -> unit
  val _dump_into_file : unit -> unit
  val _dump_dir : string -> unit

  val _args: string -> string array
  val _args0: string -> string array
end

module type Toplevel_options = sig
  include Core_options
  val _init : string -> unit
  val _noinit : unit -> unit
  val _no_version : unit -> unit
  val _prompt : unit -> unit
  val _noprompt : unit -> unit
  val _nopromptcont : unit -> unit
  val _stdin : unit -> unit
  val _args : string -> string array
  val _args0 : string -> string array
  val _color : string -> unit
  val _error_style : string -> unit
  val _eval: string -> unit
end

module type Bytecomp_options = sig
  include Core_options
  include Compiler_options
  val _compat_32 : unit -> unit
  val _custom : unit -> unit
  val _no_check_prims : unit -> unit
  val _dllib : string -> unit
  val _dllpath : string -> unit
  val _make_runtime : unit -> unit
  val _vmthread : unit -> unit
  val _use_runtime : string -> unit
  val _output_complete_exe : unit -> unit

  val _dinstr : unit -> unit
  val _dcamlprimc : unit -> unit

  val _use_prims : string -> unit
end;;

module type Bytetop_options = sig
  include Toplevel_options
  val _dinstr : unit -> unit

end;;

module type Optcommon_options = sig
  val _compact : unit -> unit
  val _inline : string -> unit
  val _inline_toplevel : string -> unit
  val _inlining_report : unit -> unit
  val _dump_pass : string -> unit
  val _inline_max_depth : string -> unit
  val _rounds : int -> unit
  val _inline_max_unroll : string -> unit
  val _classic_inlining : unit -> unit
  val _inline_call_cost : string -> unit
  val _inline_alloc_cost : string -> unit
  val _inline_prim_cost : string -> unit
  val _inline_branch_cost : string -> unit
  val _inline_indirect_cost : string -> unit
  val _inline_lifting_benefit : string -> unit
  val _unbox_closures : unit -> unit
  val _unbox_closures_factor : int -> unit
  val _inline_branch_factor : string -> unit
  val _remove_unused_arguments : unit -> unit
  val _no_unbox_free_vars_of_closures : unit -> unit
  val _no_unbox_specialised_args : unit -> unit
  val _o2 : unit -> unit
  val _o3 : unit -> unit
  val _insn_sched : unit -> unit
  val _no_insn_sched : unit -> unit
  val _linscan : unit -> unit
  val _no_float_const_prop : unit -> unit

  val _clambda_checks : unit -> unit
  val _dflambda : unit -> unit
  val _drawflambda : unit -> unit
  val _dflambda_invariants : unit -> unit
  val _dflambda_no_invariants : unit -> unit
  val _dflambda_let : int -> unit
  val _dflambda_verbose : unit -> unit
  val _drawclambda : unit -> unit
  val _dclambda : unit -> unit
  val _dcmm_invariants : unit -> unit
  val _dcmm : unit -> unit
  val _dsel : unit -> unit
  val _dcombine : unit -> unit
  val _dcse : unit -> unit
  val _dlive : unit -> unit
  val _dspill : unit -> unit
  val _dsplit : unit -> unit
  val _dinterf : unit -> unit
  val _dprefer : unit -> unit
  val _dalloc : unit -> unit
  val _dreload : unit -> unit
  val _dscheduling :  unit -> unit
  val _dlinear :  unit -> unit
  val _dinterval : unit -> unit
  val _dstartup :  unit -> unit
end;;

module type Optcomp_options = sig
  include Core_options
  include Compiler_options
  include Optcommon_options
  val _nodynlink : unit -> unit
  val _p : unit -> unit
  val _pp : string -> unit
  val _S : unit -> unit
  val _shared : unit -> unit
  val _afl_instrument : unit -> unit
  val _afl_inst_ratio : int -> unit
  val _function_sections : unit -> unit
  val _save_ir_after : string -> unit
end;;

module type Opttop_options = sig
  include Toplevel_options
  include Optcommon_options
  val _verbose : unit -> unit
  val _S : unit -> unit
end;;

module type Ocamldoc_options = sig
  include Common_options
  val _impl : string -> unit
  val _intf : string -> unit
  val _intf_suffix : string -> unit
  val _pp : string -> unit
  val _thread : unit -> unit
  val _v : unit -> unit
  val _verbose : unit -> unit
  val _vmthread : unit -> unit
end

module type Arg_list = sig
    val list : (string * Arg.spec * string) list
end;;

module Make_bytecomp_options (F : Bytecomp_options) =
struct
  let list = [
    mk_a F._a;
    mk_alert F._alert;
    mk_absname F._absname;
    mk_no_absname F._no_absname;
    mk_annot F._annot;
    mk_binannot F._binannot;
    mk_binannot_occurrences F._binannot_occurrences;
    mk_c F._c;
    mk_cc F._cc;
    mk_cclib F._cclib;
    mk_ccopt F._ccopt;
    mk_cmi_file F._cmi_file;
    mk_color F._color;
    mk_error_style F._error_style;
    mk_compat_32 F._compat_32;
    mk_config F._config;
    mk_config_var F._config_var;
    mk_custom F._custom;
    mk_dllib F._dllib;
    mk_dllpath F._dllpath;
    mk_dtypes F._annot;
    mk_for_pack_byt F._for_pack;
    mk_g_byt F._g;
    mk_no_g F._no_g;
    mk_stop_after ~native:false F._stop_after;
    mk_i F._i;
    mk_i_variance F._i_variance;
    mk_I F._I;
    mk_H F._H;
    mk_impl F._impl;
    mk_intf F._intf;
    mk_intf_suffix F._intf_suffix;
    mk_intf_suffix_2 F._intf_suffix;
    mk_keep_docs F._keep_docs;
    mk_no_keep_docs F._no_keep_docs;
    mk_keep_locs F._keep_locs;
    mk_no_keep_locs F._no_keep_locs;
    mk_keywords F._keywords;
    mk_labels F._labels;
    mk_linkall F._linkall;
    mk_make_runtime F._make_runtime;
    mk_make_runtime_2 F._make_runtime;
    mk_modern F._labels;
    mk_alias_deps F._alias_deps;
    mk_no_alias_deps F._no_alias_deps;
    mk_app_funct F._app_funct;
    mk_no_app_funct F._no_app_funct;
    mk_no_check_prims F._no_check_prims;
    mk_noassert F._noassert;
    mk_noautolink_byt F._noautolink;
    mk_nolabels F._nolabels;
    mk_nostdlib F._nostdlib;
    mk_nocwd F._nocwd;
    mk_nopervasives F._nopervasives;
    mk_o F._o;
    mk_opaque F._opaque;
    mk_open F._open;
    mk_output_obj F._output_obj;
    mk_output_complete_obj F._output_complete_obj;
    mk_output_complete_exe F._output_complete_exe;
    mk_pack_byt F._pack;
    mk_pp F._pp;
    mk_ppx F._ppx;
    mk_plugin F._plugin;
    mk_principal F._principal;
    mk_no_principal F._no_principal;
    mk_rectypes F._rectypes;
    mk_no_rectypes F._no_rectypes;
    mk_runtime_variant F._runtime_variant;
    mk_with_runtime F._with_runtime;
    mk_without_runtime F._without_runtime;
    mk_safe_string;
    mk_safer_matching F._safer_matching;
    mk_short_paths F._short_paths;
    mk_strict_sequence F._strict_sequence;
    mk_no_strict_sequence F._no_strict_sequence;
    mk_strict_formats F._strict_formats;
    mk_no_strict_formats F._no_strict_formats;
    mk_thread F._thread;
    mk_unboxed_types F._unboxed_types;
    mk_no_unboxed_types F._no_unboxed_types;
    mk_unsafe F._unsafe;
    mk_unsafe_string;
    mk_use_runtime F._use_runtime;
    mk_use_runtime_2 F._use_runtime;
    mk_v F._v;
    mk_verbose F._verbose;
    mk_version F._version;
    mk__version F._version;
    mk_vmthread F._vmthread;
    mk_vnum F._vnum;
    mk_w F._w;
    mk_warn_error F._warn_error;
    mk_warn_help F._warn_help;
    mk_where F._where;
    mk__ F.anonymous;

    mk_match_context_rows F._match_context_rows;
    mk_use_prims F._use_prims;
    mk_dno_unique_ids F._dno_unique_ids;
    mk_dunique_ids F._dunique_ids;
    mk_dno_canonical_ids F._dno_canonical_ids;
    mk_dcanonical_ids F._dcanonical_ids;
    mk_dno_locations F._dno_locations;
    mk_dlocations F._dlocations;
    mk_dsource F._dsource;
    mk_dparsetree F._dparsetree;
    mk_dparsetree_loc_ghost_invariants F._dparsetree_loc_ghost_invariants;
    mk_dtypedtree F._dtypedtree;
    mk_dshape F._dshape;
    mk_dmatchcomp F._dmatchcomp;
    mk_drawlambda F._drawlambda;
    mk_dlambda F._dlambda;
    mk_dinstr F._dinstr;
    mk_dcamlprimc F._dcamlprimc;
    mk_dtimings F._dtimings;
    mk_dprofile F._dprofile;
    mk_dump_into_file F._dump_into_file;
    mk_dump_dir F._dump_dir;

    mk_args F._args;
    mk_args0 F._args0;
  ]
end;;

module Make_bytetop_options (F : Bytetop_options) =
struct
  let list = [
    mk_absname F._absname;
    mk_no_absname F._no_absname;
    mk_alert F._alert;
    mk_i_variance F._i_variance;
    mk_I F._I;
    mk_H F._H;
    mk_init F._init;
    mk_labels F._labels;
    mk_alias_deps F._alias_deps;
    mk_no_alias_deps F._no_alias_deps;
    mk_app_funct F._app_funct;
    mk_no_app_funct F._no_app_funct;
    mk_noassert F._noassert;
    mk_noinit F._noinit;
    mk_nolabels F._nolabels;
    mk_prompt F._prompt;
    mk_noprompt F._noprompt;
    mk_nopromptcont F._nopromptcont;
    mk_nostdlib F._nostdlib;
    mk_nocwd F._nocwd;
    mk_nopervasives F._nopervasives;
    mk_open F._open;
    mk_ppx F._ppx;
    mk_keywords F._keywords;
    mk_principal F._principal;
    mk_no_principal F._no_principal;
    mk_rectypes F._rectypes;
    mk_no_rectypes F._no_rectypes;
    mk_safe_string;
    mk_safer_matching F._safer_matching;
    mk_short_paths F._short_paths;
    mk_stdin F._stdin;
    mk_strict_sequence F._strict_sequence;
    mk_no_strict_sequence F._no_strict_sequence;
    mk_strict_formats F._strict_formats;
    mk_no_strict_formats F._no_strict_formats;
    mk_unboxed_types F._unboxed_types;
    mk_no_unboxed_types F._no_unboxed_types;
    mk_unsafe F._unsafe;
    mk_unsafe_string;
    mk_version F._version;
    mk__version F._version;
    mk_no_version F._no_version;
    mk_vnum F._vnum;
    mk_w F._w;
    mk_warn_error F._warn_error;
    mk_warn_help F._warn_help;
    mk__ F.anonymous;
    mk_color F._color;
    mk_error_style F._error_style;

    mk_dno_unique_ids F._dno_unique_ids;
    mk_dunique_ids F._dunique_ids;
    mk_dno_canonical_ids F._dno_canonical_ids;
    mk_dcanonical_ids F._dcanonical_ids;
    mk_dno_locations F._dno_locations;
    mk_dlocations F._dlocations;
    mk_dsource F._dsource;
    mk_dparsetree F._dparsetree;
    mk_dparsetree_loc_ghost_invariants F._dparsetree_loc_ghost_invariants;
    mk_dtypedtree F._dtypedtree;
    mk_dshape F._dshape;
    mk_dmatchcomp F._dmatchcomp;
    mk_drawlambda F._drawlambda;
    mk_dlambda F._dlambda;
    mk_dinstr F._dinstr;

    mk_args F._args;
    mk_args0 F._args0;
    mk_eval F._eval;
  ]
end;;

module Make_optcomp_options (F : Optcomp_options) =
struct
  let list = [
    mk_a F._a;
    mk_alert F._alert;
    mk_absname F._absname;
    mk_no_absname F._no_absname;
    mk_afl_instrument F._afl_instrument;
    mk_afl_inst_ratio F._afl_inst_ratio;
    mk_annot F._annot;
    mk_binannot F._binannot;
    mk_binannot_occurrences F._binannot_occurrences;
    mk_inline_branch_factor F._inline_branch_factor;
    mk_c F._c;
    mk_cc F._cc;
    mk_cclib F._cclib;
    mk_ccopt F._ccopt;
    mk_cmi_file F._cmi_file;
    mk_clambda_checks F._clambda_checks;
    mk_classic_inlining F._classic_inlining;
    mk_color F._color;
    mk_error_style F._error_style;
    mk_compact F._compact;
    mk_config F._config;
    mk_config_var F._config_var;
    mk_dtypes F._annot;
    mk_for_pack_opt F._for_pack;
    mk_g_opt F._g;
    mk_no_g F._no_g;
    mk_function_sections F._function_sections;
    mk_stop_after ~native:true F._stop_after;
    mk_save_ir_after ~native:true F._save_ir_after;
    mk_i F._i;
    mk_i_variance F._i_variance;
    mk_I F._I;
    mk_H F._H;
    mk_impl F._impl;
    mk_inline F._inline;
    mk_inline_toplevel F._inline_toplevel;
    mk_inline_alloc_cost F._inline_alloc_cost;
    mk_inline_branch_cost F._inline_branch_cost;
    mk_inline_call_cost F._inline_call_cost;
    mk_inline_prim_cost F._inline_prim_cost;
    mk_inline_indirect_cost F._inline_indirect_cost;
    mk_inline_lifting_benefit F._inline_lifting_benefit;
    mk_inlining_report F._inlining_report;
    mk_insn_sched F._insn_sched;
    mk_intf F._intf;
    mk_intf_suffix F._intf_suffix;
    mk_keep_docs F._keep_docs;
    mk_no_keep_docs F._no_keep_docs;
    mk_keep_locs F._keep_locs;
    mk_no_keep_locs F._no_keep_locs;
    mk_keywords F._keywords;
    mk_labels F._labels;
    mk_linkall F._linkall;
    mk_inline_max_depth F._inline_max_depth;
    mk_alias_deps F._alias_deps;
    mk_no_alias_deps F._no_alias_deps;
    mk_linscan F._linscan;
    mk_app_funct F._app_funct;
    mk_no_app_funct F._no_app_funct;
    mk_no_float_const_prop F._no_float_const_prop;
    mk_noassert F._noassert;
    mk_noautolink_opt F._noautolink;
    mk_nodynlink F._nodynlink;
    mk_no_insn_sched F._no_insn_sched;
    mk_nolabels F._nolabels;
    mk_nostdlib F._nostdlib;
    mk_nocwd F._nocwd;
    mk_nopervasives F._nopervasives;
    mk_no_unbox_free_vars_of_closures F._no_unbox_free_vars_of_closures;
    mk_no_unbox_specialised_args F._no_unbox_specialised_args;
    mk_o F._o;
    mk_o2 F._o2;
    mk_o3 F._o3;
    mk_opaque F._opaque;
    mk_open F._open;
    mk_output_obj F._output_obj;
    mk_output_complete_obj F._output_complete_obj;
    mk_p F._p;
    mk_pack_opt F._pack;
    mk_plugin F._plugin;
    mk_pp F._pp;
    mk_ppx F._ppx;
    mk_principal F._principal;
    mk_no_principal F._no_principal;
    mk_rectypes F._rectypes;
    mk_no_rectypes F._no_rectypes;
    mk_remove_unused_arguments F._remove_unused_arguments;
    mk_rounds F._rounds;
    mk_runtime_variant F._runtime_variant;
    mk_with_runtime F._with_runtime;
    mk_without_runtime F._without_runtime;
    mk_S F._S;
    mk_safe_string;
    mk_safer_matching F._safer_matching;
    mk_shared F._shared;
    mk_short_paths F._short_paths;
    mk_strict_sequence F._strict_sequence;
    mk_no_strict_sequence F._no_strict_sequence;
    mk_strict_formats F._strict_formats;
    mk_no_strict_formats F._no_strict_formats;
    mk_thread F._thread;
    mk_unbox_closures F._unbox_closures;
    mk_unbox_closures_factor F._unbox_closures_factor;
    mk_inline_max_unroll F._inline_max_unroll;
    mk_unboxed_types F._unboxed_types;
    mk_no_unboxed_types F._no_unboxed_types;
    mk_unsafe F._unsafe;
    mk_unsafe_string;
    mk_v F._v;
    mk_verbose F._verbose;
    mk_version F._version;
    mk__version F._version;
    mk_vnum F._vnum;
    mk_w F._w;
    mk_warn_error F._warn_error;
    mk_warn_help F._warn_help;
    mk_where F._where;
    mk__ F.anonymous;

    mk_match_context_rows F._match_context_rows;
    mk_dno_unique_ids F._dno_unique_ids;
    mk_dunique_ids F._dunique_ids;
    mk_dno_canonical_ids F._dno_canonical_ids;
    mk_dcanonical_ids F._dcanonical_ids;
    mk_dno_locations F._dno_locations;
    mk_dlocations F._dlocations;
    mk_dsource F._dsource;
    mk_dparsetree F._dparsetree;
    mk_dparsetree_loc_ghost_invariants F._dparsetree_loc_ghost_invariants;
    mk_dtypedtree F._dtypedtree;
    mk_dshape F._dshape;
    mk_dmatchcomp F._dmatchcomp;
    mk_drawlambda F._drawlambda;
    mk_dlambda F._dlambda;
    mk_drawclambda F._drawclambda;
    mk_dclambda F._dclambda;
    mk_dcmm_invariants F._dcmm_invariants;
    mk_dflambda F._dflambda;
    mk_drawflambda F._drawflambda;
    mk_dflambda_invariants F._dflambda_invariants;
    mk_dflambda_no_invariants F._dflambda_no_invariants;
    mk_dflambda_let F._dflambda_let;
    mk_dflambda_verbose F._dflambda_verbose;
    mk_dcmm F._dcmm;
    mk_dsel F._dsel;
    mk_dcombine F._dcombine;
    mk_dcse F._dcse;
    mk_dlive F._dlive;
    mk_dspill F._dspill;
    mk_dsplit F._dsplit;
    mk_dinterf F._dinterf;
    mk_dprefer F._dprefer;
    mk_dalloc F._dalloc;
    mk_dreload F._dreload;
    mk_dscheduling F._dscheduling;
    mk_dlinear F._dlinear;
    mk_dinterval F._dinterval;
    mk_dstartup F._dstartup;
    mk_dtimings F._dtimings;
    mk_dprofile F._dprofile;
    mk_dump_into_file F._dump_into_file;
    mk_dump_dir F._dump_dir;
    mk_dump_pass F._dump_pass;

    mk_args F._args;
    mk_args0 F._args0;
  ]
end;;

module Make_opttop_options (F : Opttop_options) = struct
  let list = [
    mk_absname F._absname;
    mk_no_absname F._no_absname;
    mk_alert F._alert;
    mk_compact F._compact;
    mk_i_variance F._i_variance;
    mk_I F._I;
    mk_H F._H;
    mk_init F._init;
    mk_inline F._inline;
    mk_inline_toplevel F._inline_toplevel;
    mk_inlining_report F._inlining_report;
    mk_rounds F._rounds;
    mk_inline_max_unroll F._inline_max_unroll;
    mk_classic_inlining F._classic_inlining;
    mk_inline_call_cost F._inline_call_cost;
    mk_inline_alloc_cost F._inline_alloc_cost;
    mk_inline_prim_cost F._inline_prim_cost;
    mk_inline_branch_cost F._inline_branch_cost;
    mk_inline_indirect_cost F._inline_indirect_cost;
    mk_inline_lifting_benefit F._inline_lifting_benefit;
    mk_inline_branch_factor F._inline_branch_factor;
    mk_keywords F._keywords;
    mk_labels F._labels;
    mk_alias_deps F._alias_deps;
    mk_no_alias_deps F._no_alias_deps;
    mk_linscan F._linscan;
    mk_app_funct F._app_funct;
    mk_no_app_funct F._no_app_funct;
    mk_no_float_const_prop F._no_float_const_prop;
    mk_noassert F._noassert;
    mk_noinit F._noinit;
    mk_nolabels F._nolabels;
    mk_prompt F._prompt;
    mk_noprompt F._noprompt;
    mk_nopromptcont F._nopromptcont;
    mk_nostdlib F._nostdlib;
    mk_nocwd F._nocwd;
    mk_nopervasives F._nopervasives;
    mk_no_unbox_free_vars_of_closures F._no_unbox_free_vars_of_closures;
    mk_no_unbox_specialised_args F._no_unbox_specialised_args;
    mk_o2 F._o2;
    mk_o3 F._o3;
    mk_open F._open;
    mk_ppx F._ppx;
    mk_principal F._principal;
    mk_no_principal F._no_principal;
    mk_rectypes F._rectypes;
    mk_no_rectypes F._no_rectypes;
    mk_remove_unused_arguments F._remove_unused_arguments;
    mk_S F._S;
    mk_safe_string;
    mk_safer_matching F._safer_matching;
    mk_short_paths F._short_paths;
    mk_stdin F._stdin;
    mk_strict_sequence F._strict_sequence;
    mk_no_strict_sequence F._no_strict_sequence;
    mk_strict_formats F._strict_formats;
    mk_no_strict_formats F._no_strict_formats;
    mk_unbox_closures F._unbox_closures;
    mk_unbox_closures_factor F._unbox_closures_factor;
    mk_unboxed_types F._unboxed_types;
    mk_no_unboxed_types F._no_unboxed_types;
    mk_unsafe F._unsafe;
    mk_unsafe_string;
    mk_verbose F._verbose;
    mk_version F._version;
    mk__version F._version;
    mk_no_version F._no_version;
    mk_vnum F._vnum;
    mk_w F._w;
    mk_warn_error F._warn_error;
    mk_warn_help F._warn_help;
    mk__ F.anonymous;
    mk_color F._color;
    mk_error_style F._error_style;

    mk_dsource F._dsource;
    mk_dparsetree F._dparsetree;
    mk_dparsetree_loc_ghost_invariants F._dparsetree_loc_ghost_invariants;
    mk_dtypedtree F._dtypedtree;
    mk_dshape F._dshape;
    mk_dmatchcomp F._dmatchcomp;
    mk_drawlambda F._drawlambda;
    mk_dlambda F._dlambda;
    mk_drawclambda F._drawclambda;
    mk_dclambda F._dclambda;
    mk_dcmm_invariants F._dcmm_invariants;
    mk_drawflambda F._drawflambda;
    mk_dflambda F._dflambda;
    mk_dcmm F._dcmm;
    mk_dsel F._dsel;
    mk_dcombine F._dcombine;
    mk_dcse F._dcse;
    mk_dlive F._dlive;
    mk_dspill F._dspill;
    mk_dsplit F._dsplit;
    mk_dinterf F._dinterf;
    mk_dprefer F._dprefer;
    mk_dalloc F._dalloc;
    mk_dreload F._dreload;
    mk_dscheduling F._dscheduling;
    mk_dlinear F._dlinear;
    mk_dinterval F._dinterval;
    mk_dstartup F._dstartup;
    mk_dump_pass F._dump_pass;
    mk_eval F._eval;
  ]
end;;

module Make_ocamldoc_options (F : Ocamldoc_options) =
struct
  let list = [
    mk_absname F._absname;
    mk_no_absname F._no_absname;
    mk_alert F._alert;
    mk_i_variance F._i_variance;
    mk_I F._I;
    mk_H F._H;
    mk_impl F._impl;
    mk_intf F._intf;
    mk_intf_suffix F._intf_suffix;
    mk_intf_suffix_2 F._intf_suffix;
    mk_keywords F._keywords;
    mk_labels F._labels;
    mk_modern F._labels;
    mk_alias_deps F._alias_deps;
    mk_no_alias_deps F._no_alias_deps;
    mk_app_funct F._app_funct;
    mk_no_app_funct F._no_app_funct;
    mk_noassert F._noassert;
    mk_nolabels F._nolabels;
    mk_nostdlib F._nostdlib;
    mk_nocwd F._nocwd;
    mk_open F._open;
    mk_pp F._pp;
    mk_ppx F._ppx;
    mk_principal F._principal;
    mk_no_principal F._no_principal;
    mk_rectypes F._rectypes;
    mk_no_rectypes F._no_rectypes;
    mk_safe_string;
    mk_short_paths F._short_paths;
    mk_strict_sequence F._strict_sequence;
    mk_no_strict_sequence F._no_strict_sequence;
    mk_strict_formats F._strict_formats;
    mk_no_strict_formats F._no_strict_formats;
    mk_thread F._thread;
    mk_unboxed_types F._unboxed_types;
    mk_no_unboxed_types F._no_unboxed_types;
    mk_unsafe_string;
    mk_v F._v;
    mk_verbose F._verbose;
    mk_version F._version;
    mk__version F._version;
    mk_vmthread F._vmthread;
    mk_vnum F._vnum;
    mk_w F._w;
    mk__ F.anonymous;
  ]
end;;

[@@@ocaml.warning "-40"]
let options_with_command_line_syntax_inner r after_rest =
  let rec loop ~name_opt (spec : Arg.spec) : Arg.spec =
    let option =
      match name_opt with
      | None -> ignore
      | Some name -> (fun () -> r := name :: !r)
    in
    let arg a = r := Filename.quote a :: !r in
    let option_with_arg a = option (); arg a in
    let rest a =
      if not !after_rest then (after_rest := true; option ());
      arg a
    in
    let rest_all a = option (); List.iter arg a in
    match spec with
    | Unit f -> Unit (fun a -> f a; option ())
    | Bool f -> Bool (fun a -> f a; option_with_arg (string_of_bool a))
    | Set r -> Unit (fun () -> r := true; option ())
    | Clear r -> Unit (fun () -> r := false; option ())
    | String f -> String (fun a -> f a; option_with_arg a)
    | Set_string r -> String (fun a -> r := a; option_with_arg a)
    | Int f -> Int (fun a -> f a; option_with_arg (string_of_int a))
    | Set_int r -> Int (fun a -> r := a; option_with_arg (string_of_int a))
    | Float f -> Float (fun a -> f a; option_with_arg (string_of_float a))
    | Set_float r ->
       Float (fun a -> r := a; option_with_arg (string_of_float a))
    | Tuple [] -> Unit option
    | Tuple (hd :: tl) ->
       Tuple (loop ~name_opt hd :: List.map (loop ~name_opt:None) tl)
    | Symbol (l, f) -> Symbol (l, (fun a -> f a; option_with_arg a))
    | Rest f -> Rest (fun a -> f a; rest a)
    | Rest_all f -> Rest_all (fun a -> f a; rest_all a)
    | Expand f -> Expand f
  in
  loop

let options_with_command_line_syntax options r =
  let rest = ref false in
  List.map (fun (name, spec, doc) ->
    (name,
     options_with_command_line_syntax_inner r rest
       ~name_opt:(Some name) spec, doc)
  ) options

module Default = struct
  open Clflags
  let set r () = r := true
  let clear r () = r := false

  module Common = struct
    let _absname = set Clflags.absname
    let _alert = Warnings.parse_alert_option
    let _alias_deps = clear no_alias_deps
    let _app_funct = set applicative_functors
    let _i_variance = set print_variance
    let _labels = clear classic
    let _no_absname = clear Clflags.absname
    let _no_alias_deps = set no_alias_deps
    let _no_app_funct = clear applicative_functors
    let _no_principal = clear principal
    let _no_rectypes = clear recursive_types
    let _no_strict_formats = clear strict_formats
    let _no_strict_sequence = clear strict_sequence
    let _no_unboxed_types = clear unboxed_types
    let _noassert = set noassert
    let _nolabels = set classic
    let _nostdlib = set no_std_include
    let _nocwd = set no_cwd
    let _open s = open_modules := (s :: (!open_modules))
    let _principal = set principal
    let _rectypes = set recursive_types
    let _safer_matching = set safer_matching
    let _short_paths = clear real_paths
    let _strict_formats = set strict_formats
    let _strict_sequence = set strict_sequence
    let _unboxed_types = set unboxed_types
    let _w s =
      Warnings.parse_options false s |> Option.iter Location.(prerr_alert none)

    let anonymous = Compenv.anonymous

  end

  module Core = struct
    include Common
    let _I dir = include_dirs := dir :: (!include_dirs)
    let _H dir = hidden_include_dirs := dir :: (!hidden_include_dirs)
    let _color = Misc.set_or_ignore color_reader.parse color
    let _dlambda = set dump_lambda
    let _dparsetree = set dump_parsetree
    let _dparsetree_loc_ghost_invariants = set parsetree_ghost_loc_invariant
    let _drawlambda = set dump_rawlambda
    let _dsource = set dump_source
    let _dtypedtree = set dump_typedtree
    let _dshape = set dump_shape
    let _dmatchcomp = set dump_matchcomp
    let _dunique_ids = set unique_ids
    let _dno_unique_ids = clear unique_ids
    let _dcanonical_ids = set canonical_ids
    let _dno_canonical_ids = clear canonical_ids
    let _dlocations = set locations
    let _dno_locations = clear locations
    let _error_style =
      Misc.set_or_ignore error_style_reader.parse error_style
    let _nopervasives = set nopervasives
    let _ppx s = Compenv.first_ppx := (s :: (!Compenv.first_ppx))
    let _keywords s = Clflags.keyword_edition := (Some s)
    let _unsafe = set unsafe
    let _warn_error s =
      Warnings.parse_options true s |> Option.iter Location.(prerr_alert none)
    let _warn_help = Warnings.help_warnings
  end

  module Native = struct
    let _S = set keep_asm_file
    let _clambda_checks () = clambda_checks := true
    let _classic_inlining () = classic_inlining := true
    let _compact = clear optimize_for_speed
    let _dalloc = set dump_regalloc
    let _dclambda = set dump_clambda
    let _dcmm = set dump_cmm
    let _dcmm_invariants = set cmm_invariants
    let _dcombine = set dump_combine
    let _dcse = set dump_cse
    let _dflambda = set dump_flambda
    let _dflambda_invariants = set flambda_invariant_checks
    let _dflambda_let stamp = dump_flambda_let := (Some stamp)
    let _dflambda_no_invariants = clear flambda_invariant_checks
    let _dflambda_verbose () =
      set dump_flambda (); set dump_flambda_verbose ()
    let _dinterval = set dump_interval
    let _dinterf = set dump_interf
    let _dlinear = set dump_linear
    let _dlive () = dump_live := true
    let _dprefer = set dump_prefer
    let _drawclambda = set dump_rawclambda
    let _drawflambda = set dump_rawflambda
    let _dreload = set dump_reload
    let _dscheduling = set dump_scheduling
    let _dsel = set dump_selection
    let _dspill = set dump_spill
    let _dsplit = set dump_split
    let _dstartup = set keep_startup_file
    let _dump_pass pass = set_dumped_pass pass true
    let _inline spec =
      Float_arg_helper.parse spec "Syntax: -inline <n> | <round>=<n>[,...]"
        inline_threshold
    let _inline_alloc_cost spec =
      Int_arg_helper.parse spec
        "Syntax: -inline-alloc-cost <n> | <round>=<n>[,...]"
        inline_alloc_cost
    let _inline_branch_cost spec =
      Int_arg_helper.parse spec
        "Syntax: -inline-branch-cost <n> | <round>=<n>[,...]"
        inline_branch_cost
    let _inline_branch_factor spec =
      Float_arg_helper.parse spec
        "Syntax: -inline-branch-factor <n> | <round>=<n>[,...]"
        inline_branch_factor
    let _inline_call_cost spec =
      Int_arg_helper.parse spec
        "Syntax: -inline-call-cost <n> | <round>=<n>[,...]" inline_call_cost
    let _inline_indirect_cost spec =
      Int_arg_helper.parse spec
        "Syntax: -inline-indirect-cost <n> | <round>=<n>[,...]"
        inline_indirect_cost
    let _inline_lifting_benefit spec =
      Int_arg_helper.parse spec
        "Syntax: -inline-lifting-benefit <n> | <round>=<n>[,...]"
        inline_lifting_benefit
    let _inline_max_depth spec =
      Int_arg_helper.parse spec
        "Syntax: -inline-max-depth <n> | <round>=<n>[,...]" inline_max_depth
    let _inline_max_unroll spec =
      Int_arg_helper.parse spec
        "Syntax: -inline-max-unroll <n> | <round>=<n>[,...]"
        inline_max_unroll
    let _inline_prim_cost spec =
      Int_arg_helper.parse spec
        "Syntax: -inline-prim-cost <n> | <round>=<n>[,...]" inline_prim_cost
    let _inline_toplevel spec =
      Int_arg_helper.parse spec
        "Syntax: -inline-toplevel <n> | <round>=<n>[,...]"
        inline_toplevel_threshold
    let _inlining_report () = inlining_report := true
    let _insn_sched = set insn_sched
    let _no_insn_sched = clear insn_sched
    let _linscan = set use_linscan
    let _no_float_const_prop = clear float_const_prop
    let _no_unbox_free_vars_of_closures = clear unbox_free_vars_of_closures
    let _no_unbox_specialised_args = clear unbox_specialised_args
    (* CR-someday mshinwell: should stop e.g. -O2 -classic-inlining
       lgesbert: could be done in main() below, like for -pack and -c, but that
       would prevent overriding using OCAMLPARAM.
       mshinwell: We're going to defer this for the moment and add a note in
       the manual that the behaviour is unspecified in cases such as this.
       We should refactor the code so that the user's requirements are
       collected, then checked all at once for illegal combinations, and then
       transformed into the settings of the individual parameters.
    *)
    let _o2 () =
      default_simplify_rounds := 2;
      use_inlining_arguments_set o2_arguments;
      use_inlining_arguments_set ~round:0 o1_arguments
    let _o3 () =
      default_simplify_rounds := 3;
      use_inlining_arguments_set o3_arguments;
      use_inlining_arguments_set ~round:1 o2_arguments;
      use_inlining_arguments_set ~round:0 o1_arguments
    let _remove_unused_arguments = set remove_unused_arguments
    let _rounds n = simplify_rounds := (Some n)
    let _unbox_closures = set unbox_closures
    let _unbox_closures_factor f = unbox_closures_factor := f
    let _verbose = set verbose
  end

  module Compiler = struct
    let _a = set make_archive
    let _annot = set annotations
    let _args = Arg.read_arg
    let _args0 = Arg.read_arg0
    let _binannot = set binary_annotations
    let _binannot_occurrences = set store_occurrences
    let _c = set compile_only
    let _cc s = c_compiler := (Some s)
    let _cclib s = Compenv.defer (ProcessObjects (Misc.rev_split_words s))
    let _ccopt s = Compenv.first_ccopts := (s :: (!Compenv.first_ccopts))
    let _cmi_file s = cmi_file := (Some s)
    let _config = Misc.show_config_and_exit
    let _config_var = Misc.show_config_variable_and_exit
    let _dprofile () = profile_columns := Profile.all_columns
    let _dtimings () = profile_columns := [`Time]
    let _dump_into_file = set dump_into_file
    let _dump_dir s = dump_dir := Some s
    let _for_pack s = for_package := (Some s)
    let _g = set debug
    let _no_g = clear debug
    let _i = set print_types
    let _impl = Compenv.impl
    let _intf = Compenv.intf
    let _intf_suffix s = Config.interface_suffix := s
    let _keep_docs = set keep_docs
    let _keep_locs = set keep_locs
    let _linkall = set link_everything
    let _match_context_rows n = match_context_rows := n
    let _no_keep_docs = clear keep_docs
    let _no_keep_locs = clear keep_locs
    let _noautolink = set no_auto_link
    let _o s = output_name := (Some s)
    let _opaque = set opaque
    let _pack = set make_package
    let _plugin _p = plugin := true
    let _pp s = preprocessor := (Some s)
    let _runtime_variant s = runtime_variant := s
    let _stop_after pass =
      let module P = Compiler_pass in
        match P.of_string pass with
        | None -> () (* this should not occur as we use Arg.Symbol *)
        | Some pass ->
          match !stop_after with
          | None -> stop_after := (Some pass)
          | Some p ->
            if not (p = pass) then
              Compenv.fatal "Please specify at most one -stop-after <pass>."
    let _save_ir_after pass =
      let module P = Compiler_pass in
        match P.of_string pass with
        | None -> () (* this should not occur as we use Arg.Symbol *)
        | Some pass ->
          set_save_ir_after pass true
    let _thread = set use_threads
    let _verbose = set verbose
    let _version () = Compenv.print_version_string ()
    let _vnum () = Compenv.print_version_string ()
    let _where () = Compenv.print_standard_library ()
    let _with_runtime = set with_runtime
    let _without_runtime = clear with_runtime
  end

  module Toplevel = struct

    let print_version () =
      Printf.printf "The OCaml toplevel, version %s\n" Sys.ocaml_version;
      raise (Compenv.Exit_with_status 0);
    ;;

    let print_version_num () =
      Printf.printf "%s\n" Sys.ocaml_version;
      raise (Compenv.Exit_with_status 0);
    ;;

    let _args (_:string) = (* placeholder: wrap_expand Arg.read_arg *) [||]
    let _args0 (_:string) = (* placeholder: wrap_expand Arg.read_arg0 *) [||]
    let _init s = init_file := (Some s)
    let _no_version = set noversion
    let _noinit = set noinit
    let _prompt = clear noprompt
    let _noprompt = set noprompt
    let _nopromptcont = set nopromptcont
    let _stdin () = (* placeholder: file_argument ""*) ()
    let _version () = print_version ()
    let _vnum () = print_version_num ()
    let _eval (_:string) = ()
  end

  module Topmain = struct
    include Toplevel
    include Core
    let _dinstr = set dump_instr
  end

  module Opttopmain = struct
    include Toplevel
    include Native
    include Core
  end

  module Optmain = struct
    include Native
    include Core
    include Compiler
    let _afl_inst_ratio n = afl_inst_ratio := n
    let _afl_instrument = set afl_instrument
    let _function_sections () =
      assert Config.function_sections;
      Compenv.first_ccopts := ("-ffunction-sections" ::(!Compenv.first_ccopts));
      function_sections := true
    let _nodynlink = clear dlcode
    let _output_complete_obj () =
      set output_c_object (); set output_complete_object ()
    let _output_obj = set output_c_object
    let _p () =
      Compenv.fatal
        "Profiling with \"gprof\" (option `-p') is only supported up to \
         OCaml 4.08.0"
    let _shared () = shared := true; dlcode := true
    let _v () = Compenv.print_version_and_library "native-code compiler"
  end

  module Odoc_args = struct
    include Common
    let _I(_:string) =
      (* placeholder:
         Odoc_global.include_dirs := (s :: (!Odoc_global.include_dirs))
      *) ()
    let _H(_:string) =
      (* placeholder:
         Odoc_global.hidden_include_dirs :=
           (s :: (!Odoc_global.hidden_include_dirs))
      *) ()
    let _impl (_:string) =
      (* placeholder:
         Odoc_global.files := ((!Odoc_global.files) @ [Odoc_global.Impl_file s])
      *) ()
    let _intf (_:string) = (* placeholder:
      Odoc_global.files := ((!Odoc_global.files) @ [Odoc_global.Intf_file s])
                  *) ()
    let _intf_suffix s = Config.interface_suffix := s
    let _pp s = Clflags.preprocessor := (Some s)
    let _ppx s = Clflags.all_ppx := (s :: (!Clflags.all_ppx))
    let _keywords s = Clflags.keyword_edition := Some s
    let _thread = set Clflags.use_threads
    let _v () = Compenv.print_version_and_library "documentation generator"
    let _verbose = set Clflags.verbose
    let _version = Compenv.print_version_string
    let _vmthread = ignore
    let _vnum = Compenv.print_version_string
  end

  module Main = struct

    let vmthread_removed_message = "\
The -vmthread argument of ocamlc is no longer supported\n\
since OCaml 4.09.0.  Please switch to system threads, which have the\n\
same API. Lightweight threads with VM-level scheduling are provided by\n\
third-party libraries such as Lwt, but with a different API."

    include Core
    include Compiler
    let _compat_32 = set bytecode_compatible_32
    let _custom = set custom_runtime
    let _dcamlprimc = set keep_camlprimc_file
    let _dinstr = set dump_instr
    let _dllib s = Compenv.defer (ProcessDLLs (Misc.rev_split_words s))
    let _dllpath s = dllpaths := ((!dllpaths) @ [s])
    let _make_runtime () =
      custom_runtime := true; make_runtime := true; link_everything := true
    let _no_check_prims = set no_check_prims
    let _output_complete_obj () =
      output_c_object := true;
      output_complete_object := true;
      custom_runtime := true
    let _output_complete_exe () =
      _output_complete_obj (); output_complete_executable := true
    let _output_obj () = output_c_object := true; custom_runtime := true
    let _use_prims s = use_prims := s
    let _use_runtime s = use_runtime := s
    let _v () = Compenv.print_version_and_library "compiler"
    let _vmthread () = Compenv.fatal vmthread_removed_message
  end

end
