///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/animation/AnimManager.h>
#include <core/gui/ApplicationManager.h>
#include <core/utilities/ProgressIndicator.h>

#include "XYZParser.h"
#include "XYZParserSettingsDialog.h"
#include "../CompressedTextParserStream.h"
#include <atomviz/atoms/AtomsObject.h>
#include <atomviz/utils/ChemicalElements.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(XYZParser, MultiFileParser)

/******************************************************************************
* Opens the settings dialog for this parser.
******************************************************************************/
bool XYZParser::showSettingsDialog(QWidget* parent)
{
	XYZParserSettingsDialog dialog(this, parent);
	if(dialog.exec() != QDialog::Accepted)
		return false;
	return true;
}

/******************************************************************************
* Checks if the given file has format that can be read by this importer.
******************************************************************************/
bool XYZParser::checkFileFormat(const QString& filepath)
{
	// Open the input file for reading.
	CompressedTextParserStream stream(filepath);

	char buffer[20];
	int count = stream.getline(buffer, sizeof(buffer)/sizeof(buffer[0]));
	if(count<1) return false;

	// Skip initial whitespace.
	char* p = buffer;
	while(isspace(*p)) {
		if(*p == '\0') return false;
		++p;
	}
	if(!isdigit(*p)) return false;
	// Skip digits.
	while(isdigit(*p)) {
		if(*p == '\0') break;
		++p;
	}
	// Check trailing whitespace.
	while(*p != '\0') {
		if(!isspace(*p)) return false;
		++p;
	}

	return true;
}

/******************************************************************************
* Parses the header of the given file and returns the number of data columns contained in the file.
******************************************************************************/
bool XYZParser::inspectFileHeader(const QString& filepath, int& numberOfColumns, QStringList& columnNames)
{
	// Open the input file for reading.
	CompressedTextParserStream stream(filepath);
	setlocale(LC_NUMERIC, "C");

	// Parse number of atoms.
	int numAtoms;
	if(sscanf(stream.readline().c_str(), "%u", &numAtoms) != 1 || numAtoms < 0 || numAtoms > 1e9)
		throw Exception(tr("XYZ file parsing error: Invalid first line in XYZ file; number of atoms expected but found: %1").arg(stream.line().c_str()));
	if(numAtoms <= 0 || numAtoms > 100000000)
		throw Exception(tr("Invalid number of atoms in line 1 of XYZ file: %1").arg(stream.line().c_str()));

	// Skip comment line.
	stream.readline();

	// Read first atoms line to determine number of columns.
	QStringList tokens = QString(stream.readline().c_str()).split(QRegExp("\\s+"), QString::SkipEmptyParts);
	numberOfColumns = tokens.size();

	return true;
}

/******************************************************************************
* Scans an atoms file for the time step frames contained therein.
******************************************************************************/
bool XYZParser::scanFileForTimeSteps(const QString& filename, bool suppressDialogs)
{
	// Determine file size.
	QFile input_file(filename);
	qint64 fileSize = input_file.size();
	if(fileSize == 0)
		throw Exception(tr("The XYZ file %1 contains no data.").arg(filename));

	// Open the input file for reading.
	CompressedTextParserStream stream(filename);

	ProgressIndicator progress(tr("Opening XYZ file '%1'").arg(filename), fileSize, suppressDialogs);

	int timeStepNumber = 0;
	int numAtoms;

	while(!stream.eof()) {
		streampos byteOffset = stream.byteOffset();

		// Parse number of atoms.
		stream.readline();
		int startLineNumber = stream.lineNumber();

		if(stream.line().empty()) break;
		if(sscanf(stream.line().c_str(), "%u", &numAtoms) != 1 || numAtoms < 0 || numAtoms > 1e9)
			throw Exception(tr("Invalid number of atoms in line %1 of XYZ file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));

		// Create a new record for the time step.
		addTimeStep(filename, byteOffset, startLineNumber);
		timeStepNumber++;
		progress.setLabelText(tr("Scanning XYZ file (Frame %1)...").arg(timeStepNumber));

		// Skip comment line.
		stream.readline();

		// Read one atom per line.
		for(int i = 0; i < numAtoms; i++) {
			stream.readline();
			if(i % 4096 == 0) {
				progress.setValue(stream.byteOffset());
				if(progress.isCanceled()) {
					if(APPLICATION_MANAGER.guiMode() && QMessageBox::question(NULL, tr("Scan operation canceled"), tr("Do you want to keep the time steps that have already been scanned so far?"),
						QMessageBox::Yes | QMessageBox::No, QMessageBox::Yes) == QMessageBox::Yes)
						return true;
					else
						return false;
				}
			}
		}
	}

	return true;
}

/******************************************************************************
* Parses the atomic data of a single time step.
******************************************************************************/
EvaluationStatus XYZParser::loadTimeStep(AtomsObject* destination, int movieFrame, const QString& filename, streampos byteOffset, int lineNumber, bool suppressDialogs)
{
	CHECK_OBJECT_POINTER(destination);

	// Show the progress indicator.
	ProgressIndicator progress(tr("Opening XYZ file '%1'").arg(filename), 0, suppressDialogs);

	// Open the input file for reading.
	CompressedTextParserStream stream(filename);
	setlocale(LC_NUMERIC, "C");

	// Seek to the byte offset where the requested movie frame is stored.
	if(byteOffset != streampos(0))
		stream.seek(byteOffset);

	// Parse number of atoms.
	int numAtoms;
	if(sscanf(stream.readline().c_str(), "%u", &numAtoms) != 1 || numAtoms < 0 || numAtoms > 1e9)
		throw Exception(tr("Invalid number of atoms in line %1 of XYZ file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));

	progress.setLabelText(tr("Loading XYZ file (%1 atoms at timestep %2)").arg(numAtoms).arg(movieFrame));
	progress.setMaximum(numAtoms);

	// Extract some useful information from the comment line.
	bool hasSimulationCell = false;
	int movieMode = -1;

	Point3 cellOrigin(ORIGIN);
	Vector3 cellVector1(NULL_VECTOR);
	Vector3 cellVector2(NULL_VECTOR);
	Vector3 cellVector3(NULL_VECTOR);
	QString remainder;
	int index;

	// This is used for string -> number conversion.
	QLocale locale = QLocale::c();

	// Try to parse the simulation cell geometry from the comment line.
	QString commentLine(stream.readline().c_str());
	if((index = commentLine.indexOf("Lxyz=")) >= 0) remainder = commentLine.mid(index + 5).trimmed();
	else if((index = commentLine.indexOf("boxsize")) >= 0) remainder = commentLine.mid(index + 7).trimmed();
	if(!remainder.isEmpty()) {
		QStringList list = remainder.split(QRegExp("\\s+"));
		if(list.size() >= 3) {
			bool ok1, ok2, ok3;
			FloatType sx = (FloatType)locale.toDouble(list[0], &ok1);
			FloatType sy = (FloatType)locale.toDouble(list[1], &ok2);
			FloatType sz = (FloatType)locale.toDouble(list[2], &ok3);
			if(ok1 && ok2 && ok3) {
				VerboseLogger() << "Found box size in comment line of XYZ line: " << sx << sy << sz << endl;
				destination->simulationCell()->setCellShape(Point3(-sx/2, -sy/2, -sz/2), Vector3(sx,0,0), Vector3(0,sy,0), Vector3(0,0,sz));
				hasSimulationCell = true;
			}
		}
	}
	if((index = commentLine.indexOf("cell_orig ")) >= 0) {
		QStringList list = commentLine.mid(index + 10).split(QRegExp("\\s+"));
		for(int k = 0; k < list.size() && k < 3; k++)
			cellOrigin[k] = (FloatType)locale.toDouble(list[k]);
	}
	if((index = commentLine.indexOf("cell_vec1 ")) >= 0) {
		QStringList list = commentLine.mid(index + 10).split(QRegExp("\\s+"));
		for(int k = 0; k < list.size() && k < 3; k++)
			cellVector1[k] = (FloatType)locale.toDouble(list[k]);
	}
	if((index = commentLine.indexOf("cell_vec2 ")) >= 0) {
		QStringList list = commentLine.mid(index + 10).split(QRegExp("\\s+"));
		for(int k = 0; k < list.size() && k < 3; k++)
			cellVector2[k] = (FloatType)locale.toDouble(list[k]);
	}
	if((index = commentLine.indexOf("cell_vec3 ")) >= 0) {
		QStringList list = commentLine.mid(index + 10).split(QRegExp("\\s+"));
		for(int k = 0; k < list.size() && k < 3; k++)
			cellVector3[k] = (FloatType)locale.toDouble(list[k]);
	}
	if(cellVector1 != NULL_VECTOR && cellVector2 != NULL_VECTOR && cellVector3 != NULL_VECTOR) {
		destination->simulationCell()->setCellShape(cellOrigin, cellVector1, cellVector2, cellVector3);
		hasSimulationCell = true;
	}
	if((index = commentLine.indexOf("pbc ")) >= 0) {
		QStringList list = commentLine.mid(index + 4).split(QRegExp("\\s+"));
		bool pbc[3];
		for(int k = 0; k < list.size() && k < 3; k++)
			pbc[k] = (bool)locale.toInt(list[k]);
		destination->simulationCell()->setPeriodicity(pbc[0], pbc[1], pbc[2]);
	}

	// Resize atom array.
	destination->setAtomsCount(numAtoms);

	// Prepare the mapping between input file columns and data channels.
	DataRecordParserHelper recordParser(&columnMapping(), destination);

	// Parse atoms.
	for(int i = 0; i < numAtoms; i++) {

		// Update progress indicator.
		if((i % 4096) == 0) {
			progress.setValue(i);
			if(progress.isCanceled()) return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR);
		}

		stream.readline();
		try {
			recordParser.storeAtom(i, (char*)stream.line().c_str());
		}
		catch(Exception& ex) {
			throw ex.prependGeneralMessage(tr("Parsing error in line %1 of XYZ file.").arg(stream.lineNumber()));
		}
	}

	if(recordParser.coordinatesOutOfRange()) {
		MsgLogger() << "WARNING: At least some of the atomic coordinates are out of the valid range." << endl;
		if(APPLICATION_MANAGER.guiMode() && QMessageBox::warning(NULL, tr("Warning"), tr("At least some of the atomic coordinates are out of the valid range. Do you want to ignore this and still load the dataset?"),
				QMessageBox::Ignore | QMessageBox::Cancel, QMessageBox::Cancel) == QMessageBox::Cancel)
			return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR);
	}

	DataChannel* posChannel = destination->getStandardDataChannel(DataChannel::PositionChannel);
	if(posChannel != NULL && posChannel->size() > 1) {
		if(!hasSimulationCell) {
			// If the input file does not contain any information about the simulation cell size then compute simulation cell.
			// from the atom positions.
			Box3 boundingBox;
			boundingBox.addPoints(posChannel->constDataPoint3(), posChannel->size());
			VerboseLogger() << "XYZ file did not contain simulation cell geometry. Using bounding box of atoms as simulation cell." << endl;
			destination->simulationCell()->setBoxShape(boundingBox);
		}
		else {
			// Rescale atoms if they are given in reduced coordinates.
			const Box3& boundingBox = recordParser.boundingBox();
			if(Box3(Point3(-0.5), Point3(0.5)).containsBox(boundingBox)) {
				VerboseLogger() << "Rescaling reduced coordinates in interval [-0.5,+0.5] to absolute box size." << endl;
				AffineTransformation simCell = destination->simulationCell()->cellMatrix();
				Point3* p = posChannel->dataPoint3();
				for(size_t i = posChannel->size(); i != 0; --i, ++p)
					*p = simCell * (*p + Vector3(0.5,0.5,0.5));
			}
			else if(Box3(Point3(0.0), Point3(1.0)).containsBox(boundingBox)) {
				VerboseLogger() << "Rescaling reduced coordinates in interval [0,1] to absolute box size." << endl;
				AffineTransformation simCell = destination->simulationCell()->cellMatrix();
				Point3* p = posChannel->dataPoint3();
				for(size_t i = posChannel->size(); i != 0; --i, ++p)
					*p = simCell * (*p);
			}
		}
	}

	destination->invalidate();

	QString statusMessage = tr("Number of atoms: %1").arg(numAtoms);
	return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, statusMessage);
}

};
