/*
 * Copyright © 2012 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/** @file uniformblockbinding.c
 *
 * From the GL_ARB_uniform_buffer_object spec:
 *
 *     "Each of a program's active uniform blocks has a corresponding
 *      uniform buffer object binding point. This binding point can be
 *      assigned by calling:
 *
 *          void UniformBlockBinding(uint program,
 *                                   uint uniformBlockIndex,
 *                                   uint uniformBlockBinding);
 *
 *      <program> is a name of a program object for which the command
 *      LinkProgram has been issued in the past.
 *
 *      <uniformBlockIndex> must be an active uniform block index of
 *      the program <program>. Otherwise, INVALID_VALUE is generated.
 *
 *      <uniformBlockBinding> must be less than MAX_UNIFORM_BUFFER_BINDINGS.
 *      Otherwise, INVALID_VALUE is generated.
 *
 *      ...
 *
 *      When a program object is linked or re-linked, the uniform
 *      buffer object binding point assigned to each of its active
 *      uniform blocks is reset to zero.

 *      ...
 *
 *      The error INVALID_VALUE is generated by GetActiveUniformsiv,
 *      GetActiveUniformName, GetActiveUniformBlockiv,
 *      GetActiveUniformBlockName, and UniformBlockBinding if
 *      <program> is not a value generated by GL."
 */

#include "piglit-util-gl.h"

PIGLIT_GL_TEST_CONFIG_BEGIN

	config.supports_gl_compat_version = 10;

	config.window_width = 10;
	config.window_height = 10;
	config.window_visual = PIGLIT_GL_VISUAL_RGBA | PIGLIT_GL_VISUAL_DOUBLE;
	config.khr_no_error_support = PIGLIT_NO_ERRORS;

PIGLIT_GL_TEST_CONFIG_END

void
piglit_init(int argc, char **argv)
{
	int i;
	GLuint prog;
	const char *source =
		"#extension GL_ARB_uniform_buffer_object : enable\n"
		"uniform a { float u1; };\n"
		"uniform b { float u2; };\n"
		"void main() {\n"
		"	gl_FragColor = vec4(u1 + u2);\n"
		"}\n";
	int blocks;
	int binding, max_bindings;
	bool pass = true;

	piglit_require_extension("GL_ARB_uniform_buffer_object");

	prog = piglit_build_simple_program(NULL, source);

	glGetProgramiv(prog, GL_ACTIVE_UNIFORM_BLOCKS, &blocks);
	assert(blocks == 2);

	for (i = 0; i < blocks; i++) {
		glGetActiveUniformBlockiv(prog, i, GL_UNIFORM_BLOCK_BINDING,
					  &binding);

		if (binding != 0) {
			fprintf(stderr,
				"Linked program should have binding[%d] = %d, "
				"saw %d\n",
				i, 0, binding);
			pass = false;
		}
	}

	for (i = 0; i < blocks; i++) {
		glUniformBlockBinding(prog, i, blocks - i);
	}

	for (i = 0; i < blocks; i++) {
		glGetActiveUniformBlockiv(prog, i, GL_UNIFORM_BLOCK_BINDING,
					  &binding);

		if (binding != blocks - i) {
			fprintf(stderr,
				"Updated binding[%d] to %d, but got %d\n",
				i, blocks - i, binding);
			pass = false;
		}
	}

	glLinkProgram(prog);

	for (i = 0; i < blocks; i++) {
		glGetActiveUniformBlockiv(prog, i, GL_UNIFORM_BLOCK_BINDING,
					  &binding);

		if (binding != 0) {
			fprintf(stderr,
				"Relinked program should have binding[%d] = %d, "
				"saw %d\n",
				i, 0, binding);
			pass = false;
		}
	}

	if (!piglit_khr_no_error) {
		glUniformBlockBinding(prog, blocks, 0);
		pass = piglit_check_gl_error(GL_INVALID_VALUE) && pass;

		glGetIntegerv(GL_MAX_UNIFORM_BUFFER_BINDINGS, &max_bindings);
		glUniformBlockBinding(prog, 0, max_bindings);
		pass = piglit_check_gl_error(GL_INVALID_VALUE) && pass;

		glUniformBlockBinding(0xd0d0, 0, 0);
		pass = piglit_check_gl_error(GL_INVALID_VALUE) && pass;
	}

	glDeleteProgram(prog);

	piglit_report_result(pass ? PIGLIT_PASS : PIGLIT_FAIL);
}

enum piglit_result piglit_display(void)
{
	/* UNREACHED */
	return PIGLIT_FAIL;
}

