/***********************************************************************************
* Smooth Tasks
* Copyright (C) 2009 Marcin Baszczewski <marcin.baszczewski@gmail.com>
*                    Mathias panzenböck <grosser.meister.morti@gmx.net>
* Copyright (C) 2009-2012 Toni Dietze <smooth-tasks@derflupp.e4ward.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

// Smooth Tasks
#include "SmoothTasks/Light.h"
#include "SmoothTasks/Applet.h"
#include "SmoothTasks/TaskItem.h"
#include "SmoothTasks/TaskIcon.h"

// Qt
#include <QPainter>
#include <QPropertyAnimation>
#include <QSequentialAnimationGroup>
#include <QStyleOptionGraphicsItem>

// Plasma
#include <Plasma/Applet>
#include <Plasma/PaintUtils>

// KDE
#include <KIcon>
#include <KIconEffect>
#include <KIconLoader>

// Other
#include <cmath>

namespace SmoothTasks {

Light::Light(TaskItem *item)
	: QObject(item)
	, m_item(item)
	, m_size(1.0)
	, m_animation(0)
	, m_currentAnimation(NoAnimation)
{}

void Light::paint(QPainter *p, const QRectF& boundingRect, const QPointF& mousePos, bool mouseIn, const bool isRotated) {
	if (!mouseIn && m_currentAnimation == NoAnimation) {
		return;
	}

	qreal  cx = 0.5;
	qreal  cy = 0.5;
	QColor lightColor(m_item->icon()->highlightColor());

	if (mouseIn) {
		if (isRotated) {
			cx = 1 - (mousePos.y() - boundingRect.left()) / boundingRect.width();
			cy =     (mousePos.x() - boundingRect.top() ) / boundingRect.height();
		}
		else {
			cx = (mousePos.x() - boundingRect.left()) / boundingRect.width();
			cy = (mousePos.y() - boundingRect.top() ) / boundingRect.height();
		}
	}
	
	const LightAnchorConfig& anchorConfig = m_item->applet()->lightAnchorConfig();
	if (anchorConfig.isAnchoredHorizontally())
		cx = 0.01 * (qreal) anchorConfig.anchorHorizontal();
	if (anchorConfig.isAnchoredVertically())
		cy = 0.01 * (qreal) (100 - anchorConfig.anchorVertical());

	QRadialGradient gradient(cx, cy, m_size);
	gradient.setCoordinateMode(QRadialGradient::ObjectBoundingMode);
	
	lightColor.setAlpha(200);
	gradient.setColorAt(0.0, lightColor);
	
	lightColor.setAlpha(60);
	gradient.setColorAt(0.6, lightColor);
	
	lightColor.setAlpha(0);
	gradient.setColorAt(1.0, lightColor);
	
	p->fillRect(boundingRect, gradient);
}

void Light::startAnimation(AnimationType animation, int duration, int loopCout) {
	delete m_animation;

	QPropertyAnimation* start = new QPropertyAnimation(this, "size", this);
	start->setEasingCurve(QEasingCurve::InOutSine);
	start->setDuration(0.5 * duration);
	start->setStartValue(0.00001);  // QRadialGradient does not like a radius of 0
	start->setEndValue(1.0);

	QPropertyAnimation* down = new QPropertyAnimation(this, "size", this);
	down->setEasingCurve(QEasingCurve::InOutSine);
	down->setDuration(0.5 * duration);
	down->setStartValue(1.0);
	down->setEndValue(0.5);

	QPropertyAnimation* up = new QPropertyAnimation(this, "size", this);
	up->setEasingCurve(QEasingCurve::InOutSine);
	up->setDuration(0.5 * duration);
	up->setStartValue(0.5);
	up->setEndValue(1.0);

	QSequentialAnimationGroup* loop = new QSequentialAnimationGroup(this);
	loop->addAnimation(down);
	loop->addAnimation(up);
	loop->setLoopCount(loopCout);

	QSequentialAnimationGroup* all = new QSequentialAnimationGroup(this);
	all->addAnimation(start);
	all->addAnimation(loop);

	m_animation = all;
	m_animation->start(QAbstractAnimation::DeleteWhenStopped);
	m_currentAnimation = animation;
}

void Light::stopAnimation() {
	delete m_animation;
	m_animation = 0;
	m_currentAnimation = NoAnimation;
	setSize(1.0);
}

void Light::setSize(qreal size) {
	m_size = size;
	emit update();
}

} // namespace SmoothTasks
#include "Light.moc"
