% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/polyCub.iso.R
\name{polyCub.iso}
\alias{polyCub.iso}
\alias{.polyCub.iso}
\title{Cubature of Isotropic Functions over Polygonal Domains}
\usage{
polyCub.iso(polyregion, f, intrfr, ..., center, control = list(),
  check.intrfr = FALSE, plot = FALSE)

.polyCub.iso(polys, intrfr, ..., center, control = list(),
  .witherror = FALSE)
}
\arguments{
\item{polyregion}{a polygonal domain.
The following classes are supported:
\code{"\link[spatstat.geom]{owin}"} from package \pkg{spatstat.geom},
\code{"gpc.poly"} from \pkg{gpclib},
\code{"\linkS4class{SpatialPolygons}"}, \code{"\linkS4class{Polygons}"},
and \code{"\linkS4class{Polygon}"} from package \pkg{sp}, as well as
\code{"\link[sf:st_polygon]{(MULTI)POLYGON}"} from package \pkg{sf}.
(For these classes, \pkg{polyCub} knows how to get an \code{\link{xylist}}.)}

\item{f}{a two-dimensional real-valued function.
As its first argument it must take a coordinate matrix, i.e., a
numeric matrix with two columns, and it must return a numeric vector of
length the number of coordinates.}

\item{intrfr}{a \code{function(R, ...)}, which implements the (analytical)
antiderivative of \eqn{r f_r(r)} from 0 to \code{R}. The first argument
must be vectorized but not necessarily named \code{R}.\cr
If \code{intrfr} is missing, it will be approximated numerically via
\code{\link{integrate}(function(r, ...)
r * f(cbind(x0 + r, y0), ...), 0, R, ..., control=control)},
where \code{c(x0, y0)} is the \code{center} of isotropy.
Note that \code{f} will \emph{not} be checked for isotropy.}

\item{...}{further arguments for \code{f} or \code{intrfr}.}

\item{center}{numeric vector of length 2, the center of isotropy.}

\item{control}{list of arguments passed to \code{\link{integrate}}, the
quadrature rule used for the line integral along the polygon boundary.}

\item{check.intrfr}{logical (or numeric vector) indicating if
(for which \code{r}'s) the supplied \code{intrfr} function should be
checked against a numeric approximation. This check requires \code{f}
to be specified. If \code{TRUE}, the set of test
\code{r}'s defaults to a \code{\link{seq}} of length 20 from 1 to
the maximum absolute x or y coordinate of any edge of the \code{polyregion}.}

\item{plot}{logical indicating if an image of the function should be plotted
together with the polygonal domain, i.e.,
\code{\link{plotpolyf}(polyregion, f, \dots)}.}

\item{polys}{something like \code{owin$bdry}, but see \code{\link{xylist}}.}

\item{.witherror}{logical indicating if an upper bound for the absolute
integration error should be attached as an attribute to the result?}
}
\value{
The approximate integral of the isotropic function
\code{f} over \code{polyregion}.\cr
If the \code{intrfr} function is provided (which is assumed to be exact), an
upper bound for the absolute integration error is appended to the result as
attribute \code{"abs.error"}. It equals the sum of the absolute errors
reported by all \code{\link{integrate}} calls
(there is one for each edge of \code{polyregion}).
}
\description{
\code{polyCub.iso} numerically integrates a radially symmetric function
\eqn{f(x,y) = f_r(||(x,y)-\boldsymbol{\mu}||)}{f(x,y) = f_r(||(x,y)-\mu||)},
with \eqn{\mu} being the center of isotropy, over a polygonal domain.
It internally approximates a line integral along the polygon boundary using
\code{\link{integrate}}. The integrand requires the antiderivative of
\eqn{r f_r(r)}), which should be supplied as argument \code{intrfr}
(\code{f} itself is only required if \code{check.intrfr=TRUE}).
The two-dimensional integration problem thereby reduces to an efficient
adaptive quadrature in one dimension.
If \code{intrfr} is not available analytically, \code{polyCub.iso} can use a
numerical approximation (meaning \code{integrate} within \code{integrate}),
but the general-purpose cubature method \code{\link{polyCub.SV}} might be
more efficient in this case.
See Meyer and Held (2014, Supplement B, Section 2.4) for mathematical
details.

\code{.polyCub.iso} is a \dQuote{bare-bone} version of \code{polyCub.iso}.
}
\examples{
## we use the example polygon and f (exponential decay) from
example(plotpolyf)

## numerical approximation of 'intrfr' (not recommended)
(intISOnum <- polyCub.iso(letterR, f, center = fcenter))

## analytical 'intrfr'
## intrfr(R) = int_0^R r*f(r) dr, for f(r) = dexp(r), gives
intrfr <- function (R, rate = 1) pgamma(R, 2, rate) / rate
(intISOana <- polyCub.iso(letterR, f, intrfr = intrfr, center = fcenter,
                          check.intrfr = TRUE))
## f is only used to check 'intrfr' against a numerical approximation

stopifnot(all.equal(intISOana, intISOnum, check.attributes = FALSE))


### polygon area: f(r) = 1, f(x,y) = 1, center does not really matter

## intrfr(R) = int_0^R r*f(r) dr = int_0^R r dr = R^2/2
intrfr.const <- function (R) R^2/2
(area.ISO <- polyCub.iso(letterR, intrfr = intrfr.const, center = c(0,0)))

if (require("spatstat.geom")) { # check against area.owin()
    stopifnot(all.equal(area.owin(owin(poly = letterR)),
                        area.ISO, check.attributes = FALSE))
}
}
\references{
Hedevang, E. (2013). Personal communication at the Summer School on Topics in
Space-Time Modeling and Inference (May 2013, Aalborg, Denmark).

Meyer, S. and Held, L. (2014).
Power-law models for infectious disease spread.
\emph{The Annals of Applied Statistics}, \bold{8} (3), 1612-1639.
\doi{10.1214/14-AOAS743}
}
\seealso{
\code{system.file("include", "polyCubAPI.h", package = "polyCub")}
for a full C-implementation of this cubature method (for a \emph{single}
polygon). The corresponding C-routine \code{polyCub_iso} can be used by
other \R packages, notably \CRANpkg{surveillance}, via \samp{LinkingTo: polyCub}
(in the \file{DESCRIPTION}) and \samp{#include <polyCubAPI.h>} (in suitable
\file{/src} files). Note that the \code{intrfr} function must then also be
supplied as a C-routine. An example can be found in the package tests.

Other polyCub-methods: 
\code{\link{polyCub.SV}()},
\code{\link{polyCub.exact.Gauss}()},
\code{\link{polyCub.midpoint}()},
\code{\link{polyCub}()}
}
\author{
Sebastian Meyer

The basic mathematical formulation of this efficient integration for radially
symmetric functions was ascertained with great support by
Emil Hedevang (2013), Dept. of Mathematics, Aarhus University, Denmark.
}
\concept{polyCub-methods}
\keyword{math}
\keyword{spatial}
