///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// manage graphical render options
// e.g. print it in python for the rheolef_paraview.py script
//
// author: Pierre.Saramito@imag.fr
//
// date: 25 janv 2020
//
#include "rheolef/render_option.h"
namespace rheolef {
// ----------------------------------------------------------------------------
// option for scalar visualization with paraview
// ----------------------------------------------------------------------------
render_option::render_option()
  : showlabel(), stereo(), elevation(), iso(), cut(), grid(), fill(), volume(), view_2d(),
    view_map(), high_order(), color(), gray(), black_and_white(), have_opacity_bug(),
    n_isovalue(), n_isovalue_negative(), branch_size(1),
    scale(), f_min(), f_max(), isovalue(),
    resolution(),
    xmin(), xmax(), origin(), normal(),
    format(), mark(), label(), valued(), style()
{
}
std::ostream& operator << (std::ostream& py, const render_option& popt)
{
  popt.put_paraview (py); return py;
}
void render_option::put_paraview (std::ostream& py) const
{
  using namespace std;
  if (format == "tiff") format = "tif";
  if (format == "jpeg") format = "jpg";
  if (black_and_white)  fill = false;
  string color_str = (color ? "color" : (gray ? "gray" : "black_and_white"));
  if (isovalue != numeric_limits<Float>::max()) {
    n_isovalue = 2; // draw {min, isovalue, max} when color : two bands
  }
  if (label == "") label = mark;
  if (normal[0] == numeric_limits<Float>::max())  normal = point(1,0,0);
  py << setprecision (numeric_limits<Float>::digits10)
     << "#!/usr/bin/env paraview --script=" << endl
     << "# This is a paraview script, automatically generated by rheolef."  << endl
     << endl
     << "from paraview.simple  import *" << endl
     << "from paraview_rheolef import *  # load rheolef specific functions" << endl
     << endl
     << "opt = {                 	\\" << endl
     << "  'format'  : '" << format   << "',		\\" << endl
     << "  'resolution' : [" << resolution[0] << "," << resolution[1] << "], \\" << endl
     << "  'branch_size' : " << branch_size << ",   \\" << endl
     << "  'mark'    : '" << mark  << "',   \\" << endl
     << "  'label'   : '" << label  << "',   \\" << endl
     << "  'showlabel': " << showlabel << ",   \\" << endl
     << "  'axis'    : 1,			\\" << endl
     << "  'color'   : '" << color_str << "', \\" << endl
     << "  'have_opacity_bug'   : " << have_opacity_bug << ", \\" << endl
     << "  'stereo'  : " << stereo << ",		\\" << endl
     << "  'bbox'    : [[" << xmin[0] << "," << xmin[1] << "," << xmin[2] << "],[" 
                           << xmax[0] << "," << xmax[1] << "," << xmax[2] << "]],  \\" << endl
     << "  'scale'   : " << scale << ",		\\" << endl
     << "  'elevation'  : " << elevation   << ",	\\" << endl
     << "  'range'      : [" <<  f_min << ", " << f_max << "],  \\" << endl
     << "  'iso'        : " << iso   << ",		\\" << endl
     << "  'isovalue'   : " << isovalue   << ",		\\" << endl
     << "  'n_isovalue' : " << n_isovalue   << ",	\\" << endl
     << "  'n_isovalue_negative' : " << n_isovalue_negative   << ",		\\" << endl
     << "  'cut'        : " << cut   << ",		\\" << endl
     << "  'origin'     : " << python(origin) << ", 	\\" << endl
     << "  'normal'     : " << python(normal) << ",  	\\" << endl
     << "  'grid'    : " << grid   << ",		\\" << endl
     << "  'fill'    : " << fill   << ",		\\" << endl
     << "  'volume'  : " << volume << ",		\\" << endl
     << "  'view_1d' : 0,				\\" << endl
     << "  'view_2d' : " << view_2d << ",		\\" << endl
     << "  'view_map' : " << view_map << ",	\\" << endl
     << "  'high_order' : "<< high_order << ",   	\\" << endl
     << "  'valued'  : '"<< valued << "',   	\\" << endl
     << "  'style'   : '"<< style << "'   	\\" << endl
     << "}" << endl
     << endl
     ;
}
std::string
render_option::python (const point& x, size_t d)
{
    std::ostringstream os;
    os << "(" << x[0];
    for (size_t i = 1; i < d; i++)
      os << ", " << x[i];
    os << ")" << std::flush;
    std::string buffer = os.str();
    return buffer;
}
} // namespace rheolef
