# SPDX-License-Identifier: GPL-3.0-or-later
# (c) Emmanuel Beffara, 2002--2006
# vim: noet:ts=4
"""
This module contains the code for
the class Environment, which contains all information about a given
building process.
"""

import os.path

from rubber.util import _
import logging

msg = logging.getLogger(__name__)
import rubber.converters
from rubber.convert import Converter


class Environment:
    """
    This class contains all state information related to the building process
    for a whole document, the dependency graph and conversion rules.
    """

    def __init__(self):
        """
        Initialize the environment. The optional argument is the path to the
        reference directory for compilation, by default it is the current
        working directory.
        """
        self.path = [os.path.curdir]
        self.conv_prefs = {}
        self.converter = Converter(self)
        self.converter.read_ini(os.path.join(rubber.__path__[0], 'rules.ini'))

        self.doc_requires_shell_ = False
        self.synctex = False
        self.main = None
        self.final = None
        self.graphics_suffixes = []

    def find_file(self, name, suffix=None):
        """
        Look for a source file with the given name, and return either the
        complete path to the actual file or None if the file is not found.
        The optional argument is a suffix that may be added to the name.
        """
        for path in self.path:
            test = os.path.join(path, name)
            if suffix and os.path.exists(test + suffix) and os.path.isfile(test + suffix):
                return test + suffix
            elif os.path.exists(test) and os.path.isfile(test):
                return test
        return None

    def conv_set(self, file, vars):
        """
        Define preferences for the generation of a given file. The argument
        'file' is the name of the target and the argument 'vars' is a
        dictionary that contains imposed values for some variables.
        """
        self.conv_prefs[file] = vars

    def convert(self, target, prefixes=[""], suffixes=[""], check=None, context=None):
        """
        Use conversion rules to make a given target file, and return:
        * the file path of an existing file matching the 'target' pattern
        * the final node of a new dependency tree if a conversion is necessary
        * None if the file does not exist and cannot be built.
        The optional arguments 'prefixes' and 'suffixes'
        are lists of strings that can be added at the beginning and the end of
        the name when searching for the file. Prefixes are tried in order, and
        for each prefix, suffixes are tried in order; the first file from this
        order that exists or can be made is kept. The optional arguments
        'check' and 'context' have the same meaning as in
        'Converter.best_rule'.
        """
        # Try all suffixes and prefixes until something is found.

        last = None
        for t in [p + target + s for s in suffixes for p in prefixes]:

            # Define a check function, according to preferences.

            if t in self.conv_prefs:
                prefs = self.conv_prefs[t]

                def do_check(vars, prefs=prefs):
                    if prefs is not None:
                        for key, val in prefs.items():
                            if not (key in vars and vars[key] == val):
                                return 0
                    return 1
            else:
                prefs = None
                do_check = check

            # Find the best applicable rule.

            ans = self.converter.best_rule(t, check=do_check, context=context)
            if ans is not None:
                # TODO: I think the suppressed-in-the-following-line
                # pylint error is a false positive. Check if this is
                # correct.
                # pylint: disable=unsubscriptable-object
                if last is None or ans["cost"] < last["cost"]:
                    last = ans

            # Check if the target exists.

            if prefs is None and os.path.exists(t):
                if last is not None and last["cost"] <= 0:
                    break
                msg.debug(_("`%s' is `%s', no rule applied") % (target, t))
                return t

        if last is None:
            return None
        msg.debug(_("`%s' is `%s', made from `%s' by rule `%s'") %
                  (target, last["target"], last["source"], last["name"]))
        return self.converter.apply(last)

    def may_produce(self, name):
        """
        Return true if the given filename may be that of a file generated by
        any of the converters.
        """
        return self.converter.may_produce(name)
