// tests for function bigdecimal::parsing::parse_from_f64

use paste::paste;

use stdlib::f64;

macro_rules! impl_test {
    ($input:literal == $expected:literal) => {
        paste! { impl_test!( [< "_" $input >] : $input == $expected); }
    };
    ($name:ident : bits:$input:literal => $expected:literal) => {
        impl_test!($name : f64::from_bits($input) => $expected);
    };
    ($name:ident : $input:literal == $expected:literal) => {
        impl_test!($name : ($input as f64) => $expected);
    };
    ($name:ident : $input:expr => $expected:literal) => {
        paste! {
            #[test]
            fn [< case $name >]() {
                let n = $input;
                let d = parse_from_f64(n);
                assert_eq!(d, $expected.parse().unwrap());
            }

            #[test]
            fn [< case_neg $name >]() {
                let n = f64::from_bits($input.to_bits() | (1<<63));
                let d = parse_from_f64(n);
                assert_eq!(d, concat!("-", $expected).parse().unwrap());
            }
        }
    };
}

impl_test!(_0 : 0.0 == "0");
impl_test!(_1 : 1.0 == "1");
impl_test!(_2 : 2.0 == "2");
impl_test!(_3 : 3.0 == "3");
impl_test!(_5en1 : 0.5 == "0.5");
impl_test!(_25en2 : 0.25 == "0.25");
impl_test!(_1en1 : 0.1 == "0.1000000000000000055511151231257827021181583404541015625");
impl_test!(_1over3 : 0.333333333333333333333333333333 == "0.333333333333333314829616256247390992939472198486328125");
impl_test!(_pi : 3.141592653589793 == "3.141592653589793115997963468544185161590576171875");
impl_test!(_near_3 : 3.0000000000000004 == "3.000000000000000444089209850062616169452667236328125");
impl_test!(_8eneg306 : 8.544283616667655e-306 == "8.5442836166676545758745469881475846986178991076220674838778719735182619591847930738097459423424470941335996703553180065389909675214026779902482660710563190540056652827644969523715287333767167538014707594736533997824798692690142890189753467148541192574394234161821394612038920127719106177776787375705338074667624093006332620080979623387970617655687653904110103913103933178304212511707769987213793880764157458662751217010283883439888757033430556011326632895537144105152597427684695380215955244686097497705226475608085097617996058799189036784865947060736971859470127760066696392182317083388979882704968230500619384728741377732016919538675848783600526390429792978252568964346334556191024880163233082812954995600973750951114861484914086986464099027216434478759765625e-306");
impl_test!(_8e306 : 3e300 == "3000000000000000157514280765613260746113405743324477464747562346535407373966724587359114125241343592131113331498651634530827569706081291726934376554360120948545161602779727411213490701384364270178106859704912399835243357116902922640223958228340427483737776366460170528514347008416589160596378201620480");

impl_test!(_50 : 50. == "50");
impl_test!(_nanbits : bits:0b_0_11111111111_1000000000000000000000000000000000000000000000000001 => "269653970229347426076201969312749943170150807578117307259170330445749843759196293443300553362892619730839480672521111823337121537071529813188030913831084401350087805833926634314566788423582671529934053315387252306324360914392174188827078768228648633522131134987762597502339006422840407304422939101316534763520");
impl_test!(_3105036184601418e246 : bits:0b_0_11100000000_0000000000000000000000000000000000000000000000000000 => "3105036184601417870297958976925005110513772034233393222278104076052101905372753772661756817657292955900975461394262146412343160088229628782888574550082362278408909952041699811100530571263196889650525998387432937501785693707632115712");

impl_test!(_2_81341650018752E_308 : 2.81341650018752E-308 == "2.8134165001875198278759275525943498067505063001967969175506480744152639496835355462897889950138699429916690515722729976876607247658891051736045520063301219592298855232146428654590713004216312194773871772185068366206180596731958890086634117134422695105490626598276746331472433159429067991016548063113298957324839879447939977012897422163463450947345510093578791948321798481101505330952230105511530048812659083481787407026258844307461890753626327683153826358878159001221539330872743255707112001100520519610144879206546597846231715071742093092641158571855689231930930474890818690333095288369471228217443460522531282790309374378111440076317827545086535792316428407651758951233693496387904508572484340169054222573303301594335791590596740352481219815672375261783599853515625E-308");


#[test]
fn case_f64_min() {
    let n = f64::MIN;
    let d = parse_from_f64(n);
    assert_eq!(d, "-179769313486231570814527423731704356798070567525844996598917476803157260780028538760589558632766878171540458953514382464234321326889464182768467546703537516986049910576551282076245490090389328944075868508455133942304583236903222948165808559332123348274797826204144723168738177180919299881250404026184124858368".parse().unwrap());
}

#[test]
fn case_f64_max() {
    let n = f64::MAX;
    let d = parse_from_f64(n);
    assert_eq!(d, "179769313486231570814527423731704356798070567525844996598917476803157260780028538760589558632766878171540458953514382464234321326889464182768467546703537516986049910576551282076245490090389328944075868508455133942304583236903222948165808559332123348274797826204144723168738177180919299881250404026184124858368".parse().unwrap());
}

#[test]
fn case_f64_epsilon() {
    let n = f64::EPSILON;
    let d = parse_from_f64(n);
    assert_eq!(d, "2.220446049250313080847263336181640625e-16".parse().unwrap());
}

#[test]
fn case_f64_pi() {
    let n = f64::consts::PI;
    let d = parse_from_f64(n);
    assert_eq!(d, "3.141592653589793115997963468544185161590576171875".parse().unwrap());
}

#[test]
fn case_nan() {
    let n = f64::from_bits(0b0_11111111111_1000000000000000000000000000000000000000000000000000);
    assert!(n.is_nan());

    let d = parse_from_f64(n);
    assert_eq!(d, "269653970229347386159395778618353710042696546841345985910145121736599013708251444699062715983611304031680170819807090036488184653221624933739271145959211186566651840137298227914453329401869141179179624428127508653257226023513694322210869665811240855745025766026879447359920868907719574457253034494436336205824".parse().unwrap());
}

#[test]
fn case_try_from_nan() {
    let n = f64::NAN;
    let d = try_parse_from_f64(n);
    assert!(d.is_err());
}

#[test]
fn case_try_from_infinity() {
    let n = f64::INFINITY;
    let d = try_parse_from_f64(n);
    assert!(d.is_err());
}
