#[cfg(feature = "wasm-bindings")]
use crate::client::BoxedProfile;
use crate::{Product, Profile};
use serde_json::{json, Value};
use std::collections::HashMap;
#[cfg(feature = "wasm-bindings")]
use wasm_bindgen::prelude::wasm_bindgen;

// TODO: Fix movement?

mod products {
    use crate::{Mode, Product};
    use std::borrow::Cow;

    pub const ICE: Product = Product {
        id: Cow::Borrowed("nationalExpress"),
        mode: Mode::Train,
        bitmasks: Cow::Borrowed(&[8192]),
        name: Cow::Borrowed("Hochgeschwindigkeitszug"),
        short: Cow::Borrowed("ICE"),
    };
    pub const IC: Product = Product {
        id: Cow::Borrowed("national"),
        mode: Mode::Train,
        bitmasks: Cow::Borrowed(&[4096]),
        name: Cow::Borrowed("InterCity & EuroCity"),
        short: Cow::Borrowed("IC/EC"),
    };
    pub const IR: Product = Product {
        id: Cow::Borrowed("interregional"),
        mode: Mode::Train,
        bitmasks: Cow::Borrowed(&[2048]),
        name: Cow::Borrowed("InterRegio"),
        short: Cow::Borrowed("IR"),
    };
    pub const RB: Product = Product {
        id: Cow::Borrowed("regional"),
        mode: Mode::Train,
        bitmasks: Cow::Borrowed(&[1024]),
        name: Cow::Borrowed("Regionalzug"),
        short: Cow::Borrowed("RB"),
    };
    pub const S_BAHN: Product = Product {
        id: Cow::Borrowed("suburban"),
        mode: Mode::Train,
        bitmasks: Cow::Borrowed(&[512]),
        name: Cow::Borrowed("S-Bahn"),
        short: Cow::Borrowed("S-Bahn"),
    };
    pub const U: Product = Product {
        id: Cow::Borrowed("subway"),
        mode: Mode::Train,
        bitmasks: Cow::Borrowed(&[256]),
        name: Cow::Borrowed("U-Bahn"),
        short: Cow::Borrowed("U"),
    };
    pub const S: Product = Product {
        id: Cow::Borrowed("saarbahn"),
        mode: Mode::Train,
        bitmasks: Cow::Borrowed(&[128]),
        name: Cow::Borrowed("Saarbahn"),
        short: Cow::Borrowed("S"),
    };
    pub const BUS: Product = Product {
        id: Cow::Borrowed("bus"),
        mode: Mode::Bus,
        bitmasks: Cow::Borrowed(&[64]),
        name: Cow::Borrowed("Bus"),
        short: Cow::Borrowed("Bus"),
    };
    pub const SCHIFF: Product = Product {
        id: Cow::Borrowed("watercraft"),
        mode: Mode::Watercraft,
        bitmasks: Cow::Borrowed(&[32]), // TODO: Correct?
        name: Cow::Borrowed("Schiff"),
        short: Cow::Borrowed("Schiff"),
    };
    pub const AST: Product = Product {
        id: Cow::Borrowed("onCall"),
        mode: Mode::Bus, // TODO: Correct?
        bitmasks: Cow::Borrowed(&[16]),
        name: Cow::Borrowed("Anruf-Sammel-Taxi"),
        short: Cow::Borrowed("AST"),
    };
    pub const SCHULBUS: Product = Product {
        id: Cow::Borrowed("school-bus"),
        mode: Mode::Bus,
        bitmasks: Cow::Borrowed(&[8]),
        name: Cow::Borrowed("Schulbus"),
        short: Cow::Borrowed("Schulbus"),
    };

    // TODO: `1`, `2`, `4` bitmasks?
    pub const PRODUCTS: &[&Product] = &[
        &ICE, &IC, &IR, &RB, &S_BAHN, &U, &S, &BUS, &SCHIFF, &AST, &SCHULBUS,
    ];
}

#[derive(Debug)]
pub struct SaarvvProfile;

impl Profile for SaarvvProfile {
    fn url(&self) -> &'static str {
        "https://saarfahrplan.de/bin/mgate.exe"
    }
    fn language(&self) -> &'static str {
        "de"
    }
    fn timezone(&self) -> chrono_tz::Tz {
        chrono_tz::Europe::Berlin
    }
    fn checksum_salt(&self) -> Option<&'static str> {
        Some("HJtlubisvxiJxss")
    }

    fn products(&self) -> &'static [&'static Product] {
        products::PRODUCTS
    }

    fn prepare_body(&self, req_json: &mut Value) {
        req_json["client"] = json!({"type":"AND","id":"ZPS-SAAR","v":"","name":"Saarvv"});
        req_json["ver"] = json!("1.40");
        req_json["auth"] = json!({"type":"AID","aid":"51XfsVqgbdA6oXzHrx75jhlocRg6Xe"});
    }

    fn prepare_headers(&self, headers: &mut HashMap<&str, &str>) {
        headers.insert("User-Agent", "my-awesome-e5f276d8fe6cprogram");
    }

    fn price_currency(&self) -> &'static str {
        "EUR"
    }
}

#[cfg(feature = "wasm-bindings")]
#[wasm_bindgen]
impl SaarvvProfile {
    #[wasm_bindgen(constructor)]
    pub fn wasm_new() -> BoxedProfile {
        Self.into()
    }
}

#[cfg(test)]
mod test {
    use std::error::Error;

    use crate::profile::test::{check_journey, check_search};

    use super::*;

    #[tokio::test]
    async fn test_search() -> Result<(), Box<dyn Error>> {
        check_search(SaarvvProfile {}, "Rhe", "Rhens").await
    }

    #[tokio::test]
    async fn test_path_available() -> Result<(), Box<dyn Error>> {
        check_journey(SaarvvProfile {}, "15541", "10609").await
    }
}
